///////////////////////////////////////////////////////
// xnortext.c                                        //
// -- Routines to handle moving XNORed text.         //
///////////////////////////////////////////////////////
#include <windows.h>
#include "xnortext.h"

typedef struct
    {
    HWND      hwnd;    // Window being dragged on.
    POINT     pt;      // Last mouse position.
    POINT     dpt;     // Size of text bitmap.
    RECT      rc;      // Rectangle.
    HDC       hdc;     // Display DC while dragging.
    HDC       hdcMem;  // Memory DC for blits.
    HBITMAP   hbmpSav; // Original bitmap in memory DC.
    int       wROP2Sav;// Restore this ROP after drag.
    HBRUSH    hbrsSav; // Ditto for this brush.
    char      sz[1];
    } DRI; // DRag Info.

void _FreeDri(HLOCAL hdri);
void _DrawDri(DRI *pdri);

HLOCAL HdriBeginDrag(HWND hwnd, POINT *ppt, RECT *prc,
  char *psz)
///////////////////////////////////////////////////////
// -- Begin a drag operation.                        //
// -- Allocate, initialize, and return a drag info   //
//    struct.                                        //
///////////////////////////////////////////////////////
    {
    HLOCAL  hdri;   // Return value.
    DRI     *pdri;  // Pointer to above.
    int     cbText; // Length of text including null.
    HBITMAP hbmp;   // Monochrome text bitmap.
    RECT    rc;     // Text rectangle.
    BOOL    fOk = FALSE;

    // Allocate space for struct and string.
    cbText = NULL == psz  || 0 == *psz ?
      0 : lstrlen(psz) + 1;
    if (NULL == (hdri = LocalAlloc(LMEM_MOVEABLE,
      sizeof *pdri + cbText)))
        return NULL;

    pdri = LocalLock(hdri); // Initialize drag info.
    pdri->hwnd = hwnd;
    pdri->pt = *ppt;
    pdri->rc = *prc;

    if (0 != cbText)
        {
        DWORD lwT;

        lstrcpy(pdri->sz, psz);

        // Get DC to use while dragging.
        if (NULL == (pdri->hdc = GetDC(hwnd)))
            goto LRet;

        // Memory DC to hold bitmap.
        pdri->hdcMem = CreateCompatibleDC(pdri->hdc);
        if (NULL == pdri->hdcMem)
            goto LRet;

        // Get size of bitmap and create it.
        lwT = GetDialogBaseUnits();
        rc.left = rc.top = 0;
        pdri->dpt.x = rc.right =
          (cbText - 1) * LOWORD(lwT);
        pdri->dpt.y = rc.bottom = HIWORD(lwT);
        if (NULL == (hbmp = CreateCompatibleBitmap(
          pdri->hdcMem, rc.right, rc.bottom)))
            goto LRet;

        // Keep bitmap inside memory DC and render
        // text to it.
        pdri->hbmpSav =
          SelectObject(pdri->hdcMem, hbmp);
        ExtTextOut(pdri->hdcMem, 0, 0,
          ETO_OPAQUE | ETO_CLIPPED, &rc, pdri->sz,
          cbText - 1, NULL);

        // Initialize DC for dragging.
        pdri->wROP2Sav =
          SetROP2(pdri->hdc, R2_NOTXORPEN);
        pdri->hbrsSav = SelectObject(pdri->hdc,
          GetStockObject(NULL_BRUSH));
        }

    SetCapture(hwnd); // Want all mouse events in drag.
    _DrawDri(pdri);   // Initial rectangle and text.
    fOk = TRUE;

LRet:
    LocalUnlock(hdri);
    if (fOk)
        return hdri;
    _FreeDri(hdri);
    return NULL;
    }

void DragDri(HLOCAL hdri, POINT *ppt)
///////////////////////////////////////////////////////
// -- Move the rectangle and text.                   //
///////////////////////////////////////////////////////
    {
    POINT    dpt;
    DRI      *pdri;

    if (NULL == hdri)
        return;

    // Check that mouse is captured and has moved since
    // last message.
    pdri = LocalLock(hdri);
    dpt.x = ppt->x - pdri->pt.x;
    dpt.y = ppt->y - pdri->pt.y;
    if (GetCapture() == pdri->hwnd &&
      (0 != dpt.x || 0 != dpt.y))
        {
        _DrawDri(pdri);  // Erase old stuff.
        pdri->pt = *ppt; // Update position.
        pdri->rc.left += dpt.x;
        pdri->rc.right += dpt.x;
        pdri->rc.top += dpt.y;
        pdri->rc.bottom += dpt.y;
        _DrawDri(pdri);  // Draw new stuff.
        }
    LocalUnlock(hdri);
    }

void EndDragDri(HLOCAL hdri, POINT *ppt)
///////////////////////////////////////////////////////
// -- Finish a drag operation.                       //
// -- Remove the rectangle and text from the screen  //
//    and clean up and deallocated the drag info.    //
///////////////////////////////////////////////////////
    {
    DRI *pdri;

    if (NULL == hdri)
        return;

    pdri = LocalLock(hdri);
    if (GetCapture() == pdri->hwnd)
        {
        DragDri(hdri, ppt); // In case mouse has moved.
        _DrawDri(pdri);     // Erase last rect & text.
        ReleaseCapture();
        }
    LocalUnlock(hdri);
    _FreeDri(hdri); // All done, don't need it anymore.
    }

void _FreeDri(HLOCAL hdri)
///////////////////////////////////////////////////////
// -- Helper (nonexported) routine cleans up a drag  //
//    info then deallocates it.                      //
///////////////////////////////////////////////////////
    {
    DRI *pdri;

    if (NULL == hdri)
        return;

    pdri = LocalLock(hdri);
    if (NULL != pdri->hdc)
        { // Free GDI stuff in/and the screen DC.
        if (NULL != pdri->hbrsSav)
            SelectObject(pdri->hdc, pdri->hbrsSav);
        SetROP2(pdri->hdc, pdri->wROP2Sav);
        ReleaseDC(pdri->hwnd, pdri->hdc);
        }

    if (NULL != pdri->hdcMem)
        { // Free GDI stuff in/and the memory DC.
        if (NULL != pdri->hbmpSav)
            DeleteObject(SelectObject(pdri->hdcMem,
              pdri->hbmpSav));
        DeleteDC(pdri->hdcMem);
        }
    LocalUnlock(hdri);
    LocalFree(hdri);
    }

void _DrawDri(DRI *pdri)
///////////////////////////////////////////////////////
// -- Helper (nonexported) routine XNORs the         //
//    rectangle and text to the screen.              //
///////////////////////////////////////////////////////
    {
    if (NULL == pdri)
        return;

    // Draw the rectangle.
    Rectangle(pdri->hdc, pdri->rc.left, pdri->rc.top,
      pdri->rc.right, pdri->rc.bottom);

    // Draw the text.
    BitBlt(pdri->hdc, pdri->rc.left + 1,
      pdri->rc.top + 1, pdri->dpt.x, pdri->dpt.y,
      pdri->hdcMem, 0, 0, 0x00990066);
    }
