/*
 *  printer.c - implementation of TPrinter.
 */

#include "printer.h"
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include <commdlg.h>

char   *StringClone(const char *Input)
    {
    size_t  Size = strlen(Input)+1;
    char *Result = new char[Size];
    assert(Result != NULL);
    return (char *)memcpy(Result, Input, Size);
    }

// Constructor - someday I will make it smart enough to remember
//               current printer in a private .ini file variable.
TPrinter::TPrinter(const char *IniFilename)
    :   IniFilename_(0), Driver_(0), Port_(0), Device_(0)
    {
    if(IniFilename != NULL)
        IniFilename_    = StringClone(IniFilename);
    if(IniFilename)
        {
        
        }
    else
        {
        const int MAX_STRING = 128;
        char    Device[MAX_STRING+1];
        char    Driver[MAX_STRING+1];
        char    Port[MAX_STRING+1];

        char    Buffer[256];
        memset(Buffer, 0, sizeof(Buffer));  // be paranoid
        // get current printer from WIN.INI
        GetProfileString("windows", "device", ",,,",
                                    Buffer, sizeof(Buffer));
        // string has syntax "device,driver,port"
        strcpy(Device, "");
        strcpy(Driver, "");
        strcpy(Port, "");
        sscanf(Buffer, "%128[^,],%128[^,],%128[^,]",
                                      Device, Driver, Port);
        Device_     = StringClone(Device);
        Driver_     = StringClone(Driver);
        Port_       = StringClone(Port);
        }
    }


// copy constructor
TPrinter::TPrinter(const TPrinter& Other)
    {
    // let operator= do the work
    *this   = Other;
    }

// assignment operator
TPrinter& TPrinter::operator=(const TPrinter& Other)
    {
    if(&Other == this)  // avoid self assignment
        return *this;
    Free();
    if(Other.IniFilename_)
        IniFilename_    = StringClone(Other.IniFilename_);
    if(Other.Device_)
        Device_         = StringClone(Other.Device_);
    if(Other.Driver_)
        Driver_         = StringClone(Other.Driver_);
    if(Other.Port_)
        Port_           = StringClone(Other.Port_);
    return *this;
    }

// destructor - free up string space.
TPrinter::~TPrinter()
    {
    Free();
    }

// Free - internal function to free up strings in TPrinter object.
void    TPrinter::Free()
    {
    if(IniFilename_)
        delete[] IniFilename_;
    if(Driver_)
        delete[] Driver_;
    if(Device_)
        delete[] Device_;
    if(Port_)
        delete[] Port_;
    IniFilename_    = 0;
    Driver_         = 0;
    Device_         = 0;
    Port_           = 0;
    }

// SetupDialog - Display printer setup dialog for this printer.
int     TPrinter::SetupDialog(HWND ParentWindow)
    {
    PRINTDLG    PrintDialog;
    memset(&PrintDialog, 0, sizeof(PrintDialog));
    PrintDialog.lStructSize = sizeof(PrintDialog);
    PrintDialog.hwndOwner   = ParentWindow;
    PrintDialog.Flags       = PD_PRINTSETUP;

    HGLOBAL Names   = GlobalAlloc(GHND, sizeof(DEVNAMES)+3*128);
    assert(Names != NULL);
    char    *Block = (char *)GlobalLock(Names);
    DEVNAMES    DevNames;
    DevNames.wDriverOffset  = sizeof(DevNames);
    DevNames.wDeviceOffset  = sizeof(DevNames) + strlen(Driver_)+1;
    DevNames.wOutputOffset  = sizeof(DevNames) + strlen(Driver_)+1
                                               + strlen(Device_)+1;
    DevNames.wDefault       = 0;
    memcpy(Block, &DevNames, sizeof(DevNames));
    Block   += sizeof(DevNames);
    strcpy(Block, Driver_);
    Block   += strlen(Block)+1;
    strcpy(Block, Device_);
    Block   += strlen(Block)+1;
    strcpy(Block, Port_);
    GlobalUnlock(Names);

    BOOL    Status = PrintDlg(&PrintDialog);
    if(Status)
        {
        delete[] Driver_;
        delete[] Device_;
        delete[] Port_;
        LPDEVNAMES  Names = (LPDEVNAMES)GlobalLock(PrintDialog.hDevNames);
        Driver_     = StringClone((LPSTR)Names + Names->wDriverOffset);
        Device_     = StringClone((LPSTR)Names + Names->wDeviceOffset);
        Port_       = StringClone((LPSTR)Names + Names->wOutputOffset);
        GlobalUnlock(PrintDialog.hDevNames);
        }
    if(PrintDialog.hDevMode != NULL)
        GlobalFree(PrintDialog.hDevMode);
    if(PrintDialog.hDevNames != NULL)
        GlobalFree(PrintDialog.hDevNames);
    return FALSE;
    }

inline
size_t  MyMin(size_t a, size_t b)
    {
    return a <= b ? a : b;
    }

static
size_t  GetString(char *To, const char *From, size_t MaxSize)
    {
    if(From)
        {
        strncpy(To, From, MaxSize);
        return MyMin(MaxSize, strlen(From));
        }
    else
        {
        if(MaxSize > 0)
            *To = 0;
        return 0;
        }
    }

size_t  TPrinter::GetDriver(char *Buffer, size_t BufferSize)
    {
    return GetString(Buffer, Driver_, BufferSize);
    }

size_t  TPrinter::GetDevice(char *Buffer, size_t BufferSize)
    {
    return GetString(Buffer, Device_, BufferSize);
    }

size_t  TPrinter::GetPort(char *Buffer, size_t BufferSize)
    {
    return GetString(Buffer, Port_, BufferSize);
    }

// CreateDC - create a device context for this printer.
HDC     TPrinter::CreateDC()
    {
    assert(Driver_ != NULL);
    assert(Device_ != NULL);
    assert(Port_ != NULL);
    return ::CreateDC(Driver_, Device_, Port_, NULL);
    }
