//  uart_irq.c - determine IRQ level or IRQ conflict

#include    "uart.h"
#include    <dos.h>

void _interrupt _far common_isr(void);
static volatile unsigned char irr, isr;
static unsigned port;

void _interrupt _far common_isr(void)
    {
    outp(PIC_A0, PIC_ISR);  // prepare to read ISR of PIC
    IO_DELAY                // IBM AT Tech Ref page 9-8
    isr = inp(PIC_A0);      // read ISR
    IO_DELAY
    outp(PIC_A0, PIC_IRR);  // prepare to read IRR of PIC
    IO_DELAY
    irr = inp(PIC_A0);      // read IRR
    inp(port+IIR);          // resets thr empty
                            //               (and it arises)
    outp(PIC_A0, PIC_EOI);  // reset ISR bit
    }

unsigned char uart_irq(unsigned addr)
    {
    void (_interrupt _far *iv_save[8])(void);
    unsigned save_imr, ticks, i;
    save_imr = inp(PIC_A1); // save state of PIC
    outp(PIC_A1, 0xbc); // allow disk, keyboard & clock only
    for (i=2; i<8; ++i) // save vectors except disk, kb & clk,
        if (i != 6)     // and point them to common_isr
            {
            iv_save[i] = _dos_getvect(i+8);
            _dos_setvect(i+8, common_isr);
            }
    irr = isr = 0;        // initialize global variables ..
    port = addr;          // .. for interrupt handler routine
    outp(addr+MCR, MCR_OUT2); // permit UART interrupts
    outp(PIC_A1, 0x00);       // enable all interrupts
    ticks = *(volatile unsigned far *)TICKS + 2;
                             // wait max 1.x BIOS clock ticks
    outp(addr+IER, 0x02);    // generate a THRE interrupt
    while ((irr+isr == 0) && (ticks !=
                           *(volatile unsigned far *)TICKS))
        ;       // wait for interrupt or timeout
    outp(addr+IER, 0x00);
    for (i=2; i<8; ++i)     // restore saved intrpt vectors
        if (i != 6)
            _dos_setvect(i+8, iv_save[i]);
                            // we trashed some UART registers!
    outp(PIC_A1, save_imr); // restore imr
    return irr | isr;       // zero if no interrupt occurred
    }
