#------------------------------------------------------------------------------
# Copyright (c) 2014-2024,, Nucleic Development Team.
#
# Distributed under the terms of the Modified BSD License.
#
# The full license is in the file LICENSE, distributed with this software.
#------------------------------------------------------------------------------
from atom.api import Callable, Tuple, Dict

from .constraint_helper import ConstraintHelper


class FactoryHelper(ConstraintHelper):
    """ A constraint helper which delegates to a factory callable.

    """
    #: The callable object which will generate the list of constraints.
    #: It will be passed the constraints widget owner as the first
    #: argument, and should return the list of generated constraints.
    factory = Callable()

    #: Additional positional arguments to pass to the callable.
    args = Tuple()

    #: Additional keyword arguments to pass to the callable.
    kwargs = Dict()

    def __init__(self, factory, *args, **kwargs):
        """ Initialize a FactoryHelper.

        Parameters
        ----------
        factory : callable
            The callable object which generates the constraints.

        *args
            Additional positional arguments to pass to the factory.

        **kwargs
            Additional keyword arguments to pass to the factory.

        """
        self.factory = factory
        self.args = args
        self.kwargs = kwargs

    def constraints(self, component):
        """ Generate the constraints using the underlying factory.

        This method will automatically expand ConstraintHelper objects
        which are generated by the factory.

        """
        factory = self.factory
        if factory is None:
            return []
        cns = []  # inline `expand_constraints`
        for cn in factory(component, *self.args, **self.kwargs):
            if isinstance(cn, ConstraintHelper):
                cns.extend(cn.create_constraints(component))
            elif cn is not None:
                cns.append(cn)
        return cns
