"""
The tool to check the availability or syntax of domain, IP or URL.

::


    ██████╗ ██╗   ██╗███████╗██╗   ██╗███╗   ██╗ ██████╗███████╗██████╗ ██╗     ███████╗
    ██╔══██╗╚██╗ ██╔╝██╔════╝██║   ██║████╗  ██║██╔════╝██╔════╝██╔══██╗██║     ██╔════╝
    ██████╔╝ ╚████╔╝ █████╗  ██║   ██║██╔██╗ ██║██║     █████╗  ██████╔╝██║     █████╗
    ██╔═══╝   ╚██╔╝  ██╔══╝  ██║   ██║██║╚██╗██║██║     ██╔══╝  ██╔══██╗██║     ██╔══╝
    ██║        ██║   ██║     ╚██████╔╝██║ ╚████║╚██████╗███████╗██████╔╝███████╗███████╗
    ╚═╝        ╚═╝   ╚═╝      ╚═════╝ ╚═╝  ╚═══╝ ╚═════╝╚══════╝╚═════╝ ╚══════╝╚══════╝

Tests of our domain reputation checker.

Author:
    Nissar Chababy, @funilrys, contactTATAfunilrysTODTODcom

Special thanks:
    https://pyfunceble.github.io/special-thanks.html

Contributors:
    https://pyfunceble.github.io/contributors.html

Project link:
    https://github.com/funilrys/PyFunceble

Project documentation:
    https://docs.pyfunceble.com

Project homepage:
    https://pyfunceble.github.io/

License:
::


    Copyright 2017, 2018, 2019, 2020, 2022, 2023, 2024, 2025 Nissar Chababy

    Licensed under the Apache License, Version 2.0 (the "License");
    you may not use this file except in compliance with the License.
    You may obtain a copy of the License at

        https://www.apache.org/licenses/LICENSE-2.0

    Unless required by applicable law or agreed to in writing, software
    distributed under the License is distributed on an "AS IS" BASIS,
    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
    See the License for the specific language governing permissions and
    limitations under the License.
"""

import unittest
import unittest.mock

from PyFunceble.checker.reputation.domain import DomainReputationChecker
from PyFunceble.query.dns.query_tool import DNSQueryTool

try:
    import reputation_test_base
except ModuleNotFoundError:  # pragma: no cover
    from . import reputation_test_base


class TestDomainReputationChecker(reputation_test_base.ReputationCheckerTestBase):
    """
    Tests of the domain reputation checker.
    """

    def setUp(self) -> None:
        """
        Setups everything we need.
        """

        super().setUp()

        self.checker = DomainReputationChecker()
        self.checker.ipv4_reputation_query_tool.source_file = self.tempfile.name

        self.dns_query_tool_path = unittest.mock.patch.object(DNSQueryTool, "query")
        self.mock_query_tool = self.dns_query_tool_path.start()

        # Not needed in this scope :-)
        self.checker.do_syntax_check_first = False

    def tearDown(self) -> None:
        """
        Destroys everything previously initiated for the tests.
        """

        super().tearDown()

        self.dns_query_tool_path.stop()

        del self.dns_query_tool_path
        del self.mock_query_tool

    def test_query_status_positive(self) -> None:
        """
        Tests the method which let us query the status for the case that
        the resolved IP is known to be malicious.
        """

        self.mock_query_tool.return_value = self.fake_query_a_record()

        self.checker.subject = "example.org"
        self.checker.query_status()

        expected_status = "MALICIOUS"
        expected_source = "REPUTATION"

        actual_status = self.checker.status.status
        actual_source = self.checker.status.status_source

        self.assertEqual(expected_status, actual_status)
        self.assertEqual(expected_source, actual_source)

    def test_query_status_negative(self) -> None:
        """
        Tests the method which let us query the status for the case that
        the resolved IP is not known to be malicious.
        """

        self.mock_query_tool.return_value = self.fake_query_a_record_not_known()

        self.checker.subject = "example.org"
        self.checker.query_status()

        expected_status = "SANE"
        expected_source = "REPUTATION"

        actual_status = self.checker.status.status
        actual_source = self.checker.status.status_source

        self.assertEqual(expected_status, actual_status)
        self.assertEqual(expected_source, actual_source)


if __name__ == "__main__":
    unittest.main()
