/*$Id: Error.cpp,v 1.6 2006/08/20 16:39:04 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License Version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "Common.h"
#include "error.h"

#define MAXLINE 8192



/**
 * err_sys - Report fatal error to stderr and and terminate process. 
 * Write also error message specified by errno value.
 * @fmt: printf like format
 */
void err_sys(const char *fmt, ...)
{
	int errno_save = errno;
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	fprintf(stderr, "fatal_error: %s: %s\n", buf, strerror(errno_save));
	va_end(ap);
	exit(1);
}


/**
 * err_ret - Report error to stderr. Write also error message specified 
 * by errno value.
 * @fmt: printf like format
 */
void err_ret(const char *fmt, ...)
{
	int errno_save = errno;
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	fprintf(stderr, "error: %s: %s\n", buf, strerror(errno_save));
	va_end(ap);
}

/**
 * err_quit - The same as err_sys but doesn't print error message
 * specified by errno value. 
 */
void err_quit(const char *fmt, ...)
{
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	fprintf(stderr, "fatal error: %s\n", buf);
	va_end(ap);
	exit(1);
}

/**
 * err_msg - The same as err_ret but doesn't print error message
 * specified by errno value. 
 */
void err_msg(const char *fmt, ...)
{
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	fprintf(stderr, "error: %s\n", buf);
	va_end(ap);
}

/**
 * stderr_print - Print message to stderr.
 * @fmt: printf like format
 */
void stderr_print(const char *fmt, ...)
{
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	fprintf(stderr, "%s", buf);
	va_end(ap);
}


/*Do not use this function, use TRACE() macro instead*/
void trace_func(const char *funcname,  const char *fmt, ...)
{
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(buf, MAXLINE, fmt, ap);
	va_end(ap);
	
	fprintf(stderr, "(%s) ", funcname); 
	fprintf(stderr, "%s\n", buf);
	fflush(stderr);
}


/*Do not use this function, use TRACE() macro instead*/
void trace_func_windows(const char *fmt, ...)
{
	char buf[MAXLINE];
	va_list ap;
	va_start(ap, fmt);
	vsnprintf(buf, MAXLINE, fmt, ap);
	va_end(ap);
	
	fprintf(stderr, "(some function) "); 
	fprintf(stderr, "%s\n", buf); 
}
