/*$Id: fkHttpChannel.cpp,v 1.17 2007/07/22 21:03:11 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006,2007 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */


#include "nsCOMPtr.h"
#include "nsXPCOM.h"
#include "nsISupportsImpl.h"
#include "nsIURI.h"
#include "nsEmbedString.h"

#include "fkRule.h"
#include "fkStreamListener.h"
#include "Firekeeper.h"
#include "Error.h"

/*
 * Create new fkHttpChannel
 * @origChannel 
 *             Mozilla's Channel that will be doing almost all work here.
 * @sentence 
 *             Used to pass information about whitelisted channel that
 *             shouldn't be checked.
 * @connRules 
 *             Rules set for this channel
 */
fkHttpChannel::fkHttpChannel(nsIChannel *origChannel, fkISentence *sentence, 
			     ActiveRuleSet *connRules)
{
	TRACE("constructor %08x %08x", this, origChannel);
	nsresult rv;
	
	this->origChannel = origChannel;
	this->sentence = sentence;
	this->connRules = connRules;
	
	origRequest = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIRequest");

	origHttpChannel = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIHttpChannel");

	origRequestObserver = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIRequestObserver");

	origStreamListener = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIStreamListener");

	origCachingChannel = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsICachingChannel");

	origUploadChannel = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIUploadChannel");

	origCacheListener = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsICacheListener");

	origEncodedChannel = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIEncodedChannel");

	origHttpChannelInternal = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIHttpChannelInternal");

	origTransportEventSink = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsITransportEventSink");

	origResumableChannel = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIResumableChannel");

	origSupportsPriority = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsISupportsPriority");

	origProtocolProxyCallback = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIProtocolProxyCallback");

	origPropertyBag = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIPropertyBag");

	origPropertyBag2 = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIPropertyBag2");

	origWritablePropertyBag = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIWritablePropertyBag");

	origWritablePropertyBag2 = do_QueryInterface(origChannel, &rv); 
	checkResult(rv, "nsIWritablePropertyBag2");	

	callbacksProxy = nsnull;
	loadGroupProxy = nsnull;
	
#ifdef DEBUG
	nsCOMPtr<nsILoadGroup> lg;
	rv = origRequest->GetLoadGroup(getter_AddRefs(lg));
	FK_ASSERT(!lg);

	nsCOMPtr<nsIInterfaceRequestor> cb;	
	rv = origChannel->GetNotificationCallbacks(getter_AddRefs(cb));
	FK_ASSERT(!cb);
#endif	

}

fkHttpChannel::~fkHttpChannel()
{
	TRACE("destructor %08x", this);

#ifdef DEBUG
	FK_ASSERT(requestHashTable->Get(origHttpChannel, nsnull));
#endif	
	requestHashTable->Remove(origHttpChannel);
	TRACE("HashTable count %d", requestHashTable->Count());
	if (this->connRules)
		delete connRules;
}

NS_IMETHODIMP 
fkHttpChannel::init()
{
	return NS_OK;
}

NS_IMPL_ADDREF(fkHttpChannel)
NS_IMPL_RELEASE(fkHttpChannel)

#if 1
/*Only interfaces implemented by origLoadGroup are returned.*/
NS_IMETHODIMP
fkHttpChannel::QueryInterface(const nsIID &aIID, void **aResult)
{	
	if (aResult == NULL) {
		return NS_ERROR_NULL_POINTER;
	}
	nsISupports *foundInterface = 0;
	*aResult = NULL;

	if (aIID.Equals(NS_GET_IID(nsIRequest))){
		foundInterface = NS_STATIC_CAST(nsIRequest*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIChannel))){
		foundInterface = NS_STATIC_CAST(nsIChannel*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIRequestObserver))){
		foundInterface = NS_STATIC_CAST(nsIRequestObserver*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIStreamListener))){
		foundInterface = NS_STATIC_CAST(nsIStreamListener*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIHttpChannel))){
		foundInterface = NS_STATIC_CAST(nsIHttpChannel*, this);
	}else if (aIID.Equals(NS_GET_IID(nsICachingChannel))){
		foundInterface = NS_STATIC_CAST(nsICachingChannel*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIUploadChannel))){
		foundInterface = NS_STATIC_CAST(nsIUploadChannel*, this);
	}else if (aIID.Equals(NS_GET_IID(nsICacheListener))){
		foundInterface = NS_STATIC_CAST(nsICacheListener*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIEncodedChannel))){
		foundInterface = NS_STATIC_CAST(nsIEncodedChannel*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIHttpChannelInternal))){
		foundInterface = NS_STATIC_CAST(nsIHttpChannelInternal*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIResumableChannel))){
		foundInterface = NS_STATIC_CAST(nsIResumableChannel*, this);
	}else if (aIID.Equals(NS_GET_IID(nsITransportEventSink))){
		foundInterface = NS_STATIC_CAST(nsITransportEventSink*, this);
	}else if (aIID.Equals(NS_GET_IID(nsISupportsPriority))){
		foundInterface = NS_STATIC_CAST(nsISupportsPriority*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIProtocolProxyCallback))){
		foundInterface = NS_STATIC_CAST(nsIProtocolProxyCallback*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIWritablePropertyBag))){
		foundInterface = NS_STATIC_CAST(nsIWritablePropertyBag*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIPropertyBag))){
		foundInterface = NS_STATIC_CAST(nsIPropertyBag*, NS_STATIC_CAST(nsIWritablePropertyBag*, this));
	}else if (aIID.Equals(NS_GET_IID(nsISupports))){
		foundInterface = NS_STATIC_CAST(nsISupports*, NS_STATIC_CAST(nsIWritablePropertyBag*, this));
	}else if (aIID.Equals(NS_GET_IID(nsIPropertyBag2))){
		foundInterface = NS_STATIC_CAST(nsIPropertyBag2*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIWritablePropertyBag2))){
		foundInterface = NS_STATIC_CAST(nsIWritablePropertyBag2*, this);
	}

	
	if (!foundInterface) {
		TRACE("fkHttpChannel: interface not implemented %s .", aIID.ToString());
#ifdef DEBUG
		if (origChannel){
			void *dummy = nsnull;
			/*this code trigers assertion when aIID is nsIInterfaceRequestor (?)*/

			/*This leaks memory but only in debug mode when FK_ASSERT will be
			  triggered anyway*/
			nsresult rv = origChannel->QueryInterface(aIID, &dummy);			
			if (!NS_FAILED(rv) && dummy){
				TRACE("origChannel implements this!!!!");				
				FK_ASSERT(dummy);
			}
		}
#endif
		return NS_NOINTERFACE;
	}

	TRACE("interface implemented %s .", aIID.ToString());
	NS_ADDREF(foundInterface);
	*aResult = foundInterface;
	return NS_OK;
} 

#else
NS_INTERFACE_MAP_BEGIN(fkHttpChannel)
    NS_INTERFACE_MAP_ENTRY(nsIRequest)
    NS_INTERFACE_MAP_ENTRY(nsIChannel)
    NS_INTERFACE_MAP_ENTRY(nsIRequestObserver)
    NS_INTERFACE_MAP_ENTRY(nsIStreamListener)
    NS_INTERFACE_MAP_ENTRY(nsIHttpChannel)
    NS_INTERFACE_MAP_ENTRY(nsICachingChannel)
    NS_INTERFACE_MAP_ENTRY(nsIUploadChannel)
    NS_INTERFACE_MAP_ENTRY(nsICacheListener)
    NS_INTERFACE_MAP_ENTRY(nsIEncodedChannel)
    NS_INTERFACE_MAP_ENTRY(nsIHttpChannelInternal)
    NS_INTERFACE_MAP_ENTRY(nsIResumableChannel)
    NS_INTERFACE_MAP_ENTRY(nsITransportEventSink)
    NS_INTERFACE_MAP_ENTRY(nsISupportsPriority)
    NS_INTERFACE_MAP_ENTRY(nsIProtocolProxyCallback)
	
    NS_INTERFACE_MAP_ENTRY(nsIWritablePropertyBag)
    NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsIPropertyBag, nsIWritablePropertyBag)
    NS_INTERFACE_MAP_ENTRY_AMBIGUOUS(nsISupports, nsIWritablePropertyBag)
    NS_INTERFACE_MAP_ENTRY(nsIPropertyBag2)
    NS_INTERFACE_MAP_ENTRY(nsIWritablePropertyBag2)
NS_INTERFACE_MAP_END

#endif
//-----------------------------------------------------------------------------
// fkHttpChannel::nsIRequest
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetName(nsACString &aName)
{
	TRACE("");
	return origRequest->GetName(aName);
}

NS_IMETHODIMP
fkHttpChannel::IsPending(PRBool *value)
{
	TRACE("");
	return origRequest->IsPending(value);
}

NS_IMETHODIMP
fkHttpChannel::GetStatus(nsresult *aStatus)
{	
	TRACE("");
	return origRequest->GetStatus(aStatus);
}

NS_IMETHODIMP
fkHttpChannel::Cancel(nsresult status)
{
	TRACE("");
	nsresult rv = origRequest->Cancel(status);
	if (NS_FAILED(rv)){
		TRACE("cancel failed");
	}
	return rv;
}

NS_IMETHODIMP
fkHttpChannel::Suspend()
{
	TRACE("");
	return origRequest->Suspend();
}

NS_IMETHODIMP
fkHttpChannel::Resume()
{
	TRACE("");
	return origRequest->Resume();
}

NS_IMETHODIMP
fkHttpChannel::GetLoadGroup(nsILoadGroup **aLoadGroup)
{	
	TRACE("fkHttpChannel");
	NS_ENSURE_ARG_POINTER(aLoadGroup);
	if (!loadGroupProxy){
		*aLoadGroup = nsnull;
		return NS_OK;
	}else{
		nsresult rv;
		nsCOMPtr<fkILoadGroupProxy> p = do_QueryInterface(loadGroupProxy, &rv);
		FK_ASSERT(!NS_FAILED(rv));
		return p->GetOrigLoadGroup(aLoadGroup);
	}
}

NS_IMETHODIMP
fkHttpChannel::SetLoadGroup(nsILoadGroup *aLoadGroup)
{
	nsresult rv;
	TRACE("setting load group %08x origChannel = %08x this = %08x", 
	      aLoadGroup, &*origChannel, this);	
	loadGroupProxy = nsnull;
	
	if (aLoadGroup){
		/*If this cast succeeds aLoadGroup is already a proxy (instance of fkLoadGroup).
		  We should get and use original LoadGroup from this proxy to avoid 
		  call chains like this:		  
		  proxy1->OnStartRequest(nsRequest...) calls proxy2->OnStartRequest(fkReqeuest1)
		  calls origObserver->OnStartReqeust(fkRequest2) (This is wrong, fkReqeust1 has 
		  started not fkRequest2).
		*/
		nsCOMPtr<fkILoadGroupProxy> tmp = do_QueryInterface(aLoadGroup, &rv);
		if (!NS_FAILED(rv)){
			TRACE("proxy already used");
			loadGroupProxy = aLoadGroup;
		}
		else{
			fkLoadGroup *proxy;
			if (!groupHashTable->Get(aLoadGroup, &proxy)){
				TRACE("group not found");
				proxy = new fkLoadGroup(aLoadGroup);		
				if (!proxy || NS_FAILED(proxy->init()))
					return NS_ERROR_OUT_OF_MEMORY;
				groupHashTable->Put(aLoadGroup, proxy);

			}
			else{
				TRACE("group found");
			}
			loadGroupProxy = proxy;
		}
	}
	return origChannel->SetLoadGroup(loadGroupProxy);

}

NS_IMETHODIMP
fkHttpChannel::GetLoadFlags(nsLoadFlags *aLoadFlags)
{
	nsresult rv = origRequest->GetLoadFlags(aLoadFlags);
	TRACE("flags = %08x", *aLoadFlags);
	if (*aLoadFlags & nsIChannel::LOAD_REPLACE){
		TRACE("fkHttpChannel REPLACE FLAG SET");
	}
	FK_ASSERT(!NS_FAILED(rv));
	return rv;
}

NS_IMETHODIMP
fkHttpChannel::SetLoadFlags(nsLoadFlags aLoadFlags)
{
	nsresult rv = origRequest->SetLoadFlags(aLoadFlags);
	TRACE("fkHttpChannel flags = %08x", aLoadFlags);
	if (aLoadFlags & nsIChannel::LOAD_REPLACE){
		TRACE("fkHttpChannel REPLACE FLAG SET");
	}
	FK_ASSERT(!NS_FAILED(rv));
	return rv;
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIChannel
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetOriginalURI(nsIURI **originalURI)
{
	TRACE("");
	return origChannel->GetOriginalURI(originalURI);
}

NS_IMETHODIMP
fkHttpChannel::SetOriginalURI(nsIURI *originalURI)
{
	TRACE("");
	return origChannel->SetOriginalURI(originalURI);
}

NS_IMETHODIMP
fkHttpChannel::GetURI(nsIURI **URI)
{
	TRACE("");
	return origChannel->GetURI(URI);
}

NS_IMETHODIMP
fkHttpChannel::GetOwner(nsISupports **owner)
{
	TRACE("");
	return origChannel->GetOwner(owner);
}

NS_IMETHODIMP
fkHttpChannel::SetOwner(nsISupports *owner)
{
	TRACE("");
	return origChannel->SetOwner(owner);
}

NS_IMETHODIMP
fkHttpChannel::GetNotificationCallbacks(nsIInterfaceRequestor **callbacks)
{	
	
	TRACE("fkHttpChannel");
	NS_ENSURE_ARG_POINTER(callbacks);
	if (!callbacksProxy){
		*callbacks = nsnull;
		return NS_OK;
	}
	
	nsresult rv;
	nsCOMPtr<fkICallbacksProxy> p = do_QueryInterface(callbacksProxy, &rv);
	FK_ASSERT(!NS_FAILED(rv));
	return p->GetOrigCallbacks(callbacks);
}

NS_IMETHODIMP
fkHttpChannel::SetNotificationCallbacks(nsIInterfaceRequestor *callbacks)
{
	nsresult rv;
	TRACE("Setting callbacks %08x origChannel = %08x this = %08x", 
	      callbacks, &*origChannel, this);	
	callbacksProxy = nsnull;
	
	if (callbacks){	
		nsCOMPtr<fkICallbacksProxy> tmp = do_QueryInterface(callbacks, &rv);
		if (!NS_FAILED(rv)){
			TRACE("proxy already used");
			callbacksProxy = callbacks;
		}else{
			fkCallbacks *proxy;
			if (!callbacksHashTable->Get(callbacks, &proxy)){
				TRACE("callbacks not found");
				proxy = new fkCallbacks(nsnull, callbacks);		
				if (!proxy)
					return NS_ERROR_OUT_OF_MEMORY;
				callbacksHashTable->Put(callbacks, proxy);
			}
			else{
				TRACE("callbacks found");
			}
			callbacksProxy = proxy;
		}
	}
	return origChannel->SetNotificationCallbacks(callbacksProxy);
}
	


NS_IMETHODIMP
fkHttpChannel::GetSecurityInfo(nsISupports **securityInfo)
{
	TRACE("");
	return origChannel->GetSecurityInfo(securityInfo);
}

NS_IMETHODIMP
fkHttpChannel::GetContentType(nsACString &value)
{
	TRACE("");
	return origChannel->GetContentType(value);
#if 0	/* 
	   Remains of attempts to replace blocked traffic.
	   Maybe someday in a future ...
	 */
	int action;
	sentence->GetAction(&action);
	TRACE("action = %d", action);
	
	if (0 || action == sentence->CANCEL){
		value = nsEmbedCString("text/html");
		return NS_OK;
	}
	else{
		nsEmbedCString s;
		origChannel->GetContentType(s);
		TRACE("%s", s.get());
		value = s;
		return NS_OK;
	}
#endif
}

NS_IMETHODIMP
fkHttpChannel::SetContentType(const nsACString &value)
{
	TRACE("");
	return origChannel->SetContentType(value);
}

NS_IMETHODIMP
fkHttpChannel::GetContentCharset(nsACString &value)
{
	TRACE("");
	return origChannel->GetContentCharset(value);
}

NS_IMETHODIMP
fkHttpChannel::SetContentCharset(const nsACString &value)
{
	TRACE("");
	return origChannel->SetContentCharset(value);
}

NS_IMETHODIMP
fkHttpChannel::GetContentLength(PRInt32 *value)
{
	TRACE("");
	return origChannel->GetContentLength(value);
}

NS_IMETHODIMP
fkHttpChannel::SetContentLength(PRInt32 value)
{
	TRACE("");
	return origChannel->SetContentLength(value);
}

NS_IMETHODIMP
fkHttpChannel::Open(nsIInputStream **_retval)
{
	/*TODO: Browser doesn't use this, but some extensions may*/
	TRACE("Not tracing Open calls");
	return origChannel->Open(_retval);
}

/*
  Create fkStreamListener that will check data before passing it to the
  listener specified by this method caller.
 */
NS_IMETHODIMP
fkHttpChannel::AsyncOpen(nsIStreamListener *listener, nsISupports *context)
{	
	int action;
	nsresult rv;
	
	TRACE("");
	NS_ENSURE_ARG_POINTER(listener);
	
	sentence->GetAction(&action);
	if (action == sentence->BLOCK)
		return NS_ERROR_ABORT;
	if (action == sentence->DONT_AUDIT){		
		/*TODO: can't happen*/
		TRACE("whitelisted");
		FK_ASSERT(false);
		return origChannel->AsyncOpen(listener, context);
	}

	nsCOMPtr<fkIStreamListenerProxy> tmp = do_QueryInterface(listener, &rv);
	if (!NS_FAILED(rv)){
		TRACE("listener is already a proxy");
		/* This happens when channel is redirected. StreamListener from
		   old channel is used in AsyncOpen Call */
		return origChannel->AsyncOpen(listener, context);
	}
	
	TRACE("checking");
	
	fkStreamListener *fklistener_raw =
		new fkStreamListener(listener, this, sentence, connRules);
	
	if (!fklistener_raw)
		return NS_ERROR_OUT_OF_MEMORY;

	nsCOMPtr<nsIStreamListener> fklistener;
	rv = fklistener_raw->QueryInterface(NS_GET_IID(nsIStreamListener), 
					    getter_AddRefs(fklistener));
	if (NS_FAILED(rv))
		return rv;
	
	/*Call Mozilla AsyncOpen implementation but with Firekeeper
	  StreamListener*/
	return origChannel->AsyncOpen(fklistener, context);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIHttpChannel
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetRequestMethod(nsACString &method)
{
	TRACE("");
	return origHttpChannel->GetRequestMethod(method);
}

NS_IMETHODIMP
fkHttpChannel::SetRequestMethod(const nsACString &method)
{
	TRACE("");
	return origHttpChannel->SetRequestMethod(method);
}

NS_IMETHODIMP
fkHttpChannel::GetReferrer(nsIURI **referrer)
{
	TRACE("");
	return origHttpChannel->GetReferrer(referrer);
}

NS_IMETHODIMP
fkHttpChannel::SetReferrer(nsIURI *referrer)
{
	TRACE("");
	return origHttpChannel->SetReferrer(referrer);
}

NS_IMETHODIMP
fkHttpChannel::GetRequestHeader(const nsACString &header, nsACString &value)
{
	TRACE("");
	return origHttpChannel->GetRequestHeader(header, value);
}

NS_IMETHODIMP
fkHttpChannel::SetRequestHeader(const nsACString &header,
                                const nsACString &value,
                                PRBool merge)
{
	TRACE("");
	return origHttpChannel->SetRequestHeader(header, value, merge);
}

NS_IMETHODIMP
fkHttpChannel::VisitRequestHeaders(nsIHttpHeaderVisitor *visitor)
{
	TRACE("");
	return origHttpChannel->VisitRequestHeaders(visitor);
}


NS_IMETHODIMP
fkHttpChannel::GetResponseStatus(PRUint32 *value)
{
	TRACE("");
	return origHttpChannel->GetResponseStatus(value);
}

NS_IMETHODIMP
fkHttpChannel::GetResponseStatusText(nsACString &value)
{
	TRACE("");
	return origHttpChannel->GetResponseStatusText(value);
}

NS_IMETHODIMP
fkHttpChannel::GetRequestSucceeded(PRBool *value)
{
	TRACE("");
	return origHttpChannel->GetRequestSucceeded(value);
}

NS_IMETHODIMP
fkHttpChannel::GetResponseHeader(const nsACString &header, nsACString &value)
{
	TRACE("");
	return origHttpChannel->GetResponseHeader(header, value);
}

NS_IMETHODIMP
fkHttpChannel::SetResponseHeader(const nsACString &header,
                                 const nsACString &value,
                                 PRBool merge)
{
	TRACE("");
	return origHttpChannel->SetResponseHeader(header, value, merge);
}

NS_IMETHODIMP
fkHttpChannel::VisitResponseHeaders(nsIHttpHeaderVisitor *visitor)
{
	TRACE("");
	return origHttpChannel->VisitResponseHeaders(visitor);
}

NS_IMETHODIMP
fkHttpChannel::IsNoStoreResponse(PRBool *value)
{
	TRACE("");
	return origHttpChannel->IsNoStoreResponse(value);
}

NS_IMETHODIMP
fkHttpChannel::IsNoCacheResponse(PRBool *value)
{
	TRACE("");
	return origHttpChannel->IsNoCacheResponse(value);
}

NS_IMETHODIMP
fkHttpChannel::GetAllowPipelining(PRBool *value)
{
	TRACE("");
	return origHttpChannel->GetAllowPipelining(value);
}

NS_IMETHODIMP
fkHttpChannel::SetAllowPipelining(PRBool value)
{
	TRACE("");
	return origHttpChannel->SetAllowPipelining(value);
}

NS_IMETHODIMP
fkHttpChannel::GetRedirectionLimit(PRUint32 *value)
{
	TRACE("");
	return origHttpChannel->GetRedirectionLimit(value);
}

NS_IMETHODIMP
fkHttpChannel::SetRedirectionLimit(PRUint32 value)
{
	TRACE("");
	return origHttpChannel->SetRedirectionLimit(value);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIEncodedChannel
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkHttpChannel::GetContentEncodings(nsIUTF8StringEnumerator** aEncodings)
{
	return origEncodedChannel->GetContentEncodings(aEncodings);
}

NS_IMETHODIMP
fkHttpChannel::GetApplyConversion(PRBool *value)
{
	return origEncodedChannel->GetApplyConversion(value);
}


NS_IMETHODIMP
fkHttpChannel::SetApplyConversion(PRBool value)
{
	return origEncodedChannel->SetApplyConversion(value);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIUploadChannel
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetUploadStream(nsIInputStream **stream)
{
	TRACE("");
	return origUploadChannel->GetUploadStream(stream);
}



NS_IMETHODIMP
fkHttpChannel::SetUploadStream(nsIInputStream *stream, 
			       const nsACString &contentType, PRInt32 contentLength)
{
	TRACE("");
	return origUploadChannel->SetUploadStream(stream, contentType, contentLength);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIHttpChannelInternal
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetDocumentURI(nsIURI **aDocumentURI)
{
	TRACE("");
	return origHttpChannelInternal->GetDocumentURI(aDocumentURI);
}

NS_IMETHODIMP
fkHttpChannel::SetDocumentURI(nsIURI *aDocumentURI)
{
	TRACE("");
	return origHttpChannelInternal->SetDocumentURI(aDocumentURI);
}

NS_IMETHODIMP
fkHttpChannel::GetRequestVersion(PRUint32 *major, PRUint32 *minor)
{
	return origHttpChannelInternal->GetRequestVersion(major, minor);
}

NS_IMETHODIMP
fkHttpChannel::GetResponseVersion(PRUint32 *major, PRUint32 *minor)
{
	return origHttpChannelInternal->GetResponseVersion(major, minor);
}

NS_IMETHODIMP
fkHttpChannel::SetCookie(const char *aCookieHeader)
{
	TRACE("");
	return origHttpChannelInternal->SetCookie(aCookieHeader);
}

NS_IMETHODIMP
fkHttpChannel::GetProxyInfo(nsIProxyInfo **result)
{
	TRACE("");
	return origHttpChannelInternal->GetProxyInfo(result);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsISupportsPriority
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetPriority(PRInt32 *value)
{
	TRACE("");
	return origSupportsPriority->GetPriority(value);
}

NS_IMETHODIMP
fkHttpChannel::SetPriority(PRInt32 value)
{
	TRACE("");
	return origSupportsPriority->SetPriority(value);
}

NS_IMETHODIMP
fkHttpChannel::AdjustPriority(PRInt32 delta)
{
	TRACE("");
	return origSupportsPriority->AdjustPriority(delta);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIProtocolProxyCallback
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::OnProxyAvailable(nsICancelable *request, nsIURI *uri,
                                nsIProxyInfo *pi, nsresult status)
{
	TRACE("");
	return origProtocolProxyCallback->OnProxyAvailable(request, uri,
							   pi, status);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIRequestObserver
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::OnStartRequest(nsIRequest *request, nsISupports *ctxt)
{
	TRACE("");
	return origRequestObserver->OnStartRequest(this, ctxt);
}

NS_IMETHODIMP
fkHttpChannel::OnStopRequest(nsIRequest *request, nsISupports *ctxt, nsresult status)
{
	TRACE("");
	return origRequestObserver->OnStopRequest(this, ctxt, status);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIStreamListener
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::OnDataAvailable(nsIRequest *request, nsISupports *ctxt,
                               nsIInputStream *input,
                               PRUint32 offset, PRUint32 count)
{
	TRACE("");
	return origStreamListener->OnDataAvailable(this, ctxt, input,
						   offset, count);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsITransportEventSink
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::OnTransportStatus(nsITransport *trans, nsresult status,
                                 PRUint64 progress, PRUint64 progressMax)
{
	TRACE("");
	return origTransportEventSink->OnTransportStatus(trans, status, progress,
							 progressMax);
} 

//-----------------------------------------------------------------------------
// fkHttpChannel::nsICachingChannel
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::GetCacheToken(nsISupports **token)
{
	TRACE("");
	return origCachingChannel->GetCacheToken(token);
}

NS_IMETHODIMP
fkHttpChannel::SetCacheToken(nsISupports *token)
{
	TRACE("");
	return origCachingChannel->SetCacheToken(token);
}

NS_IMETHODIMP
fkHttpChannel::GetCacheKey(nsISupports **key)
{
	return origCachingChannel->GetCacheKey(key);
}

NS_IMETHODIMP
fkHttpChannel::SetCacheKey(nsISupports *key)
{
	return origCachingChannel->SetCacheKey(key);
}

NS_IMETHODIMP
fkHttpChannel::GetCacheAsFile(PRBool *value)
{
	return origCachingChannel->GetCacheAsFile(value);
}

NS_IMETHODIMP
fkHttpChannel::SetCacheAsFile(PRBool value)
{
	return origCachingChannel->SetCacheAsFile(value);
}

NS_IMETHODIMP
fkHttpChannel::GetCacheFile(nsIFile **cacheFile)
{
	return origCachingChannel->GetCacheFile(cacheFile);
}

NS_IMETHODIMP
fkHttpChannel::IsFromCache(PRBool *value)
{
	return origCachingChannel->IsFromCache(value);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIResumableChannel
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::ResumeAt(PRUint64 aStartPos,
                        const nsACString& aEntityID)
{
	TRACE("");
	return origResumableChannel->ResumeAt(aStartPos,aEntityID);
}

NS_IMETHODIMP
fkHttpChannel::GetEntityID(nsACString& aEntityID)
{
	return origResumableChannel->GetEntityID(aEntityID);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsICacheListener
//-----------------------------------------------------------------------------

NS_IMETHODIMP
fkHttpChannel::OnCacheEntryAvailable(nsICacheEntryDescriptor *entry,
                                     nsCacheAccessMode access,
                                     nsresult status)
{
	return origCacheListener->OnCacheEntryAvailable(entry, access, status);
}

/*
void
fkHttpChannel::ClearPasswordManagerEntry(const char      *scheme,
                                         const char      *host,
                                         PRInt32          port,
                                         const char      *realm,
                                         const PRUnichar *user)
{
	return origCacheListener->ClearPasswordManagerEntry(scheme, host, port,
							    realm, user);
}

nsresult
fkHttpChannel::DoAuthRetry(nsAHttpConnection *conn)
{
	return origCacheListener->DoAuthRetry(conn);
}
*/



//-----------------------------------------------------------------------------
// fkHttpChannel::nsIPropertyBag
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkHttpChannel::GetProperty(const nsAString& name, nsIVariant* *_retval)
{
    return origPropertyBag->GetProperty(name, _retval);
}

NS_IMETHODIMP
fkHttpChannel::GetEnumerator(nsISimpleEnumerator* *_retval)
{
	return origPropertyBag->GetEnumerator(_retval);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIWritablePropertyBag
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkHttpChannel::SetProperty(const nsAString& name, nsIVariant *value)
{
	return origWritablePropertyBag->SetProperty(name, value);
}

NS_IMETHODIMP
fkHttpChannel::DeleteProperty(const nsAString& name)
{
	return origWritablePropertyBag->DeleteProperty(name);
}

//-----------------------------------------------------------------------------
// fkHttpChannel::nsIPropertyBag2 && fkHttpChannel::nsIWritablePropertyBag2
//-----------------------------------------------------------------------------
#define IMPL_GETSETPROPERTY_AS(Name, Type) \
NS_IMETHODIMP \
fkHttpChannel::GetPropertyAs ## Name (const nsAString & prop, Type *_retval) \
{ \
    return origPropertyBag2->GetPropertyAs ## Name(prop, _retval); \
} \
\
NS_IMETHODIMP \
fkHttpChannel::SetPropertyAs ## Name (const nsAString & prop, Type value) \
{ \
    return origWritablePropertyBag2->SetPropertyAs ## Name(prop, value); \
}

IMPL_GETSETPROPERTY_AS(Bool, PRBool)
IMPL_GETSETPROPERTY_AS(Double, double)
IMPL_GETSETPROPERTY_AS(Int32, PRInt32)
IMPL_GETSETPROPERTY_AS(Int64, PRInt64)
IMPL_GETSETPROPERTY_AS(Uint32, PRUint32)
IMPL_GETSETPROPERTY_AS(Uint64, PRUint64)

NS_IMETHODIMP
fkHttpChannel::GetPropertyAsAString(const nsAString & prop, nsAString &_retval)
{
	return origPropertyBag2->GetPropertyAsAString(prop, _retval);
}

NS_IMETHODIMP
fkHttpChannel::GetPropertyAsACString(const nsAString & prop, nsACString &_retval)
{
	return origPropertyBag2->GetPropertyAsACString(prop, _retval);
}

NS_IMETHODIMP
fkHttpChannel::GetPropertyAsAUTF8String(const nsAString & prop, nsACString &_retval)
{
	return origPropertyBag2->GetPropertyAsAUTF8String(prop, _retval);
}

NS_IMETHODIMP
fkHttpChannel::GetPropertyAsInterface(const nsAString & prop,
                                          const nsIID & aIID,
                                          void** _retval)
{
	return origPropertyBag2->GetPropertyAsInterface(prop, aIID, _retval);
}


NS_IMETHODIMP
fkHttpChannel::SetPropertyAsAString(const nsAString & prop, const nsAString & value)
{
	return origWritablePropertyBag2->SetPropertyAsAString(prop, value);
}

NS_IMETHODIMP
fkHttpChannel::SetPropertyAsACString(const nsAString & prop, const nsACString & value)
{
	return origWritablePropertyBag2->SetPropertyAsACString(prop, value);
}

NS_IMETHODIMP
fkHttpChannel::SetPropertyAsAUTF8String(const nsAString & prop, const nsACString & value)
{
	return origWritablePropertyBag2->SetPropertyAsAUTF8String(prop, value);
}

NS_IMETHODIMP
fkHttpChannel::SetPropertyAsInterface(const nsAString & prop, nsISupports* value)
{
	return origWritablePropertyBag2->SetPropertyAsInterface(prop, value);
}

