/*$Id: fkCallbacks.cpp,v 1.4 2007/07/22 21:03:11 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2007 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "nsStringAPI.h"
#include "nsEmbedString.h"
#include "nsISupportsImpl.h"
#include "nsIURI.h"

#include "nsCOMPtr.h"
#include "Firekeeper.h"
#include "fkCallbacks.h"
#include "fkHttpHandler.h"
#include "fkRule.h"

fkCallbacks::fkCallbacks(nsILoadGroup *loadGroup, nsIInterfaceRequestor *callbacks)
{
	TRACE("constructor %08x", this);
	nsresult rv;

	origCallbacks = callbacks;
	parrentGroup = loadGroup;
	
#ifdef DEBUG	
	/*If this cast succeeds callbacks is already a proxy (instance of fkCallbacks).
	  We should get and use original callbacks from this proxy.
	*/
	nsCOMPtr<fkICallbacksProxy> proxy = do_QueryInterface(origCallbacks, &rv);
	FK_ASSERT(NS_FAILED(rv));
#endif
	
	origWebProgressListener = do_QueryInterface(callbacks, &rv);
	checkResult(rv, "nsIWebProgressListener");
		
	origSecurityEventSink = do_QueryInterface(callbacks, &rv);
	checkResult(rv, "nsISecurityEventSink");
	
	origProgressEventSink = do_QueryInterface(callbacks, &rv);
	checkResult(rv, "nsIProgressEventSink");	
		
	origHttpEventSink = do_QueryInterface(callbacks, &rv);
	checkResult(rv, "nsIHttpEventSink");
		
	origChannelEventSink = do_QueryInterface(callbacks, &rv);
	checkResult(rv, "nsIChannelEventSink");
		
	origPrompt = do_QueryInterface(callbacks, &rv);
	checkResult(rv, "nsIPrompt");
}


fkCallbacks::~fkCallbacks()
{
	TRACE("destructor %08x", this);
	
#ifdef DEBUG
	FK_ASSERT(callbacksHashTable->Get(origCallbacks, nsnull));
#endif	
	callbacksHashTable->Remove(origCallbacks);
	TRACE("HashTable count %d", callbacksHashTable->Count());
}

NS_IMPL_ADDREF(fkCallbacks)
NS_IMPL_RELEASE(fkCallbacks)


NS_IMETHODIMP
fkCallbacks::QueryInterface(const nsIID &aIID, void **aResult)
{	
	if (aResult == NULL) {
		return NS_ERROR_NULL_POINTER;
	}
	nsISupports *foundInterface = 0;
	*aResult = NULL;

	if (aIID.Equals(NS_GET_IID(nsISupportsWeakReference))){
		foundInterface = NS_STATIC_CAST(nsISupportsWeakReference*, this);
	}
	else if (aIID.Equals(NS_GET_IID(fkICallbacksProxy))){
		foundInterface = NS_STATIC_CAST(fkICallbacksProxy*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIWebProgressListener)) && origWebProgressListener){
		foundInterface = NS_STATIC_CAST(nsIWebProgressListener*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIInterfaceRequestor)) && origCallbacks){
		foundInterface = NS_STATIC_CAST(nsIInterfaceRequestor*, this);
	}else if (aIID.Equals(NS_GET_IID(nsISecurityEventSink)) && origSecurityEventSink){
		foundInterface = NS_STATIC_CAST(nsISecurityEventSink*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIProgressEventSink)) && origProgressEventSink){
		foundInterface = NS_STATIC_CAST(nsIProgressEventSink*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIHttpEventSink)) && origHttpEventSink){
		foundInterface = NS_STATIC_CAST(nsIHttpEventSink*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIChannelEventSink)) && origChannelEventSink){
		foundInterface = NS_STATIC_CAST(nsIChannelEventSink*, this);
	}else if (aIID.Equals(NS_GET_IID(nsIPrompt)) && origPrompt){
		foundInterface = NS_STATIC_CAST(nsIPrompt*, this);
	}
	
	if (!foundInterface) {
		TRACE("interface not implemented %s .", aIID.ToString());
#ifdef DEBUG
		if (origCallbacks){
			void *dummy = nsnull;
			/*Let's leak some memory. Only in debug mode when assertion is 
			 triggered anyway*/
			/*this code trigers assertion when aIID is nsIInterfaceRequestor(?)*/
			nsresult rv = origCallbacks->QueryInterface(aIID, &dummy);
			if (!NS_FAILED(rv)){
				TRACE("origCallbacks implement this!");				
				FK_ASSERT(dummy);
			}
		}
#endif
		return NS_NOINTERFACE;
	}

	TRACE("fkCallbacks interface implemented %s .", aIID.ToString());
	NS_ADDREF(foundInterface);
	*aResult = foundInterface;
	return NS_OK;
} 


//-----------------------------------------------------------------------------
// fkCallbacks::fkICallbacksProxy
//-----------------------------------------------------------------------------
NS_IMETHODIMP 
fkCallbacks::GetOrigCallbacks(nsIInterfaceRequestor **aCallbacks)
{
	TRACE("fkCallbacks::fkICallbacksProxy");
	NS_ENSURE_ARG_POINTER(aCallbacks);		
	FK_ASSERT(origCallbacks);
	
	NS_IF_ADDREF(*aCallbacks = origCallbacks);
	return NS_OK;
}


//-----------------------------------------------------------------------------
// fkCallbacks::nsIWebProgressListener
//-----------------------------------------------------------------------------

NS_IMETHODIMP 
fkCallbacks::OnStateChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, 
			     PRUint32 aStateFlags, nsresult aStatus)
{
	TRACE("fkCallbacks::nsIWebProgressListener");
	FK_ASSERT(origWebProgressListener);
	if (!origWebProgressListener)
	    return NS_ERROR_FAILURE;
	return origWebProgressListener->OnStateChange(aWebProgress, getRequestProxy(aRequest), 
						      aStateFlags, aStatus);
}

NS_IMETHODIMP 
fkCallbacks::OnProgressChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, 
				PRInt32 aCurSelfProgress, PRInt32 aMaxSelfProgress, 
				PRInt32 aCurTotalProgress, PRInt32 aMaxTotalProgress)
{
	TRACE("fkCallbacks::nsIWebProgressListener");
	FK_ASSERT(origWebProgressListener);
	if (!origWebProgressListener)
	    return NS_ERROR_FAILURE;
	return origWebProgressListener->OnProgressChange(aWebProgress, getRequestProxy(aRequest), 
							 aCurSelfProgress, aMaxSelfProgress, 
							 aCurTotalProgress, aMaxTotalProgress);
}

NS_IMETHODIMP 
fkCallbacks::OnLocationChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, 
				nsIURI *aLocation)
{
	TRACE("fkCallbacks::nsIWebProgressListener");
	FK_ASSERT(origWebProgressListener);
	if (!origWebProgressListener)
	    return NS_ERROR_FAILURE;
	return origWebProgressListener->OnLocationChange(aWebProgress, getRequestProxy(aRequest), 
							 aLocation);
}

NS_IMETHODIMP 
fkCallbacks::OnStatusChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, 
			      nsresult aStatus, const PRUnichar *aMessage)
{
	TRACE("fkCallbacks::nsIWebProgressListener");
	FK_ASSERT(origWebProgressListener);
	if (!origWebProgressListener)
	    return NS_ERROR_FAILURE;
	return origWebProgressListener->OnStatusChange(aWebProgress, getRequestProxy(aRequest), 
						       aStatus, aMessage);
} 

NS_IMETHODIMP 
fkCallbacks::OnSecurityChange(nsIWebProgress *aWebProgress, nsIRequest *aRequest, 
				PRUint32 aState)
{
	TRACE("fkCallbacks::nsIWebProgressListener");
	FK_ASSERT(origWebProgressListener);
	if (!origWebProgressListener)
		return NS_ERROR_FAILURE;
	return origWebProgressListener->OnSecurityChange(aWebProgress, getRequestProxy(aRequest), 
							 aState);
}

//-----------------------------------------------------------------------------
// fkCallbacks::nsIProgressEventSink
//-----------------------------------------------------------------------------

NS_IMETHODIMP 
fkCallbacks::OnProgress(nsIRequest *aRequest, nsISupports *aContext, PRUint64 aProgress, PRUint64 aProgressMax)
{
	TRACE("fkCallbacks::nsIProgressEventSink");
	FK_ASSERT(origProgressEventSink);
	if (!origProgressEventSink)
		return NS_ERROR_FAILURE;

	return origProgressEventSink->OnProgress(getRequestProxy(aRequest), aContext, aProgress, aProgressMax);
	
}

NS_IMETHODIMP 
fkCallbacks::OnStatus(nsIRequest *aRequest, nsISupports *aContext, nsresult aStatus, const PRUnichar *aStatusArg)
{
	TRACE("fkCallbacks::nsIProgressEventSink");
	FK_ASSERT(origProgressEventSink);
	if (!origProgressEventSink)
		return NS_ERROR_FAILURE;
	return origProgressEventSink->OnStatus(getRequestProxy(aRequest), aContext, aStatus, aStatusArg);
		
}

//-----------------------------------------------------------------------------
// fkCallbacks::nsISecurityEventSink  
//-----------------------------------------------------------------------------
NS_IMETHODIMP
fkCallbacks::OnSecurityChange(nsISupports *i_Context, PRUint32 state)
{
	TRACE("fkCallbacks::nsISecurityEventSink");
	FK_ASSERT(origSecurityEventSink);
	if (!origSecurityEventSink)
		return NS_ERROR_FAILURE;
	return origSecurityEventSink->OnSecurityChange(i_Context, state);
}

//-----------------------------------------------------------------------------
// fkCallbacks::nsInterfaceRequestor
//-----------------------------------------------------------------------------

NS_IMETHODIMP 
fkCallbacks::GetInterface(const nsIID& aIID, void** aSink)
{
	TRACE("fkCallbacks::nsInterfaceRequestor");
	nsresult rv = NS_ERROR_NO_INTERFACE;

	/*
	  from nsDocLoader.cpp
	  NS_ENSURE_ARG_POINTER(aSink);
	  
	  if(aIID.Equals(NS_GET_IID(nsILoadGroup))) {
	  *aSink = mLoadGroup;
	  NS_IF_ADDREF((nsISupports*)*aSink);
	  rv = NS_OK;
	  } else {
	*/
	
	rv = QueryInterface(aIID, aSink);
	return rv;
}




//-----------------------------------------------------------------------------
// fkCallbacks::nsIHttpEventSink
//-----------------------------------------------------------------------------
NS_IMETHODIMP 
fkCallbacks::OnRedirect(nsIHttpChannel *httpChannel, nsIChannel *newChannel)
{
	FK_ASSERT(false);
	return NS_ERROR_NOT_IMPLEMENTED;
}

//-----------------------------------------------------------------------------
// fkCallbacks::nsIChannelEventSink
//-----------------------------------------------------------------------------
NS_IMETHODIMP 
fkCallbacks::OnChannelRedirect(nsIChannel *origOldChannel, nsIChannel *origNewChannel, PRUint32 flags)
{
	nsresult rv;	
	
	if (!origChannelEventSink){
		FK_ASSERT(false);
		return NS_ERROR_FAILURE;
	}
	
	nsCOMPtr<nsIChannel> oldChannel = do_QueryInterface(getRequestProxy(origOldChannel));
	if (!oldChannel)
		/* There is no firekeeper proxy for origOldChannel */
		oldChannel = origOldChannel;
#ifdef DEBUG
	nsCOMPtr<nsIChannel> newChannel = do_QueryInterface(getRequestProxy(origNewChannel));
	TRACE("fkCallbacks::nsIChannelEventSink old %08x new %08x oldproxy %08x newproxy %08x", 
	      origOldChannel, origNewChannel, &*oldChannel, newChannel ? &*newChannel : 0);
#endif
	
	return origChannelEventSink->OnChannelRedirect(oldChannel, origNewChannel, flags);
}

//-----------------------------------------------------------------------------
// fkCallbacks::nsIPrompt
//-----------------------------------------------------------------------------
NS_IMETHODIMP 
fkCallbacks::Alert(const PRUnichar *dialogTitle, const PRUnichar *text)
{
	TRACE("fkCallbacks::nsIPrompt");
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->Alert(dialogTitle, text);
}

NS_IMETHODIMP 
fkCallbacks::AlertCheck(const PRUnichar *dialogTitle, const PRUnichar *text, 
			const PRUnichar *checkMsg, PRBool *checkValue)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->AlertCheck(dialogTitle, text, checkMsg, checkValue);
}

NS_IMETHODIMP 
fkCallbacks::Confirm(const PRUnichar *dialogTitle, const PRUnichar *text, 
		     PRBool *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->Confirm(dialogTitle, text, _retval);
}

NS_IMETHODIMP 
fkCallbacks::ConfirmCheck(const PRUnichar *dialogTitle, const PRUnichar *text, 
			  const PRUnichar *checkMsg, PRBool *checkValue, PRBool *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->ConfirmCheck(dialogTitle, text, checkMsg, checkValue, _retval);
}

NS_IMETHODIMP 
fkCallbacks::ConfirmEx(const PRUnichar *dialogTitle, const PRUnichar *text, 
		       PRUint32 buttonFlags, const PRUnichar *button0Title, 
		       const PRUnichar *button1Title, const PRUnichar *button2Title, 
		       const PRUnichar *checkMsg, PRBool *checkValue, PRInt32 *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->ConfirmEx(dialogTitle, text, buttonFlags, button0Title, 
				     button1Title, button2Title, checkMsg, 
				     checkValue, _retval);
}

NS_IMETHODIMP 
fkCallbacks::Prompt(const PRUnichar *dialogTitle, const PRUnichar *text, PRUnichar **value, 
		    const PRUnichar *checkMsg, PRBool *checkValue, PRBool *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->Prompt(dialogTitle, text, value, checkMsg, checkValue, _retval);
}

NS_IMETHODIMP 
fkCallbacks::PromptPassword(const PRUnichar *dialogTitle, const PRUnichar *text, 
			    PRUnichar **password, const PRUnichar *checkMsg, 
			    PRBool *checkValue, PRBool *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->PromptPassword(dialogTitle, text, password, 
					  checkMsg, checkValue, _retval);
}

NS_IMETHODIMP 
fkCallbacks::PromptUsernameAndPassword(const PRUnichar *dialogTitle, const PRUnichar *text, 
				       PRUnichar **username, PRUnichar **password, 
				       const PRUnichar *checkMsg, PRBool *checkValue, 
				       PRBool *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->PromptUsernameAndPassword(dialogTitle, text, username, password, 
						     checkMsg, checkValue, _retval);
}

NS_IMETHODIMP 
fkCallbacks::Select(const PRUnichar *dialogTitle, const PRUnichar *text, PRUint32 count, 
		    const PRUnichar **selectList, PRInt32 *outSelection, PRBool *_retval)
{
	TRACE("fkCallbacks::nsIPrompt");
	FK_ASSERT(origPrompt);
	if (!origPrompt)
		return NS_ERROR_FAILURE;
	return origPrompt->Select(dialogTitle, text, count, selectList, outSelection, _retval);
}

