/*  Motti -- a strategy game
    Copyright (C) 1999 Free Software Foundation

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
*/
#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <getopt.h>
#include <dirent.h>

#include "wrappers.h"
#include "thread.h"
#include "map.h"
#ifdef HAVE_LIBCURSES
#include "curses_if.h"
#endif

static int parse_numbers (const char **, const int);
static void parse_players (int, char **, thread_func ***, const char ***);
static void help_msg (void);
static void version_msg (void);
static void get_type (const char, thread_func **);
static const char **display_name = NULL;
static const char *display_name_local;
static enum player_type *connection_types;

#if defined HAVE_LIBCURSES && defined HAVE_LIBX11
static int use_curses;
#endif

#ifdef HAVE_LIBPTHREAD
static void
get_type (type, func)
     const char type;
     thread_func **func;
{
#ifdef HAVE_LIBX11
  if (type == ':')
    {
      *func = x_game_init;
      n_x_connect++;
    }
  else
    die ("invalid player type");
#endif
}

static int
parse_numbers (string, this_connect)
     const char **string;
     const int this_connect;
{
  int value, this_mask = 0;
  const char *ptr = *string;
  char *endptr;
  do
    {
      value = strtol (ptr, &endptr, 10);
      if (ptr == endptr)
	{
	  if (*ptr == ',')
	    {
	      ptr++;
	      continue;
	    }
	  else if (this_mask)
	    {
	      *string = ptr+1;
	      return this_mask;
	    }
	  else
	    die ("invalid player string");
	}
      ptr = endptr;
      if (value <= game_map.players && value > 0)
	this_mask |= 1<<(value-1);
      else
	die ("invalid player number");
    }
  while (*ptr != '\0');
  die ("invalid player string: %s", string);
}

static void
parse_players (argc, argv, type, name)
     int argc;
     char *argv[];
     thread_func ***type;
     const char ***name;
{
  int players_marked = 0, local_player, this_connect = 0;
  const char *ptr;
  *name = my_malloc (sizeof (char *) * argc);
  *type = my_malloc (sizeof (thread_func *) * argc);
  n_connect = 0;
  /* Enough in all cases.  */
  connect_mask = my_malloc (sizeof (int) * (argc+1));
  global_conn_info = my_malloc (sizeof (union conn_info *) *
				(argc+1));
  global_conn_type = my_malloc (sizeof (enum conn_type) * (argc+1));

  while (optind < argc)
    {
      int this_display_mask;
      ptr = argv[optind++];
      this_display_mask = parse_numbers (&ptr, this_connect);
      if (this_display_mask == 0)
	die ("invalid player string");
      if (this_display_mask & players_marked)
	die ("player set multiple times");

      players_marked |= this_display_mask;
      (*name)[this_connect] = ptr;	/* Damn C!  :-) */
      get_type (*(ptr-1), &((*type)[this_connect++]));
      connect_mask[n_connect++] = this_display_mask;
    }

  /* All players, which aren't explicitly set, are local players.  */
  local_player = ~players_marked & ((1<<game_map.players)-1);
  if (local_player)
    {
#ifdef HAVE_LIBX11
      (*type)[n_connect] = x_game_init;
      (*name)[n_connect] = "";
#endif
      connect_mask[n_connect++] = local_player;
    }
}
#endif	/* HAVE_LIBPTHREAD */

static void
help_msg ()
{
#ifdef HAVE_LIBPTHREAD
  puts ("Usage: motti map [options] [player, ...]");
#ifdef HAVE_LIBCURSES
  puts ("       motti map --curses\n");
#else
  puts ("");
#endif
#else
  puts ("Usage: motti map [options]\n");
#endif
#if defined HAVE_LIBCURSES && defined HAVE_LIBX11
  puts ("  -c, --curses                 use curses interface");
#endif
  puts ("      --help                   display this help and exit");
  puts ("      --version                output version information and exit\n");
#ifdef HAVE_LIBPTHREAD
  puts ("Player specified with format 1,2,...,n:display");
#endif /* have_libx11 */
  puts ("Report bugs to bug-motti@gnu.org");
  exit (0);
}

static void
version_msg ()
{
  puts (VERSION_STR);
  puts ("Copyright (C) 1999 Free Software Foundation, Inc.");
  puts ("Motti comes with NO WARRANTY,");
  puts ("to the extent permitted by law.");
  puts ("You may redistribute copies of motti");
  puts ("under the terms of the GNU General Public License.");
  puts ("For more information about these matters,");
  puts ("see the files named COPYING.");
  exit (0);
}

static void
list_maps ()
{
  DIR *dir;
  struct dirent *dp;
  printf ("No map file specified, maps available at %s.\n", MAPPATH);
  dir = opendir (MAPPATH);
  if (!dir)
    die ("can't read directory %s", MAPPATH);
  while (dp = readdir (dir))
    if (*dp->d_name != '.')
      printf ("%s\n", dp->d_name);
  exit (0);
}

int
main (argc, argv)
     int argc;
     char *argv[];
{
  char *status_msg;
  int use_x = 0;
  thread_func **connection_type;
  const char **connection_name;

  while (1)
    {
      int c;
      const struct option long_options[] =
      {
#if defined HAVE_LIBCURSES && defined HAVE_LIBX11
	{"curses", 0, 0, 'c'},
#endif
	{"help", 0, 0, 1000},
	{"version", 0, 0, 1001},
	{0, 0, 0, 0}
      };

      c = getopt_long (argc, argv, "c",
		       long_options, (int *) NULL);
      if (c == -1)
	break;

      switch (c)
	{
#if defined HAVE_LIBCURSES && defined HAVE_LIBX11
	case 'c':
	  use_curses = 1;
	  break;
#endif
	case 1000:
	default:
	  help_msg ();
	case 1001:
	  version_msg ();
	}
    }

  puts (VERSION_STR);
  puts ("Copyright 1999 Free Software Foundation, Inc.");
  puts ("This is free software with ABSOLUTELY NO WARRANTY.");
  puts ("For details read file 'COPYING'.");

  if (argc == optind)
    list_maps ();
  load_map (argv[optind++]);

#ifdef HAVE_LIBPTHREAD
#ifdef HAVE_LIBCURSES
  if (!use_curses)
#endif
    parse_players (argc, argv, &connection_type, &connection_name);
#endif /* HAVE_LIBPTHREAD */
#ifdef HAVE_LIBX11
  global_argc = argc;
  global_argv = argv;
#endif /* HAVE_LIBX11 */

  srand (time ((time_t *) NULL));

#ifdef HAVE_LIBX11
#ifdef HAVE_LIBCURSES
  if (!use_curses)
#endif
    create_threads (connection_type, connection_name);
#endif
#if HAVE_LIBCURSES
  curses_game_init ();
#endif
  return 0;	/* Never executed.  */
}
