/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: SeaOfGatesCellBuilder.java
 *
 * Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.sun.electric.tool.routing.seaOfGates;

import com.sun.electric.database.CellBackup;
import com.sun.electric.database.CellRevision;
import com.sun.electric.database.CellTree;
import com.sun.electric.database.EditingPreferences;
import com.sun.electric.database.Environment;
import com.sun.electric.database.ImmutableArcInst;
import com.sun.electric.database.ImmutableCell;
import com.sun.electric.database.ImmutableNodeInst;
import com.sun.electric.database.Snapshot;
import com.sun.electric.database.geometry.EPoint;
import com.sun.electric.database.id.ArcProtoId;
import com.sun.electric.database.id.CellId;
import com.sun.electric.database.id.PortProtoId;
import com.sun.electric.database.id.PrimitiveNodeId;
import com.sun.electric.database.text.Name;
import com.sun.electric.database.topology.ArcInst;
import com.sun.electric.database.variable.TextDescriptor;
import com.sun.electric.technology.TechPool;
import com.sun.electric.tool.Tool;
import com.sun.electric.util.math.Orientation;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.TreeMap;

/**
 *
 * @author dn146861
 */
class SeaOfGatesCellBuilder {

    private final EditingPreferences ep;
    // oldSnapshot
    private final Snapshot oldSnapshot;
    private final Tool oldTool;
    private final Environment oldEnvironment;
    private final CellTree oldCellTree;
    private final CellBackup oldCellBackup;
    private final CellTree[] oldCellTrees;
    private final TechPool oldTechPool;
    private final CellRevision oldCellRevision;
    private final ImmutableCell oldCell;
    final CellId cellId;
    // curSnapshot
    private Snapshot curSnapshot;
    private final CellBackup[] curCellBackups;
    private CellTree curCellTree;
    private CellBackup curCellBackup;
    // nodes
    private final BitSet curNodes = new BitSet();
    private int curNodesCount;
    private int lastNodeId;
    private final TreeMap<Name, MaxNodeSuffix> maxNodeSuffixesOrdered = new TreeMap<Name, MaxNodeSuffix>();
    private final IdentityHashMap<PrimitiveNodeId, MaxNodeSuffix> maxNodeSuffixes = new IdentityHashMap<PrimitiveNodeId, MaxNodeSuffix>();
    private final IdentityHashMap<SeaOfGatesEngine.RouteNode, ImmutableNodeInst> addedNodesToNodeInst = new IdentityHashMap<SeaOfGatesEngine.RouteNode, ImmutableNodeInst>();
    // arcs
    private final BitSet curArcs = new BitSet();
    private int curArcsCount;
    private int lastArcId;
    private int maxArcSuffix;
    private final int arcInsertionPoint;
    private final List<ImmutableArcInst> addedArcs = new ArrayList<ImmutableArcInst>();

    SeaOfGatesCellBuilder(Snapshot snapshot, CellId cellId, EditingPreferences ep) {
        this.ep = ep;

        oldSnapshot = snapshot;
        oldTool = oldSnapshot.tool;
        oldEnvironment = oldSnapshot.environment;
        oldCellTree = snapshot.getCellTree(cellId);
        oldCellBackup = oldCellTree.top;
        oldCellTrees = oldCellTree.getSubTrees();
        oldTechPool = oldCellTree.techPool;
        oldCellRevision = oldCellBackup.cellRevision;
        oldCell = oldCellRevision.d;
        this.cellId = cellId;

        int maxCellIndex = -1;
        for (CellTree cellTree : oldSnapshot.cellTrees) {
            if (cellTree != null) {
                maxCellIndex = Math.max(maxCellIndex, cellTree.top.cellRevision.d.cellId.cellIndex);
            }
        }
        curCellBackups = new CellBackup[maxCellIndex + 1];
        for (CellTree cellTree : oldSnapshot.cellTrees) {
            if (cellTree != null) {
                curCellBackups[cellTree.top.cellRevision.d.cellId.cellIndex] = cellTree.top;
            }
        }

        lastNodeId = -1;
        for (ImmutableNodeInst n : oldCellRevision.nodes) {
            int nodeId = n.nodeId;
            assert !curNodes.get(nodeId);
            curNodes.set(nodeId);
            lastNodeId = Math.max(lastNodeId, nodeId);
        }
        curNodesCount = oldCellRevision.nodes.size();

        lastArcId = -1;
        maxArcSuffix = -1;
        for (ImmutableArcInst a : oldCellRevision.arcs) {
            int arcId = a.arcId;
            assert !curArcs.get(arcId);
            curArcs.set(arcId);
            lastArcId = Math.max(lastArcId, arcId);
        }
        curArcsCount = oldCellRevision.arcs.size();
        arcInsertionPoint = searchArcInsertionPoint(ImmutableArcInst.BASENAME.toString());
        maxArcSuffix = -1;
        if (arcInsertionPoint > 0) {
            Name name = oldCellRevision.arcs.get(arcInsertionPoint - 1).name;
            if (name.isTempname()) {
                assert name.getBasename() == ImmutableArcInst.BASENAME;
                maxArcSuffix = name.getNumSuffix();
            }
        }

        curSnapshot = oldSnapshot;
        curCellTree = oldCellTree;
        curCellBackup = oldCellBackup;
    }

    /**
     * Method to instantiate RouteResolution
     * @param resolution RouteResolution
     */
    synchronized void instantiate(SeaOfGatesEngine.RouteResolution resolution) {
        assert resolution.cellId == cellId;
        for (int nodeId : resolution.nodesIDsToKill) {
            assert curNodes.get(nodeId);
            curNodes.clear(nodeId);
            curNodesCount--;
        }

        for (int arcId : resolution.arcsIDsToKill) {
            assert curArcs.get(arcId);
            curArcs.clear(arcId);
            curArcsCount--;
        }

        for (SeaOfGatesEngine.RouteNode rn : resolution.nodesToRoute) {
            if (rn.exists()) {
                continue;
            }
            int nodeId = ++lastNodeId;
            PrimitiveNodeId protoId = (PrimitiveNodeId) rn.getProtoId();
            MaxNodeSuffix maxSuffix = maxNodeSuffixes.get(protoId);
            if (maxSuffix == null) {
                Name baseName = rn.getBaseName();
                maxSuffix = maxNodeSuffixesOrdered.get(baseName);
                if (maxSuffix == null) {
                    maxSuffix = new MaxNodeSuffix(this, baseName);
                    maxNodeSuffixesOrdered.put(baseName, maxSuffix);
                }
                maxNodeSuffixes.put(protoId, maxSuffix);
            }
            Name name = maxSuffix.getNextName();
            TextDescriptor nameTd = ep.getNodeTextDescriptor();
            Orientation orient = rn.getOrient();
            EPoint anchor = rn.getLoc();
            EPoint size = rn.getSize();
            int flags = 0;
            int techBits = rn.getTechBits();
            TextDescriptor protoTd = ep.getInstanceTextDescriptor();
            ImmutableNodeInst n = ImmutableNodeInst.newInstance(nodeId, protoId, name, nameTd,
                    orient, anchor, size, flags, techBits, protoTd);
            maxSuffix.add(n);
            assert !curNodes.get(nodeId);
            curNodes.set(nodeId);
            curNodesCount++;
            addedNodesToNodeInst.put(rn, n);
        }

        for (SeaOfGatesEngine.RouteArc ra : resolution.arcsToRoute) {
            int arcId = ++lastArcId;
            ArcProtoId protoId = ra.getProtoId();
            assert maxArcSuffix < Integer.MAX_VALUE;
            Name name = ImmutableArcInst.BASENAME.findSuffixed(++maxArcSuffix);
            TextDescriptor nameTd = ep.getArcTextDescriptor();

            SeaOfGatesEngine.RouteNode tail = ra.getTail();
            int tailNodeId = tail.exists() ? tail.getNodeId() : addedNodesToNodeInst.get(tail).nodeId;
            PortProtoId tailProtoId = tail.getPortProtoId();
            EPoint tailLocation = tail.getLoc();

            SeaOfGatesEngine.RouteNode head = ra.getHead();
            int headNodeId = head.exists() ? head.getNodeId() : addedNodesToNodeInst.get(head).nodeId;
            PortProtoId headProtoId = head.getPortProtoId();
            EPoint headLocation = head.getLoc();

            long gridExtendOverMin = ra.getGridExtendOverMin();
            int angle = ArcInst.DEFAULTANGLE;
            int flags = ra.getFlags(ep);

            ImmutableArcInst a = ImmutableArcInst.newInstance(arcId, protoId, name, nameTd,
                    tailNodeId, tailProtoId, tailLocation,
                    headNodeId, headProtoId, headLocation,
                    gridExtendOverMin, angle, flags);
            addedArcs.add(a);
            assert !curArcs.get(arcId);
            curArcs.set(arcId);
            curArcsCount++;
        }

        ArcProtoId unroutedId = oldTechPool.getGeneric().unrouted_arc.getId();
        long unroutedGridExtend = oldTechPool.getGeneric().unrouted_arc.getDefaultInst(ep).getGridExtendOverMin();
        int unroutedFlags = oldTechPool.getGeneric().unrouted_arc.getDefaultInst(ep).flags;
        for (SeaOfGatesEngine.RouteAddUnrouted rau : resolution.unroutedToAdd) {
            int arcId = ++lastArcId;
            ArcProtoId protoId = unroutedId;
            assert maxArcSuffix < Integer.MAX_VALUE;
            Name name = ImmutableArcInst.BASENAME.findSuffixed(++maxArcSuffix);
            TextDescriptor nameTd = ep.getArcTextDescriptor();

            int tailNodeId = rau.getTailId();
            PortProtoId tailProtoId = rau.getTailPortProtoId();
            EPoint tailLocation = rau.getTailLocation();

            int headNodeId = rau.getHeadId();
            PortProtoId headProtoId = rau.getHeadPortProtoId();
            EPoint headLocation = rau.getHeadLocation();

            long gridExtendOverMin = unroutedGridExtend;
            int angle = ArcInst.DEFAULTANGLE;
            int flags = unroutedFlags;

            ImmutableArcInst a = ImmutableArcInst.newInstance(arcId, protoId, name, nameTd,
                    tailNodeId, tailProtoId, tailLocation,
                    headNodeId, headProtoId, headLocation,
                    gridExtendOverMin, angle, flags);
            addedArcs.add(a);
            assert !curArcs.get(arcId);
            curArcs.set(arcId);
            curArcsCount++;
        }
        resolution.nodesToRoute.clear();
        resolution.arcsToRoute.clear();
        resolution.nodesIDsToKill.clear();
        resolution.arcsIDsToKill.clear();
        resolution.unroutedToAdd.clear();
    }

    Snapshot commit() {
        ImmutableNodeInst[] newNodes;
        ImmutableArcInst[] newArcs;

        synchronized (this) {
            newNodes = new ImmutableNodeInst[curNodesCount];
            int oldNodeIndex = 0;
            int newNodeIndex = 0;
            for (MaxNodeSuffix maxSuffix : maxNodeSuffixesOrdered.values()) {
                while (oldNodeIndex < maxSuffix.insertionPoint) {
                    ImmutableNodeInst n = oldCellRevision.nodes.get(oldNodeIndex++);
                    if (curNodes.get(n.nodeId)) {
                        newNodes[newNodeIndex++] = n;
                    }
                }
                for (ImmutableNodeInst n : maxSuffix.addedNodes) {
                    assert curNodes.get(n.nodeId);
                    newNodes[newNodeIndex++] = n;
                }
            }
            while (oldNodeIndex < oldCellRevision.nodes.size()) {
                ImmutableNodeInst n = oldCellRevision.nodes.get(oldNodeIndex++);
                if (curNodes.get(n.nodeId)) {
                    newNodes[newNodeIndex++] = n;
                }
            }
            assert newNodeIndex == newNodes.length;

            newArcs = new ImmutableArcInst[curArcsCount];
            int newArcIndex = 0;
            for (int oldArcIndex = 0; oldArcIndex < arcInsertionPoint; oldArcIndex++) {
                ImmutableArcInst a = oldCellRevision.arcs.get(oldArcIndex);
                if (curArcs.get(a.arcId)) {
                    newArcs[newArcIndex++] = a;
                }
            }
            for (ImmutableArcInst a : addedArcs) {
                assert curArcs.get(a.arcId);
                newArcs[newArcIndex++] = a;
            }
            for (int oldArcIndex = arcInsertionPoint; oldArcIndex < oldCellRevision.arcs.size(); oldArcIndex++) {
                ImmutableArcInst a = oldCellRevision.arcs.get(oldArcIndex);
                if (curArcs.get(a.arcId)) {
                    newArcs[newArcIndex++] = a;
                }
            }
            assert newArcIndex == newArcs.length;
        }

        curCellBackup = curCellBackup.with(oldCell, newNodes, newArcs, null, oldTechPool);

        curCellBackups[cellId.cellIndex] = curCellBackup;
        curSnapshot = curSnapshot.with(oldTool, oldEnvironment, curCellBackups, null);

        return curSnapshot;
    }

    private static class MaxNodeSuffix {

        final Name basename;
        final int insertionPoint;
        int maxSuffix;
        List<ImmutableNodeInst> addedNodes = new ArrayList<ImmutableNodeInst>();

        private MaxNodeSuffix(SeaOfGatesCellBuilder b, Name basename) {
            this.basename = basename;
            insertionPoint = b.searchNodeInsertionPoint(basename.toString());
            maxSuffix = -1;
            if (insertionPoint > 0) {
                Name name = b.oldCellRevision.nodes.get(insertionPoint - 1).name;
                if (name.isTempname() && name.getBasename() == basename) {
                    maxSuffix = name.getNumSuffix();
                }

            }
        }

        private Name getNextName() {
            return basename.findSuffixed(maxSuffix + 1);
        }

        private void add(ImmutableNodeInst n) {
            assert n.name.getBasename() == basename;
            assert n.name.getNumSuffix() == ++maxSuffix;
            addedNodes.add(n);
        }
    }

    private int searchNodeInsertionPoint(String basename) {
        assert basename.endsWith("@0");
        char nextChar = (char) (basename.charAt(basename.length() - 2) + 1);
        String nextName = basename.substring(0, basename.length() - 2) + nextChar;
        int index = oldCellRevision.nodes.searchByName(nextName);
        return index >= 0 ? index : -(index + 1);
    }

    private int searchArcInsertionPoint(String basename) {
        assert basename.endsWith("@0");
        char nextChar = (char) (basename.charAt(basename.length() - 2) + 1);
        String nextName = basename.substring(0, basename.length() - 2) + nextChar;
        int index = oldCellRevision.arcs.searchByName(nextName);
        return index >= 0 ? index : -(index + 1);
    }
}
