/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iotdb.db.engine.compaction.inner.utils;

import com.google.common.util.concurrent.RateLimiter;
import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import org.apache.iotdb.db.conf.IoTDBDescriptor;
import org.apache.iotdb.db.engine.compaction.CompactionMetricsManager;
import org.apache.iotdb.db.engine.compaction.CompactionTaskManager;
import org.apache.iotdb.db.engine.compaction.constant.CompactionType;
import org.apache.iotdb.db.engine.compaction.constant.ProcessChunkType;
import org.apache.iotdb.db.engine.storagegroup.TsFileResource;
import org.apache.iotdb.db.metadata.path.PartialPath;
import org.apache.iotdb.metrics.config.MetricConfigDescriptor;
import org.apache.iotdb.tsfile.file.header.ChunkHeader;
import org.apache.iotdb.tsfile.file.metadata.ChunkMetadata;
import org.apache.iotdb.tsfile.read.TimeValuePair;
import org.apache.iotdb.tsfile.read.TsFileSequenceReader;
import org.apache.iotdb.tsfile.read.common.BatchData;
import org.apache.iotdb.tsfile.read.common.Chunk;
import org.apache.iotdb.tsfile.read.reader.chunk.ChunkReaderByTimestamp;
import org.apache.iotdb.tsfile.utils.Pair;
import org.apache.iotdb.tsfile.write.chunk.ChunkWriterImpl;
import org.apache.iotdb.tsfile.write.schema.IMeasurementSchema;
import org.apache.iotdb.tsfile.write.schema.MeasurementSchema;
import org.apache.iotdb.tsfile.write.writer.TsFileIOWriter;

public class SingleSeriesCompactionExecutor {
    private PartialPath series;
    private IMeasurementSchema schema;
    private String device;
    private LinkedList<Pair<TsFileSequenceReader, List<ChunkMetadata>>> readerAndChunkMetadataList;
    private TsFileIOWriter fileWriter;
    private TsFileResource targetResource;
    private ChunkWriterImpl chunkWriter;
    private Chunk cachedChunk;
    private ChunkMetadata cachedChunkMetadata;
    private RateLimiter compactionRateLimiter = CompactionTaskManager.getInstance().getMergeWriteRateLimiter();
    private long minStartTimestamp = Long.MAX_VALUE;
    private long maxEndTimestamp = Long.MIN_VALUE;
    private long pointCountInChunkWriter = 0L;
    private final long targetChunkSize = IoTDBDescriptor.getInstance().getConfig().getTargetChunkSize();
    private final long targetChunkPointNum = IoTDBDescriptor.getInstance().getConfig().getTargetChunkPointNum();
    private final long chunkSizeLowerBound = IoTDBDescriptor.getInstance().getConfig().getChunkSizeLowerBoundInCompaction();
    private final long chunkPointNumLowerBound = IoTDBDescriptor.getInstance().getConfig().getChunkPointNumLowerBoundInCompaction();
    private final boolean enableMetrics = MetricConfigDescriptor.getInstance().getMetricConfig().getEnableMetric();

    public SingleSeriesCompactionExecutor(PartialPath series, LinkedList<Pair<TsFileSequenceReader, List<ChunkMetadata>>> readerAndChunkMetadataList, TsFileIOWriter fileWriter, TsFileResource targetResource) {
        this.device = series.getDevice();
        this.readerAndChunkMetadataList = readerAndChunkMetadataList;
        this.fileWriter = fileWriter;
        this.series = series;
        this.chunkWriter = null;
        this.cachedChunk = null;
        this.cachedChunkMetadata = null;
        this.targetResource = targetResource;
    }

    public void execute() throws IOException {
        while (this.readerAndChunkMetadataList.size() > 0) {
            Pair<TsFileSequenceReader, List<ChunkMetadata>> readerListPair = this.readerAndChunkMetadataList.removeFirst();
            TsFileSequenceReader reader = (TsFileSequenceReader)readerListPair.left;
            List chunkMetadataList = (List)readerListPair.right;
            for (ChunkMetadata chunkMetadata : chunkMetadataList) {
                Chunk currentChunk = reader.readMemChunk(chunkMetadata);
                if (this.chunkWriter == null) {
                    this.constructChunkWriterFromReadChunk(currentChunk);
                }
                CompactionMetricsManager.recordReadInfo(currentChunk.getHeader().getSerializedSize() + currentChunk.getHeader().getDataSize());
                if (chunkMetadata.getDeleteIntervalList() != null) {
                    this.processModifiedChunk(currentChunk);
                    continue;
                }
                long chunkSize = this.getChunkSize(currentChunk);
                long chunkPointNum = currentChunk.getChunkStatistic().getCount();
                if (chunkSize >= this.targetChunkSize || chunkPointNum >= this.targetChunkPointNum) {
                    this.processLargeChunk(currentChunk, chunkMetadata);
                    continue;
                }
                if (chunkSize < this.chunkSizeLowerBound && chunkPointNum < this.chunkPointNumLowerBound) {
                    this.processSmallChunk(currentChunk);
                    continue;
                }
                this.processMiddleChunk(currentChunk, chunkMetadata);
            }
        }
        if (this.cachedChunk != null) {
            this.flushChunkToFileWriter(this.cachedChunk, this.cachedChunkMetadata, true);
            this.cachedChunk = null;
            this.cachedChunkMetadata = null;
        } else if (this.pointCountInChunkWriter != 0L) {
            this.flushChunkWriter();
        }
        this.targetResource.updateStartTime(this.device, this.minStartTimestamp);
        this.targetResource.updateEndTime(this.device, this.maxEndTimestamp);
    }

    private void constructChunkWriterFromReadChunk(Chunk chunk) {
        ChunkHeader chunkHeader = chunk.getHeader();
        this.schema = new MeasurementSchema(this.series.getMeasurement(), chunkHeader.getDataType(), chunkHeader.getEncodingType(), chunkHeader.getCompressionType());
        this.chunkWriter = new ChunkWriterImpl(this.schema);
    }

    private long getChunkSize(Chunk chunk) {
        return chunk.getHeader().getSerializedSize() + chunk.getHeader().getDataSize();
    }

    private void processModifiedChunk(Chunk chunk) throws IOException {
        if (this.cachedChunk != null) {
            this.writeCachedChunkIntoChunkWriter();
        }
        this.writeChunkIntoChunkWriter(chunk);
        this.flushChunkWriterIfLargeEnough();
    }

    private void processLargeChunk(Chunk chunk, ChunkMetadata chunkMetadata) throws IOException {
        if (this.pointCountInChunkWriter != 0L) {
            this.writeChunkIntoChunkWriter(chunk);
            this.flushChunkWriterIfLargeEnough();
        } else if (this.cachedChunk != null) {
            this.mergeWithCachedChunk(chunk, chunkMetadata);
            this.flushCachedChunkIfLargeEnough();
        } else {
            this.flushChunkToFileWriter(chunk, chunkMetadata, false);
        }
    }

    private void processMiddleChunk(Chunk chunk, ChunkMetadata chunkMetadata) throws IOException {
        if (this.pointCountInChunkWriter != 0L) {
            this.writeChunkIntoChunkWriter(chunk);
            this.flushChunkWriterIfLargeEnough();
        } else if (this.cachedChunk != null) {
            this.mergeWithCachedChunk(chunk, chunkMetadata);
            this.flushCachedChunkIfLargeEnough();
        } else {
            this.cachedChunk = chunk;
            this.cachedChunkMetadata = chunkMetadata;
        }
    }

    private void processSmallChunk(Chunk chunk) throws IOException {
        if (this.cachedChunk != null) {
            this.writeCachedChunkIntoChunkWriter();
        }
        this.writeChunkIntoChunkWriter(chunk);
        this.flushChunkWriterIfLargeEnough();
    }

    private void writeChunkIntoChunkWriter(Chunk chunk) throws IOException {
        ChunkReaderByTimestamp chunkReader = new ChunkReaderByTimestamp(chunk);
        while (chunkReader.hasNextSatisfiedPage()) {
            BatchData.BatchDataIterator batchIterator = chunkReader.nextPageData().getBatchDataIterator();
            while (batchIterator.hasNextTimeValuePair()) {
                TimeValuePair timeValuePair = batchIterator.nextTimeValuePair();
                this.writeTimeAndValueToChunkWriter(timeValuePair);
                if (timeValuePair.getTimestamp() > this.maxEndTimestamp) {
                    this.maxEndTimestamp = timeValuePair.getTimestamp();
                }
                if (timeValuePair.getTimestamp() >= this.minStartTimestamp) continue;
                this.minStartTimestamp = timeValuePair.getTimestamp();
            }
        }
        this.pointCountInChunkWriter += chunk.getChunkStatistic().getCount();
    }

    private void writeCachedChunkIntoChunkWriter() throws IOException {
        if (this.cachedChunk.getData().position() != 0) {
            this.cachedChunk.getData().flip();
        }
        this.writeChunkIntoChunkWriter(this.cachedChunk);
        this.cachedChunk = null;
        this.cachedChunkMetadata = null;
    }

    private void mergeWithCachedChunk(Chunk currentChunk, ChunkMetadata currentChunkMetadata) throws IOException {
        this.cachedChunk.mergeChunkByAppendPage(currentChunk);
        this.cachedChunkMetadata.mergeChunkMetadata(currentChunkMetadata);
    }

    private void writeTimeAndValueToChunkWriter(TimeValuePair timeValuePair) {
        switch (this.chunkWriter.getDataType()) {
            case TEXT: {
                this.chunkWriter.write(timeValuePair.getTimestamp(), timeValuePair.getValue().getBinary());
                break;
            }
            case FLOAT: {
                this.chunkWriter.write(timeValuePair.getTimestamp(), timeValuePair.getValue().getFloat());
                break;
            }
            case DOUBLE: {
                this.chunkWriter.write(timeValuePair.getTimestamp(), timeValuePair.getValue().getDouble());
                break;
            }
            case BOOLEAN: {
                this.chunkWriter.write(timeValuePair.getTimestamp(), timeValuePair.getValue().getBoolean());
                break;
            }
            case INT64: {
                this.chunkWriter.write(timeValuePair.getTimestamp(), timeValuePair.getValue().getLong());
                break;
            }
            case INT32: {
                this.chunkWriter.write(timeValuePair.getTimestamp(), timeValuePair.getValue().getInt());
                break;
            }
            default: {
                throw new UnsupportedOperationException("Unknown data type " + this.chunkWriter.getDataType());
            }
        }
    }

    private void flushChunkToFileWriter(Chunk chunk, ChunkMetadata chunkMetadata, boolean isCachedChunk) throws IOException {
        CompactionTaskManager.mergeRateLimiterAcquire(this.compactionRateLimiter, this.getChunkSize(chunk));
        if (chunkMetadata.getStartTime() < this.minStartTimestamp) {
            this.minStartTimestamp = chunkMetadata.getStartTime();
        }
        if (chunkMetadata.getEndTime() > this.maxEndTimestamp) {
            this.maxEndTimestamp = chunkMetadata.getEndTime();
        }
        CompactionMetricsManager.recordWriteInfo(CompactionType.INNER_SEQ_COMPACTION, isCachedChunk ? ProcessChunkType.MERGE_CHUNK : ProcessChunkType.FLUSH_CHUNK, false, this.getChunkSize(chunk));
        this.fileWriter.writeChunk(chunk, chunkMetadata);
    }

    private void flushChunkWriterIfLargeEnough() throws IOException {
        if (this.pointCountInChunkWriter >= this.targetChunkPointNum || this.chunkWriter.estimateMaxSeriesMemSize() >= this.targetChunkSize) {
            CompactionTaskManager.mergeRateLimiterAcquire(this.compactionRateLimiter, this.chunkWriter.estimateMaxSeriesMemSize());
            CompactionMetricsManager.recordWriteInfo(CompactionType.INNER_SEQ_COMPACTION, ProcessChunkType.DESERIALIZE_CHUNK, false, this.chunkWriter.estimateMaxSeriesMemSize());
            this.chunkWriter.writeToFileWriter(this.fileWriter);
            this.pointCountInChunkWriter = 0L;
        }
    }

    private void flushCachedChunkIfLargeEnough() throws IOException {
        if (this.cachedChunk.getChunkStatistic().getCount() >= this.targetChunkPointNum || this.getChunkSize(this.cachedChunk) >= this.targetChunkSize) {
            this.flushChunkToFileWriter(this.cachedChunk, this.cachedChunkMetadata, true);
            this.cachedChunk = null;
            this.cachedChunkMetadata = null;
        }
    }

    private void flushChunkWriter() throws IOException {
        CompactionTaskManager.mergeRateLimiterAcquire(this.compactionRateLimiter, this.chunkWriter.estimateMaxSeriesMemSize());
        CompactionMetricsManager.recordWriteInfo(CompactionType.INNER_SEQ_COMPACTION, ProcessChunkType.DESERIALIZE_CHUNK, false, this.chunkWriter.estimateMaxSeriesMemSize());
        this.chunkWriter.writeToFileWriter(this.fileWriter);
        this.pointCountInChunkWriter = 0L;
    }
}

