/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

// author Kevin Lang, Oath Research

#ifndef CPC_COMPRESSION_DATA_HPP_
#define CPC_COMPRESSION_DATA_HPP_

namespace datasketches {

/*
   The 23 length-limited Huffman codes in this file were created
   by the ocaml program "generateHuffmanCodes.ml", which was 
   compiled and run as follows:

~/ocaml-4.03.0/bin/ocamlopt -o generateHuffmanCodes columnProbabilities.ml generateHuffmanCodes.ml

./generateHuffmanCodes > raw-encoding-tables.c

Some manual cutting and pasting was then done to transfer the contents
of that file into this one.

Only the encoding tables are defined by this file. The decoding tables (which are exact inverses)
are created at library startup time.
*/

static const uint16_t encoding_tables_for_high_entropy_byte [22][256] = {
 // Sixteen Encoding Tables for the Steady State.

 // (table 0 of 22) (steady 0 of 16) (phase = 0.031250000 = 1.0 / 32.0)
 // entropy:    4.4619200780464778333
 // avg_length: 4.5415773046232610355; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x9017, // ( 9,   23)   0
 0x5009, // ( 5,    9)   1
 0x7033, // ( 7,   51)   2
 0x3002, // ( 3,    2)   3
 0x9117, // ( 9,  279)   4
 0x5019, // ( 5,   25)   5
 0x7073, // ( 7,  115)   6
 0x2000, // ( 2,    0)   7
 0xa177, // (10,  375)   8
 0x601d, // ( 6,   29)   9
 0x803b, // ( 8,   59)  10
 0x4001, // ( 4,    1)  11
 0xa377, // (10,  887)  12
 0x5005, // ( 5,    5)  13
 0x80bb, // ( 8,  187)  14
 0x3006, // ( 3,    6)  15
 0xb0cf, // (11,  207)  16
 0x700b, // ( 7,   11)  17
 0xa0f7, // (10,  247)  18
 0x5015, // ( 5,   21)  19
 0xb4cf, // (11, 1231)  20
 0x704b, // ( 7,   75)  21
 0x9097, // ( 9,  151)  22
 0x500d, // ( 5,   13)  23
 0xc4af, // (12, 1199)  24
 0x807b, // ( 8,  123)  25
 0xa2f7, // (10,  759)  26
 0x603d, // ( 6,   61)  27
 0xccaf, // (12, 3247)  28
 0x80fb, // ( 8,  251)  29
 0xa1f7, // (10,  503)  30
 0x6003, // ( 6,    3)  31
 0xc2af, // (12,  687)  32
 0x8007, // ( 8,    7)  33
 0xb2cf, // (11,  719)  34
 0x6023, // ( 6,   35)  35
 0xcaaf, // (12, 2735)  36
 0x8087, // ( 8,  135)  37
 0xa3f7, // (10, 1015)  38
 0x6013, // ( 6,   19)  39
 0xc6af, // (12, 1711)  40
 0x9197, // ( 9,  407)  41
 0xceaf, // (12, 3759)  42
 0x702b, // ( 7,   43)  43
 0xc1af, // (12,  431)  44
 0x9057, // ( 9,   87)  45
 0xb6cf, // (11, 1743)  46
 0x706b, // ( 7,  107)  47
 0xc9af, // (12, 2479)  48
 0xa00f, // (10,   15)  49
 0xc5af, // (12, 1455)  50
 0x8047, // ( 8,   71)  51
 0xcdaf, // (12, 3503)  52
 0xa20f, // (10,  527)  53
 0xc3af, // (12,  943)  54
 0x80c7, // ( 8,  199)  55
 0xcbaf, // (12, 2991)  56
 0xb1cf, // (11,  463)  57
 0xc7af, // (12, 1967)  58
 0x9157, // ( 9,  343)  59
 0xcfaf, // (12, 4015)  60
 0xb5cf, // (11, 1487)  61
 0xc06f, // (12,  111)  62
 0x90d7, // ( 9,  215)  63
 0xc86f, // (12, 2159)  64
 0x91d7, // ( 9,  471)  65
 0xc46f, // (12, 1135)  66
 0x701b, // ( 7,   27)  67
 0xcc6f, // (12, 3183)  68
 0x9037, // ( 9,   55)  69
 0xb3cf, // (11,  975)  70
 0x705b, // ( 7,   91)  71
 0xc26f, // (12,  623)  72
 0xa10f, // (10,  271)  73
 0xca6f, // (12, 2671)  74
 0x8027, // ( 8,   39)  75
 0xc66f, // (12, 1647)  76
 0xa30f, // (10,  783)  77
 0xce6f, // (12, 3695)  78
 0x80a7, // ( 8,  167)  79
 0xc16f, // (12,  367)  80
 0xb7cf, // (11, 1999)  81
 0xc96f, // (12, 2415)  82
 0x9137, // ( 9,  311)  83
 0xc56f, // (12, 1391)  84
 0xb02f, // (11,   47)  85
 0xcd6f, // (12, 3439)  86
 0x90b7, // ( 9,  183)  87
 0xc36f, // (12,  879)  88
 0xcb6f, // (12, 2927)  89
 0xc76f, // (12, 1903)  90
 0xa08f, // (10,  143)  91
 0xcf6f, // (12, 3951)  92
 0xc0ef, // (12,  239)  93
 0xc8ef, // (12, 2287)  94
 0xa28f, // (10,  655)  95
 0xc4ef, // (12, 1263)  96
 0xccef, // (12, 3311)  97
 0xc2ef, // (12,  751)  98
 0xa18f, // (10,  399)  99
 0xcaef, // (12, 2799) 100
 0xc6ef, // (12, 1775) 101
 0xceef, // (12, 3823) 102
 0xa38f, // (10,  911) 103
 0xc1ef, // (12,  495) 104
 0xc9ef, // (12, 2543) 105
 0xc5ef, // (12, 1519) 106
 0xb42f, // (11, 1071) 107
 0xcdef, // (12, 3567) 108
 0xc3ef, // (12, 1007) 109
 0xcbef, // (12, 3055) 110
 0xb22f, // (11,  559) 111
 0xc7ef, // (12, 2031) 112
 0xcfef, // (12, 4079) 113
 0xc01f, // (12,   31) 114
 0xc81f, // (12, 2079) 115
 0xc41f, // (12, 1055) 116
 0xcc1f, // (12, 3103) 117
 0xc21f, // (12,  543) 118
 0xca1f, // (12, 2591) 119
 0xc61f, // (12, 1567) 120
 0xce1f, // (12, 3615) 121
 0xc11f, // (12,  287) 122
 0xc91f, // (12, 2335) 123
 0xc51f, // (12, 1311) 124
 0xcd1f, // (12, 3359) 125
 0xc31f, // (12,  799) 126
 0xcb1f, // (12, 2847) 127
 0xc71f, // (12, 1823) 128
 0xa04f, // (10,   79) 129
 0xcf1f, // (12, 3871) 130
 0x8067, // ( 8,  103) 131
 0xc09f, // (12,  159) 132
 0xa24f, // (10,  591) 133
 0xc89f, // (12, 2207) 134
 0x80e7, // ( 8,  231) 135
 0xc49f, // (12, 1183) 136
 0xb62f, // (11, 1583) 137
 0xcc9f, // (12, 3231) 138
 0x91b7, // ( 9,  439) 139
 0xc29f, // (12,  671) 140
 0xb12f, // (11,  303) 141
 0xca9f, // (12, 2719) 142
 0x9077, // ( 9,  119) 143
 0xc69f, // (12, 1695) 144
 0xce9f, // (12, 3743) 145
 0xc19f, // (12,  415) 146
 0xa14f, // (10,  335) 147
 0xc99f, // (12, 2463) 148
 0xc59f, // (12, 1439) 149
 0xcd9f, // (12, 3487) 150
 0xa34f, // (10,  847) 151
 0xc39f, // (12,  927) 152
 0xcb9f, // (12, 2975) 153
 0xc79f, // (12, 1951) 154
 0xb52f, // (11, 1327) 155
 0xcf9f, // (12, 3999) 156
 0xc05f, // (12,   95) 157
 0xc85f, // (12, 2143) 158
 0xb32f, // (11,  815) 159
 0xc45f, // (12, 1119) 160
 0xcc5f, // (12, 3167) 161
 0xc25f, // (12,  607) 162
 0xb72f, // (11, 1839) 163
 0xca5f, // (12, 2655) 164
 0xc65f, // (12, 1631) 165
 0xce5f, // (12, 3679) 166
 0xb0af, // (11,  175) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 1 of 22) (steady 1 of 16) (phase = 0.093750000 = 3.0 / 32.0)
 // entropy:    4.4574755684414029133
 // avg_length: 4.5336306265208552446; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0xa177, // (10,  375)   0
 0x5009, // ( 5,    9)   1
 0x803b, // ( 8,   59)   2
 0x3002, // ( 3,    2)   3
 0x9017, // ( 9,   23)   4
 0x5019, // ( 5,   25)   5
 0x700b, // ( 7,   11)   6
 0x2000, // ( 2,    0)   7
 0xb34f, // (11,  847)   8
 0x601d, // ( 6,   29)   9
 0x9117, // ( 9,  279)  10
 0x4001, // ( 4,    1)  11
 0xa377, // (10,  887)  12
 0x603d, // ( 6,   61)  13
 0x80bb, // ( 8,  187)  14
 0x3006, // ( 3,    6)  15
 0xc4af, // (12, 1199)  16
 0x704b, // ( 7,   75)  17
 0xa0f7, // (10,  247)  18
 0x5005, // ( 5,    5)  19
 0xb74f, // (11, 1871)  20
 0x702b, // ( 7,   43)  21
 0x9097, // ( 9,  151)  22
 0x5015, // ( 5,   21)  23
 0xccaf, // (12, 3247)  24
 0x807b, // ( 8,  123)  25
 0xb0cf, // (11,  207)  26
 0x6003, // ( 6,    3)  27
 0xc2af, // (12,  687)  28
 0x80fb, // ( 8,  251)  29
 0xa2f7, // (10,  759)  30
 0x500d, // ( 5,   13)  31
 0xcaaf, // (12, 2735)  32
 0x8007, // ( 8,    7)  33
 0xb4cf, // (11, 1231)  34
 0x6023, // ( 6,   35)  35
 0xc6af, // (12, 1711)  36
 0x8087, // ( 8,  135)  37
 0xa1f7, // (10,  503)  38
 0x6013, // ( 6,   19)  39
 0xceaf, // (12, 3759)  40
 0x9197, // ( 9,  407)  41
 0xc1af, // (12,  431)  42
 0x706b, // ( 7,  107)  43
 0xc9af, // (12, 2479)  44
 0x9057, // ( 9,   87)  45
 0xb2cf, // (11,  719)  46
 0x6033, // ( 6,   51)  47
 0xc5af, // (12, 1455)  48
 0xa3f7, // (10, 1015)  49
 0xcdaf, // (12, 3503)  50
 0x8047, // ( 8,   71)  51
 0xc3af, // (12,  943)  52
 0xa00f, // (10,   15)  53
 0xcbaf, // (12, 2991)  54
 0x80c7, // ( 8,  199)  55
 0xc7af, // (12, 1967)  56
 0xb6cf, // (11, 1743)  57
 0xcfaf, // (12, 4015)  58
 0x9157, // ( 9,  343)  59
 0xc06f, // (12,  111)  60
 0xb1cf, // (11,  463)  61
 0xc86f, // (12, 2159)  62
 0x90d7, // ( 9,  215)  63
 0xc46f, // (12, 1135)  64
 0x91d7, // ( 9,  471)  65
 0xcc6f, // (12, 3183)  66
 0x701b, // ( 7,   27)  67
 0xc26f, // (12,  623)  68
 0x9037, // ( 9,   55)  69
 0xb5cf, // (11, 1487)  70
 0x705b, // ( 7,   91)  71
 0xca6f, // (12, 2671)  72
 0xa20f, // (10,  527)  73
 0xc66f, // (12, 1647)  74
 0x8027, // ( 8,   39)  75
 0xce6f, // (12, 3695)  76
 0xa10f, // (10,  271)  77
 0xc16f, // (12,  367)  78
 0x80a7, // ( 8,  167)  79
 0xc96f, // (12, 2415)  80
 0xb3cf, // (11,  975)  81
 0xc56f, // (12, 1391)  82
 0x9137, // ( 9,  311)  83
 0xcd6f, // (12, 3439)  84
 0xb7cf, // (11, 1999)  85
 0xc36f, // (12,  879)  86
 0x90b7, // ( 9,  183)  87
 0xcb6f, // (12, 2927)  88
 0xc76f, // (12, 1903)  89
 0xcf6f, // (12, 3951)  90
 0xa30f, // (10,  783)  91
 0xc0ef, // (12,  239)  92
 0xc8ef, // (12, 2287)  93
 0xc4ef, // (12, 1263)  94
 0xa08f, // (10,  143)  95
 0xccef, // (12, 3311)  96
 0xc2ef, // (12,  751)  97
 0xcaef, // (12, 2799)  98
 0xa28f, // (10,  655)  99
 0xc6ef, // (12, 1775) 100
 0xceef, // (12, 3823) 101
 0xc1ef, // (12,  495) 102
 0xa18f, // (10,  399) 103
 0xc9ef, // (12, 2543) 104
 0xc5ef, // (12, 1519) 105
 0xcdef, // (12, 3567) 106
 0xb02f, // (11,   47) 107
 0xc3ef, // (12, 1007) 108
 0xcbef, // (12, 3055) 109
 0xc7ef, // (12, 2031) 110
 0xb42f, // (11, 1071) 111
 0xcfef, // (12, 4079) 112
 0xc01f, // (12,   31) 113
 0xc81f, // (12, 2079) 114
 0xc41f, // (12, 1055) 115
 0xcc1f, // (12, 3103) 116
 0xc21f, // (12,  543) 117
 0xca1f, // (12, 2591) 118
 0xc61f, // (12, 1567) 119
 0xce1f, // (12, 3615) 120
 0xc11f, // (12,  287) 121
 0xc91f, // (12, 2335) 122
 0xc51f, // (12, 1311) 123
 0xcd1f, // (12, 3359) 124
 0xc31f, // (12,  799) 125
 0xcb1f, // (12, 2847) 126
 0xc71f, // (12, 1823) 127
 0xcf1f, // (12, 3871) 128
 0xa38f, // (10,  911) 129
 0xc09f, // (12,  159) 130
 0x8067, // ( 8,  103) 131
 0xc89f, // (12, 2207) 132
 0xa04f, // (10,   79) 133
 0xc49f, // (12, 1183) 134
 0x80e7, // ( 8,  231) 135
 0xcc9f, // (12, 3231) 136
 0xb22f, // (11,  559) 137
 0xc29f, // (12,  671) 138
 0x91b7, // ( 9,  439) 139
 0xca9f, // (12, 2719) 140
 0xb62f, // (11, 1583) 141
 0xc69f, // (12, 1695) 142
 0x9077, // ( 9,  119) 143
 0xce9f, // (12, 3743) 144
 0xc19f, // (12,  415) 145
 0xc99f, // (12, 2463) 146
 0xa24f, // (10,  591) 147
 0xc59f, // (12, 1439) 148
 0xcd9f, // (12, 3487) 149
 0xc39f, // (12,  927) 150
 0xa14f, // (10,  335) 151
 0xcb9f, // (12, 2975) 152
 0xc79f, // (12, 1951) 153
 0xcf9f, // (12, 3999) 154
 0xb12f, // (11,  303) 155
 0xc05f, // (12,   95) 156
 0xc85f, // (12, 2143) 157
 0xc45f, // (12, 1119) 158
 0xb52f, // (11, 1327) 159
 0xcc5f, // (12, 3167) 160
 0xc25f, // (12,  607) 161
 0xca5f, // (12, 2655) 162
 0xb32f, // (11,  815) 163
 0xc65f, // (12, 1631) 164
 0xce5f, // (12, 3679) 165
 0xc15f, // (12,  351) 166
 0xb72f, // (11, 1839) 167
 0xc95f, // (12, 2399) 168
 0xc55f, // (12, 1375) 169
 0xcd5f, // (12, 3423) 170
 0xc35f, // (12,  863) 171
 0xcb5f, // (12, 2911) 172
 0xc75f, // (12, 1887) 173
 0xcf5f, // (12, 3935) 174
 0xb0af, // (11,  175) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 2 of 22) (steady 2 of 16) (phase = 0.156250000 = 5.0 / 32.0)
 // entropy:    4.4520619712441886762
 // avg_length: 4.5253989110544479146; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0xa177, // (10,  375)   0
 0x5009, // ( 5,    9)   1
 0x803b, // ( 8,   59)   2
 0x3002, // ( 3,    2)   3
 0xa377, // (10,  887)   4
 0x5019, // ( 5,   25)   5
 0x80bb, // ( 8,  187)   6
 0x2000, // ( 2,    0)   7
 0xb34f, // (11,  847)   8
 0x601d, // ( 6,   29)   9
 0x9057, // ( 9,   87)  10
 0x4001, // ( 4,    1)  11
 0xb74f, // (11, 1871)  12
 0x603d, // ( 6,   61)  13
 0x807b, // ( 8,  123)  14
 0x3006, // ( 3,    6)  15
 0xc72f, // (12, 1839)  16
 0x700b, // ( 7,   11)  17
 0xa0f7, // (10,  247)  18
 0x5005, // ( 5,    5)  19
 0xcf2f, // (12, 3887)  20
 0x704b, // ( 7,   75)  21
 0xa2f7, // (10,  759)  22
 0x5015, // ( 5,   21)  23
 0xc0af, // (12,  175)  24
 0x80fb, // ( 8,  251)  25
 0xb0cf, // (11,  207)  26
 0x6003, // ( 6,    3)  27
 0xc8af, // (12, 2223)  28
 0x8007, // ( 8,    7)  29
 0xa1f7, // (10,  503)  30
 0x500d, // ( 5,   13)  31
 0xc4af, // (12, 1199)  32
 0x8087, // ( 8,  135)  33
 0xb4cf, // (11, 1231)  34
 0x6023, // ( 6,   35)  35
 0xccaf, // (12, 3247)  36
 0x8047, // ( 8,   71)  37
 0xb2cf, // (11,  719)  38
 0x6013, // ( 6,   19)  39
 0xc2af, // (12,  687)  40
 0x9157, // ( 9,  343)  41
 0xcaaf, // (12, 2735)  42
 0x702b, // ( 7,   43)  43
 0xc6af, // (12, 1711)  44
 0x90d7, // ( 9,  215)  45
 0xceaf, // (12, 3759)  46
 0x6033, // ( 6,   51)  47
 0xc1af, // (12,  431)  48
 0xa3f7, // (10, 1015)  49
 0xc9af, // (12, 2479)  50
 0x80c7, // ( 8,  199)  51
 0xc5af, // (12, 1455)  52
 0xa00f, // (10,   15)  53
 0xcdaf, // (12, 3503)  54
 0x8027, // ( 8,   39)  55
 0xc3af, // (12,  943)  56
 0xb6cf, // (11, 1743)  57
 0xcbaf, // (12, 2991)  58
 0x91d7, // ( 9,  471)  59
 0xc7af, // (12, 1967)  60
 0xb1cf, // (11,  463)  61
 0xcfaf, // (12, 4015)  62
 0x80a7, // ( 8,  167)  63
 0xc06f, // (12,  111)  64
 0x9037, // ( 9,   55)  65
 0xc86f, // (12, 2159)  66
 0x706b, // ( 7,  107)  67
 0xc46f, // (12, 1135)  68
 0x9137, // ( 9,  311)  69
 0xcc6f, // (12, 3183)  70
 0x701b, // ( 7,   27)  71
 0xc26f, // (12,  623)  72
 0xa20f, // (10,  527)  73
 0xca6f, // (12, 2671)  74
 0x8067, // ( 8,  103)  75
 0xc66f, // (12, 1647)  76
 0xa10f, // (10,  271)  77
 0xce6f, // (12, 3695)  78
 0x705b, // ( 7,   91)  79
 0xc16f, // (12,  367)  80
 0xb5cf, // (11, 1487)  81
 0xc96f, // (12, 2415)  82
 0x90b7, // ( 9,  183)  83
 0xc56f, // (12, 1391)  84
 0xb3cf, // (11,  975)  85
 0xcd6f, // (12, 3439)  86
 0x91b7, // ( 9,  439)  87
 0xc36f, // (12,  879)  88
 0xcb6f, // (12, 2927)  89
 0xc76f, // (12, 1903)  90
 0xa30f, // (10,  783)  91
 0xcf6f, // (12, 3951)  92
 0xc0ef, // (12,  239)  93
 0xc8ef, // (12, 2287)  94
 0xa08f, // (10,  143)  95
 0xc4ef, // (12, 1263)  96
 0xccef, // (12, 3311)  97
 0xc2ef, // (12,  751)  98
 0xa28f, // (10,  655)  99
 0xcaef, // (12, 2799) 100
 0xc6ef, // (12, 1775) 101
 0xceef, // (12, 3823) 102
 0xa18f, // (10,  399) 103
 0xc1ef, // (12,  495) 104
 0xc9ef, // (12, 2543) 105
 0xc5ef, // (12, 1519) 106
 0xb7cf, // (11, 1999) 107
 0xcdef, // (12, 3567) 108
 0xc3ef, // (12, 1007) 109
 0xcbef, // (12, 3055) 110
 0xb02f, // (11,   47) 111
 0xc7ef, // (12, 2031) 112
 0xcfef, // (12, 4079) 113
 0xc01f, // (12,   31) 114
 0xc81f, // (12, 2079) 115
 0xc41f, // (12, 1055) 116
 0xcc1f, // (12, 3103) 117
 0xc21f, // (12,  543) 118
 0xca1f, // (12, 2591) 119
 0xc61f, // (12, 1567) 120
 0xce1f, // (12, 3615) 121
 0xc11f, // (12,  287) 122
 0xc91f, // (12, 2335) 123
 0xc51f, // (12, 1311) 124
 0xcd1f, // (12, 3359) 125
 0xc31f, // (12,  799) 126
 0xcb1f, // (12, 2847) 127
 0xc71f, // (12, 1823) 128
 0xa38f, // (10,  911) 129
 0xcf1f, // (12, 3871) 130
 0x80e7, // ( 8,  231) 131
 0xc09f, // (12,  159) 132
 0xa04f, // (10,   79) 133
 0xc89f, // (12, 2207) 134
 0x8017, // ( 8,   23) 135
 0xc49f, // (12, 1183) 136
 0xb42f, // (11, 1071) 137
 0xcc9f, // (12, 3231) 138
 0x9077, // ( 9,  119) 139
 0xc29f, // (12,  671) 140
 0xb22f, // (11,  559) 141
 0xca9f, // (12, 2719) 142
 0x8097, // ( 8,  151) 143
 0xc69f, // (12, 1695) 144
 0xce9f, // (12, 3743) 145
 0xc19f, // (12,  415) 146
 0xa24f, // (10,  591) 147
 0xc99f, // (12, 2463) 148
 0xc59f, // (12, 1439) 149
 0xcd9f, // (12, 3487) 150
 0xa14f, // (10,  335) 151
 0xc39f, // (12,  927) 152
 0xcb9f, // (12, 2975) 153
 0xc79f, // (12, 1951) 154
 0xb62f, // (11, 1583) 155
 0xcf9f, // (12, 3999) 156
 0xc05f, // (12,   95) 157
 0xc85f, // (12, 2143) 158
 0xb12f, // (11,  303) 159
 0xc45f, // (12, 1119) 160
 0xcc5f, // (12, 3167) 161
 0xc25f, // (12,  607) 162
 0xb52f, // (11, 1327) 163
 0xca5f, // (12, 2655) 164
 0xc65f, // (12, 1631) 165
 0xce5f, // (12, 3679) 166
 0xb32f, // (11,  815) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 3 of 22) (steady 3 of 16) (phase = 0.218750000 = 7.0 / 32.0)
 // entropy:    4.4457680500675866853
 // avg_length: 4.5181192844586535173; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0xb24f, // (11,  591)   0
 0x601d, // ( 6,   29)   1
 0x9097, // ( 9,  151)   2
 0x3002, // ( 3,    2)   3
 0xa1f7, // (10,  503)   4
 0x5005, // ( 5,    5)   5
 0x807b, // ( 8,  123)   6
 0x2000, // ( 2,    0)   7
 0xc52f, // (12, 1327)   8
 0x603d, // ( 6,   61)   9
 0x9197, // ( 9,  407)  10
 0x4001, // ( 4,    1)  11
 0xb64f, // (11, 1615)  12
 0x6003, // ( 6,    3)  13
 0x9057, // ( 9,   87)  14
 0x3006, // ( 3,    6)  15
 0xcd2f, // (12, 3375)  16
 0x80fb, // ( 8,  251)  17
 0xb14f, // (11,  335)  18
 0x5015, // ( 5,   21)  19
 0xc32f, // (12,  815)  20
 0x702b, // ( 7,   43)  21
 0xa3f7, // (10, 1015)  22
 0x4009, // ( 4,    9)  23
 0xcb2f, // (12, 2863)  24
 0x8007, // ( 8,    7)  25
 0xb54f, // (11, 1359)  26
 0x6023, // ( 6,   35)  27
 0xc72f, // (12, 1839)  28
 0x8087, // ( 8,  135)  29
 0xb34f, // (11,  847)  30
 0x500d, // ( 5,   13)  31
 0xcf2f, // (12, 3887)  32
 0x9157, // ( 9,  343)  33
 0xc0af, // (12,  175)  34
 0x6013, // ( 6,   19)  35
 0xc8af, // (12, 2223)  36
 0x8047, // ( 8,   71)  37
 0xb74f, // (11, 1871)  38
 0x6033, // ( 6,   51)  39
 0xc4af, // (12, 1199)  40
 0x90d7, // ( 9,  215)  41
 0xccaf, // (12, 3247)  42
 0x706b, // ( 7,  107)  43
 0xc2af, // (12,  687)  44
 0x91d7, // ( 9,  471)  45
 0xcaaf, // (12, 2735)  46
 0x600b, // ( 6,   11)  47
 0xc6af, // (12, 1711)  48
 0xb0cf, // (11,  207)  49
 0xceaf, // (12, 3759)  50
 0x80c7, // ( 8,  199)  51
 0xc1af, // (12,  431)  52
 0xa00f, // (10,   15)  53
 0xc9af, // (12, 2479)  54
 0x8027, // ( 8,   39)  55
 0xc5af, // (12, 1455)  56
 0xb4cf, // (11, 1231)  57
 0xcdaf, // (12, 3503)  58
 0x9037, // ( 9,   55)  59
 0xc3af, // (12,  943)  60
 0xb2cf, // (11,  719)  61
 0xcbaf, // (12, 2991)  62
 0x80a7, // ( 8,  167)  63
 0xc7af, // (12, 1967)  64
 0xa20f, // (10,  527)  65
 0xcfaf, // (12, 4015)  66
 0x701b, // ( 7,   27)  67
 0xc06f, // (12,  111)  68
 0x9137, // ( 9,  311)  69
 0xc86f, // (12, 2159)  70
 0x705b, // ( 7,   91)  71
 0xc46f, // (12, 1135)  72
 0xb6cf, // (11, 1743)  73
 0xcc6f, // (12, 3183)  74
 0x8067, // ( 8,  103)  75
 0xc26f, // (12,  623)  76
 0xa10f, // (10,  271)  77
 0xca6f, // (12, 2671)  78
 0x703b, // ( 7,   59)  79
 0xc66f, // (12, 1647)  80
 0xce6f, // (12, 3695)  81
 0xc16f, // (12,  367)  82
 0x90b7, // ( 9,  183)  83
 0xc96f, // (12, 2415)  84
 0xb1cf, // (11,  463)  85
 0xc56f, // (12, 1391)  86
 0x91b7, // ( 9,  439)  87
 0xcd6f, // (12, 3439)  88
 0xc36f, // (12,  879)  89
 0xcb6f, // (12, 2927)  90
 0xa30f, // (10,  783)  91
 0xc76f, // (12, 1903)  92
 0xcf6f, // (12, 3951)  93
 0xc0ef, // (12,  239)  94
 0x9077, // ( 9,  119)  95
 0xc8ef, // (12, 2287)  96
 0xc4ef, // (12, 1263)  97
 0xccef, // (12, 3311)  98
 0xa08f, // (10,  143)  99
 0xc2ef, // (12,  751) 100
 0xcaef, // (12, 2799) 101
 0xc6ef, // (12, 1775) 102
 0xa28f, // (10,  655) 103
 0xceef, // (12, 3823) 104
 0xc1ef, // (12,  495) 105
 0xc9ef, // (12, 2543) 106
 0xb5cf, // (11, 1487) 107
 0xc5ef, // (12, 1519) 108
 0xcdef, // (12, 3567) 109
 0xc3ef, // (12, 1007) 110
 0xb3cf, // (11,  975) 111
 0xcbef, // (12, 3055) 112
 0xc7ef, // (12, 2031) 113
 0xcfef, // (12, 4079) 114
 0xc01f, // (12,   31) 115
 0xc81f, // (12, 2079) 116
 0xc41f, // (12, 1055) 117
 0xcc1f, // (12, 3103) 118
 0xc21f, // (12,  543) 119
 0xca1f, // (12, 2591) 120
 0xc61f, // (12, 1567) 121
 0xce1f, // (12, 3615) 122
 0xc11f, // (12,  287) 123
 0xc91f, // (12, 2335) 124
 0xc51f, // (12, 1311) 125
 0xcd1f, // (12, 3359) 126
 0xc31f, // (12,  799) 127
 0xcb1f, // (12, 2847) 128
 0xb7cf, // (11, 1999) 129
 0xc71f, // (12, 1823) 130
 0x80e7, // ( 8,  231) 131
 0xcf1f, // (12, 3871) 132
 0xa18f, // (10,  399) 133
 0xc09f, // (12,  159) 134
 0x8017, // ( 8,   23) 135
 0xc89f, // (12, 2207) 136
 0xc49f, // (12, 1183) 137
 0xcc9f, // (12, 3231) 138
 0x9177, // ( 9,  375) 139
 0xc29f, // (12,  671) 140
 0xb02f, // (11,   47) 141
 0xca9f, // (12, 2719) 142
 0x90f7, // ( 9,  247) 143
 0xc69f, // (12, 1695) 144
 0xce9f, // (12, 3743) 145
 0xc19f, // (12,  415) 146
 0xa38f, // (10,  911) 147
 0xc99f, // (12, 2463) 148
 0xc59f, // (12, 1439) 149
 0xcd9f, // (12, 3487) 150
 0xa04f, // (10,   79) 151
 0xc39f, // (12,  927) 152
 0xcb9f, // (12, 2975) 153
 0xc79f, // (12, 1951) 154
 0xb42f, // (11, 1071) 155
 0xcf9f, // (12, 3999) 156
 0xc05f, // (12,   95) 157
 0xc85f, // (12, 2143) 158
 0xb22f, // (11,  559) 159
 0xc45f, // (12, 1119) 160
 0xcc5f, // (12, 3167) 161
 0xc25f, // (12,  607) 162
 0xb62f, // (11, 1583) 163
 0xca5f, // (12, 2655) 164
 0xc65f, // (12, 1631) 165
 0xce5f, // (12, 3679) 166
 0xb12f, // (11,  303) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 4 of 22) (steady 4 of 16) (phase = 0.281250000 = 9.0 / 32.0)
 // entropy:    4.4386754570568340839
 // avg_length: 4.5071584786605640716; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0xb24f, // (11,  591)   0
 0x601d, // ( 6,   29)   1
 0x9057, // ( 9,   87)   2
 0x3002, // ( 3,    2)   3
 0xb64f, // (11, 1615)   4
 0x5005, // ( 5,    5)   5
 0x807b, // ( 8,  123)   6
 0x2000, // ( 2,    0)   7
 0xc32f, // (12,  815)   8
 0x700b, // ( 7,   11)   9
 0xa0f7, // (10,  247)  10
 0x4001, // ( 4,    1)  11
 0xb14f, // (11,  335)  12
 0x603d, // ( 6,   61)  13
 0x9157, // ( 9,  343)  14
 0x3006, // ( 3,    6)  15
 0xcb2f, // (12, 2863)  16
 0x80fb, // ( 8,  251)  17
 0xb54f, // (11, 1359)  18
 0x5015, // ( 5,   21)  19
 0xc72f, // (12, 1839)  20
 0x704b, // ( 7,   75)  21
 0xa2f7, // (10,  759)  22
 0x4009, // ( 4,    9)  23
 0xcf2f, // (12, 3887)  24
 0x8007, // ( 8,    7)  25
 0xb34f, // (11,  847)  26
 0x6003, // ( 6,    3)  27
 0xc0af, // (12,  175)  28
 0x8087, // ( 8,  135)  29
 0xb74f, // (11, 1871)  30
 0x500d, // ( 5,   13)  31
 0xc8af, // (12, 2223)  32
 0x90d7, // ( 9,  215)  33
 0xc4af, // (12, 1199)  34
 0x6023, // ( 6,   35)  35
 0xccaf, // (12, 3247)  36
 0x8047, // ( 8,   71)  37
 0xb0cf, // (11,  207)  38
 0x6013, // ( 6,   19)  39
 0xc2af, // (12,  687)  40
 0xa1f7, // (10,  503)  41
 0xcaaf, // (12, 2735)  42
 0x702b, // ( 7,   43)  43
 0xc6af, // (12, 1711)  44
 0x91d7, // ( 9,  471)  45
 0xceaf, // (12, 3759)  46
 0x6033, // ( 6,   51)  47
 0xc1af, // (12,  431)  48
 0xb4cf, // (11, 1231)  49
 0xc9af, // (12, 2479)  50
 0x80c7, // ( 8,  199)  51
 0xc5af, // (12, 1455)  52
 0xa3f7, // (10, 1015)  53
 0xcdaf, // (12, 3503)  54
 0x706b, // ( 7,  107)  55
 0xc3af, // (12,  943)  56
 0xb2cf, // (11,  719)  57
 0xcbaf, // (12, 2991)  58
 0x9037, // ( 9,   55)  59
 0xc7af, // (12, 1967)  60
 0xb6cf, // (11, 1743)  61
 0xcfaf, // (12, 4015)  62
 0x8027, // ( 8,   39)  63
 0xc06f, // (12,  111)  64
 0xa00f, // (10,   15)  65
 0xc86f, // (12, 2159)  66
 0x701b, // ( 7,   27)  67
 0xc46f, // (12, 1135)  68
 0x9137, // ( 9,  311)  69
 0xcc6f, // (12, 3183)  70
 0x705b, // ( 7,   91)  71
 0xc26f, // (12,  623)  72
 0xb1cf, // (11,  463)  73
 0xca6f, // (12, 2671)  74
 0x80a7, // ( 8,  167)  75
 0xc66f, // (12, 1647)  76
 0xa20f, // (10,  527)  77
 0xce6f, // (12, 3695)  78
 0x703b, // ( 7,   59)  79
 0xc16f, // (12,  367)  80
 0xc96f, // (12, 2415)  81
 0xc56f, // (12, 1391)  82
 0x90b7, // ( 9,  183)  83
 0xcd6f, // (12, 3439)  84
 0xb5cf, // (11, 1487)  85
 0xc36f, // (12,  879)  86
 0x8067, // ( 8,  103)  87
 0xcb6f, // (12, 2927)  88
 0xc76f, // (12, 1903)  89
 0xcf6f, // (12, 3951)  90
 0xa10f, // (10,  271)  91
 0xc0ef, // (12,  239)  92
 0xc8ef, // (12, 2287)  93
 0xc4ef, // (12, 1263)  94
 0x91b7, // ( 9,  439)  95
 0xccef, // (12, 3311)  96
 0xc2ef, // (12,  751)  97
 0xcaef, // (12, 2799)  98
 0xa30f, // (10,  783)  99
 0xc6ef, // (12, 1775) 100
 0xceef, // (12, 3823) 101
 0xc1ef, // (12,  495) 102
 0xa08f, // (10,  143) 103
 0xc9ef, // (12, 2543) 104
 0xc5ef, // (12, 1519) 105
 0xcdef, // (12, 3567) 106
 0xb3cf, // (11,  975) 107
 0xc3ef, // (12, 1007) 108
 0xcbef, // (12, 3055) 109
 0xc7ef, // (12, 2031) 110
 0xa28f, // (10,  655) 111
 0xcfef, // (12, 4079) 112
 0xc01f, // (12,   31) 113
 0xc81f, // (12, 2079) 114
 0xc41f, // (12, 1055) 115
 0xcc1f, // (12, 3103) 116
 0xc21f, // (12,  543) 117
 0xca1f, // (12, 2591) 118
 0xb7cf, // (11, 1999) 119
 0xc61f, // (12, 1567) 120
 0xce1f, // (12, 3615) 121
 0xc11f, // (12,  287) 122
 0xc91f, // (12, 2335) 123
 0xc51f, // (12, 1311) 124
 0xcd1f, // (12, 3359) 125
 0xc31f, // (12,  799) 126
 0xcb1f, // (12, 2847) 127
 0xc71f, // (12, 1823) 128
 0xb02f, // (11,   47) 129
 0xcf1f, // (12, 3871) 130
 0x80e7, // ( 8,  231) 131
 0xc09f, // (12,  159) 132
 0xa18f, // (10,  399) 133
 0xc89f, // (12, 2207) 134
 0x8017, // ( 8,   23) 135
 0xc49f, // (12, 1183) 136
 0xcc9f, // (12, 3231) 137
 0xc29f, // (12,  671) 138
 0x9077, // ( 9,  119) 139
 0xca9f, // (12, 2719) 140
 0xb42f, // (11, 1071) 141
 0xc69f, // (12, 1695) 142
 0x8097, // ( 8,  151) 143
 0xce9f, // (12, 3743) 144
 0xc19f, // (12,  415) 145
 0xc99f, // (12, 2463) 146
 0xa38f, // (10,  911) 147
 0xc59f, // (12, 1439) 148
 0xcd9f, // (12, 3487) 149
 0xc39f, // (12,  927) 150
 0x9177, // ( 9,  375) 151
 0xcb9f, // (12, 2975) 152
 0xc79f, // (12, 1951) 153
 0xcf9f, // (12, 3999) 154
 0xb22f, // (11,  559) 155
 0xc05f, // (12,   95) 156
 0xc85f, // (12, 2143) 157
 0xc45f, // (12, 1119) 158
 0xa04f, // (10,   79) 159
 0xcc5f, // (12, 3167) 160
 0xc25f, // (12,  607) 161
 0xca5f, // (12, 2655) 162
 0xb62f, // (11, 1583) 163
 0xc65f, // (12, 1631) 164
 0xce5f, // (12, 3679) 165
 0xc15f, // (12,  351) 166
 0xb12f, // (11,  303) 167
 0xc95f, // (12, 2399) 168
 0xc55f, // (12, 1375) 169
 0xcd5f, // (12, 3423) 170
 0xc35f, // (12,  863) 171
 0xcb5f, // (12, 2911) 172
 0xc75f, // (12, 1887) 173
 0xcf5f, // (12, 3935) 174
 0xb52f, // (11, 1327) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 5 of 22) (steady 5 of 16) (phase = 0.343750000 = 11.0 / 32.0)
 // entropy:    4.4308578632493116345
 // avg_length: 4.4996166821663301505; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0xc12f, // (12,  303)   0
 0x601d, // ( 6,   29)   1
 0x9057, // ( 9,   87)   2
 0x3002, // ( 3,    2)   3
 0xb14f, // (11,  335)   4
 0x5005, // ( 5,    5)   5
 0x807b, // ( 8,  123)   6
 0x2000, // ( 2,    0)   7
 0xc92f, // (12, 2351)   8
 0x700b, // ( 7,   11)   9
 0xa1f7, // (10,  503)  10
 0x4001, // ( 4,    1)  11
 0xc52f, // (12, 1327)  12
 0x603d, // ( 6,   61)  13
 0x9157, // ( 9,  343)  14
 0x3006, // ( 3,    6)  15
 0xcd2f, // (12, 3375)  16
 0x80fb, // ( 8,  251)  17
 0xb54f, // (11, 1359)  18
 0x5015, // ( 5,   21)  19
 0xc32f, // (12,  815)  20
 0x704b, // ( 7,   75)  21
 0xa3f7, // (10, 1015)  22
 0x4009, // ( 4,    9)  23
 0xcb2f, // (12, 2863)  24
 0x8007, // ( 8,    7)  25
 0xc72f, // (12, 1839)  26
 0x6003, // ( 6,    3)  27
 0xcf2f, // (12, 3887)  28
 0x8087, // ( 8,  135)  29
 0xb34f, // (11,  847)  30
 0x500d, // ( 5,   13)  31
 0xc0af, // (12,  175)  32
 0x90d7, // ( 9,  215)  33
 0xc8af, // (12, 2223)  34
 0x6023, // ( 6,   35)  35
 0xc4af, // (12, 1199)  36
 0x8047, // ( 8,   71)  37
 0xb74f, // (11, 1871)  38
 0x6013, // ( 6,   19)  39
 0xccaf, // (12, 3247)  40
 0xa00f, // (10,   15)  41
 0xc2af, // (12,  687)  42
 0x702b, // ( 7,   43)  43
 0xcaaf, // (12, 2735)  44
 0x91d7, // ( 9,  471)  45
 0xc6af, // (12, 1711)  46
 0x6033, // ( 6,   51)  47
 0xceaf, // (12, 3759)  48
 0xb0cf, // (11,  207)  49
 0xc1af, // (12,  431)  50
 0x80c7, // ( 8,  199)  51
 0xc9af, // (12, 2479)  52
 0xa20f, // (10,  527)  53
 0xc5af, // (12, 1455)  54
 0x706b, // ( 7,  107)  55
 0xcdaf, // (12, 3503)  56
 0xc3af, // (12,  943)  57
 0xcbaf, // (12, 2991)  58
 0x9037, // ( 9,   55)  59
 0xc7af, // (12, 1967)  60
 0xb4cf, // (11, 1231)  61
 0xcfaf, // (12, 4015)  62
 0x8027, // ( 8,   39)  63
 0xc06f, // (12,  111)  64
 0xa10f, // (10,  271)  65
 0xc86f, // (12, 2159)  66
 0x701b, // ( 7,   27)  67
 0xc46f, // (12, 1135)  68
 0x9137, // ( 9,  311)  69
 0xcc6f, // (12, 3183)  70
 0x705b, // ( 7,   91)  71
 0xc26f, // (12,  623)  72
 0xb2cf, // (11,  719)  73
 0xca6f, // (12, 2671)  74
 0x80a7, // ( 8,  167)  75
 0xc66f, // (12, 1647)  76
 0xa30f, // (10,  783)  77
 0xce6f, // (12, 3695)  78
 0x703b, // ( 7,   59)  79
 0xc16f, // (12,  367)  80
 0xc96f, // (12, 2415)  81
 0xc56f, // (12, 1391)  82
 0x90b7, // ( 9,  183)  83
 0xcd6f, // (12, 3439)  84
 0xb6cf, // (11, 1743)  85
 0xc36f, // (12,  879)  86
 0x8067, // ( 8,  103)  87
 0xcb6f, // (12, 2927)  88
 0xc76f, // (12, 1903)  89
 0xcf6f, // (12, 3951)  90
 0xa08f, // (10,  143)  91
 0xc0ef, // (12,  239)  92
 0xc8ef, // (12, 2287)  93
 0xc4ef, // (12, 1263)  94
 0x91b7, // ( 9,  439)  95
 0xccef, // (12, 3311)  96
 0xc2ef, // (12,  751)  97
 0xcaef, // (12, 2799)  98
 0xa28f, // (10,  655)  99
 0xc6ef, // (12, 1775) 100
 0xceef, // (12, 3823) 101
 0xc1ef, // (12,  495) 102
 0x9077, // ( 9,  119) 103
 0xc9ef, // (12, 2543) 104
 0xc5ef, // (12, 1519) 105
 0xcdef, // (12, 3567) 106
 0xb1cf, // (11,  463) 107
 0xc3ef, // (12, 1007) 108
 0xcbef, // (12, 3055) 109
 0xc7ef, // (12, 2031) 110
 0xa18f, // (10,  399) 111
 0xcfef, // (12, 4079) 112
 0xc01f, // (12,   31) 113
 0xc81f, // (12, 2079) 114
 0xc41f, // (12, 1055) 115
 0xcc1f, // (12, 3103) 116
 0xc21f, // (12,  543) 117
 0xca1f, // (12, 2591) 118
 0xb5cf, // (11, 1487) 119
 0xc61f, // (12, 1567) 120
 0xce1f, // (12, 3615) 121
 0xc11f, // (12,  287) 122
 0xc91f, // (12, 2335) 123
 0xc51f, // (12, 1311) 124
 0xcd1f, // (12, 3359) 125
 0xc31f, // (12,  799) 126
 0xcb1f, // (12, 2847) 127
 0xc71f, // (12, 1823) 128
 0xb3cf, // (11,  975) 129
 0xcf1f, // (12, 3871) 130
 0x80e7, // ( 8,  231) 131
 0xc09f, // (12,  159) 132
 0xa38f, // (10,  911) 133
 0xc89f, // (12, 2207) 134
 0x8017, // ( 8,   23) 135
 0xc49f, // (12, 1183) 136
 0xcc9f, // (12, 3231) 137
 0xc29f, // (12,  671) 138
 0x9177, // ( 9,  375) 139
 0xca9f, // (12, 2719) 140
 0xb7cf, // (11, 1999) 141
 0xc69f, // (12, 1695) 142
 0x8097, // ( 8,  151) 143
 0xce9f, // (12, 3743) 144
 0xc19f, // (12,  415) 145
 0xc99f, // (12, 2463) 146
 0xa04f, // (10,   79) 147
 0xc59f, // (12, 1439) 148
 0xcd9f, // (12, 3487) 149
 0xc39f, // (12,  927) 150
 0x90f7, // ( 9,  247) 151
 0xcb9f, // (12, 2975) 152
 0xc79f, // (12, 1951) 153
 0xcf9f, // (12, 3999) 154
 0xb02f, // (11,   47) 155
 0xc05f, // (12,   95) 156
 0xc85f, // (12, 2143) 157
 0xc45f, // (12, 1119) 158
 0xa24f, // (10,  591) 159
 0xcc5f, // (12, 3167) 160
 0xc25f, // (12,  607) 161
 0xca5f, // (12, 2655) 162
 0xb42f, // (11, 1071) 163
 0xc65f, // (12, 1631) 164
 0xce5f, // (12, 3679) 165
 0xc15f, // (12,  351) 166
 0xb22f, // (11,  559) 167
 0xc95f, // (12, 2399) 168
 0xc55f, // (12, 1375) 169
 0xcd5f, // (12, 3423) 170
 0xc35f, // (12,  863) 171
 0xcb5f, // (12, 2911) 172
 0xc75f, // (12, 1887) 173
 0xcf5f, // (12, 3935) 174
 0xb62f, // (11, 1583) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 6 of 22) (steady 6 of 16) (phase = 0.406250000 = 13.0 / 32.0)
 // entropy:    4.4310364988500126060
 // avg_length: 4.5051134111084252254; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x601d, // ( 6,   29)   0
 0x3002, // ( 3,    2)   1
 0x603d, // ( 6,   61)   2
 0x2000, // ( 2,    0)   3
 0x700b, // ( 7,   11)   4
 0x4001, // ( 4,    1)   5
 0x6003, // ( 6,    3)   6
 0x3006, // ( 3,    6)   7
 0x807b, // ( 8,  123)   8
 0x5005, // ( 5,    5)   9
 0x704b, // ( 7,   75)  10
 0x4009, // ( 4,    9)  11
 0x9097, // ( 9,  151)  12
 0x6023, // ( 6,   35)  13
 0x80fb, // ( 8,  251)  14
 0x5015, // ( 5,   21)  15
 0x9197, // ( 9,  407)  16
 0x6013, // ( 6,   19)  17
 0x8007, // ( 8,    7)  18
 0x500d, // ( 5,   13)  19
 0xa0f7, // (10,  247)  20
 0x702b, // ( 7,   43)  21
 0x9057, // ( 9,   87)  22
 0x6033, // ( 6,   51)  23
 0xb14f, // (11,  335)  24
 0x8087, // ( 8,  135)  25
 0xa2f7, // (10,  759)  26
 0x706b, // ( 7,  107)  27
 0xb54f, // (11, 1359)  28
 0x9157, // ( 9,  343)  29
 0xa1f7, // (10,  503)  30
 0x8047, // ( 8,   71)  31
 0xa3f7, // (10, 1015)  32
 0x701b, // ( 7,   27)  33
 0x90d7, // ( 9,  215)  34
 0x705b, // ( 7,   91)  35
 0xb34f, // (11,  847)  36
 0x80c7, // ( 8,  199)  37
 0xa00f, // (10,   15)  38
 0x703b, // ( 7,   59)  39
 0xc32f, // (12,  815)  40
 0x91d7, // ( 9,  471)  41
 0xb74f, // (11, 1871)  42
 0x8027, // ( 8,   39)  43
 0xcb2f, // (12, 2863)  44
 0xa20f, // (10,  527)  45
 0xb0cf, // (11,  207)  46
 0x9037, // ( 9,   55)  47
 0xc72f, // (12, 1839)  48
 0xa10f, // (10,  271)  49
 0xcf2f, // (12, 3887)  50
 0x9137, // ( 9,  311)  51
 0xc0af, // (12,  175)  52
 0xb4cf, // (11, 1231)  53
 0xc8af, // (12, 2223)  54
 0xa30f, // (10,  783)  55
 0xc4af, // (12, 1199)  56
 0xccaf, // (12, 3247)  57
 0xc2af, // (12,  687)  58
 0xb2cf, // (11,  719)  59
 0xcaaf, // (12, 2735)  60
 0xc6af, // (12, 1711)  61
 0xceaf, // (12, 3759)  62
 0xb6cf, // (11, 1743)  63
 0xb1cf, // (11,  463)  64
 0x80a7, // ( 8,  167)  65
 0xa08f, // (10,  143)  66
 0x8067, // ( 8,  103)  67
 0xc1af, // (12,  431)  68
 0x90b7, // ( 9,  183)  69
 0xb5cf, // (11, 1487)  70
 0x80e7, // ( 8,  231)  71
 0xc9af, // (12, 2479)  72
 0xa28f, // (10,  655)  73
 0xc5af, // (12, 1455)  74
 0x91b7, // ( 9,  439)  75
 0xcdaf, // (12, 3503)  76
 0xb3cf, // (11,  975)  77
 0xc3af, // (12,  943)  78
 0xa18f, // (10,  399)  79
 0xcbaf, // (12, 2991)  80
 0xb7cf, // (11, 1999)  81
 0xc7af, // (12, 1967)  82
 0xa38f, // (10,  911)  83
 0xcfaf, // (12, 4015)  84
 0xc06f, // (12,  111)  85
 0xc86f, // (12, 2159)  86
 0xb02f, // (11,   47)  87
 0xc46f, // (12, 1135)  88
 0xcc6f, // (12, 3183)  89
 0xc26f, // (12,  623)  90
 0xca6f, // (12, 2671)  91
 0xc66f, // (12, 1647)  92
 0xce6f, // (12, 3695)  93
 0xc16f, // (12,  367)  94
 0xc96f, // (12, 2415)  95
 0xc56f, // (12, 1391)  96
 0xcd6f, // (12, 3439)  97
 0xc36f, // (12,  879)  98
 0xb42f, // (11, 1071)  99
 0xcb6f, // (12, 2927) 100
 0xc76f, // (12, 1903) 101
 0xcf6f, // (12, 3951) 102
 0xc0ef, // (12,  239) 103
 0xc8ef, // (12, 2287) 104
 0xc4ef, // (12, 1263) 105
 0xccef, // (12, 3311) 106
 0xc2ef, // (12,  751) 107
 0xcaef, // (12, 2799) 108
 0xc6ef, // (12, 1775) 109
 0xceef, // (12, 3823) 110
 0xc1ef, // (12,  495) 111
 0xc9ef, // (12, 2543) 112
 0xc5ef, // (12, 1519) 113
 0xcdef, // (12, 3567) 114
 0xc3ef, // (12, 1007) 115
 0xcbef, // (12, 3055) 116
 0xc7ef, // (12, 2031) 117
 0xcfef, // (12, 4079) 118
 0xc01f, // (12,   31) 119
 0xc81f, // (12, 2079) 120
 0xc41f, // (12, 1055) 121
 0xcc1f, // (12, 3103) 122
 0xc21f, // (12,  543) 123
 0xca1f, // (12, 2591) 124
 0xc61f, // (12, 1567) 125
 0xce1f, // (12, 3615) 126
 0xc11f, // (12,  287) 127
 0xc91f, // (12, 2335) 128
 0x9077, // ( 9,  119) 129
 0xb22f, // (11,  559) 130
 0x8017, // ( 8,   23) 131
 0xc51f, // (12, 1311) 132
 0xa04f, // (10,   79) 133
 0xcd1f, // (12, 3359) 134
 0x9177, // ( 9,  375) 135
 0xc31f, // (12,  799) 136
 0xb62f, // (11, 1583) 137
 0xcb1f, // (12, 2847) 138
 0xa24f, // (10,  591) 139
 0xc71f, // (12, 1823) 140
 0xcf1f, // (12, 3871) 141
 0xc09f, // (12,  159) 142
 0xb12f, // (11,  303) 143
 0xc89f, // (12, 2207) 144
 0xc49f, // (12, 1183) 145
 0xcc9f, // (12, 3231) 146
 0xb52f, // (11, 1327) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 7 of 22) (steady 7 of 16) (phase = 0.468750000 = 15.0 / 32.0)
 // entropy:    4.4417871821766841123
 // avg_length: 4.5206419191518980583; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x700b, // ( 7,   11)   0
 0x3002, // ( 3,    2)   1
 0x601d, // ( 6,   29)   2
 0x2000, // ( 2,    0)   3
 0x704b, // ( 7,   75)   4
 0x4001, // ( 4,    1)   5
 0x603d, // ( 6,   61)   6
 0x3006, // ( 3,    6)   7
 0x8007, // ( 8,    7)   8
 0x5005, // ( 5,    5)   9
 0x702b, // ( 7,   43)  10
 0x4009, // ( 4,    9)  11
 0x9097, // ( 9,  151)  12
 0x6003, // ( 6,    3)  13
 0x8087, // ( 8,  135)  14
 0x5015, // ( 5,   21)  15
 0x9197, // ( 9,  407)  16
 0x6023, // ( 6,   35)  17
 0x8047, // ( 8,   71)  18
 0x500d, // ( 5,   13)  19
 0xa0f7, // (10,  247)  20
 0x706b, // ( 7,  107)  21
 0x9057, // ( 9,   87)  22
 0x6013, // ( 6,   19)  23
 0xb14f, // (11,  335)  24
 0x80c7, // ( 8,  199)  25
 0xa2f7, // (10,  759)  26
 0x701b, // ( 7,   27)  27
 0xc52f, // (12, 1327)  28
 0x9157, // ( 9,  343)  29
 0xb54f, // (11, 1359)  30
 0x8027, // ( 8,   39)  31
 0xa1f7, // (10,  503)  32
 0x705b, // ( 7,   91)  33
 0x90d7, // ( 9,  215)  34
 0x6033, // ( 6,   51)  35
 0xb34f, // (11,  847)  36
 0x80a7, // ( 8,  167)  37
 0xa3f7, // (10, 1015)  38
 0x703b, // ( 7,   59)  39
 0xcd2f, // (12, 3375)  40
 0x91d7, // ( 9,  471)  41
 0xb74f, // (11, 1871)  42
 0x8067, // ( 8,  103)  43
 0xc32f, // (12,  815)  44
 0xa00f, // (10,   15)  45
 0xcb2f, // (12, 2863)  46
 0x9037, // ( 9,   55)  47
 0xc72f, // (12, 1839)  48
 0xa20f, // (10,  527)  49
 0xcf2f, // (12, 3887)  50
 0x9137, // ( 9,  311)  51
 0xc0af, // (12,  175)  52
 0xb0cf, // (11,  207)  53
 0xc8af, // (12, 2223)  54
 0xa10f, // (10,  271)  55
 0xc4af, // (12, 1199)  56
 0xccaf, // (12, 3247)  57
 0xc2af, // (12,  687)  58
 0xb4cf, // (11, 1231)  59
 0xcaaf, // (12, 2735)  60
 0xc6af, // (12, 1711)  61
 0xceaf, // (12, 3759)  62
 0xb2cf, // (11,  719)  63
 0xb6cf, // (11, 1743)  64
 0x80e7, // ( 8,  231)  65
 0xa30f, // (10,  783)  66
 0x707b, // ( 7,  123)  67
 0xc1af, // (12,  431)  68
 0x90b7, // ( 9,  183)  69
 0xb1cf, // (11,  463)  70
 0x8017, // ( 8,   23)  71
 0xc9af, // (12, 2479)  72
 0xa08f, // (10,  143)  73
 0xc5af, // (12, 1455)  74
 0x91b7, // ( 9,  439)  75
 0xcdaf, // (12, 3503)  76
 0xb5cf, // (11, 1487)  77
 0xc3af, // (12,  943)  78
 0xa28f, // (10,  655)  79
 0xcbaf, // (12, 2991)  80
 0xb3cf, // (11,  975)  81
 0xc7af, // (12, 1967)  82
 0xa18f, // (10,  399)  83
 0xcfaf, // (12, 4015)  84
 0xc06f, // (12,  111)  85
 0xc86f, // (12, 2159)  86
 0xb7cf, // (11, 1999)  87
 0xc46f, // (12, 1135)  88
 0xcc6f, // (12, 3183)  89
 0xc26f, // (12,  623)  90
 0xca6f, // (12, 2671)  91
 0xc66f, // (12, 1647)  92
 0xce6f, // (12, 3695)  93
 0xc16f, // (12,  367)  94
 0xc96f, // (12, 2415)  95
 0xc56f, // (12, 1391)  96
 0xcd6f, // (12, 3439)  97
 0xc36f, // (12,  879)  98
 0xb02f, // (11,   47)  99
 0xcb6f, // (12, 2927) 100
 0xc76f, // (12, 1903) 101
 0xcf6f, // (12, 3951) 102
 0xc0ef, // (12,  239) 103
 0xc8ef, // (12, 2287) 104
 0xc4ef, // (12, 1263) 105
 0xccef, // (12, 3311) 106
 0xc2ef, // (12,  751) 107
 0xcaef, // (12, 2799) 108
 0xc6ef, // (12, 1775) 109
 0xceef, // (12, 3823) 110
 0xc1ef, // (12,  495) 111
 0xc9ef, // (12, 2543) 112
 0xc5ef, // (12, 1519) 113
 0xcdef, // (12, 3567) 114
 0xc3ef, // (12, 1007) 115
 0xcbef, // (12, 3055) 116
 0xc7ef, // (12, 2031) 117
 0xcfef, // (12, 4079) 118
 0xc01f, // (12,   31) 119
 0xc81f, // (12, 2079) 120
 0xc41f, // (12, 1055) 121
 0xcc1f, // (12, 3103) 122
 0xc21f, // (12,  543) 123
 0xca1f, // (12, 2591) 124
 0xc61f, // (12, 1567) 125
 0xce1f, // (12, 3615) 126
 0xc11f, // (12,  287) 127
 0xc91f, // (12, 2335) 128
 0xa38f, // (10,  911) 129
 0xb42f, // (11, 1071) 130
 0x9077, // ( 9,  119) 131
 0xc51f, // (12, 1311) 132
 0xa04f, // (10,   79) 133
 0xcd1f, // (12, 3359) 134
 0x9177, // ( 9,  375) 135
 0xc31f, // (12,  799) 136
 0xb22f, // (11,  559) 137
 0xcb1f, // (12, 2847) 138
 0xa24f, // (10,  591) 139
 0xc71f, // (12, 1823) 140
 0xcf1f, // (12, 3871) 141
 0xc09f, // (12,  159) 142
 0xb62f, // (11, 1583) 143
 0xc89f, // (12, 2207) 144
 0xc49f, // (12, 1183) 145
 0xcc9f, // (12, 3231) 146
 0xb12f, // (11,  303) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 8 of 22) (steady 8 of 16) (phase = 0.531250000 = 17.0 / 32.0)
 // entropy:    4.4505873338397474726
 // avg_length: 4.5270058771550303334; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x7033, // ( 7,   51)   0
 0x4006, // ( 4,    6)   1
 0x601d, // ( 6,   29)   2
 0x2000, // ( 2,    0)   3
 0x7073, // ( 7,  115)   4
 0x400e, // ( 4,   14)   5
 0x603d, // ( 6,   61)   6
 0x3002, // ( 3,    2)   7
 0x807b, // ( 8,  123)   8
 0x5005, // ( 5,    5)   9
 0x700b, // ( 7,   11)  10
 0x4001, // ( 4,    1)  11
 0x9097, // ( 9,  151)  12
 0x5015, // ( 5,   21)  13
 0x80fb, // ( 8,  251)  14
 0x4009, // ( 4,    9)  15
 0xa0f7, // (10,  247)  16
 0x6003, // ( 6,    3)  17
 0x8007, // ( 8,    7)  18
 0x500d, // ( 5,   13)  19
 0xa2f7, // (10,  759)  20
 0x704b, // ( 7,   75)  21
 0x9197, // ( 9,  407)  22
 0x6023, // ( 6,   35)  23
 0xb34f, // (11,  847)  24
 0x8087, // ( 8,  135)  25
 0xa1f7, // (10,  503)  26
 0x702b, // ( 7,   43)  27
 0xb74f, // (11, 1871)  28
 0x8047, // ( 8,   71)  29
 0xa3f7, // (10, 1015)  30
 0x706b, // ( 7,  107)  31
 0xb0cf, // (11,  207)  32
 0x701b, // ( 7,   27)  33
 0x9057, // ( 9,   87)  34
 0x6013, // ( 6,   19)  35
 0xb4cf, // (11, 1231)  36
 0x80c7, // ( 8,  199)  37
 0xa00f, // (10,   15)  38
 0x705b, // ( 7,   91)  39
 0xc72f, // (12, 1839)  40
 0x9157, // ( 9,  343)  41
 0xb2cf, // (11,  719)  42
 0x8027, // ( 8,   39)  43
 0xcf2f, // (12, 3887)  44
 0x90d7, // ( 9,  215)  45
 0xb6cf, // (11, 1743)  46
 0x80a7, // ( 8,  167)  47
 0xc0af, // (12,  175)  48
 0xa20f, // (10,  527)  49
 0xc8af, // (12, 2223)  50
 0x91d7, // ( 9,  471)  51
 0xc4af, // (12, 1199)  52
 0xa10f, // (10,  271)  53
 0xccaf, // (12, 3247)  54
 0x9037, // ( 9,   55)  55
 0xc2af, // (12,  687)  56
 0xcaaf, // (12, 2735)  57
 0xc6af, // (12, 1711)  58
 0xb1cf, // (11,  463)  59
 0xceaf, // (12, 3759)  60
 0xc1af, // (12,  431)  61
 0xc9af, // (12, 2479)  62
 0xb5cf, // (11, 1487)  63
 0xc5af, // (12, 1455)  64
 0x8067, // ( 8,  103)  65
 0xa30f, // (10,  783)  66
 0x703b, // ( 7,   59)  67
 0xcdaf, // (12, 3503)  68
 0x9137, // ( 9,  311)  69
 0xb3cf, // (11,  975)  70
 0x80e7, // ( 8,  231)  71
 0xc3af, // (12,  943)  72
 0xa08f, // (10,  143)  73
 0xcbaf, // (12, 2991)  74
 0x90b7, // ( 9,  183)  75
 0xc7af, // (12, 1967)  76
 0xa28f, // (10,  655)  77
 0xcfaf, // (12, 4015)  78
 0x91b7, // ( 9,  439)  79
 0xc06f, // (12,  111)  80
 0xb7cf, // (11, 1999)  81
 0xc86f, // (12, 2159)  82
 0xa18f, // (10,  399)  83
 0xc46f, // (12, 1135)  84
 0xb02f, // (11,   47)  85
 0xcc6f, // (12, 3183)  86
 0xa38f, // (10,  911)  87
 0xc26f, // (12,  623)  88
 0xca6f, // (12, 2671)  89
 0xc66f, // (12, 1647)  90
 0xce6f, // (12, 3695)  91
 0xc16f, // (12,  367)  92
 0xc96f, // (12, 2415)  93
 0xc56f, // (12, 1391)  94
 0xcd6f, // (12, 3439)  95
 0xc36f, // (12,  879)  96
 0xcb6f, // (12, 2927)  97
 0xc76f, // (12, 1903)  98
 0xb42f, // (11, 1071)  99
 0xcf6f, // (12, 3951) 100
 0xc0ef, // (12,  239) 101
 0xc8ef, // (12, 2287) 102
 0xb22f, // (11,  559) 103
 0xc4ef, // (12, 1263) 104
 0xccef, // (12, 3311) 105
 0xc2ef, // (12,  751) 106
 0xcaef, // (12, 2799) 107
 0xc6ef, // (12, 1775) 108
 0xceef, // (12, 3823) 109
 0xc1ef, // (12,  495) 110
 0xc9ef, // (12, 2543) 111
 0xc5ef, // (12, 1519) 112
 0xcdef, // (12, 3567) 113
 0xc3ef, // (12, 1007) 114
 0xcbef, // (12, 3055) 115
 0xc7ef, // (12, 2031) 116
 0xcfef, // (12, 4079) 117
 0xc01f, // (12,   31) 118
 0xc81f, // (12, 2079) 119
 0xc41f, // (12, 1055) 120
 0xcc1f, // (12, 3103) 121
 0xc21f, // (12,  543) 122
 0xca1f, // (12, 2591) 123
 0xc61f, // (12, 1567) 124
 0xce1f, // (12, 3615) 125
 0xc11f, // (12,  287) 126
 0xc91f, // (12, 2335) 127
 0xc51f, // (12, 1311) 128
 0x9077, // ( 9,  119) 129
 0xcd1f, // (12, 3359) 130
 0x8017, // ( 8,   23) 131
 0xc31f, // (12,  799) 132
 0xa04f, // (10,   79) 133
 0xcb1f, // (12, 2847) 134
 0x9177, // ( 9,  375) 135
 0xc71f, // (12, 1823) 136
 0xb62f, // (11, 1583) 137
 0xcf1f, // (12, 3871) 138
 0xa24f, // (10,  591) 139
 0xc09f, // (12,  159) 140
 0xb12f, // (11,  303) 141
 0xc89f, // (12, 2207) 142
 0xa14f, // (10,  335) 143
 0xc49f, // (12, 1183) 144
 0xcc9f, // (12, 3231) 145
 0xc29f, // (12,  671) 146
 0xb52f, // (11, 1327) 147
 0xca9f, // (12, 2719) 148
 0xc69f, // (12, 1695) 149
 0xce9f, // (12, 3743) 150
 0xb32f, // (11,  815) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 9 of 22) (steady 9 of 16) (phase = 0.593750000 = 19.0 / 32.0)
 // entropy:    4.4575203029748040606
 // avg_length: 4.5315465600684730063; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x7033, // ( 7,   51)   0
 0x4006, // ( 4,    6)   1
 0x601d, // ( 6,   29)   2
 0x2000, // ( 2,    0)   3
 0x7073, // ( 7,  115)   4
 0x400e, // ( 4,   14)   5
 0x603d, // ( 6,   61)   6
 0x3002, // ( 3,    2)   7
 0x9097, // ( 9,  151)   8
 0x5005, // ( 5,    5)   9
 0x700b, // ( 7,   11)  10
 0x4001, // ( 4,    1)  11
 0x9197, // ( 9,  407)  12
 0x6003, // ( 6,    3)  13
 0x807b, // ( 8,  123)  14
 0x4009, // ( 4,    9)  15
 0xa0f7, // (10,  247)  16
 0x6023, // ( 6,   35)  17
 0x80fb, // ( 8,  251)  18
 0x5015, // ( 5,   21)  19
 0xa2f7, // (10,  759)  20
 0x704b, // ( 7,   75)  21
 0x9057, // ( 9,   87)  22
 0x500d, // ( 5,   13)  23
 0xb34f, // (11,  847)  24
 0x8007, // ( 8,    7)  25
 0xa1f7, // (10,  503)  26
 0x702b, // ( 7,   43)  27
 0xc72f, // (12, 1839)  28
 0x8087, // ( 8,  135)  29
 0xa3f7, // (10, 1015)  30
 0x706b, // ( 7,  107)  31
 0xb74f, // (11, 1871)  32
 0x701b, // ( 7,   27)  33
 0x9157, // ( 9,  343)  34
 0x6013, // ( 6,   19)  35
 0xb0cf, // (11,  207)  36
 0x8047, // ( 8,   71)  37
 0xa00f, // (10,   15)  38
 0x705b, // ( 7,   91)  39
 0xcf2f, // (12, 3887)  40
 0x90d7, // ( 9,  215)  41
 0xb4cf, // (11, 1231)  42
 0x80c7, // ( 8,  199)  43
 0xc0af, // (12,  175)  44
 0x91d7, // ( 9,  471)  45
 0xb2cf, // (11,  719)  46
 0x8027, // ( 8,   39)  47
 0xc8af, // (12, 2223)  48
 0xa20f, // (10,  527)  49
 0xc4af, // (12, 1199)  50
 0x9037, // ( 9,   55)  51
 0xccaf, // (12, 3247)  52
 0xa10f, // (10,  271)  53
 0xc2af, // (12,  687)  54
 0x9137, // ( 9,  311)  55
 0xcaaf, // (12, 2735)  56
 0xc6af, // (12, 1711)  57
 0xceaf, // (12, 3759)  58
 0xa30f, // (10,  783)  59
 0xc1af, // (12,  431)  60
 0xc9af, // (12, 2479)  61
 0xc5af, // (12, 1455)  62
 0xb6cf, // (11, 1743)  63
 0xcdaf, // (12, 3503)  64
 0x80a7, // ( 8,  167)  65
 0xb1cf, // (11,  463)  66
 0x703b, // ( 7,   59)  67
 0xc3af, // (12,  943)  68
 0x90b7, // ( 9,  183)  69
 0xb5cf, // (11, 1487)  70
 0x8067, // ( 8,  103)  71
 0xcbaf, // (12, 2991)  72
 0xa08f, // (10,  143)  73
 0xc7af, // (12, 1967)  74
 0x91b7, // ( 9,  439)  75
 0xcfaf, // (12, 4015)  76
 0xa28f, // (10,  655)  77
 0xc06f, // (12,  111)  78
 0x9077, // ( 9,  119)  79
 0xc86f, // (12, 2159)  80
 0xb3cf, // (11,  975)  81
 0xc46f, // (12, 1135)  82
 0xa18f, // (10,  399)  83
 0xcc6f, // (12, 3183)  84
 0xb7cf, // (11, 1999)  85
 0xc26f, // (12,  623)  86
 0xa38f, // (10,  911)  87
 0xca6f, // (12, 2671)  88
 0xc66f, // (12, 1647)  89
 0xce6f, // (12, 3695)  90
 0xb02f, // (11,   47)  91
 0xc16f, // (12,  367)  92
 0xc96f, // (12, 2415)  93
 0xc56f, // (12, 1391)  94
 0xcd6f, // (12, 3439)  95
 0xc36f, // (12,  879)  96
 0xcb6f, // (12, 2927)  97
 0xc76f, // (12, 1903)  98
 0xb42f, // (11, 1071)  99
 0xcf6f, // (12, 3951) 100
 0xc0ef, // (12,  239) 101
 0xc8ef, // (12, 2287) 102
 0xb22f, // (11,  559) 103
 0xc4ef, // (12, 1263) 104
 0xccef, // (12, 3311) 105
 0xc2ef, // (12,  751) 106
 0xcaef, // (12, 2799) 107
 0xc6ef, // (12, 1775) 108
 0xceef, // (12, 3823) 109
 0xc1ef, // (12,  495) 110
 0xc9ef, // (12, 2543) 111
 0xc5ef, // (12, 1519) 112
 0xcdef, // (12, 3567) 113
 0xc3ef, // (12, 1007) 114
 0xcbef, // (12, 3055) 115
 0xc7ef, // (12, 2031) 116
 0xcfef, // (12, 4079) 117
 0xc01f, // (12,   31) 118
 0xc81f, // (12, 2079) 119
 0xc41f, // (12, 1055) 120
 0xcc1f, // (12, 3103) 121
 0xc21f, // (12,  543) 122
 0xca1f, // (12, 2591) 123
 0xc61f, // (12, 1567) 124
 0xce1f, // (12, 3615) 125
 0xc11f, // (12,  287) 126
 0xc91f, // (12, 2335) 127
 0xc51f, // (12, 1311) 128
 0x9177, // ( 9,  375) 129
 0xcd1f, // (12, 3359) 130
 0x80e7, // ( 8,  231) 131
 0xc31f, // (12,  799) 132
 0xa04f, // (10,   79) 133
 0xcb1f, // (12, 2847) 134
 0x8017, // ( 8,   23) 135
 0xc71f, // (12, 1823) 136
 0xb62f, // (11, 1583) 137
 0xcf1f, // (12, 3871) 138
 0xa24f, // (10,  591) 139
 0xc09f, // (12,  159) 140
 0xb12f, // (11,  303) 141
 0xc89f, // (12, 2207) 142
 0xa14f, // (10,  335) 143
 0xc49f, // (12, 1183) 144
 0xcc9f, // (12, 3231) 145
 0xc29f, // (12,  671) 146
 0xb52f, // (11, 1327) 147
 0xca9f, // (12, 2719) 148
 0xc69f, // (12, 1695) 149
 0xce9f, // (12, 3743) 150
 0xb32f, // (11,  815) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 10 of 22) (steady 10 of 16) (phase = 0.656250000 = 21.0 / 32.0)
 // entropy:    4.4626765653088611430
 // avg_length: 4.5373141251902122661; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x700b, // ( 7,   11)   0
 0x4006, // ( 4,    6)   1
 0x601d, // ( 6,   29)   2
 0x2000, // ( 2,    0)   3
 0x807b, // ( 8,  123)   4
 0x400e, // ( 4,   14)   5
 0x603d, // ( 6,   61)   6
 0x3002, // ( 3,    2)   7
 0x9017, // ( 9,   23)   8
 0x5005, // ( 5,    5)   9
 0x704b, // ( 7,   75)  10
 0x4001, // ( 4,    1)  11
 0x9117, // ( 9,  279)  12
 0x6003, // ( 6,    3)  13
 0x80fb, // ( 8,  251)  14
 0x4009, // ( 4,    9)  15
 0xa177, // (10,  375)  16
 0x6023, // ( 6,   35)  17
 0x9097, // ( 9,  151)  18
 0x5015, // ( 5,   21)  19
 0xa377, // (10,  887)  20
 0x702b, // ( 7,   43)  21
 0x9197, // ( 9,  407)  22
 0x500d, // ( 5,   13)  23
 0xb34f, // (11,  847)  24
 0x8007, // ( 8,    7)  25
 0xa0f7, // (10,  247)  26
 0x706b, // ( 7,  107)  27
 0xc0af, // (12,  175)  28
 0x8087, // ( 8,  135)  29
 0xa2f7, // (10,  759)  30
 0x701b, // ( 7,   27)  31
 0xb74f, // (11, 1871)  32
 0x8047, // ( 8,   71)  33
 0xa1f7, // (10,  503)  34
 0x6013, // ( 6,   19)  35
 0xb0cf, // (11,  207)  36
 0x80c7, // ( 8,  199)  37
 0xa3f7, // (10, 1015)  38
 0x6033, // ( 6,   51)  39
 0xc8af, // (12, 2223)  40
 0x9057, // ( 9,   87)  41
 0xb4cf, // (11, 1231)  42
 0x8027, // ( 8,   39)  43
 0xc4af, // (12, 1199)  44
 0x9157, // ( 9,  343)  45
 0xb2cf, // (11,  719)  46
 0x80a7, // ( 8,  167)  47
 0xccaf, // (12, 3247)  48
 0xa00f, // (10,   15)  49
 0xc2af, // (12,  687)  50
 0x90d7, // ( 9,  215)  51
 0xcaaf, // (12, 2735)  52
 0xa20f, // (10,  527)  53
 0xc6af, // (12, 1711)  54
 0x91d7, // ( 9,  471)  55
 0xceaf, // (12, 3759)  56
 0xb6cf, // (11, 1743)  57
 0xc1af, // (12,  431)  58
 0xa10f, // (10,  271)  59
 0xc9af, // (12, 2479)  60
 0xc5af, // (12, 1455)  61
 0xcdaf, // (12, 3503)  62
 0xa30f, // (10,  783)  63
 0xc3af, // (12,  943)  64
 0x9037, // ( 9,   55)  65
 0xb1cf, // (11,  463)  66
 0x705b, // ( 7,   91)  67
 0xcbaf, // (12, 2991)  68
 0x9137, // ( 9,  311)  69
 0xb5cf, // (11, 1487)  70
 0x703b, // ( 7,   59)  71
 0xc7af, // (12, 1967)  72
 0xa08f, // (10,  143)  73
 0xcfaf, // (12, 4015)  74
 0x90b7, // ( 9,  183)  75
 0xc06f, // (12,  111)  76
 0xa28f, // (10,  655)  77
 0xc86f, // (12, 2159)  78
 0x91b7, // ( 9,  439)  79
 0xc46f, // (12, 1135)  80
 0xb3cf, // (11,  975)  81
 0xcc6f, // (12, 3183)  82
 0xa18f, // (10,  399)  83
 0xc26f, // (12,  623)  84
 0xb7cf, // (11, 1999)  85
 0xca6f, // (12, 2671)  86
 0xa38f, // (10,  911)  87
 0xc66f, // (12, 1647)  88
 0xce6f, // (12, 3695)  89
 0xc16f, // (12,  367)  90
 0xb02f, // (11,   47)  91
 0xc96f, // (12, 2415)  92
 0xc56f, // (12, 1391)  93
 0xcd6f, // (12, 3439)  94
 0xb42f, // (11, 1071)  95
 0xc36f, // (12,  879)  96
 0xcb6f, // (12, 2927)  97
 0xc76f, // (12, 1903)  98
 0xb22f, // (11,  559)  99
 0xcf6f, // (12, 3951) 100
 0xc0ef, // (12,  239) 101
 0xc8ef, // (12, 2287) 102
 0xb62f, // (11, 1583) 103
 0xc4ef, // (12, 1263) 104
 0xccef, // (12, 3311) 105
 0xc2ef, // (12,  751) 106
 0xcaef, // (12, 2799) 107
 0xc6ef, // (12, 1775) 108
 0xceef, // (12, 3823) 109
 0xc1ef, // (12,  495) 110
 0xc9ef, // (12, 2543) 111
 0xc5ef, // (12, 1519) 112
 0xcdef, // (12, 3567) 113
 0xc3ef, // (12, 1007) 114
 0xcbef, // (12, 3055) 115
 0xc7ef, // (12, 2031) 116
 0xcfef, // (12, 4079) 117
 0xc01f, // (12,   31) 118
 0xc81f, // (12, 2079) 119
 0xc41f, // (12, 1055) 120
 0xcc1f, // (12, 3103) 121
 0xc21f, // (12,  543) 122
 0xca1f, // (12, 2591) 123
 0xc61f, // (12, 1567) 124
 0xce1f, // (12, 3615) 125
 0xc11f, // (12,  287) 126
 0xc91f, // (12, 2335) 127
 0xc51f, // (12, 1311) 128
 0x9077, // ( 9,  119) 129
 0xcd1f, // (12, 3359) 130
 0x8067, // ( 8,  103) 131
 0xc31f, // (12,  799) 132
 0xa04f, // (10,   79) 133
 0xcb1f, // (12, 2847) 134
 0x80e7, // ( 8,  231) 135
 0xc71f, // (12, 1823) 136
 0xb12f, // (11,  303) 137
 0xcf1f, // (12, 3871) 138
 0xa24f, // (10,  591) 139
 0xc09f, // (12,  159) 140
 0xb52f, // (11, 1327) 141
 0xc89f, // (12, 2207) 142
 0xa14f, // (10,  335) 143
 0xc49f, // (12, 1183) 144
 0xcc9f, // (12, 3231) 145
 0xc29f, // (12,  671) 146
 0xb32f, // (11,  815) 147
 0xca9f, // (12, 2719) 148
 0xc69f, // (12, 1695) 149
 0xce9f, // (12, 3743) 150
 0xb72f, // (11, 1839) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 11 of 22) (steady 11 of 16) (phase = 0.718750000 = 23.0 / 32.0)
 // entropy:    4.4661524304421691411
 // avg_length: 4.5443750890419041255; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x803b, // ( 8,   59)   0
 0x4006, // ( 4,    6)   1
 0x601d, // ( 6,   29)   2
 0x2000, // ( 2,    0)   3
 0x80bb, // ( 8,  187)   4
 0x400e, // ( 4,   14)   5
 0x603d, // ( 6,   61)   6
 0x3002, // ( 3,    2)   7
 0x9017, // ( 9,   23)   8
 0x5005, // ( 5,    5)   9
 0x807b, // ( 8,  123)  10
 0x4001, // ( 4,    1)  11
 0x9117, // ( 9,  279)  12
 0x6003, // ( 6,    3)  13
 0x80fb, // ( 8,  251)  14
 0x4009, // ( 4,    9)  15
 0xa177, // (10,  375)  16
 0x6023, // ( 6,   35)  17
 0x9097, // ( 9,  151)  18
 0x5015, // ( 5,   21)  19
 0xa377, // (10,  887)  20
 0x702b, // ( 7,   43)  21
 0x9197, // ( 9,  407)  22
 0x500d, // ( 5,   13)  23
 0xb34f, // (11,  847)  24
 0x8007, // ( 8,    7)  25
 0xa0f7, // (10,  247)  26
 0x6013, // ( 6,   19)  27
 0xc0af, // (12,  175)  28
 0x8087, // ( 8,  135)  29
 0xa2f7, // (10,  759)  30
 0x706b, // ( 7,  107)  31
 0xb74f, // (11, 1871)  32
 0x8047, // ( 8,   71)  33
 0xa1f7, // (10,  503)  34
 0x6033, // ( 6,   51)  35
 0xb0cf, // (11,  207)  36
 0x80c7, // ( 8,  199)  37
 0xa3f7, // (10, 1015)  38
 0x600b, // ( 6,   11)  39
 0xc8af, // (12, 2223)  40
 0x9057, // ( 9,   87)  41
 0xb4cf, // (11, 1231)  42
 0x8027, // ( 8,   39)  43
 0xc4af, // (12, 1199)  44
 0x9157, // ( 9,  343)  45
 0xb2cf, // (11,  719)  46
 0x80a7, // ( 8,  167)  47
 0xccaf, // (12, 3247)  48
 0xa00f, // (10,   15)  49
 0xc2af, // (12,  687)  50
 0x90d7, // ( 9,  215)  51
 0xcaaf, // (12, 2735)  52
 0xa20f, // (10,  527)  53
 0xc6af, // (12, 1711)  54
 0x91d7, // ( 9,  471)  55
 0xceaf, // (12, 3759)  56
 0xb6cf, // (11, 1743)  57
 0xc1af, // (12,  431)  58
 0xa10f, // (10,  271)  59
 0xc9af, // (12, 2479)  60
 0xc5af, // (12, 1455)  61
 0xcdaf, // (12, 3503)  62
 0xa30f, // (10,  783)  63
 0xc3af, // (12,  943)  64
 0x9037, // ( 9,   55)  65
 0xb1cf, // (11,  463)  66
 0x701b, // ( 7,   27)  67
 0xcbaf, // (12, 2991)  68
 0x9137, // ( 9,  311)  69
 0xb5cf, // (11, 1487)  70
 0x705b, // ( 7,   91)  71
 0xc7af, // (12, 1967)  72
 0xa08f, // (10,  143)  73
 0xcfaf, // (12, 4015)  74
 0x90b7, // ( 9,  183)  75
 0xc06f, // (12,  111)  76
 0xa28f, // (10,  655)  77
 0xc86f, // (12, 2159)  78
 0x91b7, // ( 9,  439)  79
 0xc46f, // (12, 1135)  80
 0xb3cf, // (11,  975)  81
 0xcc6f, // (12, 3183)  82
 0xa18f, // (10,  399)  83
 0xc26f, // (12,  623)  84
 0xb7cf, // (11, 1999)  85
 0xca6f, // (12, 2671)  86
 0xa38f, // (10,  911)  87
 0xc66f, // (12, 1647)  88
 0xce6f, // (12, 3695)  89
 0xc16f, // (12,  367)  90
 0xb02f, // (11,   47)  91
 0xc96f, // (12, 2415)  92
 0xc56f, // (12, 1391)  93
 0xcd6f, // (12, 3439)  94
 0xb42f, // (11, 1071)  95
 0xc36f, // (12,  879)  96
 0xcb6f, // (12, 2927)  97
 0xc76f, // (12, 1903)  98
 0xb22f, // (11,  559)  99
 0xcf6f, // (12, 3951) 100
 0xc0ef, // (12,  239) 101
 0xc8ef, // (12, 2287) 102
 0xb62f, // (11, 1583) 103
 0xc4ef, // (12, 1263) 104
 0xccef, // (12, 3311) 105
 0xc2ef, // (12,  751) 106
 0xcaef, // (12, 2799) 107
 0xc6ef, // (12, 1775) 108
 0xceef, // (12, 3823) 109
 0xc1ef, // (12,  495) 110
 0xc9ef, // (12, 2543) 111
 0xc5ef, // (12, 1519) 112
 0xcdef, // (12, 3567) 113
 0xc3ef, // (12, 1007) 114
 0xcbef, // (12, 3055) 115
 0xc7ef, // (12, 2031) 116
 0xcfef, // (12, 4079) 117
 0xc01f, // (12,   31) 118
 0xc81f, // (12, 2079) 119
 0xc41f, // (12, 1055) 120
 0xcc1f, // (12, 3103) 121
 0xc21f, // (12,  543) 122
 0xca1f, // (12, 2591) 123
 0xc61f, // (12, 1567) 124
 0xce1f, // (12, 3615) 125
 0xc11f, // (12,  287) 126
 0xc91f, // (12, 2335) 127
 0xc51f, // (12, 1311) 128
 0xa04f, // (10,   79) 129
 0xcd1f, // (12, 3359) 130
 0x8067, // ( 8,  103) 131
 0xc31f, // (12,  799) 132
 0xa24f, // (10,  591) 133
 0xcb1f, // (12, 2847) 134
 0x80e7, // ( 8,  231) 135
 0xc71f, // (12, 1823) 136
 0xb12f, // (11,  303) 137
 0xcf1f, // (12, 3871) 138
 0x9077, // ( 9,  119) 139
 0xc09f, // (12,  159) 140
 0xb52f, // (11, 1327) 141
 0xc89f, // (12, 2207) 142
 0xa14f, // (10,  335) 143
 0xc49f, // (12, 1183) 144
 0xcc9f, // (12, 3231) 145
 0xc29f, // (12,  671) 146
 0xb32f, // (11,  815) 147
 0xca9f, // (12, 2719) 148
 0xc69f, // (12, 1695) 149
 0xce9f, // (12, 3743) 150
 0xb72f, // (11, 1839) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 12 of 22) (steady 12 of 16) (phase = 0.781250000 = 25.0 / 32.0)
 // entropy:    4.4680486273043946710
 // avg_length: 4.5521643785256946657; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x807b, // ( 8,  123)   0
 0x4006, // ( 4,    6)   1
 0x601d, // ( 6,   29)   2
 0x2000, // ( 2,    0)   3
 0x80fb, // ( 8,  251)   4
 0x400e, // ( 4,   14)   5
 0x700b, // ( 7,   11)   6
 0x3002, // ( 3,    2)   7
 0x9097, // ( 9,  151)   8
 0x5005, // ( 5,    5)   9
 0x8007, // ( 8,    7)  10
 0x4001, // ( 4,    1)  11
 0x9197, // ( 9,  407)  12
 0x603d, // ( 6,   61)  13
 0x8087, // ( 8,  135)  14
 0x4009, // ( 4,    9)  15
 0xa177, // (10,  375)  16
 0x704b, // ( 7,   75)  17
 0x9057, // ( 9,   87)  18
 0x5015, // ( 5,   21)  19
 0xb34f, // (11,  847)  20
 0x702b, // ( 7,   43)  21
 0x9157, // ( 9,  343)  22
 0x500d, // ( 5,   13)  23
 0xc72f, // (12, 1839)  24
 0x8047, // ( 8,   71)  25
 0xa377, // (10,  887)  26
 0x6003, // ( 6,    3)  27
 0xcf2f, // (12, 3887)  28
 0x80c7, // ( 8,  199)  29
 0xa0f7, // (10,  247)  30
 0x6023, // ( 6,   35)  31
 0xc0af, // (12,  175)  32
 0x8027, // ( 8,   39)  33
 0xa2f7, // (10,  759)  34
 0x6013, // ( 6,   19)  35
 0xc8af, // (12, 2223)  36
 0x80a7, // ( 8,  167)  37
 0xa1f7, // (10,  503)  38
 0x6033, // ( 6,   51)  39
 0xc4af, // (12, 1199)  40
 0x90d7, // ( 9,  215)  41
 0xb74f, // (11, 1871)  42
 0x706b, // ( 7,  107)  43
 0xccaf, // (12, 3247)  44
 0x91d7, // ( 9,  471)  45
 0xb0cf, // (11,  207)  46
 0x701b, // ( 7,   27)  47
 0xc2af, // (12,  687)  48
 0xa3f7, // (10, 1015)  49
 0xcaaf, // (12, 2735)  50
 0x9037, // ( 9,   55)  51
 0xc6af, // (12, 1711)  52
 0xa00f, // (10,   15)  53
 0xceaf, // (12, 3759)  54
 0x9137, // ( 9,  311)  55
 0xc1af, // (12,  431)  56
 0xb4cf, // (11, 1231)  57
 0xc9af, // (12, 2479)  58
 0xa20f, // (10,  527)  59
 0xc5af, // (12, 1455)  60
 0xb2cf, // (11,  719)  61
 0xcdaf, // (12, 3503)  62
 0xa10f, // (10,  271)  63
 0xc3af, // (12,  943)  64
 0x90b7, // ( 9,  183)  65
 0xb6cf, // (11, 1743)  66
 0x705b, // ( 7,   91)  67
 0xcbaf, // (12, 2991)  68
 0x91b7, // ( 9,  439)  69
 0xb1cf, // (11,  463)  70
 0x703b, // ( 7,   59)  71
 0xc7af, // (12, 1967)  72
 0xa30f, // (10,  783)  73
 0xcfaf, // (12, 4015)  74
 0x8067, // ( 8,  103)  75
 0xc06f, // (12,  111)  76
 0xa08f, // (10,  143)  77
 0xc86f, // (12, 2159)  78
 0x9077, // ( 9,  119)  79
 0xc46f, // (12, 1135)  80
 0xb5cf, // (11, 1487)  81
 0xcc6f, // (12, 3183)  82
 0xa28f, // (10,  655)  83
 0xc26f, // (12,  623)  84
 0xb3cf, // (11,  975)  85
 0xca6f, // (12, 2671)  86
 0xa18f, // (10,  399)  87
 0xc66f, // (12, 1647)  88
 0xce6f, // (12, 3695)  89
 0xc16f, // (12,  367)  90
 0xb7cf, // (11, 1999)  91
 0xc96f, // (12, 2415)  92
 0xc56f, // (12, 1391)  93
 0xcd6f, // (12, 3439)  94
 0xb02f, // (11,   47)  95
 0xc36f, // (12,  879)  96
 0xcb6f, // (12, 2927)  97
 0xc76f, // (12, 1903)  98
 0xb42f, // (11, 1071)  99
 0xcf6f, // (12, 3951) 100
 0xc0ef, // (12,  239) 101
 0xc8ef, // (12, 2287) 102
 0xb22f, // (11,  559) 103
 0xc4ef, // (12, 1263) 104
 0xccef, // (12, 3311) 105
 0xc2ef, // (12,  751) 106
 0xcaef, // (12, 2799) 107
 0xc6ef, // (12, 1775) 108
 0xceef, // (12, 3823) 109
 0xc1ef, // (12,  495) 110
 0xc9ef, // (12, 2543) 111
 0xc5ef, // (12, 1519) 112
 0xcdef, // (12, 3567) 113
 0xc3ef, // (12, 1007) 114
 0xcbef, // (12, 3055) 115
 0xc7ef, // (12, 2031) 116
 0xcfef, // (12, 4079) 117
 0xc01f, // (12,   31) 118
 0xc81f, // (12, 2079) 119
 0xc41f, // (12, 1055) 120
 0xcc1f, // (12, 3103) 121
 0xc21f, // (12,  543) 122
 0xca1f, // (12, 2591) 123
 0xc61f, // (12, 1567) 124
 0xce1f, // (12, 3615) 125
 0xc11f, // (12,  287) 126
 0xc91f, // (12, 2335) 127
 0xc51f, // (12, 1311) 128
 0xa38f, // (10,  911) 129
 0xcd1f, // (12, 3359) 130
 0x80e7, // ( 8,  231) 131
 0xc31f, // (12,  799) 132
 0xa04f, // (10,   79) 133
 0xcb1f, // (12, 2847) 134
 0x8017, // ( 8,   23) 135
 0xc71f, // (12, 1823) 136
 0xb62f, // (11, 1583) 137
 0xcf1f, // (12, 3871) 138
 0xa24f, // (10,  591) 139
 0xc09f, // (12,  159) 140
 0xb12f, // (11,  303) 141
 0xc89f, // (12, 2207) 142
 0xa14f, // (10,  335) 143
 0xc49f, // (12, 1183) 144
 0xcc9f, // (12, 3231) 145
 0xc29f, // (12,  671) 146
 0xb52f, // (11, 1327) 147
 0xca9f, // (12, 2719) 148
 0xc69f, // (12, 1695) 149
 0xce9f, // (12, 3743) 150
 0xb32f, // (11,  815) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 13 of 22) (steady 13 of 16) (phase = 0.843750000 = 27.0 / 32.0)
 // entropy:    4.4684687952964843305
 // avg_length: 4.5509169030369793774; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x803b, // ( 8,   59)   0
 0x4006, // ( 4,    6)   1
 0x7033, // ( 7,   51)   2
 0x3002, // ( 3,    2)   3
 0x80bb, // ( 8,  187)   4
 0x400e, // ( 4,   14)   5
 0x7073, // ( 7,  115)   6
 0x2000, // ( 2,    0)   7
 0xa0f7, // (10,  247)   8
 0x601d, // ( 6,   29)   9
 0x807b, // ( 8,  123)  10
 0x4001, // ( 4,    1)  11
 0xa2f7, // (10,  759)  12
 0x5005, // ( 5,    5)  13
 0x80fb, // ( 8,  251)  14
 0x4009, // ( 4,    9)  15
 0xb34f, // (11,  847)  16
 0x700b, // ( 7,   11)  17
 0x9057, // ( 9,   87)  18
 0x5015, // ( 5,   21)  19
 0xb74f, // (11, 1871)  20
 0x704b, // ( 7,   75)  21
 0x9157, // ( 9,  343)  22
 0x500d, // ( 5,   13)  23
 0xc72f, // (12, 1839)  24
 0x8007, // ( 8,    7)  25
 0xa1f7, // (10,  503)  26
 0x603d, // ( 6,   61)  27
 0xcf2f, // (12, 3887)  28
 0x8087, // ( 8,  135)  29
 0xa3f7, // (10, 1015)  30
 0x6003, // ( 6,    3)  31
 0xc0af, // (12,  175)  32
 0x8047, // ( 8,   71)  33
 0xa00f, // (10,   15)  34
 0x6023, // ( 6,   35)  35
 0xc8af, // (12, 2223)  36
 0x80c7, // ( 8,  199)  37
 0xa20f, // (10,  527)  38
 0x6013, // ( 6,   19)  39
 0xc4af, // (12, 1199)  40
 0x90d7, // ( 9,  215)  41
 0xb0cf, // (11,  207)  42
 0x702b, // ( 7,   43)  43
 0xccaf, // (12, 3247)  44
 0x91d7, // ( 9,  471)  45
 0xb4cf, // (11, 1231)  46
 0x706b, // ( 7,  107)  47
 0xc2af, // (12,  687)  48
 0xa10f, // (10,  271)  49
 0xcaaf, // (12, 2735)  50
 0x8027, // ( 8,   39)  51
 0xc6af, // (12, 1711)  52
 0xa30f, // (10,  783)  53
 0xceaf, // (12, 3759)  54
 0x80a7, // ( 8,  167)  55
 0xc1af, // (12,  431)  56
 0xb2cf, // (11,  719)  57
 0xc9af, // (12, 2479)  58
 0xa08f, // (10,  143)  59
 0xc5af, // (12, 1455)  60
 0xb6cf, // (11, 1743)  61
 0xcdaf, // (12, 3503)  62
 0xa28f, // (10,  655)  63
 0xc3af, // (12,  943)  64
 0x9037, // ( 9,   55)  65
 0xb1cf, // (11,  463)  66
 0x701b, // ( 7,   27)  67
 0xcbaf, // (12, 2991)  68
 0x9137, // ( 9,  311)  69
 0xb5cf, // (11, 1487)  70
 0x705b, // ( 7,   91)  71
 0xc7af, // (12, 1967)  72
 0xa18f, // (10,  399)  73
 0xcfaf, // (12, 4015)  74
 0x8067, // ( 8,  103)  75
 0xc06f, // (12,  111)  76
 0xa38f, // (10,  911)  77
 0xc86f, // (12, 2159)  78
 0x80e7, // ( 8,  231)  79
 0xc46f, // (12, 1135)  80
 0xb3cf, // (11,  975)  81
 0xcc6f, // (12, 3183)  82
 0x90b7, // ( 9,  183)  83
 0xc26f, // (12,  623)  84
 0xb7cf, // (11, 1999)  85
 0xca6f, // (12, 2671)  86
 0x91b7, // ( 9,  439)  87
 0xc66f, // (12, 1647)  88
 0xce6f, // (12, 3695)  89
 0xc16f, // (12,  367)  90
 0xb02f, // (11,   47)  91
 0xc96f, // (12, 2415)  92
 0xc56f, // (12, 1391)  93
 0xcd6f, // (12, 3439)  94
 0xb42f, // (11, 1071)  95
 0xc36f, // (12,  879)  96
 0xcb6f, // (12, 2927)  97
 0xc76f, // (12, 1903)  98
 0xb22f, // (11,  559)  99
 0xcf6f, // (12, 3951) 100
 0xc0ef, // (12,  239) 101
 0xc8ef, // (12, 2287) 102
 0xb62f, // (11, 1583) 103
 0xc4ef, // (12, 1263) 104
 0xccef, // (12, 3311) 105
 0xc2ef, // (12,  751) 106
 0xcaef, // (12, 2799) 107
 0xc6ef, // (12, 1775) 108
 0xceef, // (12, 3823) 109
 0xc1ef, // (12,  495) 110
 0xc9ef, // (12, 2543) 111
 0xc5ef, // (12, 1519) 112
 0xcdef, // (12, 3567) 113
 0xc3ef, // (12, 1007) 114
 0xcbef, // (12, 3055) 115
 0xc7ef, // (12, 2031) 116
 0xcfef, // (12, 4079) 117
 0xc01f, // (12,   31) 118
 0xc81f, // (12, 2079) 119
 0xc41f, // (12, 1055) 120
 0xcc1f, // (12, 3103) 121
 0xc21f, // (12,  543) 122
 0xca1f, // (12, 2591) 123
 0xc61f, // (12, 1567) 124
 0xce1f, // (12, 3615) 125
 0xc11f, // (12,  287) 126
 0xc91f, // (12, 2335) 127
 0xc51f, // (12, 1311) 128
 0xa04f, // (10,   79) 129
 0xcd1f, // (12, 3359) 130
 0x8017, // ( 8,   23) 131
 0xc31f, // (12,  799) 132
 0xa24f, // (10,  591) 133
 0xcb1f, // (12, 2847) 134
 0x8097, // ( 8,  151) 135
 0xc71f, // (12, 1823) 136
 0xb12f, // (11,  303) 137
 0xcf1f, // (12, 3871) 138
 0x9077, // ( 9,  119) 139
 0xc09f, // (12,  159) 140
 0xb52f, // (11, 1327) 141
 0xc89f, // (12, 2207) 142
 0x9177, // ( 9,  375) 143
 0xc49f, // (12, 1183) 144
 0xcc9f, // (12, 3231) 145
 0xc29f, // (12,  671) 146
 0xb32f, // (11,  815) 147
 0xca9f, // (12, 2719) 148
 0xc69f, // (12, 1695) 149
 0xce9f, // (12, 3743) 150
 0xa14f, // (10,  335) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 14 of 22) (steady 14 of 16) (phase = 0.906250000 = 29.0 / 32.0)
 // entropy:    4.4675179140944036860
 // avg_length: 4.5477235350841240802; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x9017, // ( 9,   23)   0
 0x4006, // ( 4,    6)   1
 0x7033, // ( 7,   51)   2
 0x3002, // ( 3,    2)   3
 0x9117, // ( 9,  279)   4
 0x400e, // ( 4,   14)   5
 0x7073, // ( 7,  115)   6
 0x2000, // ( 2,    0)   7
 0xa177, // (10,  375)   8
 0x601d, // ( 6,   29)   9
 0x803b, // ( 8,   59)  10
 0x4001, // ( 4,    1)  11
 0xa377, // (10,  887)  12
 0x5005, // ( 5,    5)  13
 0x80bb, // ( 8,  187)  14
 0x4009, // ( 4,    9)  15
 0xb0cf, // (11,  207)  16
 0x700b, // ( 7,   11)  17
 0x9097, // ( 9,  151)  18
 0x5015, // ( 5,   21)  19
 0xb4cf, // (11, 1231)  20
 0x704b, // ( 7,   75)  21
 0x9197, // ( 9,  407)  22
 0x500d, // ( 5,   13)  23
 0xc4af, // (12, 1199)  24
 0x807b, // ( 8,  123)  25
 0xa0f7, // (10,  247)  26
 0x603d, // ( 6,   61)  27
 0xccaf, // (12, 3247)  28
 0x80fb, // ( 8,  251)  29
 0xa2f7, // (10,  759)  30
 0x6003, // ( 6,    3)  31
 0xc2af, // (12,  687)  32
 0x8007, // ( 8,    7)  33
 0xa1f7, // (10,  503)  34
 0x6023, // ( 6,   35)  35
 0xcaaf, // (12, 2735)  36
 0x8087, // ( 8,  135)  37
 0xa3f7, // (10, 1015)  38
 0x6013, // ( 6,   19)  39
 0xc6af, // (12, 1711)  40
 0x9057, // ( 9,   87)  41
 0xb2cf, // (11,  719)  42
 0x702b, // ( 7,   43)  43
 0xceaf, // (12, 3759)  44
 0x9157, // ( 9,  343)  45
 0xb6cf, // (11, 1743)  46
 0x706b, // ( 7,  107)  47
 0xc1af, // (12,  431)  48
 0xa00f, // (10,   15)  49
 0xc9af, // (12, 2479)  50
 0x8047, // ( 8,   71)  51
 0xc5af, // (12, 1455)  52
 0xa20f, // (10,  527)  53
 0xcdaf, // (12, 3503)  54
 0x80c7, // ( 8,  199)  55
 0xc3af, // (12,  943)  56
 0xb1cf, // (11,  463)  57
 0xcbaf, // (12, 2991)  58
 0xa10f, // (10,  271)  59
 0xc7af, // (12, 1967)  60
 0xb5cf, // (11, 1487)  61
 0xcfaf, // (12, 4015)  62
 0x90d7, // ( 9,  215)  63
 0xc06f, // (12,  111)  64
 0x91d7, // ( 9,  471)  65
 0xb3cf, // (11,  975)  66
 0x701b, // ( 7,   27)  67
 0xc86f, // (12, 2159)  68
 0x9037, // ( 9,   55)  69
 0xb7cf, // (11, 1999)  70
 0x705b, // ( 7,   91)  71
 0xc46f, // (12, 1135)  72
 0xa30f, // (10,  783)  73
 0xcc6f, // (12, 3183)  74
 0x8027, // ( 8,   39)  75
 0xc26f, // (12,  623)  76
 0xa08f, // (10,  143)  77
 0xca6f, // (12, 2671)  78
 0x80a7, // ( 8,  167)  79
 0xc66f, // (12, 1647)  80
 0xb02f, // (11,   47)  81
 0xce6f, // (12, 3695)  82
 0x9137, // ( 9,  311)  83
 0xc16f, // (12,  367)  84
 0xb42f, // (11, 1071)  85
 0xc96f, // (12, 2415)  86
 0x90b7, // ( 9,  183)  87
 0xc56f, // (12, 1391)  88
 0xcd6f, // (12, 3439)  89
 0xc36f, // (12,  879)  90
 0xb22f, // (11,  559)  91
 0xcb6f, // (12, 2927)  92
 0xc76f, // (12, 1903)  93
 0xcf6f, // (12, 3951)  94
 0xa28f, // (10,  655)  95
 0xc0ef, // (12,  239)  96
 0xc8ef, // (12, 2287)  97
 0xc4ef, // (12, 1263)  98
 0xa18f, // (10,  399)  99
 0xccef, // (12, 3311) 100
 0xc2ef, // (12,  751) 101
 0xcaef, // (12, 2799) 102
 0xa38f, // (10,  911) 103
 0xc6ef, // (12, 1775) 104
 0xceef, // (12, 3823) 105
 0xc1ef, // (12,  495) 106
 0xc9ef, // (12, 2543) 107
 0xc5ef, // (12, 1519) 108
 0xcdef, // (12, 3567) 109
 0xc3ef, // (12, 1007) 110
 0xb62f, // (11, 1583) 111
 0xcbef, // (12, 3055) 112
 0xc7ef, // (12, 2031) 113
 0xcfef, // (12, 4079) 114
 0xc01f, // (12,   31) 115
 0xc81f, // (12, 2079) 116
 0xc41f, // (12, 1055) 117
 0xcc1f, // (12, 3103) 118
 0xc21f, // (12,  543) 119
 0xca1f, // (12, 2591) 120
 0xc61f, // (12, 1567) 121
 0xce1f, // (12, 3615) 122
 0xc11f, // (12,  287) 123
 0xc91f, // (12, 2335) 124
 0xc51f, // (12, 1311) 125
 0xcd1f, // (12, 3359) 126
 0xc31f, // (12,  799) 127
 0xcb1f, // (12, 2847) 128
 0xa04f, // (10,   79) 129
 0xc71f, // (12, 1823) 130
 0x8067, // ( 8,  103) 131
 0xcf1f, // (12, 3871) 132
 0xa24f, // (10,  591) 133
 0xc09f, // (12,  159) 134
 0x80e7, // ( 8,  231) 135
 0xc89f, // (12, 2207) 136
 0xb12f, // (11,  303) 137
 0xc49f, // (12, 1183) 138
 0x91b7, // ( 9,  439) 139
 0xcc9f, // (12, 3231) 140
 0xb52f, // (11, 1327) 141
 0xc29f, // (12,  671) 142
 0x9077, // ( 9,  119) 143
 0xca9f, // (12, 2719) 144
 0xc69f, // (12, 1695) 145
 0xce9f, // (12, 3743) 146
 0xa14f, // (10,  335) 147
 0xc19f, // (12,  415) 148
 0xc99f, // (12, 2463) 149
 0xc59f, // (12, 1439) 150
 0xa34f, // (10,  847) 151
 0xcd9f, // (12, 3487) 152
 0xc39f, // (12,  927) 153
 0xcb9f, // (12, 2975) 154
 0xc79f, // (12, 1951) 155
 0xcf9f, // (12, 3999) 156
 0xc05f, // (12,   95) 157
 0xc85f, // (12, 2143) 158
 0xb32f, // (11,  815) 159
 0xc45f, // (12, 1119) 160
 0xcc5f, // (12, 3167) 161
 0xc25f, // (12,  607) 162
 0xb72f, // (11, 1839) 163
 0xca5f, // (12, 2655) 164
 0xc65f, // (12, 1631) 165
 0xce5f, // (12, 3679) 166
 0xb0af, // (11,  175) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 15 of 22) (steady 15 of 16) (phase = 0.968750000 = 31.0 / 32.0)
 // entropy:    4.4653007097343397902
 // avg_length: 4.5480722016259509388; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x9017, // ( 9,   23)   0
 0x4006, // ( 4,    6)   1
 0x7033, // ( 7,   51)   2
 0x3002, // ( 3,    2)   3
 0x9117, // ( 9,  279)   4
 0x400e, // ( 4,   14)   5
 0x7073, // ( 7,  115)   6
 0x2000, // ( 2,    0)   7
 0xa0f7, // (10,  247)   8
 0x601d, // ( 6,   29)   9
 0x803b, // ( 8,   59)  10
 0x4001, // ( 4,    1)  11
 0xa2f7, // (10,  759)  12
 0x5005, // ( 5,    5)  13
 0x80bb, // ( 8,  187)  14
 0x4009, // ( 4,    9)  15
 0xb0cf, // (11,  207)  16
 0x700b, // ( 7,   11)  17
 0x9097, // ( 9,  151)  18
 0x5015, // ( 5,   21)  19
 0xb4cf, // (11, 1231)  20
 0x704b, // ( 7,   75)  21
 0x9197, // ( 9,  407)  22
 0x500d, // ( 5,   13)  23
 0xc0af, // (12,  175)  24
 0x807b, // ( 8,  123)  25
 0xb2cf, // (11,  719)  26
 0x603d, // ( 6,   61)  27
 0xc8af, // (12, 2223)  28
 0x80fb, // ( 8,  251)  29
 0xa1f7, // (10,  503)  30
 0x6003, // ( 6,    3)  31
 0xc4af, // (12, 1199)  32
 0x8007, // ( 8,    7)  33
 0xb6cf, // (11, 1743)  34
 0x6023, // ( 6,   35)  35
 0xccaf, // (12, 3247)  36
 0x8087, // ( 8,  135)  37
 0xa3f7, // (10, 1015)  38
 0x6013, // ( 6,   19)  39
 0xc2af, // (12,  687)  40
 0x9057, // ( 9,   87)  41
 0xcaaf, // (12, 2735)  42
 0x702b, // ( 7,   43)  43
 0xc6af, // (12, 1711)  44
 0x9157, // ( 9,  343)  45
 0xb1cf, // (11,  463)  46
 0x706b, // ( 7,  107)  47
 0xceaf, // (12, 3759)  48
 0xa00f, // (10,   15)  49
 0xc1af, // (12,  431)  50
 0x8047, // ( 8,   71)  51
 0xc9af, // (12, 2479)  52
 0xa20f, // (10,  527)  53
 0xc5af, // (12, 1455)  54
 0x80c7, // ( 8,  199)  55
 0xcdaf, // (12, 3503)  56
 0xb5cf, // (11, 1487)  57
 0xc3af, // (12,  943)  58
 0x90d7, // ( 9,  215)  59
 0xcbaf, // (12, 2991)  60
 0xb3cf, // (11,  975)  61
 0xc7af, // (12, 1967)  62
 0x91d7, // ( 9,  471)  63
 0xcfaf, // (12, 4015)  64
 0x9037, // ( 9,   55)  65
 0xc06f, // (12,  111)  66
 0x701b, // ( 7,   27)  67
 0xc86f, // (12, 2159)  68
 0x9137, // ( 9,  311)  69
 0xb7cf, // (11, 1999)  70
 0x705b, // ( 7,   91)  71
 0xc46f, // (12, 1135)  72
 0xa10f, // (10,  271)  73
 0xcc6f, // (12, 3183)  74
 0x8027, // ( 8,   39)  75
 0xc26f, // (12,  623)  76
 0xa30f, // (10,  783)  77
 0xca6f, // (12, 2671)  78
 0x80a7, // ( 8,  167)  79
 0xc66f, // (12, 1647)  80
 0xb02f, // (11,   47)  81
 0xce6f, // (12, 3695)  82
 0x90b7, // ( 9,  183)  83
 0xc16f, // (12,  367)  84
 0xb42f, // (11, 1071)  85
 0xc96f, // (12, 2415)  86
 0x91b7, // ( 9,  439)  87
 0xc56f, // (12, 1391)  88
 0xcd6f, // (12, 3439)  89
 0xc36f, // (12,  879)  90
 0xa08f, // (10,  143)  91
 0xcb6f, // (12, 2927)  92
 0xc76f, // (12, 1903)  93
 0xcf6f, // (12, 3951)  94
 0xa28f, // (10,  655)  95
 0xc0ef, // (12,  239)  96
 0xc8ef, // (12, 2287)  97
 0xc4ef, // (12, 1263)  98
 0xa18f, // (10,  399)  99
 0xccef, // (12, 3311) 100
 0xc2ef, // (12,  751) 101
 0xcaef, // (12, 2799) 102
 0xa38f, // (10,  911) 103
 0xc6ef, // (12, 1775) 104
 0xceef, // (12, 3823) 105
 0xc1ef, // (12,  495) 106
 0xc9ef, // (12, 2543) 107
 0xc5ef, // (12, 1519) 108
 0xcdef, // (12, 3567) 109
 0xc3ef, // (12, 1007) 110
 0xb22f, // (11,  559) 111
 0xcbef, // (12, 3055) 112
 0xc7ef, // (12, 2031) 113
 0xcfef, // (12, 4079) 114
 0xc01f, // (12,   31) 115
 0xc81f, // (12, 2079) 116
 0xc41f, // (12, 1055) 117
 0xcc1f, // (12, 3103) 118
 0xc21f, // (12,  543) 119
 0xca1f, // (12, 2591) 120
 0xc61f, // (12, 1567) 121
 0xce1f, // (12, 3615) 122
 0xc11f, // (12,  287) 123
 0xc91f, // (12, 2335) 124
 0xc51f, // (12, 1311) 125
 0xcd1f, // (12, 3359) 126
 0xc31f, // (12,  799) 127
 0xcb1f, // (12, 2847) 128
 0xa04f, // (10,   79) 129
 0xc71f, // (12, 1823) 130
 0x8067, // ( 8,  103) 131
 0xcf1f, // (12, 3871) 132
 0xa24f, // (10,  591) 133
 0xc09f, // (12,  159) 134
 0x80e7, // ( 8,  231) 135
 0xc89f, // (12, 2207) 136
 0xb62f, // (11, 1583) 137
 0xc49f, // (12, 1183) 138
 0x9077, // ( 9,  119) 139
 0xcc9f, // (12, 3231) 140
 0xb12f, // (11,  303) 141
 0xc29f, // (12,  671) 142
 0x9177, // ( 9,  375) 143
 0xca9f, // (12, 2719) 144
 0xc69f, // (12, 1695) 145
 0xce9f, // (12, 3743) 146
 0xa14f, // (10,  335) 147
 0xc19f, // (12,  415) 148
 0xc99f, // (12, 2463) 149
 0xc59f, // (12, 1439) 150
 0xa34f, // (10,  847) 151
 0xcd9f, // (12, 3487) 152
 0xc39f, // (12,  927) 153
 0xcb9f, // (12, 2975) 154
 0xc79f, // (12, 1951) 155
 0xcf9f, // (12, 3999) 156
 0xc05f, // (12,   95) 157
 0xc85f, // (12, 2143) 158
 0xb52f, // (11, 1327) 159
 0xc45f, // (12, 1119) 160
 0xcc5f, // (12, 3167) 161
 0xc25f, // (12,  607) 162
 0xb32f, // (11,  815) 163
 0xca5f, // (12, 2655) 164
 0xc65f, // (12, 1631) 165
 0xce5f, // (12, 3679) 166
 0xb72f, // (11, 1839) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // Six Encoding Tables for the Midrange.

 // (table 16 of 22) (midrange 0 of 6) (c/k = 0.500000000 = 3.0 / 6.0)
 // entropy:    2.1627885076675394949
 // avg_length: 2.2704182849800043087; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x1000, // ( 1,    0)   0
 0x2001, // ( 2,    1)   1
 0x4003, // ( 4,    3)   2
 0x500b, // ( 5,   11)   3
 0x501b, // ( 5,   27)   4
 0x6007, // ( 6,    7)   5
 0x8057, // ( 8,   87)   6
 0x9077, // ( 9,  119)   7
 0x6027, // ( 6,   39)   8
 0x80d7, // ( 8,  215)   9
 0x9177, // ( 9,  375)  10
 0xa1f7, // (10,  503)  11
 0xa3f7, // (10, 1015)  12
 0xb08f, // (11,  143)  13
 0xc58f, // (12, 1423)  14
 0xcd8f, // (12, 3471)  15
 0x7017, // ( 7,   23)  16
 0x8037, // ( 8,   55)  17
 0xa00f, // (10,   15)  18
 0xb48f, // (11, 1167)  19
 0xb28f, // (11,  655)  20
 0xc38f, // (12,  911)  21
 0xcb8f, // (12, 2959)  22
 0xc78f, // (12, 1935)  23
 0xcf8f, // (12, 3983)  24
 0xc04f, // (12,   79)  25
 0xc84f, // (12, 2127)  26
 0xc44f, // (12, 1103)  27
 0xcc4f, // (12, 3151)  28
 0xc24f, // (12,  591)  29
 0xca4f, // (12, 2639)  30
 0xc64f, // (12, 1615)  31
 0x80b7, // ( 8,  183)  32
 0xa20f, // (10,  527)  33
 0xb68f, // (11, 1679)  34
 0xce4f, // (12, 3663)  35
 0xc14f, // (12,  335)  36
 0xc94f, // (12, 2383)  37
 0xc54f, // (12, 1359)  38
 0xcd4f, // (12, 3407)  39
 0xc34f, // (12,  847)  40
 0xcb4f, // (12, 2895)  41
 0xc74f, // (12, 1871)  42
 0xcf4f, // (12, 3919)  43
 0xc0cf, // (12,  207)  44
 0xc8cf, // (12, 2255)  45
 0xc4cf, // (12, 1231)  46
 0xcccf, // (12, 3279)  47
 0xc2cf, // (12,  719)  48
 0xcacf, // (12, 2767)  49
 0xc6cf, // (12, 1743)  50
 0xcecf, // (12, 3791)  51
 0xc1cf, // (12,  463)  52
 0xc9cf, // (12, 2511)  53
 0xc5cf, // (12, 1487)  54
 0xcdcf, // (12, 3535)  55
 0xc3cf, // (12,  975)  56
 0xcbcf, // (12, 3023)  57
 0xc7cf, // (12, 1999)  58
 0xcfcf, // (12, 4047)  59
 0xc02f, // (12,   47)  60
 0xc82f, // (12, 2095)  61
 0xc42f, // (12, 1071)  62
 0xcc2f, // (12, 3119)  63
 0x90f7, // ( 9,  247)  64
 0xa10f, // (10,  271)  65
 0xc22f, // (12,  559)  66
 0xca2f, // (12, 2607)  67
 0xc62f, // (12, 1583)  68
 0xce2f, // (12, 3631)  69
 0xc12f, // (12,  303)  70
 0xc92f, // (12, 2351)  71
 0xc52f, // (12, 1327)  72
 0xcd2f, // (12, 3375)  73
 0xc32f, // (12,  815)  74
 0xcb2f, // (12, 2863)  75
 0xc72f, // (12, 1839)  76
 0xcf2f, // (12, 3887)  77
 0xc0af, // (12,  175)  78
 0xc8af, // (12, 2223)  79
 0xc4af, // (12, 1199)  80
 0xccaf, // (12, 3247)  81
 0xc2af, // (12,  687)  82
 0xcaaf, // (12, 2735)  83
 0xc6af, // (12, 1711)  84
 0xceaf, // (12, 3759)  85
 0xc1af, // (12,  431)  86
 0xc9af, // (12, 2479)  87
 0xc5af, // (12, 1455)  88
 0xcdaf, // (12, 3503)  89
 0xc3af, // (12,  943)  90
 0xcbaf, // (12, 2991)  91
 0xc7af, // (12, 1967)  92
 0xcfaf, // (12, 4015)  93
 0xc06f, // (12,  111)  94
 0xc86f, // (12, 2159)  95
 0xc46f, // (12, 1135)  96
 0xcc6f, // (12, 3183)  97
 0xc26f, // (12,  623)  98
 0xca6f, // (12, 2671)  99
 0xc66f, // (12, 1647) 100
 0xce6f, // (12, 3695) 101
 0xc16f, // (12,  367) 102
 0xc96f, // (12, 2415) 103
 0xc56f, // (12, 1391) 104
 0xcd6f, // (12, 3439) 105
 0xc36f, // (12,  879) 106
 0xcb6f, // (12, 2927) 107
 0xc76f, // (12, 1903) 108
 0xcf6f, // (12, 3951) 109
 0xc0ef, // (12,  239) 110
 0xc8ef, // (12, 2287) 111
 0xc4ef, // (12, 1263) 112
 0xccef, // (12, 3311) 113
 0xc2ef, // (12,  751) 114
 0xcaef, // (12, 2799) 115
 0xc6ef, // (12, 1775) 116
 0xceef, // (12, 3823) 117
 0xc1ef, // (12,  495) 118
 0xc9ef, // (12, 2543) 119
 0xc5ef, // (12, 1519) 120
 0xcdef, // (12, 3567) 121
 0xc3ef, // (12, 1007) 122
 0xcbef, // (12, 3055) 123
 0xc7ef, // (12, 2031) 124
 0xcfef, // (12, 4079) 125
 0xc01f, // (12,   31) 126
 0xc81f, // (12, 2079) 127
 0xa30f, // (10,  783) 128
 0xb18f, // (11,  399) 129
 0xc41f, // (12, 1055) 130
 0xcc1f, // (12, 3103) 131
 0xc21f, // (12,  543) 132
 0xca1f, // (12, 2591) 133
 0xc61f, // (12, 1567) 134
 0xce1f, // (12, 3615) 135
 0xc11f, // (12,  287) 136
 0xc91f, // (12, 2335) 137
 0xc51f, // (12, 1311) 138
 0xcd1f, // (12, 3359) 139
 0xc31f, // (12,  799) 140
 0xcb1f, // (12, 2847) 141
 0xc71f, // (12, 1823) 142
 0xcf1f, // (12, 3871) 143
 0xc09f, // (12,  159) 144
 0xc89f, // (12, 2207) 145
 0xc49f, // (12, 1183) 146
 0xcc9f, // (12, 3231) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 17 of 22) (midrange 1 of 6) (c/k = 0.833333333 = 5.0 / 6.0)
 // entropy:    2.9553294756640680063
 // avg_length: 3.0766035704232641557; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x2000, // ( 2,    0)   0
 0x2002, // ( 2,    2)   1
 0x3001, // ( 3,    1)   2
 0x4005, // ( 4,    5)   3
 0x400d, // ( 4,   13)   4
 0x5003, // ( 5,    3)   5
 0x600b, // ( 6,   11)   6
 0x602b, // ( 6,   43)   7
 0x5013, // ( 5,   19)   8
 0x601b, // ( 6,   27)   9
 0x7007, // ( 7,    7)  10
 0x7047, // ( 7,   71)  11
 0x8017, // ( 8,   23)  12
 0x90b7, // ( 9,  183)  13
 0xa1f7, // (10,  503)  14
 0xa3f7, // (10, 1015)  15
 0x603b, // ( 6,   59)  16
 0x7027, // ( 7,   39)  17
 0x8097, // ( 8,  151)  18
 0x8057, // ( 8,   87)  19
 0x91b7, // ( 9,  439)  20
 0xa00f, // (10,   15)  21
 0xb18f, // (11,  399)  22
 0xb58f, // (11, 1423)  23
 0xa20f, // (10,  527)  24
 0xb38f, // (11,  911)  25
 0xc54f, // (12, 1359)  26
 0xcd4f, // (12, 3407)  27
 0xc34f, // (12,  847)  28
 0xcb4f, // (12, 2895)  29
 0xc74f, // (12, 1871)  30
 0xcf4f, // (12, 3919)  31
 0x7067, // ( 7,  103)  32
 0x80d7, // ( 8,  215)  33
 0x9077, // ( 9,  119)  34
 0xa10f, // (10,  271)  35
 0xa30f, // (10,  783)  36
 0xb78f, // (11, 1935)  37
 0xc0cf, // (12,  207)  38
 0xc8cf, // (12, 2255)  39
 0xb04f, // (11,   79)  40
 0xc4cf, // (12, 1231)  41
 0xcccf, // (12, 3279)  42
 0xc2cf, // (12,  719)  43
 0xcacf, // (12, 2767)  44
 0xc6cf, // (12, 1743)  45
 0xcecf, // (12, 3791)  46
 0xc1cf, // (12,  463)  47
 0xc9cf, // (12, 2511)  48
 0xc5cf, // (12, 1487)  49
 0xcdcf, // (12, 3535)  50
 0xc3cf, // (12,  975)  51
 0xcbcf, // (12, 3023)  52
 0xc7cf, // (12, 1999)  53
 0xcfcf, // (12, 4047)  54
 0xc02f, // (12,   47)  55
 0xc82f, // (12, 2095)  56
 0xc42f, // (12, 1071)  57
 0xcc2f, // (12, 3119)  58
 0xc22f, // (12,  559)  59
 0xca2f, // (12, 2607)  60
 0xc62f, // (12, 1583)  61
 0xce2f, // (12, 3631)  62
 0xc12f, // (12,  303)  63
 0x8037, // ( 8,   55)  64
 0x9177, // ( 9,  375)  65
 0xa08f, // (10,  143)  66
 0xb44f, // (11, 1103)  67
 0xb24f, // (11,  591)  68
 0xc92f, // (12, 2351)  69
 0xc52f, // (12, 1327)  70
 0xcd2f, // (12, 3375)  71
 0xc32f, // (12,  815)  72
 0xcb2f, // (12, 2863)  73
 0xc72f, // (12, 1839)  74
 0xcf2f, // (12, 3887)  75
 0xc0af, // (12,  175)  76
 0xc8af, // (12, 2223)  77
 0xc4af, // (12, 1199)  78
 0xccaf, // (12, 3247)  79
 0xc2af, // (12,  687)  80
 0xcaaf, // (12, 2735)  81
 0xc6af, // (12, 1711)  82
 0xceaf, // (12, 3759)  83
 0xc1af, // (12,  431)  84
 0xc9af, // (12, 2479)  85
 0xc5af, // (12, 1455)  86
 0xcdaf, // (12, 3503)  87
 0xc3af, // (12,  943)  88
 0xcbaf, // (12, 2991)  89
 0xc7af, // (12, 1967)  90
 0xcfaf, // (12, 4015)  91
 0xc06f, // (12,  111)  92
 0xc86f, // (12, 2159)  93
 0xc46f, // (12, 1135)  94
 0xcc6f, // (12, 3183)  95
 0xc26f, // (12,  623)  96
 0xca6f, // (12, 2671)  97
 0xc66f, // (12, 1647)  98
 0xce6f, // (12, 3695)  99
 0xc16f, // (12,  367) 100
 0xc96f, // (12, 2415) 101
 0xc56f, // (12, 1391) 102
 0xcd6f, // (12, 3439) 103
 0xc36f, // (12,  879) 104
 0xcb6f, // (12, 2927) 105
 0xc76f, // (12, 1903) 106
 0xcf6f, // (12, 3951) 107
 0xc0ef, // (12,  239) 108
 0xc8ef, // (12, 2287) 109
 0xc4ef, // (12, 1263) 110
 0xccef, // (12, 3311) 111
 0xc2ef, // (12,  751) 112
 0xcaef, // (12, 2799) 113
 0xc6ef, // (12, 1775) 114
 0xceef, // (12, 3823) 115
 0xc1ef, // (12,  495) 116
 0xc9ef, // (12, 2543) 117
 0xc5ef, // (12, 1519) 118
 0xcdef, // (12, 3567) 119
 0xc3ef, // (12, 1007) 120
 0xcbef, // (12, 3055) 121
 0xc7ef, // (12, 2031) 122
 0xcfef, // (12, 4079) 123
 0xc01f, // (12,   31) 124
 0xc81f, // (12, 2079) 125
 0xc41f, // (12, 1055) 126
 0xcc1f, // (12, 3103) 127
 0x90f7, // ( 9,  247) 128
 0xa28f, // (10,  655) 129
 0xb64f, // (11, 1615) 130
 0xb14f, // (11,  335) 131
 0xc21f, // (12,  543) 132
 0xca1f, // (12, 2591) 133
 0xc61f, // (12, 1567) 134
 0xce1f, // (12, 3615) 135
 0xc11f, // (12,  287) 136
 0xc91f, // (12, 2335) 137
 0xc51f, // (12, 1311) 138
 0xcd1f, // (12, 3359) 139
 0xc31f, // (12,  799) 140
 0xcb1f, // (12, 2847) 141
 0xc71f, // (12, 1823) 142
 0xcf1f, // (12, 3871) 143
 0xc09f, // (12,  159) 144
 0xc89f, // (12, 2207) 145
 0xc49f, // (12, 1183) 146
 0xcc9f, // (12, 3231) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 18 of 22) (midrange 2 of 6) (c/k = 1.166666667 = 7.0 / 6.0)
 // entropy:    3.5218672531711128215
 // avg_length: 3.6153551492375441967; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x2000, // ( 2,    0)   0
 0x2002, // ( 2,    2)   1
 0x4005, // ( 4,    5)   2
 0x3001, // ( 3,    1)   3
 0x5003, // ( 5,    3)   4
 0x400d, // ( 4,   13)   5
 0x600b, // ( 6,   11)   6
 0x602b, // ( 6,   43)   7
 0x601b, // ( 6,   27)   8
 0x5013, // ( 5,   19)   9
 0x703b, // ( 7,   59)  10
 0x707b, // ( 7,  123)  11
 0x8067, // ( 8,  103)  12
 0x80e7, // ( 8,  231)  13
 0x90d7, // ( 9,  215)  14
 0x91d7, // ( 9,  471)  15
 0x7007, // ( 7,    7)  16
 0x7047, // ( 7,   71)  17
 0x8017, // ( 8,   23)  18
 0x8097, // ( 8,  151)  19
 0x9037, // ( 9,   55)  20
 0x9137, // ( 9,  311)  21
 0xa1f7, // (10,  503)  22
 0xa3f7, // (10, 1015)  23
 0xa00f, // (10,   15)  24
 0xa20f, // (10,  527)  25
 0xb38f, // (11,  911)  26
 0xb78f, // (11, 1935)  27
 0xc0cf, // (12,  207)  28
 0xc8cf, // (12, 2255)  29
 0xc4cf, // (12, 1231)  30
 0xcccf, // (12, 3279)  31
 0x8057, // ( 8,   87)  32
 0x7027, // ( 7,   39)  33
 0x90b7, // ( 9,  183)  34
 0x91b7, // ( 9,  439)  35
 0xa10f, // (10,  271)  36
 0xa30f, // (10,  783)  37
 0xb04f, // (11,   79)  38
 0xb44f, // (11, 1103)  39
 0xb24f, // (11,  591)  40
 0xb64f, // (11, 1615)  41
 0xc2cf, // (12,  719)  42
 0xcacf, // (12, 2767)  43
 0xc6cf, // (12, 1743)  44
 0xcecf, // (12, 3791)  45
 0xc1cf, // (12,  463)  46
 0xc9cf, // (12, 2511)  47
 0xc5cf, // (12, 1487)  48
 0xcdcf, // (12, 3535)  49
 0xc3cf, // (12,  975)  50
 0xcbcf, // (12, 3023)  51
 0xc7cf, // (12, 1999)  52
 0xcfcf, // (12, 4047)  53
 0xc02f, // (12,   47)  54
 0xc82f, // (12, 2095)  55
 0xc42f, // (12, 1071)  56
 0xcc2f, // (12, 3119)  57
 0xc22f, // (12,  559)  58
 0xca2f, // (12, 2607)  59
 0xc62f, // (12, 1583)  60
 0xce2f, // (12, 3631)  61
 0xc12f, // (12,  303)  62
 0xc92f, // (12, 2351)  63
 0x9077, // ( 9,  119)  64
 0x9177, // ( 9,  375)  65
 0xa08f, // (10,  143)  66
 0xa28f, // (10,  655)  67
 0xb14f, // (11,  335)  68
 0xb54f, // (11, 1359)  69
 0xc52f, // (12, 1327)  70
 0xcd2f, // (12, 3375)  71
 0xc32f, // (12,  815)  72
 0xcb2f, // (12, 2863)  73
 0xc72f, // (12, 1839)  74
 0xcf2f, // (12, 3887)  75
 0xc0af, // (12,  175)  76
 0xc8af, // (12, 2223)  77
 0xc4af, // (12, 1199)  78
 0xccaf, // (12, 3247)  79
 0xc2af, // (12,  687)  80
 0xcaaf, // (12, 2735)  81
 0xc6af, // (12, 1711)  82
 0xceaf, // (12, 3759)  83
 0xc1af, // (12,  431)  84
 0xc9af, // (12, 2479)  85
 0xc5af, // (12, 1455)  86
 0xcdaf, // (12, 3503)  87
 0xc3af, // (12,  943)  88
 0xcbaf, // (12, 2991)  89
 0xc7af, // (12, 1967)  90
 0xcfaf, // (12, 4015)  91
 0xc06f, // (12,  111)  92
 0xc86f, // (12, 2159)  93
 0xc46f, // (12, 1135)  94
 0xcc6f, // (12, 3183)  95
 0xc26f, // (12,  623)  96
 0xca6f, // (12, 2671)  97
 0xc66f, // (12, 1647)  98
 0xce6f, // (12, 3695)  99
 0xc16f, // (12,  367) 100
 0xc96f, // (12, 2415) 101
 0xc56f, // (12, 1391) 102
 0xcd6f, // (12, 3439) 103
 0xc36f, // (12,  879) 104
 0xcb6f, // (12, 2927) 105
 0xc76f, // (12, 1903) 106
 0xcf6f, // (12, 3951) 107
 0xc0ef, // (12,  239) 108
 0xc8ef, // (12, 2287) 109
 0xc4ef, // (12, 1263) 110
 0xccef, // (12, 3311) 111
 0xc2ef, // (12,  751) 112
 0xcaef, // (12, 2799) 113
 0xc6ef, // (12, 1775) 114
 0xceef, // (12, 3823) 115
 0xc1ef, // (12,  495) 116
 0xc9ef, // (12, 2543) 117
 0xc5ef, // (12, 1519) 118
 0xcdef, // (12, 3567) 119
 0xc3ef, // (12, 1007) 120
 0xcbef, // (12, 3055) 121
 0xc7ef, // (12, 2031) 122
 0xcfef, // (12, 4079) 123
 0xc01f, // (12,   31) 124
 0xc81f, // (12, 2079) 125
 0xc41f, // (12, 1055) 126
 0xcc1f, // (12, 3103) 127
 0xa18f, // (10,  399) 128
 0x90f7, // ( 9,  247) 129
 0xb34f, // (11,  847) 130
 0xb74f, // (11, 1871) 131
 0xc21f, // (12,  543) 132
 0xca1f, // (12, 2591) 133
 0xc61f, // (12, 1567) 134
 0xce1f, // (12, 3615) 135
 0xc11f, // (12,  287) 136
 0xc91f, // (12, 2335) 137
 0xc51f, // (12, 1311) 138
 0xcd1f, // (12, 3359) 139
 0xc31f, // (12,  799) 140
 0xcb1f, // (12, 2847) 141
 0xc71f, // (12, 1823) 142
 0xcf1f, // (12, 3871) 143
 0xc09f, // (12,  159) 144
 0xc89f, // (12, 2207) 145
 0xc49f, // (12, 1183) 146
 0xcc9f, // (12, 3231) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 19 of 22) (midrange 3 of 6) (c/k = 1.500000000 = 9.0 / 6.0)
 // entropy:    3.9228873257934386842
 // avg_length: 3.9989687586992346269; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x3002, // ( 3,    2)   0
 0x2000, // ( 2,    0)   1
 0x4001, // ( 4,    1)   2
 0x3006, // ( 3,    6)   3
 0x500d, // ( 5,   13)   4
 0x4009, // ( 4,    9)   5
 0x501d, // ( 5,   29)   6
 0x4005, // ( 4,    5)   7
 0x6013, // ( 6,   19)   8
 0x5003, // ( 5,    3)   9
 0x6033, // ( 6,   51)  10
 0x600b, // ( 6,   11)  11
 0x8027, // ( 8,   39)  12
 0x701b, // ( 7,   27)  13
 0x80a7, // ( 8,  167)  14
 0x705b, // ( 7,   91)  15
 0x703b, // ( 7,   59)  16
 0x602b, // ( 6,   43)  17
 0x707b, // ( 7,  123)  18
 0x7007, // ( 7,    7)  19
 0x90d7, // ( 9,  215)  20
 0x8067, // ( 8,  103)  21
 0x91d7, // ( 9,  471)  22
 0x80e7, // ( 8,  231)  23
 0xa1f7, // (10,  503)  24
 0x9037, // ( 9,   55)  25
 0xa3f7, // (10, 1015)  26
 0xa00f, // (10,   15)  27
 0xc5cf, // (12, 1487)  28
 0xb04f, // (11,   79)  29
 0xcdcf, // (12, 3535)  30
 0xb44f, // (11, 1103)  31
 0x8017, // ( 8,   23)  32
 0x7047, // ( 7,   71)  33
 0x9137, // ( 9,  311)  34
 0x8097, // ( 8,  151)  35
 0xa20f, // (10,  527)  36
 0x90b7, // ( 9,  183)  37
 0xa10f, // (10,  271)  38
 0x91b7, // ( 9,  439)  39
 0xb24f, // (11,  591)  40
 0xa30f, // (10,  783)  41
 0xb64f, // (11, 1615)  42
 0xb14f, // (11,  335)  43
 0xc3cf, // (12,  975)  44
 0xcbcf, // (12, 3023)  45
 0xc7cf, // (12, 1999)  46
 0xcfcf, // (12, 4047)  47
 0xc02f, // (12,   47)  48
 0xb54f, // (11, 1359)  49
 0xc82f, // (12, 2095)  50
 0xc42f, // (12, 1071)  51
 0xcc2f, // (12, 3119)  52
 0xc22f, // (12,  559)  53
 0xca2f, // (12, 2607)  54
 0xc62f, // (12, 1583)  55
 0xce2f, // (12, 3631)  56
 0xc12f, // (12,  303)  57
 0xc92f, // (12, 2351)  58
 0xc52f, // (12, 1327)  59
 0xcd2f, // (12, 3375)  60
 0xc32f, // (12,  815)  61
 0xcb2f, // (12, 2863)  62
 0xc72f, // (12, 1839)  63
 0x9077, // ( 9,  119)  64
 0x8057, // ( 8,   87)  65
 0xa08f, // (10,  143)  66
 0x9177, // ( 9,  375)  67
 0xb34f, // (11,  847)  68
 0xa28f, // (10,  655)  69
 0xb74f, // (11, 1871)  70
 0xb0cf, // (11,  207)  71
 0xcf2f, // (12, 3887)  72
 0xb4cf, // (11, 1231)  73
 0xc0af, // (12,  175)  74
 0xc8af, // (12, 2223)  75
 0xc4af, // (12, 1199)  76
 0xccaf, // (12, 3247)  77
 0xc2af, // (12,  687)  78
 0xcaaf, // (12, 2735)  79
 0xc6af, // (12, 1711)  80
 0xceaf, // (12, 3759)  81
 0xc1af, // (12,  431)  82
 0xc9af, // (12, 2479)  83
 0xc5af, // (12, 1455)  84
 0xcdaf, // (12, 3503)  85
 0xc3af, // (12,  943)  86
 0xcbaf, // (12, 2991)  87
 0xc7af, // (12, 1967)  88
 0xcfaf, // (12, 4015)  89
 0xc06f, // (12,  111)  90
 0xc86f, // (12, 2159)  91
 0xc46f, // (12, 1135)  92
 0xcc6f, // (12, 3183)  93
 0xc26f, // (12,  623)  94
 0xca6f, // (12, 2671)  95
 0xc66f, // (12, 1647)  96
 0xce6f, // (12, 3695)  97
 0xc16f, // (12,  367)  98
 0xc96f, // (12, 2415)  99
 0xc56f, // (12, 1391) 100
 0xcd6f, // (12, 3439) 101
 0xc36f, // (12,  879) 102
 0xcb6f, // (12, 2927) 103
 0xc76f, // (12, 1903) 104
 0xcf6f, // (12, 3951) 105
 0xc0ef, // (12,  239) 106
 0xc8ef, // (12, 2287) 107
 0xc4ef, // (12, 1263) 108
 0xccef, // (12, 3311) 109
 0xc2ef, // (12,  751) 110
 0xcaef, // (12, 2799) 111
 0xc6ef, // (12, 1775) 112
 0xceef, // (12, 3823) 113
 0xc1ef, // (12,  495) 114
 0xc9ef, // (12, 2543) 115
 0xc5ef, // (12, 1519) 116
 0xcdef, // (12, 3567) 117
 0xc3ef, // (12, 1007) 118
 0xcbef, // (12, 3055) 119
 0xc7ef, // (12, 2031) 120
 0xcfef, // (12, 4079) 121
 0xc01f, // (12,   31) 122
 0xc81f, // (12, 2079) 123
 0xc41f, // (12, 1055) 124
 0xcc1f, // (12, 3103) 125
 0xc21f, // (12,  543) 126
 0xca1f, // (12, 2591) 127
 0xa18f, // (10,  399) 128
 0x90f7, // ( 9,  247) 129
 0xb2cf, // (11,  719) 130
 0xa38f, // (10,  911) 131
 0xc61f, // (12, 1567) 132
 0xb6cf, // (11, 1743) 133
 0xce1f, // (12, 3615) 134
 0xb1cf, // (11,  463) 135
 0xc11f, // (12,  287) 136
 0xc91f, // (12, 2335) 137
 0xc51f, // (12, 1311) 138
 0xcd1f, // (12, 3359) 139
 0xc31f, // (12,  799) 140
 0xcb1f, // (12, 2847) 141
 0xc71f, // (12, 1823) 142
 0xcf1f, // (12, 3871) 143
 0xc09f, // (12,  159) 144
 0xc89f, // (12, 2207) 145
 0xc49f, // (12, 1183) 146
 0xcc9f, // (12, 3231) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 20 of 22) (midrange 4 of 6) (c/k = 1.833333333 = 11.0 / 6.0)
 // entropy:    4.1937026483207340277
 // avg_length: 4.2809622975207295426; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x4006, // ( 4,    6)   0
 0x2000, // ( 2,    0)   1
 0x400e, // ( 4,   14)   2
 0x3002, // ( 3,    2)   3
 0x5005, // ( 5,    5)   4
 0x4001, // ( 4,    1)   5
 0x5015, // ( 5,   21)   6
 0x4009, // ( 4,    9)   7
 0x6003, // ( 6,    3)   8
 0x500d, // ( 5,   13)   9
 0x6023, // ( 6,   35)  10
 0x501d, // ( 5,   29)  11
 0x8047, // ( 8,   71)  12
 0x6013, // ( 6,   19)  13
 0x80c7, // ( 8,  199)  14
 0x6033, // ( 6,   51)  15
 0x701b, // ( 7,   27)  16
 0x600b, // ( 6,   11)  17
 0x8027, // ( 8,   39)  18
 0x602b, // ( 6,   43)  19
 0x90d7, // ( 9,  215)  20
 0x705b, // ( 7,   91)  21
 0x91d7, // ( 9,  471)  22
 0x703b, // ( 7,   59)  23
 0xa1f7, // (10,  503)  24
 0x80a7, // ( 8,  167)  25
 0xa3f7, // (10, 1015)  26
 0x8067, // ( 8,  103)  27
 0xb24f, // (11,  591)  28
 0xa00f, // (10,   15)  29
 0xb64f, // (11, 1615)  30
 0xa20f, // (10,  527)  31
 0x9037, // ( 9,   55)  32
 0x707b, // ( 7,  123)  33
 0x9137, // ( 9,  311)  34
 0x7007, // ( 7,    7)  35
 0xa10f, // (10,  271)  36
 0x80e7, // ( 8,  231)  37
 0xa30f, // (10,  783)  38
 0x8017, // ( 8,   23)  39
 0xb14f, // (11,  335)  40
 0x90b7, // ( 9,  183)  41
 0xb54f, // (11, 1359)  42
 0xa08f, // (10,  143)  43
 0xc02f, // (12,   47)  44
 0xb34f, // (11,  847)  45
 0xc82f, // (12, 2095)  46
 0xb74f, // (11, 1871)  47
 0xc42f, // (12, 1071)  48
 0xb0cf, // (11,  207)  49
 0xcc2f, // (12, 3119)  50
 0xb4cf, // (11, 1231)  51
 0xc22f, // (12,  559)  52
 0xca2f, // (12, 2607)  53
 0xc62f, // (12, 1583)  54
 0xce2f, // (12, 3631)  55
 0xc12f, // (12,  303)  56
 0xc92f, // (12, 2351)  57
 0xc52f, // (12, 1327)  58
 0xcd2f, // (12, 3375)  59
 0xc32f, // (12,  815)  60
 0xcb2f, // (12, 2863)  61
 0xc72f, // (12, 1839)  62
 0xcf2f, // (12, 3887)  63
 0xa28f, // (10,  655)  64
 0x8097, // ( 8,  151)  65
 0xa18f, // (10,  399)  66
 0x8057, // ( 8,   87)  67
 0xb2cf, // (11,  719)  68
 0x91b7, // ( 9,  439)  69
 0xb6cf, // (11, 1743)  70
 0x9077, // ( 9,  119)  71
 0xc0af, // (12,  175)  72
 0xb1cf, // (11,  463)  73
 0xc8af, // (12, 2223)  74
 0xb5cf, // (11, 1487)  75
 0xc4af, // (12, 1199)  76
 0xccaf, // (12, 3247)  77
 0xc2af, // (12,  687)  78
 0xcaaf, // (12, 2735)  79
 0xc6af, // (12, 1711)  80
 0xceaf, // (12, 3759)  81
 0xc1af, // (12,  431)  82
 0xc9af, // (12, 2479)  83
 0xc5af, // (12, 1455)  84
 0xcdaf, // (12, 3503)  85
 0xc3af, // (12,  943)  86
 0xcbaf, // (12, 2991)  87
 0xc7af, // (12, 1967)  88
 0xcfaf, // (12, 4015)  89
 0xc06f, // (12,  111)  90
 0xc86f, // (12, 2159)  91
 0xc46f, // (12, 1135)  92
 0xcc6f, // (12, 3183)  93
 0xc26f, // (12,  623)  94
 0xca6f, // (12, 2671)  95
 0xc66f, // (12, 1647)  96
 0xce6f, // (12, 3695)  97
 0xc16f, // (12,  367)  98
 0xc96f, // (12, 2415)  99
 0xc56f, // (12, 1391) 100
 0xcd6f, // (12, 3439) 101
 0xc36f, // (12,  879) 102
 0xcb6f, // (12, 2927) 103
 0xc76f, // (12, 1903) 104
 0xcf6f, // (12, 3951) 105
 0xc0ef, // (12,  239) 106
 0xc8ef, // (12, 2287) 107
 0xc4ef, // (12, 1263) 108
 0xccef, // (12, 3311) 109
 0xc2ef, // (12,  751) 110
 0xcaef, // (12, 2799) 111
 0xc6ef, // (12, 1775) 112
 0xceef, // (12, 3823) 113
 0xc1ef, // (12,  495) 114
 0xc9ef, // (12, 2543) 115
 0xc5ef, // (12, 1519) 116
 0xcdef, // (12, 3567) 117
 0xc3ef, // (12, 1007) 118
 0xcbef, // (12, 3055) 119
 0xc7ef, // (12, 2031) 120
 0xcfef, // (12, 4079) 121
 0xc01f, // (12,   31) 122
 0xc81f, // (12, 2079) 123
 0xc41f, // (12, 1055) 124
 0xcc1f, // (12, 3103) 125
 0xc21f, // (12,  543) 126
 0xca1f, // (12, 2591) 127
 0xb3cf, // (11,  975) 128
 0x9177, // ( 9,  375) 129
 0xb7cf, // (11, 1999) 130
 0x90f7, // ( 9,  247) 131
 0xc61f, // (12, 1567) 132
 0xa38f, // (10,  911) 133
 0xce1f, // (12, 3615) 134
 0xa04f, // (10,   79) 135
 0xc11f, // (12,  287) 136
 0xc91f, // (12, 2335) 137
 0xc51f, // (12, 1311) 138
 0xcd1f, // (12, 3359) 139
 0xc31f, // (12,  799) 140
 0xcb1f, // (12, 2847) 141
 0xc71f, // (12, 1823) 142
 0xcf1f, // (12, 3871) 143
 0xc09f, // (12,  159) 144
 0xc89f, // (12, 2207) 145
 0xc49f, // (12, 1183) 146
 0xcc9f, // (12, 3231) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
},

 // (table 21 of 22) (midrange 5 of 6) (c/k = 2.166666667 = 13.0 / 6.0)
 // entropy:    4.3601926041863263706
 // avg_length: 4.4384101723259572481; max_length = 12; num_symbols = 256
{
//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x5009, // ( 5,    9)   0
 0x3002, // ( 3,    2)   1
 0x5019, // ( 5,   25)   2
 0x2000, // ( 2,    0)   3
 0x6003, // ( 6,    3)   4
 0x4001, // ( 4,    1)   5
 0x5005, // ( 5,    5)   6
 0x3006, // ( 3,    6)   7
 0x702b, // ( 7,   43)   8
 0x5015, // ( 5,   21)   9
 0x706b, // ( 7,  107)  10
 0x500d, // ( 5,   13)  11
 0x8007, // ( 8,    7)  12
 0x6023, // ( 6,   35)  13
 0x8087, // ( 8,  135)  14
 0x501d, // ( 5,   29)  15
 0x8047, // ( 8,   71)  16
 0x6013, // ( 6,   19)  17
 0x80c7, // ( 8,  199)  18
 0x6033, // ( 6,   51)  19
 0x9097, // ( 9,  151)  20
 0x701b, // ( 7,   27)  21
 0x9197, // ( 9,  407)  22
 0x600b, // ( 6,   11)  23
 0xa0f7, // (10,  247)  24
 0x8027, // ( 8,   39)  25
 0xa2f7, // (10,  759)  26
 0x80a7, // ( 8,  167)  27
 0xb14f, // (11,  335)  28
 0x9057, // ( 9,   87)  29
 0xb54f, // (11, 1359)  30
 0x9157, // ( 9,  343)  31
 0x90d7, // ( 9,  215)  32
 0x705b, // ( 7,   91)  33
 0x91d7, // ( 9,  471)  34
 0x703b, // ( 7,   59)  35
 0xa1f7, // (10,  503)  36
 0x8067, // ( 8,  103)  37
 0xa3f7, // (10, 1015)  38
 0x707b, // ( 7,  123)  39
 0xb34f, // (11,  847)  40
 0x9037, // ( 9,   55)  41
 0xb74f, // (11, 1871)  42
 0x9137, // ( 9,  311)  43
 0xc12f, // (12,  303)  44
 0xa00f, // (10,   15)  45
 0xc92f, // (12, 2351)  46
 0xa20f, // (10,  527)  47
 0xc52f, // (12, 1327)  48
 0xa10f, // (10,  271)  49
 0xcd2f, // (12, 3375)  50
 0xa30f, // (10,  783)  51
 0xc32f, // (12,  815)  52
 0xb0cf, // (11,  207)  53
 0xcb2f, // (12, 2863)  54
 0xb4cf, // (11, 1231)  55
 0xc72f, // (12, 1839)  56
 0xcf2f, // (12, 3887)  57
 0xc0af, // (12,  175)  58
 0xc8af, // (12, 2223)  59
 0xc4af, // (12, 1199)  60
 0xccaf, // (12, 3247)  61
 0xc2af, // (12,  687)  62
 0xcaaf, // (12, 2735)  63
 0xa08f, // (10,  143)  64
 0x80e7, // ( 8,  231)  65
 0xa28f, // (10,  655)  66
 0x8017, // ( 8,   23)  67
 0xb2cf, // (11,  719)  68
 0x90b7, // ( 9,  183)  69
 0xb6cf, // (11, 1743)  70
 0x91b7, // ( 9,  439)  71
 0xc6af, // (12, 1711)  72
 0xa18f, // (10,  399)  73
 0xceaf, // (12, 3759)  74
 0xa38f, // (10,  911)  75
 0xc1af, // (12,  431)  76
 0xb1cf, // (11,  463)  77
 0xc9af, // (12, 2479)  78
 0xb5cf, // (11, 1487)  79
 0xc5af, // (12, 1455)  80
 0xb3cf, // (11,  975)  81
 0xcdaf, // (12, 3503)  82
 0xb7cf, // (11, 1999)  83
 0xc3af, // (12,  943)  84
 0xcbaf, // (12, 2991)  85
 0xc7af, // (12, 1967)  86
 0xcfaf, // (12, 4015)  87
 0xc06f, // (12,  111)  88
 0xc86f, // (12, 2159)  89
 0xc46f, // (12, 1135)  90
 0xcc6f, // (12, 3183)  91
 0xc26f, // (12,  623)  92
 0xca6f, // (12, 2671)  93
 0xc66f, // (12, 1647)  94
 0xce6f, // (12, 3695)  95
 0xc16f, // (12,  367)  96
 0xc96f, // (12, 2415)  97
 0xc56f, // (12, 1391)  98
 0xcd6f, // (12, 3439)  99
 0xc36f, // (12,  879) 100
 0xcb6f, // (12, 2927) 101
 0xc76f, // (12, 1903) 102
 0xcf6f, // (12, 3951) 103
 0xc0ef, // (12,  239) 104
 0xc8ef, // (12, 2287) 105
 0xc4ef, // (12, 1263) 106
 0xccef, // (12, 3311) 107
 0xc2ef, // (12,  751) 108
 0xcaef, // (12, 2799) 109
 0xc6ef, // (12, 1775) 110
 0xceef, // (12, 3823) 111
 0xc1ef, // (12,  495) 112
 0xc9ef, // (12, 2543) 113
 0xc5ef, // (12, 1519) 114
 0xcdef, // (12, 3567) 115
 0xc3ef, // (12, 1007) 116
 0xcbef, // (12, 3055) 117
 0xc7ef, // (12, 2031) 118
 0xcfef, // (12, 4079) 119
 0xc01f, // (12,   31) 120
 0xc81f, // (12, 2079) 121
 0xc41f, // (12, 1055) 122
 0xcc1f, // (12, 3103) 123
 0xc21f, // (12,  543) 124
 0xca1f, // (12, 2591) 125
 0xc61f, // (12, 1567) 126
 0xce1f, // (12, 3615) 127
 0xb02f, // (11,   47) 128
 0x9077, // ( 9,  119) 129
 0xb42f, // (11, 1071) 130
 0x9177, // ( 9,  375) 131
 0xc11f, // (12,  287) 132
 0xa04f, // (10,   79) 133
 0xc91f, // (12, 2335) 134
 0xa24f, // (10,  591) 135
 0xc51f, // (12, 1311) 136
 0xb22f, // (11,  559) 137
 0xcd1f, // (12, 3359) 138
 0xb62f, // (11, 1583) 139
 0xc31f, // (12,  799) 140
 0xcb1f, // (12, 2847) 141
 0xc71f, // (12, 1823) 142
 0xcf1f, // (12, 3871) 143
 0xc09f, // (12,  159) 144
 0xc89f, // (12, 2207) 145
 0xc49f, // (12, 1183) 146
 0xcc9f, // (12, 3231) 147
 0xc29f, // (12,  671) 148
 0xca9f, // (12, 2719) 149
 0xc69f, // (12, 1695) 150
 0xce9f, // (12, 3743) 151
 0xc19f, // (12,  415) 152
 0xc99f, // (12, 2463) 153
 0xc59f, // (12, 1439) 154
 0xcd9f, // (12, 3487) 155
 0xc39f, // (12,  927) 156
 0xcb9f, // (12, 2975) 157
 0xc79f, // (12, 1951) 158
 0xcf9f, // (12, 3999) 159
 0xc05f, // (12,   95) 160
 0xc85f, // (12, 2143) 161
 0xc45f, // (12, 1119) 162
 0xcc5f, // (12, 3167) 163
 0xc25f, // (12,  607) 164
 0xca5f, // (12, 2655) 165
 0xc65f, // (12, 1631) 166
 0xce5f, // (12, 3679) 167
 0xc15f, // (12,  351) 168
 0xc95f, // (12, 2399) 169
 0xc55f, // (12, 1375) 170
 0xcd5f, // (12, 3423) 171
 0xc35f, // (12,  863) 172
 0xcb5f, // (12, 2911) 173
 0xc75f, // (12, 1887) 174
 0xcf5f, // (12, 3935) 175
 0xc0df, // (12,  223) 176
 0xc8df, // (12, 2271) 177
 0xc4df, // (12, 1247) 178
 0xccdf, // (12, 3295) 179
 0xc2df, // (12,  735) 180
 0xcadf, // (12, 2783) 181
 0xc6df, // (12, 1759) 182
 0xcedf, // (12, 3807) 183
 0xc1df, // (12,  479) 184
 0xc9df, // (12, 2527) 185
 0xc5df, // (12, 1503) 186
 0xcddf, // (12, 3551) 187
 0xc3df, // (12,  991) 188
 0xcbdf, // (12, 3039) 189
 0xc7df, // (12, 2015) 190
 0xcfdf, // (12, 4063) 191
 0xc03f, // (12,   63) 192
 0xc83f, // (12, 2111) 193
 0xc43f, // (12, 1087) 194
 0xcc3f, // (12, 3135) 195
 0xc23f, // (12,  575) 196
 0xca3f, // (12, 2623) 197
 0xc63f, // (12, 1599) 198
 0xce3f, // (12, 3647) 199
 0xc13f, // (12,  319) 200
 0xc93f, // (12, 2367) 201
 0xc53f, // (12, 1343) 202
 0xcd3f, // (12, 3391) 203
 0xc33f, // (12,  831) 204
 0xcb3f, // (12, 2879) 205
 0xc73f, // (12, 1855) 206
 0xcf3f, // (12, 3903) 207
 0xc0bf, // (12,  191) 208
 0xc8bf, // (12, 2239) 209
 0xc4bf, // (12, 1215) 210
 0xccbf, // (12, 3263) 211
 0xc2bf, // (12,  703) 212
 0xcabf, // (12, 2751) 213
 0xc6bf, // (12, 1727) 214
 0xcebf, // (12, 3775) 215
 0xc1bf, // (12,  447) 216
 0xc9bf, // (12, 2495) 217
 0xc5bf, // (12, 1471) 218
 0xcdbf, // (12, 3519) 219
 0xc3bf, // (12,  959) 220
 0xcbbf, // (12, 3007) 221
 0xc7bf, // (12, 1983) 222
 0xcfbf, // (12, 4031) 223
 0xc07f, // (12,  127) 224
 0xc87f, // (12, 2175) 225
 0xc47f, // (12, 1151) 226
 0xcc7f, // (12, 3199) 227
 0xc27f, // (12,  639) 228
 0xca7f, // (12, 2687) 229
 0xc67f, // (12, 1663) 230
 0xce7f, // (12, 3711) 231
 0xc17f, // (12,  383) 232
 0xc97f, // (12, 2431) 233
 0xc57f, // (12, 1407) 234
 0xcd7f, // (12, 3455) 235
 0xc37f, // (12,  895) 236
 0xcb7f, // (12, 2943) 237
 0xc77f, // (12, 1919) 238
 0xcf7f, // (12, 3967) 239
 0xc0ff, // (12,  255) 240
 0xc8ff, // (12, 2303) 241
 0xc4ff, // (12, 1279) 242
 0xccff, // (12, 3327) 243
 0xc2ff, // (12,  767) 244
 0xcaff, // (12, 2815) 245
 0xc6ff, // (12, 1791) 246
 0xceff, // (12, 3839) 247
 0xc1ff, // (12,  511) 248
 0xc9ff, // (12, 2559) 249
 0xc5ff, // (12, 1535) 250
 0xcdff, // (12, 3583) 251
 0xc3ff, // (12, 1023) 252
 0xcbff, // (12, 3071) 253
 0xc7ff, // (12, 2047) 254
 0xcfff  // (12, 4095) 255
}
};

/************************************************************************************************************/

/* Notice that there are only 65 symbols here, which is different from our
   usual 8->12 coding scheme which handles 256 symbols. */

static const uint16_t length_limited_unary_encoding_table65[65] = {
 // Length-limited "unary" code with 65 symbols.
 // entropy:    2.0
 // avg_length: 2.0249023437500000000; max_length = 12; num_symbols = 65

//table, // (4 bits,  12 bits) symbol
//entry, // (length, codeword) [byte]
 0x1000, // ( 1,    0)   0
 0x2001, // ( 2,    1)   1
 0x3003, // ( 3,    3)   2
 0x4007, // ( 4,    7)   3
 0x500f, // ( 5,   15)   4
 0x701f, // ( 7,   31)   5
 0x805f, // ( 8,   95)   6
 0x80df, // ( 8,  223)   7
 0xa03f, // (10,   63)   8
 0xa23f, // (10,  575)   9
 0xb13f, // (11,  319)  10
 0xc53f, // (12, 1343)  11
 0xcd3f, // (12, 3391)  12
 0xc33f, // (12,  831)  13
 0xcb3f, // (12, 2879)  14
 0xc73f, // (12, 1855)  15
 0xcf3f, // (12, 3903)  16
 0xc0bf, // (12,  191)  17
 0xc8bf, // (12, 2239)  18
 0xc4bf, // (12, 1215)  19
 0xccbf, // (12, 3263)  20
 0xc2bf, // (12,  703)  21
 0xcabf, // (12, 2751)  22
 0xc6bf, // (12, 1727)  23
 0xcebf, // (12, 3775)  24
 0xc1bf, // (12,  447)  25
 0xc9bf, // (12, 2495)  26
 0xc5bf, // (12, 1471)  27
 0xcdbf, // (12, 3519)  28
 0xc3bf, // (12,  959)  29
 0xcbbf, // (12, 3007)  30
 0xc7bf, // (12, 1983)  31
 0xcfbf, // (12, 4031)  32
 0xc07f, // (12,  127)  33
 0xc87f, // (12, 2175)  34
 0xc47f, // (12, 1151)  35
 0xcc7f, // (12, 3199)  36
 0xc27f, // (12,  639)  37
 0xca7f, // (12, 2687)  38
 0xc67f, // (12, 1663)  39
 0xce7f, // (12, 3711)  40
 0xc17f, // (12,  383)  41
 0xc97f, // (12, 2431)  42
 0xc57f, // (12, 1407)  43
 0xcd7f, // (12, 3455)  44
 0xc37f, // (12,  895)  45
 0xcb7f, // (12, 2943)  46
 0xc77f, // (12, 1919)  47
 0xcf7f, // (12, 3967)  48
 0xc0ff, // (12,  255)  49
 0xc8ff, // (12, 2303)  50
 0xc4ff, // (12, 1279)  51
 0xccff, // (12, 3327)  52
 0xc2ff, // (12,  767)  53
 0xcaff, // (12, 2815)  54
 0xc6ff, // (12, 1791)  55
 0xceff, // (12, 3839)  56
 0xc1ff, // (12,  511)  57
 0xc9ff, // (12, 2559)  58
 0xc5ff, // (12, 1535)  59
 0xcdff, // (12, 3583)  60
 0xc3ff, // (12, 1023)  61
 0xcbff, // (12, 3071)  62
 0xc7ff, // (12, 2047)  63
 0xcfff  // (12, 4095)  64
};

/*
Note: these column permutations are part of the encoding scheme for sketches where C >= 3.375 * K.
In each row, we identify the (0-based) column indices of all surprising bits
outside of the high-entropy byte. 

These indices are "rotated right" via the formula
new = (old - (8+shift_by) + 64) mod 64 = (old + 56 - shift_by) mod 64.
resulting in canonicalized indices between 0 and 55 inclusive. 

These are then mapped through the forwards permutation specified below (and selected
by the phase of C / K). Finally, the remapped indices are encoding with a unary code 
(with delta encoding for rows containing more than one surprising bit).
*/

// These permutations were created by
// the ocaml program "generatePermutationsForSLIDING.ml".

static const uint8_t column_permutations_for_encoding[16][56] = {
  // for phase = 1 / 32
  {0, 1, 2, 3, 5, 6, 7, 8, 9, 10, 11, 12, 13, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 34, 14, 4},
  // for phase = 3 / 32
  {0, 1, 2, 3, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 35, 15, 4},
  // for phase = 5 / 32
  {0, 1, 2, 3, 4, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 37, 16, 5},
  // for phase = 7 / 32
  {0, 1, 2, 3, 4, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 39, 17, 5},
  // for phase = 9 / 32
  {0, 1, 2, 3, 4, 5, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
   40, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 41, 18, 6},
  // for phase = 11 / 32
  {0, 1, 2, 3, 4, 5, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39,
   40, 41, 42, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 43, 19, 6},
  // for phase = 13 / 32
  {1, 2, 3, 4, 5, 6, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 21, 22,
   23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 45, 20, 7, 0},
  // for phase = 15 / 32
  {1, 2, 3, 4, 5, 6, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 22,
   23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 48, 49, 50, 51, 52, 53, 54, 55, 47, 21, 7, 0},
  // for phase = 17 / 32
  {1, 2, 3, 4, 5, 6, 7, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 51, 52, 53, 54, 55, 50, 22, 8, 0},
  // for phase = 19 / 32
  {0, 2, 3, 4, 5, 6, 7, 8, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 53, 54, 55, 52, 23, 9, 1},
  // for phase = 21 / 32
  {0, 2, 3, 4, 5, 6, 7, 8, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 25, 9, 1},
  // for phase = 23 / 32
  {0, 2, 3, 4, 5, 6, 7, 8, 9, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 26, 10, 1},
  // for phase = 25 / 32
  {0, 1, 3, 4, 5, 6, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 27, 11, 2},
  // for phase = 27 / 32
  {0, 1, 3, 4, 5, 6, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 29, 11, 2},
  // for phase = 29 / 32
  {0, 1, 2, 4, 5, 6, 7, 8, 9, 10, 11, 13, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 30, 12, 3},
  // for phase = 31 / 32
  {0, 1, 2, 4, 5, 6, 7, 8, 9, 10, 11, 12, 14, 15, 16, 17, 18, 19, 20, 21,
   22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 33, 34, 35, 36, 37, 38, 39, 40,
   41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 32, 13, 3}
};

} /* namespace datasketches */

#endif
