/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.kotlin.components

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Unit
import org.apache.camel.kotlin.CamelDslMarker
import org.apache.camel.kotlin.UriDsl

/**
 * Read and write nodes to/from a JCR compliant content repository.
 */
public fun UriDsl.jcr(i: JcrUriDsl.() -> Unit) {
  JcrUriDsl(this).apply(i)
}

@CamelDslMarker
public class JcrUriDsl(
  it: UriDsl,
) {
  private val it: UriDsl

  init {
    this.it = it
    this.it.component("jcr")
  }

  private var host: String = ""

  private var base: String = ""

  /**
   * Name of the javax.jcr.Repository to lookup from the Camel registry to be used.
   */
  public fun host(host: String) {
    this.host = host
    it.url("$host/$base")
  }

  /**
   * Get the base node when accessing the repository
   */
  public fun base(base: String) {
    this.base = base
    it.url("$host/$base")
  }

  /**
   * When isDeep is true, events whose associated parent node is at absPath or within its subgraph
   * are received.
   */
  public fun deep(deep: String) {
    it.property("deep", deep)
  }

  /**
   * When isDeep is true, events whose associated parent node is at absPath or within its subgraph
   * are received.
   */
  public fun deep(deep: Boolean) {
    it.property("deep", deep.toString())
  }

  /**
   * eventTypes (a combination of one or more event types encoded as a bit mask value such as
   * javax.jcr.observation.Event.NODE_ADDED, javax.jcr.observation.Event.NODE_REMOVED, etc.).
   */
  public fun eventTypes(eventTypes: String) {
    it.property("eventTypes", eventTypes)
  }

  /**
   * eventTypes (a combination of one or more event types encoded as a bit mask value such as
   * javax.jcr.observation.Event.NODE_ADDED, javax.jcr.observation.Event.NODE_REMOVED, etc.).
   */
  public fun eventTypes(eventTypes: Int) {
    it.property("eventTypes", eventTypes.toString())
  }

  /**
   * When a comma separated nodeTypeName list string is set, only events whose associated parent
   * node has one of the node types (or a subtype of one of the node types) in this list will be
   * received.
   */
  public fun nodeTypeNames(nodeTypeNames: String) {
    it.property("nodeTypeNames", nodeTypeNames)
  }

  /**
   * If noLocal is true, then events generated by the session through which the listener was
   * registered are ignored. Otherwise, they are not ignored.
   */
  public fun noLocal(noLocal: String) {
    it.property("noLocal", noLocal)
  }

  /**
   * If noLocal is true, then events generated by the session through which the listener was
   * registered are ignored. Otherwise, they are not ignored.
   */
  public fun noLocal(noLocal: Boolean) {
    it.property("noLocal", noLocal.toString())
  }

  /**
   * Password for login
   */
  public fun password(password: String) {
    it.property("password", password)
  }

  /**
   * Interval in milliseconds to wait before each session live checking The default value is 60000
   * ms.
   */
  public fun sessionLiveCheckInterval(sessionLiveCheckInterval: String) {
    it.property("sessionLiveCheckInterval", sessionLiveCheckInterval)
  }

  /**
   * Interval in milliseconds to wait before the first session live checking. The default value is
   * 3000 ms.
   */
  public fun sessionLiveCheckIntervalOnStart(sessionLiveCheckIntervalOnStart: String) {
    it.property("sessionLiveCheckIntervalOnStart", sessionLiveCheckIntervalOnStart)
  }

  /**
   * Username for login
   */
  public fun username(username: String) {
    it.property("username", username)
  }

  /**
   * When a comma separated uuid list string is set, only events whose associated parent node has
   * one of the identifiers in the comma separated uuid list will be received.
   */
  public fun uuids(uuids: String) {
    it.property("uuids", uuids)
  }

  /**
   * The workspace to access. If it's not specified then the default one will be used
   */
  public fun workspaceName(workspaceName: String) {
    it.property("workspaceName", workspaceName)
  }

  /**
   * Allows for bridging the consumer to the Camel routing Error Handler, which mean any exceptions
   * (if possible) occurred while the Camel consumer is trying to pickup incoming messages, or the
   * likes, will now be processed as a message and handled by the routing Error Handler. Important:
   * This is only possible if the 3rd party component allows Camel to be alerted if an exception was
   * thrown. Some components handle this internally only, and therefore bridgeErrorHandler is not
   * possible. In other situations we may improve the Camel component to hook into the 3rd party
   * component and make this possible for future releases. By default the consumer will use the
   * org.apache.camel.spi.ExceptionHandler to deal with exceptions, that will be logged at WARN or
   * ERROR level and ignored.
   */
  public fun bridgeErrorHandler(bridgeErrorHandler: String) {
    it.property("bridgeErrorHandler", bridgeErrorHandler)
  }

  /**
   * Allows for bridging the consumer to the Camel routing Error Handler, which mean any exceptions
   * (if possible) occurred while the Camel consumer is trying to pickup incoming messages, or the
   * likes, will now be processed as a message and handled by the routing Error Handler. Important:
   * This is only possible if the 3rd party component allows Camel to be alerted if an exception was
   * thrown. Some components handle this internally only, and therefore bridgeErrorHandler is not
   * possible. In other situations we may improve the Camel component to hook into the 3rd party
   * component and make this possible for future releases. By default the consumer will use the
   * org.apache.camel.spi.ExceptionHandler to deal with exceptions, that will be logged at WARN or
   * ERROR level and ignored.
   */
  public fun bridgeErrorHandler(bridgeErrorHandler: Boolean) {
    it.property("bridgeErrorHandler", bridgeErrorHandler.toString())
  }

  /**
   * To let the consumer use a custom ExceptionHandler. Notice if the option bridgeErrorHandler is
   * enabled then this option is not in use. By default the consumer will deal with exceptions, that
   * will be logged at WARN or ERROR level and ignored.
   */
  public fun exceptionHandler(exceptionHandler: String) {
    it.property("exceptionHandler", exceptionHandler)
  }

  /**
   * Sets the exchange pattern when the consumer creates an exchange.
   */
  public fun exchangePattern(exchangePattern: String) {
    it.property("exchangePattern", exchangePattern)
  }

  /**
   * Whether the producer should be started lazy (on the first message). By starting lazy you can
   * use this to allow CamelContext and routes to startup in situations where a producer may otherwise
   * fail during starting and cause the route to fail being started. By deferring this startup to be
   * lazy then the startup failure can be handled during routing messages via Camel's routing error
   * handlers. Beware that when the first message is processed then creating and starting the producer
   * may take a little time and prolong the total processing time of the processing.
   */
  public fun lazyStartProducer(lazyStartProducer: String) {
    it.property("lazyStartProducer", lazyStartProducer)
  }

  /**
   * Whether the producer should be started lazy (on the first message). By starting lazy you can
   * use this to allow CamelContext and routes to startup in situations where a producer may otherwise
   * fail during starting and cause the route to fail being started. By deferring this startup to be
   * lazy then the startup failure can be handled during routing messages via Camel's routing error
   * handlers. Beware that when the first message is processed then creating and starting the producer
   * may take a little time and prolong the total processing time of the processing.
   */
  public fun lazyStartProducer(lazyStartProducer: Boolean) {
    it.property("lazyStartProducer", lazyStartProducer.toString())
  }
}
