/*
    This file is part of KOrganizer.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <qcstring.h>
#include <qwhatsthis.h>
#include <qdialog.h>
#include <qapplication.h>
#include <qlabel.h>
#include <qlayout.h>

#include <klocale.h>
#include <kapplication.h>
#include <libkcal/event.h>
#include <libkcal/todo.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <krun.h>
#include <kglobal.h>
#include <kprocess.h>
#include "koprefs.h"

#include <libkdepim/addresseeview.h>
#include <kabc/stdaddressbook.h>

#ifndef KORG_NODCOP
#include <dcopclient.h>
#include "korganizer.h"
#include "koprefs.h"
#include "actionmanager.h"
#endif

#include "koeventviewer.h"
#ifndef KORG_NOKABC
#include <kabc/stdaddressbook.h>
#define size count
#endif

#ifdef DESKTOP_VERSION
#include <kabc/addresseedialog.h>
#else //DESKTOP_VERSION
#include <externalapphandler.h>
#include <qtopia/qcopenvelope_qws.h> 
#endif //DESKTOP_VERSION

KOEventViewer::KOEventViewer(QWidget *parent,const char *name)
  : QTextBrowser(parent,name)
{
    mSyncMode = false;
    mColorMode = 0;
}

KOEventViewer::~KOEventViewer()
{
}

void KOEventViewer::setSource(const QString& n)
{

    if ( n.left(3) == "uid" ) 
#ifdef DESKTOP_VERSION
        {
        KABC::StdAddressBook* AddressBook = KABC::StdAddressBook::self( true );
        KABC::AddressBook::Iterator it;
        for( it = AddressBook->begin(); it != AddressBook->end(); ++it ) {
            // LR I do not understand, why the uid string is different on zaurus and desktop
            QString uid = "uid://"+(*it).uid();

            //qDebug("for *%s*          +%s+ ", n.latin1(), uid.latin1());
            if (n == uid ) {
                //qDebug("found %s ",(*it).mobileHomePhone().latin1() );
                QDialog dia( this,"dia123", true );
                dia.setCaption( i18n("Details of attendee") );
                QVBoxLayout lay ( &dia );
                KPIM::AddresseeView av ( &dia );
                av.setAddressee( (*it) );
                lay.addWidget( &av );
                if ( QApplication::desktop()->width() < 480 )
                    dia.resize( 220, 240);
                else {
                    dia.resize( 400,400);
                }
                dia.exec();
                break;
            } 
        }
        return;
    }
#else  
    {
        if (  "uid:organizer" == n ) {
            ExternalAppHandler::instance()->requestDetailsFromKAPI("", mCurrentIncidence->organizer(),"");
            return;
        } 
        QPtrList<Attendee> attendees = mCurrentIncidence->attendees();
        if (attendees.count()) {  
            Attendee *a;
            for(a=attendees.first();a;a=attendees.next()) {
                if ( "uid:"+a->uid() == n ) {
                    bool res = ExternalAppHandler::instance()->requestDetailsFromKAPI(a->name(), a->email(), a->uid());
                    return;
                }
            }
        }
        return;
    }
    //requestNameEmailUidListFromKAPI("QPE/Application/kopi", this->name() /* name is here the unique uid*/);
  // the result should now arrive through method insertAttendees
    //QString uid = "uid:"+(*it).uid();
#endif
    if ( n.left(6) == "mailto" ) {
        // qDebug("KOEventViewer::setSource %s ", n.mid(7).latin1());
#ifndef DESKTOP_VERSION
        QCopEnvelope e("QPE/Application/ompi", "newMail(QString)" );
        e << n.mid(7);
#endif

    }


#ifndef KORG_NODCOP
    kdDebug() << "KOEventViewer::setSource(): " << n << endl;
    QString tmpStr;
    if (n.startsWith("mailto:")) {
        KApplication::kApplication()->invokeMailer(n.mid(7),QString::null);
        //emit showIncidence(n);
        return;
    } else if (n.startsWith("uid:")) {
        DCOPClient *client = KApplication::kApplication()->dcopClient();
        const QByteArray noParamData;
        const QByteArray paramData;
        QByteArray replyData;
        QCString replyTypeStr;
#define PING_ABBROWSER (client->call("kaddressbook", "KAddressBookIface", "interfaces()",  noParamData, replyTypeStr, replyData))
        bool foundAbbrowser = PING_ABBROWSER;

        if (foundAbbrowser) {
            //KAddressbook is already running, so just DCOP to it to bring up the contact editor
            //client->send("kaddressbook","KAddressBookIface",
            QDataStream arg(paramData, IO_WriteOnly);
            arg << n.mid(6);
            client->send("kaddressbook", "KAddressBookIface", "showContactEditor( QString )",  paramData);
            return;
        } else {
            /*
              KaddressBook is not already running.  Pass it the UID of the contact via the command line while starting it - its neater.
              We start it without its main interface
            */
            KIconLoader* iconLoader = new KIconLoader();
            QString iconPath = iconLoader->iconPath("go",KIcon::Small);
            ActionManager::setStartedKAddressBook(true);
            tmpStr = "kaddressbook --editor-only --uid ";
            tmpStr += KProcess::quote(n.mid(6));
            KRun::runCommand(tmpStr,"KAddressBook",iconPath);
            return;
        }
    } else {
        //QTextBrowser::setSource(n);
    }
#endif
}

void KOEventViewer::addTag(const QString & tag,const QString & text)
{
    int number=text.contains("\n");
    QString str = "<" + tag + ">";
    QString tmpText=text;
    QString tmpStr=str;
    if(number !=-1) 
        {
            if (number > 0) {
                int pos=0;
                QString tmp;
                for(int i=0;i<=number;i++) {
                    pos=tmpText.find("\n");
                    tmp=tmpText.left(pos);
                    tmpText=tmpText.right(tmpText.length()-pos-1);
                    tmpStr+=tmp+"<br>";
                }
            }
            else tmpStr += tmpText;
            tmpStr+="</" + tag + ">";
            mText.append(tmpStr);
        }
    else
        {
            str += text + "</" + tag + ">";
            mText.append(str);
        }
}

void KOEventViewer::setColorMode( int m )
{
    mColorMode = m;
}
void KOEventViewer::appendEvent(Event *event, int mode )
{  
    mMailSubject = "";
    mCurrentIncidence = event;
    bool shortDate = KOPrefs::instance()->mShortDateInViewer;
    topLevelWidget()->setCaption(i18n("Event Viewer"));
    if ( mode == 0 ) {
        addTag("h2",event->summary());
    }
    else {
        if ( mColorMode == 1 ) {
            mText +="<font color=\"#00A000\">";
        }
        if ( mColorMode == 2 ) {
            mText +="<font color=\"#C00000\">";
        }
        // mText +="<font color=\"#F00000\">" + i18n("O-due!") + "</font>";
        if ( mode == 1 ) {
            addTag("h2",i18n( "Local: " ) +event->summary());
        } else {
            addTag("h2",i18n( "Remote: " ) +event->summary());
        } 
        addTag("h3",i18n( "Last modified: " ) + KGlobal::locale()->formatDateTime(event->lastModified(),shortDate, true ) );
        if ( mColorMode )
            mText += "</font>";
    } 
    mMailSubject += i18n( "Meeting " )+ event->summary();
    if (event->cancelled ()) {
        mText +="<font color=\"#B00000\">";
        addTag("i",i18n("This event has been cancelled!"));
        mText.append("<br>");
        mText += "</font>";
        mMailSubject += i18n("(cancelled)");
    }
    if (!event->location().isEmpty()) {
        addTag("b",i18n("Location: "));
        mText.append(event->location()+"<br>");
        mMailSubject += i18n(" at ") + event->location();
    }
    if (event->doesFloat()) {
        if (event->isMultiDay()) {
            mText.append(i18n("<p><b>From:</b> %1 </p><p><b>To:</b> %2</p>")
                         .arg(event->dtStartDateStr(shortDate))
                         .arg(event->dtEndDateStr(shortDate)));
        } else {
            mText.append(i18n("<p><b>On:</b> %1</p>").arg(event->dtStartDateStr( shortDate )));
        }
    } else {
        if (event->isMultiDay()) {
            mText.append(i18n("<p><b>From:</b> %1</p> ")
                         .arg(event->dtStartStr( shortDate)));
            mText.append(i18n("<p><b>To:</b> %1</p>")
                         .arg(event->dtEndStr(shortDate)));
        } else {
            mText.append(i18n("<p><b>On:</b> %1</p> ")
                         .arg(event->dtStartDateStr( shortDate )));
            mText.append(i18n("<p><b>From:</b> %1 <b>To:</b> %2</p>")
                         .arg(event->dtStartTimeStr())
                         .arg(event->dtEndTimeStr()));
        }
    }

    if (event->recurrence()->doesRecur()) {

        QString recurText = event->recurrence()->recurrenceText();
        addTag("p","<em>" + i18n("This is a %1 recurring event.").arg(recurText ) + "</em>");
        bool last;
        QDate start = QDate::currentDate();
        QDate next;
        next = event->recurrence()->getPreviousDate( start , &last );
        if ( !last ) {
            next = event->recurrence()->getNextDate( start.addDays( - 1 ) );
            addTag("p",i18n("<b>Next recurrence is on:</b>")  );
            addTag("p", KGlobal::locale()->formatDate( next, shortDate ));
            QDateTime nextdt = QDateTime( next, event->dtStart().time());
            mMailSubject += i18n(" - " )+ KGlobal::locale()->formatDateTime( nextdt, true );

        } else {
            addTag("p",i18n("<b>Last recurrence was on:</b>")  );
            addTag("p", KGlobal::locale()->formatDate( next, shortDate ));
        }
    } else {
        mMailSubject += i18n(" - " )+event->dtStartStr( true );

    }


    if (event->isAlarmEnabled()) {
        Alarm *alarm =event->alarms().first() ;
        QDateTime t = alarm->time();
        int min = t.secsTo( event->dtStart() )/60;
        QString s =i18n("( %1 min before )").arg( min );
        addTag("p",i18n("<b>Alarm on: ") + s  +" </b>");
        addTag("p", KGlobal::locale()->formatDateTime( t, shortDate ));
        //addTag("p",s);
    }

    addTag("b",i18n("Access: "));
    mText.append(event->secrecyStr()+"<br>");
    if (!event->description().isEmpty()) {
        addTag("p",i18n("<b>Details: </b>"));
        addTag("p",event->description());
    }

    formatCategories(event);

    formatReadOnly(event);
    formatAttendees(event);

    setText(mText);
    //QWhatsThis::add(this,mText);

}

void KOEventViewer::appendTodo(Todo *event, int mode )
{
    mMailSubject = "";
    mCurrentIncidence = event;
    topLevelWidget()->setCaption(i18n("Todo Viewer"));
    bool shortDate = KOPrefs::instance()->mShortDateInViewer;
    if (mode == 0 )
        addTag("h2",event->summary());
    else { 
        if ( mColorMode == 1 ) {
            mText +="<font color=\"#00A000\">";
        }
        if ( mColorMode == 2 ) {
            mText +="<font color=\"#B00000\">";
        }
        if ( mode == 1 ) {
            addTag("h2",i18n( "Local: " ) +event->summary());
        } else {
            addTag("h2",i18n( "Remote: " ) +event->summary());
        }
        addTag("h3",i18n( "Last modified: " ) + KGlobal::locale()->formatDateTime(event->lastModified(),shortDate, true ) );
        if ( mColorMode )
            mText += "</font>";
    } 
    mMailSubject += i18n( "Todo " )+ event->summary();
    if (event->cancelled ()) {
        mText +="<font color=\"#B00000\">";
        addTag("i",i18n("This todo has been cancelled!"));
        mText.append("<br>");
        mText += "</font>";
        mMailSubject += i18n("(cancelled)");
    }
   
    if (!event->location().isEmpty()) {
        addTag("b",i18n("Location: "));
        mText.append(event->location()+"<br>");
        mMailSubject += i18n(" at ") + event->location();
    }
    if (event->hasDueDate()) {
        mText.append(i18n("<p><b>Due on:</b> %1</p>").arg(event->dtDueStr(KOPrefs::instance()->mShortDateInViewer)));
        mMailSubject += i18n(" - " )+event->dtDueStr(  true );
    }
    addTag("b",i18n("Access: "));
    mText.append(event->secrecyStr()+"<br>");
    if (!event->description().isEmpty()) {
        addTag("p",i18n("<b>Details: </b>"));
        addTag("p",event->description());
    }

    formatCategories(event);

    mText.append(i18n("<p><b>Priority:</b> %2</p>")
                 .arg(QString::number(event->priority())));

    mText.append(i18n("<p><i>%1 % completed</i></p>")
                 .arg(event->percentComplete()));

    formatReadOnly(event);
    formatAttendees(event);

    setText(mText);
}

void KOEventViewer::formatCategories(Incidence *event)
{
    if (!event->categoriesStr().isEmpty()) {
        if (event->categories().count() == 1) {
            addTag("h3",i18n("Category"));
        } else {
            addTag("h3",i18n("Categories"));
        }
        addTag("p",event->categoriesStr());
    }
}
void KOEventViewer::formatAttendees(Incidence *event)
{
    QPtrList<Attendee> attendees = event->attendees();
    if (attendees.count()) {
        
  
        QString iconPath = KGlobal::iconLoader()->iconPath("mailappt",KIcon::Small);
        addTag("h3",i18n("Organizer"));
        mText.append("<ul><li>");
#ifndef KORG_NOKABC
       
#ifdef DESKTOP_VERSION
        KABC::AddressBook *add_book = KABC::StdAddressBook::self();
        KABC::Addressee::List addressList;
        addressList = add_book->findByEmail(event->organizer());
        KABC::Addressee o = addressList.first();
        if (!o.isEmpty() && addressList.size()<2) {
            mText += "<a href=\"uid:" + o.uid() + "\">";
            mText += o.formattedName();
            mText += "</a>\n";
        } else {
            mText.append(event->organizer());
        }
#else //DESKTOP_VERSION
                    mText += "<a href=\"uid:organizer\">";
                    mText += event->organizer();
                    mText += "</a>\n";
#endif //DESKTOP_VERSION


#else
        mText.append(event->organizer());
#endif
       
        if (iconPath) {
            mText += " <a href=\"mailto:" + event->organizer() + "\">";
            mText += "<IMG src=\"" + iconPath + "\">";
            mText += "</a>\n";
        }
        mText.append("</li></ul>");

        addTag("h3",i18n("Attendees"));
        Attendee *a;
        mText.append("<ul>");
        for(a=attendees.first();a;a=attendees.next()) {
#ifndef KORG_NOKABC
#ifdef DESKTOP_VERSION
            if (a->name().isEmpty()) {
                addressList = add_book->findByEmail(a->email());
                KABC::Addressee o = addressList.first();
                if (!o.isEmpty() && addressList.size()<2) {
                    mText += "<a href=\"uid:" + o.uid() + "\">";
                    mText += o.formattedName();
                    mText += "</a>\n";
                } else {
                    mText += "<li>";
                    mText.append(a->email());
                    mText += "\n";
                }
            } else {
                mText += "<li><a href=\"uid:" + a->uid() + "\">";
                if (!a->name().isEmpty()) mText += a->name();
                else mText += a->email();
                mText += "</a>\n";
            }
#else //DESKTOP_VERSION
                mText += "<li><a href=\"uid:" + a->uid() + "\">";
                if (!a->name().isEmpty()) mText += a->name();
                else mText += a->email();
                mText += "</a>\n";
#endif //DESKTOP_VERSION
#else
            //qDebug("nokabc ");
            mText += "<li><a href=\"uid:" + a->uid() + "\">";
            if (!a->name().isEmpty()) mText += a->name();
            else mText += a->email();
            mText += "</a>\n";
#endif

          
            if (!a->email().isEmpty()) {
                if (iconPath) {
                    mText += "<a href=\"mailto:" + a->name() +" <" + a->email() + ">:" + mMailSubject + "\">";
                    mText += "<IMG src=\"" + iconPath + "\">";
                    mText += "</a>\n";
                }
            }
            if (a->status() != Attendee::NeedsAction )
                mText +="[" + a->statusStr() + "] ";
            if (a->role() == Attendee::Chair )
                mText +="(" + a->roleStr().left(1) + ".)";
        }
        mText.append("</li></ul>");
    }

}
void KOEventViewer::appendJournal(Journal *jour, int mode )
{ 
    bool shortDate = KOPrefs::instance()->mShortDateInViewer;
    if (mode == 0 )
        addTag("h2",i18n("Journal from: "));
    else {
        if ( mode == 1 ) {
            addTag("h2",i18n( "Local: " ) +i18n("Journal from: "));
        } else {
            addTag("h2",i18n( "Remote: " ) +i18n("Journal from: ")); 
        } 
        addTag("h3",i18n( "Last modified " ) + KGlobal::locale()->formatDateTime(jour->lastModified(),shortDate ) );
    }
    topLevelWidget()->setCaption("Journal Viewer");
    mText.append(i18n("<h3> %1 </h3> ").arg(jour->dtStartDateStr(KOPrefs::instance()->mShortDateInViewer)));
    if (!jour->description().isEmpty()) {
        addTag("p",jour->description());
    }
    setText(mText);
}

void KOEventViewer::formatReadOnly(Incidence *event)
{
    if (event->isReadOnly()) {
        addTag("p","<em>(" + i18n("read-only") + ")</em>");
    }
}
void KOEventViewer::setSyncMode( bool b )
{
    mSyncMode = b;
}


void KOEventViewer::setTodo(Todo *event, bool clearV )
{
    if ( clearV )
        clearEvents();
    if ( mSyncMode ) {
        if ( clearV )
            appendTodo(event,1 );
        else
            appendTodo(event,2);
    } else
        appendTodo(event);
}
void KOEventViewer::setJournal(Journal *event, bool clearV )
{
    if ( clearV )
        clearEvents();
    if ( mSyncMode ) {
        if ( clearV )
            appendJournal(event, 1); 
        else
            appendJournal(event, 2); 
    } else
        appendJournal(event);
}

void KOEventViewer::setEvent(Event *event)
{
    clearEvents();
    if ( mSyncMode )
        appendEvent(event, 1);
    else
        appendEvent(event);
}

void KOEventViewer::addEvent(Event *event)
{  
    if ( mSyncMode )
        appendEvent(event, 2);
    else
        appendEvent(event);
}

void KOEventViewer::clearEvents(bool now)
{
    mText = "";
    if (now) setText(mText);
}

void KOEventViewer::addText(QString text)
{
    mText.append(text);
    setText(mText);
}
