#include <lct/local.h>

#include <errno.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>

#include <lct/font.h>

static int bdf_char_read(FILE* fontfile, char* chardata_ptr, unsigned short bitmapwidth);

int bdf_read (FILE* fontfile, simple_font* font_ptr)
{
  char* buf = NULL;
  size_t bufsize = 0;
  enum { st_header,				  /* reading header */
	 st_props,				  /* reading font props */
	 st_chars,				  /* reading char data */
	 st_end					  /* waiting for ENDFONT */
  } state = st_header;
  int done = 0;
  unsigned short bitmapwidth;			  /* width in bytes to store a bitmap line */

  while ((!done) && (-1 != getline (&buf, &bufsize, fontfile)))
    switch (state)
      {
      case st_header:
	if (0 == strncmp (buf, "STARTPROPERTIES ", strlen("STARTPROPERTIES ")))
	  /* header is finished */
	  state = st_props;
	else if (0 == strncmp (buf, "FONTBOUNDINGBOX ", strlen("FONTBOUNDINGBOX ")))
	  /* get font metrics */
	  {
	    int dummy;
	    if (4 > sscanf(buf, "FONTBOUNDINGBOX %hu %hu %d %d",
			   &font_ptr->font.charwidth,
			   &font_ptr->font.charheight,
			   &dummy, &dummy))
	      {
		fprintf (stderr, "Bad FONTBOUNDINGBOX line in BDF file\n");
		errno = EBFONT;
		return -1;
	      }

	    /* FIXME: there may be a kernel bug here for width in [17..24] */
	    bitmapwidth = (font_ptr->font.charwidth + 7) / 8;
						  /* from drivers/chars/console.h in 2.2.10 */
  
	    if (font_ptr->font.charheight > 32)
	      {
		fprintf (stderr, _("Font too high\n"));
		errno = EINVAL;
		return -1;
	      }
		
	  }
	break;
	

      case st_props:
	/* FIXME: should reject `SPACING "P"' fonts */
	
	if (0 == strncmp (buf, "ENDPROPERTIES", strlen("ENDPROPERTIES")))
	  /* header is finished */
	  state = st_chars;
	break;


      case st_chars:
	/* number of chars */
	if (1 > sscanf (buf, "CHARS %hu", &font_ptr->font.charcount))
	  {
	    fprintf (stderr, _("Expected CHARS line not found after ENDPROPERTIES\n"));
	    errno = EBFONT;
	    return -1;
	  }

	/* FIXME: chardata already malloc'd assuming <= 512 */
	assert (font_ptr->font.charcount <= 512);

	{
	  int i;
	  int encoding;
	  int sfm_slots = 0;			  /* size allocated in font_ptr->sfm */
	  
	  for (i = 0; i < font_ptr->font.charcount; i++)
	    {
	      if (-1 == (encoding = bdf_char_read(fontfile,
						  font_ptr->font.chardata + bitmapwidth * 32 * i,
						  bitmapwidth)))
		return -1;
	      /* FIXME: assumes ISO8859-1 */
	      unimapdesc_addpair (i, (unicode)encoding, &font_ptr->sfm, &sfm_slots);
	    }

	  unimapdesc_adjust (&font_ptr->sfm);
	}
	
	state = st_end;
	break;

      case st_end:
	if (0 == strncmp (buf, "ENDFONT", strlen("ENDFONT")))
	  done = 1;
	else
	  {
	    fprintf (stderr, _("ENDFONT not found : `%s'\n"), buf);
	    errno = EBFONT;
	    return -1;
	  }
	break;

      default:
	abort();
      }
  
  return 0;
}


/*
 * returns ENCODING field
 */
static int bdf_char_read(FILE* fontfile, char* chardata_ptr, unsigned short bitmapwidth)
{
  enum { st_start,				  /* not yet started */
	 st_attr,				  /* reading char attributes */
	 st_bitmap				  /* reading bitmap */
  } state = st_start;
  int done = 0;
  char* buf;
  int bufsize;
  char charname[16];				  /* BFD specifies 14 chars max */
  unsigned short encoding;			  /* char index in REGISTRY-ENCODING */

  /* clear char bitmap on all 32 lines */
  memset (chardata_ptr, 0, bitmapwidth * 32);
	    
  while (!done)
    {
      if (-1 == getline (&buf, &bufsize, fontfile))
	{
	  fprintf (stderr, _("Error reading character data"));
	  errno = EBFONT;
	  return -1;
	}
      
      switch (state)
	{
	case st_start:
	  if (1 > sscanf (buf, "STARTCHAR %14s", charname))
	    {
	      fprintf (stderr, _("Expected STARTCHAR line not found\n"));
	      free (buf);
	      errno = EBFONT;
	      return -1;
	    }

	  state = st_attr;
	  break;

	case st_attr:
	  /* FIXME: should maybe read BBX and use it to adjust on y axis */
	  if (0 == strncmp (buf, "BITMAP", strlen ("BITMAP")))
	    state = st_bitmap;
	  else if (1 == sscanf (buf, "ENCODING %hu", &encoding))
	      ;
	  break;

	case st_bitmap:
	  if (0 == strncmp (buf, "ENDCHAR", strlen ("ENDCHAR")))
	    done = 1;
	  else
	    {
	      char* ptr;
	      char* bitmap_ptr;
	      unsigned char value;

	      /* for each byte read its value and store it as data */
	      for (ptr = buf, bitmap_ptr = chardata_ptr;
		   ptr[0] != '\n';
		   ptr += 2, bitmap_ptr++)
		{
		  /* decode 1st nibble */
		  if ((ptr[0] >= '0') && (ptr[0] <= '9'))
		    value = (ptr[0] - '0') << 4;
		  else if ((ptr[0] >= 'A') && (ptr[0] <= 'F'))
		    value = (ptr[0] - 'A' + 10) << 4;
		  else if ((ptr[0] >= 'a') && (ptr[0] <= 'f'))
		    value = (ptr[0] - 'a' + 10) << 4;
		  else
		    {
		      fprintf (stderr, "Bad 1st nibble 0x%c\n", ptr[0]);
		      free (buf);
		      errno = EBFONT;
		      return -1;
		    }
		
		  /* decode 2nd nibble */
		  if ((ptr[1] >= '0') && (ptr[1] <= '9'))
		    value |= (ptr[1] - '0');
		  else if ((ptr[1] >= 'A') && (ptr[1] <= 'F'))
		    value |= (ptr[1] - 'A' + 10);
		  else if ((ptr[1] >= 'a') && (ptr[1] <= 'f'))
		    value |= (ptr[1] - 'a' + 10);
		  else
		    {
		      fprintf (stderr, "Bad 2nd nibble 0x%c\n", ptr[1]);
		      free (buf);
		      errno = EBFONT;
		      return -1;
		    }

		  /* store */
		  *bitmap_ptr = value;
		}

	      /* position for next line */
	      chardata_ptr += bitmapwidth;
	    }

	  break;

	default:
	  abort();
	}
    }

  return encoding;
}
