/*** -*- Mode: Javascript; tab-width: 2;

The contents of this file are subject to the Mozilla Public
License Version 1.1 (the "License"); you may not use this file
except in compliance with the License. You may obtain a copy of
the License at http://www.mozilla.org/MPL/

Software distributed under the License is distributed on an "AS
IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
implied. See the License for the specific language governing
rights and limitations under the License.

The Original Code is Collabnet code.
The Initial Developer of the Original Code is Collabnet.

Portions created by Collabnet are
Copyright (C) 2000 Collabnet.  All
Rights Reserved.

Contributor(s): Pete Collins, Doug Turner, Brendan Eich, Warren Harris, Eric Plaster


JS FileUtils IO API (The purpose of this file is to make it a little easier to do file IO from js) 

    fileUtils.js

Function List

    chrome_to_path(aPath)            // Converts a chrome://bob/content uri to a path.  NOTE: although this gives you the
                                     // path to a file in the chrome directory, you will most likely not have permisions
                                     // to create or write to files there.
    URL_to_path(aPath)               // Converts a file:// url to a path
    exists(aPath);                   // check to see if a file exists
    append(aDirPath, aFileName);     // append is for abstracting platform specific file paths
    rm(aPath);                       // remove a file
    copy(aSource, aDest);            // copy a file from source to destination
    leaf(aPath);                     // leaf is the endmost file string ex: foo.html in /myDir/foo.html
    permissions(aPath);              // returns the files permissions
    dateModified(aPath);             // returns the last modified date in locale string
    size(aPath);                     // returns the file size
    extension(aPath);                // returns a file extension if there is one
    dirPath(aPath)                   // returns the dir part of a path
    spawn(aPath, aArgs)              // spawns another program 
    nsIFIle(aPath)                   // returns an nsIFile obj 
    help();                          // currently dumps a list of available functions 

Instructions:

  First include this js file 

   var file = new FileUtils();

  Examples:

   var path='/usr/X11R6/bin/Eterm';
   file.spawn(path, ['-e/usr/bin/vi']); 
   *note* all args passed to spawn must be in the form of an array

  Warning: these API's are not for religious types

*/


/****************** Globals **********************/

const JSFUTILS_LOCAL_CID               = "@mozilla.org/file/local;1";
const JSFUTILS_FILESPEC_PROGID         = '@mozilla.org/filespec;1';
const JSFUTILS_NETWORK_STD_CID         = '@mozilla.org/network/standard-url;1';
const SIMPLEURI_CONTRACTID             = "@mozilla.org/network/simple-uri;1";
const JSFUTILS_LOCATOR_PROGID          = '@mozilla.org/filelocator;1'; 
const JSFUTILS_CHROME_REG_PROGID       = '@mozilla.org/chrome/chrome-registry;1';

const JSFUTILS_I_LOCAL_FILE            = "nsILocalFile";
const JSFUTILS_INIT_W_PATH             = "initWithPath";

const JSFUTILS_OK                      = true;
const JSFUTILS_FilePath                = new Components.Constructor( JSFUTILS_LOCAL_CID, JSFUTILS_I_LOCAL_FILE, JSFUTILS_INIT_W_PATH);

const nsIURI                           = Components.interfaces.nsIURI;
    

/****************** FileUtils Object Class *********************/
function FileUtils() {
} // constructor

FileUtils.prototype  = {
  fileInst        : null,

/********************* CHROME_TO_PATH ***************************/
chrome_to_path : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:nativePath:ERROR: Missing path argument\n");
    return null;
  }

  var uri = Components.classes[SIMPLEURI_CONTRACTID].createInstance(nsIURI);
  var path;

  if(aPath.search(/chrome:/) == 0) {
    try {
      var chromeRegistry = Components.classes["@mozilla.org/chrome/chrome-registry;1"].getService();
      if ( chromeRegistry )
        chromeRegistry = chromeRegistry.QueryInterface( Components.interfaces.nsIChromeRegistry );
    }
    catch(e) {}

    dump("aPath: "+aPath+"\n");
    uri.spec = aPath;
    uri.spec = chromeRegistry.convertChromeURL(uri);
    path = uri.path;


    // The magic number to find the profile directory 
    // Value represents the decimal value of nsFileLocations::App_PrefsDirectory50 
    var CHROME_DIRECTORY = 0x0001001F; 


    var locator_service = Components.classes[JSFUTILS_LOCATOR_PROGID].getService(); 
    locator_service = locator_service.QueryInterface(Components.interfaces.nsIFileLocator); 
    var chrome_directory = locator_service.GetFileLocation(CHROME_DIRECTORY); 
    dump("...chrome_directory: " + chrome_directory.URLString + "\n"); 
    uri.spec = chrome_directory.URLString;

    path = uri.path.slice(2,uri.path.length)+path.slice(8,path.length);
  } else
  if(aPath.search(/file:/) == 0) {
    uri.spec = aPath;
    path = uri.path.slice(2,uri.path.length);
  }
  
  return path;
},

/********************* URL_TO_PATH ***************************/
URL_to_path : function (aPath)
{
  if(!aPath)
  {
    dump("fileUtils.js:URL_to_path:ERROR: Missing path argument\n");
    return null;
  }

  var retval;
  if(aPath.search(/file:/) == 0) {
    try
    {
      var FileURL=new  Components.Constructor(JSFUTILS_NETWORK_STD_CID, "nsIURL");
      var fileURL=new FileURL();
      fileURL.spec=aPath;
      retval = fileURL.path;
    }

    catch (e)
    { 
      dump("fileUtils.js:URL_to_path:ERROR: "+e.name+"\n"+e.message+'\n');
      retval=null;
    }
  }

  return retval;
},

/********************* EXISTS ***************************/
exists : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:exists:ERROR: Missing path argument\n");
    return null;
  }

  var retval;
  try
  { 
    var file                = new JSFUTILS_FilePath(aPath);
    retval=file.exists();
  }
  catch(e) { 
    dump("fileUtils.js:exists:ERROR: "+e.name+"\n"+e.message+'\n'); 
    retval=null;
  }

  return retval;
},

/********************* RM *******************************/
rm : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:rm:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:rm:ERROR: Sorry, file "+aPath+" doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    this.fileInst            = new JSFUTILS_FilePath(aPath);
    if(this.fileInst.isDirectory())
    {
      dump("fileUtils.js:rm:ERROR:Sorry file is a directory. Try rmdir() instead\n");
      return null;
    }

    this.fileInst['delete'](false);
    retval=JSFUTILS_OK;
  }
  catch (e)
  { 
    dump("fileUtils.js:rm:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* COPY *****************************/
copy  : function (aSource, aDest) 
{
  if(!aSource || !aDest)
  {
    dump("fileUtils.js:copy:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aSource))
  {
    dump("fileUtils.js:copy:ERROR: Sorry, source file "+source+" doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    var fileInst      = new JSFUTILS_FilePath(aSource);
    var dir           = new JSFUTILS_FilePath(aDest);
    var copyName      = fileInst.leafName;

    if(fileInst.isDirectory())
    {
      dump("fileUtils.js:copy:File is a directory ignoring . . .\n");
      return null;
    }

    if(!this.exists(aDest) || !dir.isDirectory())
    {
      copyName          = dir.leafName;
      dir               = new JSFUTILS_FilePath(dir.path.replace(copyName,''));

      if(!this.exists(dir.path))
      {
        dump("fileUtils.js:copy:ERROR: Sorry, dest directory "+dir.path+" doesn't exist\n");
        return null;
      }

      if(!dir.isDirectory())
      {
        dump("fileUtils.js:copy:ERROR: Sorry, destination dir "+dir.path+" is not a valid dir path\n");
        return null;
      }
    }

    if(this.exists(this.append(dir.path, copyName)))
    {
      dump("fileUtils.js:copy:ERROR: Sorry destination file "+this.append(dir.path, copyName)+" already exists\n");
      return null;
    }

    fileInst.copyTo(dir, copyName);
    dump('fileUtils.js:copy successful!\n');
    retval=JSFUTILS_OK;
  }

  catch (e)
  { 
    dump("fileUtils.js:copy:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* LEAF *****************************/
leaf  : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:leaf:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:leaf:ERROR: File doesn't exist\n");
    return null;
  }

  var retval;
  try
  {
    var fileInst = new JSFUTILS_FilePath(aPath);
    retval=fileInst.leafName;
  }

  catch(e)
  { 
    dump("fileUtils.js:leaf:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* APPEND ***************************/
append : function (aDirPath, aFileName) 
{
  if(!aDirPath || !aFileName)
  {
    dump("fileUtils.js:append:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aDirPath))
  {
    dump("fileUtils.js:append:ERROR: File doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    this.fileInst           = new JSFUTILS_FilePath(aDirPath);
    var fileAppended        = this.fileInst.append(aFileName);
    fileAppended            = this.fileInst.path;
    retval=fileAppended;
  }

  catch(e)
  { 
    dump("fileUtils.js:append:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* VALIDATE PERMISSIONS *************/
validatePermissions : function(aNum) 
{
  if ( parseInt(aNum.toString(10).length) < 3 ) 
    return false;

  return JSFUTILS_OK;
},

/********************* PERMISSIONS **********************/
permissions : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:permissions:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:permissions:ERROR: File doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    var fileInst  = new JSFUTILS_FilePath(aPath); 
    retval=fileInst.permissions.toString(8);
  }

  catch(e)
  { 
    dump("fileUtils.js:permissions:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* MODIFIED *************************/
dateModified  : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:dateModified:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:dateModified:ERROR: File doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    var fileInst  = new JSFUTILS_FilePath(aPath); 
    var date = new Date(fileInst.lastModificationDate).toLocaleString();
    retval=date;
  }

  catch(e)
  { 
    dump("fileUtils.js:dateModified:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* SIZE *****************************/
size  : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:size:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:size:ERROR: File doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    var fileInst    = new JSFUTILS_FilePath(aPath); 
    retval          = fileInst.fileSize;
  }

  catch(e)
  { 
    dump("fileUtils.js:size:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=0;
  }

  return retval;
},

/********************* EXTENSION ************************/
extension  : function (aPath)
{
  if(!aPath)
  {
    dump("fileUtils.js:extension:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:extension:ERROR: File doesn't exist\n");
    return null;
  }

  var retval;
  try
  { 
    var fileInst  = new JSFUTILS_FilePath(aPath); 
    var leafName  = fileInst.leafName; 
    var dotIndex  = leafName.lastIndexOf('.'); 
    retval=(dotIndex >= 0) ? leafName.substring(dotIndex+1) : ""; 
  }

  catch(e)
  { 
    dump("fileUtils.js:extension:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* DIRPATH **************************/
dirPath   : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:dirPath:ERROR: Missing path argument\n");
    return null;
  }

  var retval;
  try
  { 
    var fileInst            = new JSFUTILS_FilePath(aPath);

    if(fileInst.isFile())
      retval=fileInst.path.replace(this.leaf(aPath), "");

    if(fileInst.isDirectory())
      retval=fileInst.path;
  }

  catch (e)
  { 
    dump("fileUtils.js:dirPath:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* SPAWN ****************************/
spawn : function (aPath, aArgs) 
{
  if(!aPath)
  {
    dump("fileUtils.js:spawn:ERROR: Missing path argument\n");
    return null;
  }

  if(!this.exists(aPath))
  {
    dump("fileUtils.js:spawn:ERROR: File doesn't exist\n");
    return null;
  }

  var len=0;

  if(aArgs)
    len = aArgs.length;

  else
    aArgs=null;

  var retval;
  try
  { 
    var fileInst            = new JSFUTILS_FilePath(aPath);

    if(!fileInst.isExecutable())
    {
      dump("fileUtils.js:spawn:ERROR: File is not executable\n");
      retval=null;
    }

    if(fileInst.isDirectory())
    {
      dump("fileUtils.js:spawn:ERROR: File is not a program\n");
      retval=null
    }

    else
      return fileInst.spawn(aArgs, len);
  }

  catch (e)
  { 
    dump("fileUtils.js:spawn:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* nsIFILE **************************/
nsIFile : function (aPath) 
{
  if(!aPath)
  {
    dump("fileUtils.js:nsIFile:ERROR: Missing path argument\n");
    return null;
  }

  var retval;
  try
  {
    retval = new JSFUTILS_FilePath(aPath);
  }

  catch (e)
  { 
    dump("fileUtils.js:nsIFile:ERROR: "+e.name+"\n"+e.message+'\n');
    retval=null;
  }

  return retval;
},

/********************* HELP *****************************/
help  : function() 
{
  var help =

    "\n\nFunction List:\n"                  +
    "\n"                                    +
    "   exists(aPath);\n"                   +
    "   chrome_to_path(aPath);\n"           +
    "   URL_to_path(aPath);\n"              +
    "   append(aDirPath, aFileName);\n"     +
    "   rm(aPath);\n"                       +
    "   copy(aSource, aDest);\n"            +
    "   leaf(aPath);\n"                     +
    "   permissions(aPath);\n"              +
    "   dateModified(aPath);\n"             +
    "   size(aPath);\n"                     +
    "   extension(aPath);\n"                +
    "   dirPath(aPath);\n"                  + 
    "   spawn(aPath, aArgs);\n"             + 
    "   nsIFile(aPath);\n"                  + 
    "   help();\n";

  dump(help+'\n');

  return "";
},

};

