// The PasspetFolder XPCOM component provides read and write access to files
// in the "passpet" subdirectory of the user profile directory.

// Shorthand for getting XPConnect interfaces, classes, objects, and services.
const XPI = Components.interfaces, XPC = Components.classes;
function XPC_(c) { return XPC[c.match(/^@/) ? c : '@mozilla.org/' + c]; }
function XPO(c, i) { return XPC_(c) ? XPC_(c).createInstance(i) : null; }
function XPS(c, i) { return XPC_(c) ? XPC_(c).getService(i) : null; }
function XQI(o, i) { return o.QueryInterface(i); }

const XPR = Components.results;
const CID = Components.ID('{49bc08de-3d48-563b-a0b5-ab98ce64757e}');
const NAME = 'Passpet Folder Service';
const CONTRACT = '@passpet.org/folder;1';

// The module registers and unregisters the PasspetHash component interface.
function NSGetModule(manager, filespec) {
    return new function() {
        this.registerSelf = function(manager, filespec, location, type) {
            manager = XQI(manager, XPI.nsIComponentRegistrar);
            manager.registerFactoryLocation(
                CID, NAME, CONTRACT, filespec, location, type);
        };

        this.unregisterSelf = function(manager, location, type) {
            manager = XQI(manager, XPI.nsIComponentRegistrar);
            manager.unregisterFactoryLocation(CID, location);
        };

        this.canUnload = function(manager) {
            return true;
        };

        this.getClassObject = function(manager, cid, iid) {
            if (!iid.equals(XPI.nsIFactory)) throw XPR.NS_ERROR_NOT_IMPLEMENTED;
            if (!cid.equals(CID)) throw XPR.NS_ERROR_NO_INTERFACE;
            return new function() {
                this.createInstance = function(outer, iid) {
                    if (outer != null) throw XPR.NS_ERROR_NO_AGGREGATION;
                    return XQI(new service(), iid);
                };
            };
        };
    };
};

const service = function() {
    const OPEN_READ = 1, OPEN_WRITE = 2, OPEN_READWRITE = 4;
    const OPEN_CREATE = 8, OPEN_APPEND = 16, OPEN_TRUNCATE = 32;
    const props = XPS('file/directory_service;1', XPI.nsIProperties);
    const directory = props.get('ProfD', XPI.nsIFile);
    directory.append('passpet');
    if (!directory.exists()) {
        directory.create(XPI.nsIFile.DIRECTORY_TYPE, 0755);
    }

    function getFile(name) {
        var file = directory.clone();
        file.append(name);
        return file;
    }

    this.list = function() {
        var filenames = [];
        var iterator = directory.directoryEntries;
        while (iterator.hasMoreElements()) {
            var file = XQI(iterator.getNext(), XPI.nsIFile);
            filenames.push(file.leafName);
        }
        return filenames.sort().join('\0');
    }

    this.contains = function(name) {
        return getFile(name).exists();
    }

    this.read = function(name) {
        var input = XPO('network/file-input-stream;1',
                        XPI.nsIFileInputStream);
        input.init(getFile(name), OPEN_READ, 0644, 0);
        var stream = XPO('scriptableinputstream;1',
                         XPI.nsIScriptableInputStream);
        stream.init(input);
        var content = stream.read(stream.available());
        stream.close();
        return content;
    }

    this.write = function(name, content) {
        var output = XPO(
            'network/file-output-stream;1', XPI.nsIFileOutputStream);
        var outfile = getFile(name + '.tmp');
        output.init(outfile, OPEN_WRITE | OPEN_CREATE | OPEN_TRUNCATE, 0644, 0);
        output.write(content, content.length);
        output.close();
        outfile.moveTo(directory, name);
    }

    this.remove = function(name) {
        getFile(name).remove(false);
    }

    // This method implements the nsISupports interface.
    this.QueryInterface = function(iid) {
        if (iid.equals(XPI.IPasspetFolder)) return this;
        if (iid.equals(XPI.nsISupports)) return this;
        throw XPR.NS_ERROR_NO_INTERFACE;
    };
};
