/***
  * mzContentBlocker XPCOM component version: v1.1.2
  *
  * Copyright (c) 2002-2006 by HJ van Rantwijk - All Rights Reserved!
  * Warning: this is NOT some free/Public Domain/Open Source Software!
  *
  */

/***
  * Constants
  */

const AMD_BLOCK_BUTTON_SELECTED  = 1;
const AMD_ALLOW_BUTTON_SELECTED  = 2;
const AMD_MANUAL_BUTTON_SELECTED = 4;
const AMD_CHECKBOX_CHECKED       = 8;

const BEHAVIOR_ALLOW             = 1;
const BEHAVIOR_ALLOW_SAME_DOMAIN = 2;
const BEHAVIOR_BLOCK             = 4;
const BEHAVIOR_ASK_ME            = 8;
const BEHAVIOR_USE_REGEX         = 16;

const MIMETYPE_USE_REGEX         = 1;
const MIMETYPE_STORE             = 2;

const UNLISTED_PERMISSION_ENTRY  = -1;

/***
  * source: http://lxr.mozilla.org/seamonkey/source/netwerk/base/public/nsIPermissionManager.idl
  *
  * const PRUint32 UNKNOWN_ACTION = 0;
  * const PRUint32 ALLOW_ACTION   = 1;
  * const PRUint32 DENY_ACTION    = 2;
  *
  */

/***
  * source: http://lxr.mozilla.org/seamonkey/source/content/base/public/nsIContentPolicy.idl
  */

const REJECT_REQUEST = Components.interfaces.nsIContentPolicy.REJECT_REQUEST;
const REJECT_TYPE    = Components.interfaces.nsIContentPolicy.REJECT_TYPE;
const REJECT_SERVER  = Components.interfaces.nsIContentPolicy.REJECT_SERVER;
const REJECT_OTHER   = Components.interfaces.nsIContentPolicy.REJECT_OTHER;
const ACCEPT         = Components.interfaces.nsIContentPolicy.ACCEPT;

/***
  * List with the supported permission types:
  * source: http://lxr.mozilla.org/seamonkey/source/content/base/public/nsIContentPolicy.idl
  *
  * const unsigned long TYPE_OTHER       = 1; // unsupported in MultiZilla version 1.7.0.0 and up
  * const unsigned long TYPE_SCRIPT      = 2;
  * const unsigned long TYPE_IMAGE       = 3;
  * const unsigned long TYPE_STYLESHEET  = 4; // unsupported in MultiZilla version 1.7.0.0 and up
  * const unsigned long TYPE_OBJECT      = 5;
  * const unsigned long TYPE_DOCUMENT    = 6;
  * const unsigned long TYPE_SUBDOCUMENT = 7;
  * const unsigned long TYPE_REFRESH     = 8;
  */

const TYPE_OBJECT      = Components.interfaces.nsIContentPolicy.TYPE_OBJECT;
const TYPE_DOCUMENT    = Components.interfaces.nsIContentPolicy.TYPE_DOCUMENT;
const TYPE_SUBDOCUMENT = Components.interfaces.nsIContentPolicy.TYPE_SUBDOCUMENT;
const TYPE_STYLESHEET  = Components.interfaces.nsIContentPolicy.TYPE_STYLESHEET;
const TYPE_IMAGE       = Components.interfaces.nsIContentPolicy.TYPE_IMAGE;

var permissionTypes    = [ "unsupported", "unsupported", "script", "image", "document", "object", "document", "frame", "meta" ];
var eventTypes         = [ "unsupported", "unsupported", "MZScriptBlocked", "MZImageBlocked", "MZDocumentBlocked", "MZObjectBlocked", "MZDocumentBlocked", "MZFrameBlocked", "MZMetaRedirectBlocked" ];
var behaviorPrefValues = [ -1, -1, -1, -1, -1, -1, -1, -1, -1 ];
var mimePrefValues     = [ -1, -1, -1, -1, -1, -1, -1, -1, -1 ];
var patterns           = new Array();
var counter            = 0;
var cbIsEnabled        = false;
var cbDebug            = false;
var cbPrefBranch       = "multizilla.content-blocker";
var cbJavaEnabled      = false;
var cbCheckFavicons    = true;

/***
  * Used interfaces
  */

const permissionmanager = Components.classes["@mozilla.org/permissionmanager;1"]
                                    .getService(Components.interfaces.nsIPermissionManager);
const gIOService        = Components.classes["@mozilla.org/network/io-service;1"]
                                    .getService(Components.interfaces.nsIIOService);
const prefs             = Components.classes["@mozilla.org/preferences-service;1"]
                                    .getService(Components.interfaces.nsIPrefBranch);
const promptService     = Components.classes["@mozilla.org/embedcomp/prompt-service;1"]
                                    .getService(Components.interfaces.nsIPromptService);
const cacheService      = Components.classes["@mozilla.org/network/cache-service;1"]
                                    .getService(Components.interfaces.nsICacheService);
const gMIMEService      = Components.classes["@mozilla.org/mime;1"]
                                    .getService(Components.interfaces.nsIMIMEService);

/*** 
  * Preferences
  * pref("multizilla.content-blocker.enabled", "bool", [true/false]);
  * pref("multizilla.content-blocker.behavior.xxx", "int", [1=disabled/2=allow/4=same domain/8=block/16=ask me/32=use regex]/64=keep);
  * pref("multizilla.content-blocker.mimetype.behavior.xxx", "int", [1=disabled/2=enabled/4=ask me/8=use regex/16=keep]);
  * pref("multizilla.content-blocker.mimetypes.for.xxx", "string", "video/quicktime, application/x-shockwave-flash, application/futuresplash")
  */

/***
  * Prefs observer
  */
  
const cbPrefListener =
{
  observe: function(aSubject, aTopic, aPrefName) {
    // debug("\naTopic: " + aTopic);
    if (aTopic != "nsPref:changed")
      return;
    if (aPrefName == cbPrefBranch + ".enabled") {
      try {
        cbIsEnabled = prefs.getBoolPref(aPrefName);
        debug("\nPref Changed to: " + cbIsEnabled);
      } catch(ex) {
        // this might fail, but without throwing a JS error please!
      }
    }
    else if (aPrefName == cbPrefBranch + ".java-enabled") {
      try {
        cbJavaEnabled = prefs.getBoolPref(aPrefName);
        debug("\nJava Pref Changed to: " + cbJavaEnabled);
      } catch(ex){
        // this might fail, but without throwing a JS error please!
      }
    }
    else if (aPrefName == cbPrefBranch + ".check-favicons") {
      try {
        cbCheckFavicons = prefs.getBoolPref(aPrefName);
        debug("\nCheck Favicon Pref Changed to: " + cbCheckFavicons);
      } catch(ex){
        // this might fail, but without throwing a JS error please!
      }      
    }
    else if (aPrefName.match(cbPrefBranch + ".behavior.")) {
      initBehaviorFor(aPrefName, "behavior");
    }
    /* else if (aPrefName.match(cbPrefBranch + ".mimetype.behavior.")) {
      initBehaviorFor(aPrefName, "mimetype.behavior");
    } */
    else if (aPrefName.match(cbPrefBranch + ".pattern.for.")) {
      initPatternFor(aPrefName);
    }
  }
};
////////////////////////////////////////////////////////////////////////
// The mzContentBlocker class constructor.
////////////////////////////////////////////////////////////////////////
function mzContentBlocker()
{
  for (i in permissionTypes) {
    var regExContainer = (permissionTypes[i] == "unsupported") ? null : new Array();
    patterns.push(regExContainer);
  }
  addPrefListener(cbPrefListener);

  initPrefs();
}
////////////////////////////////////////////////////////////////////////
// We need to initialize our preferences manually, since this doesn't 
// happen without the profile manager opened on startup!
////////////////////////////////////////////////////////////////////////
function initPrefs()
{
  try {
    cbIsEnabled = prefs.getBoolPref(cbPrefBranch + ".enabled");

    if (cbIsEnabled) { // Only initialize when we're enabled!
      for (i in permissionTypes) {
        if (permissionTypes[i] != "unsupported") {
          initBehaviorFor(cbPrefBranch + ".behavior." + permissionTypes[i], "behavior");
        }
      }
      for (i in permissionTypes) {
        if (permissionTypes[i] != "unsupported") {
          initPatternFor(cbPrefBranch + ".pattern.for." + permissionTypes[i]);
        }
      }
    }
    cbJavaEnabled = prefs.getBoolPref(cbPrefBranch + ".java-enabled");
    cbCheckFavicons = prefs.getBoolPref(cbPrefBranch + ".check-favicons");
  }
  catch(ex) {
    // die silently
    // dump("\nERROR: " + ex);
  }
  debug(cbIsEnabled);
}
////////////////////////////////////////////////////////////////////////
// mzContentBlocker : nsIContentPolicy, nsISupports
////////////////////////////////////////////////////////////////////////
mzContentBlocker.prototype =
{
    getVersionInfo: function () 
    {
      return("MultiZilla's Content Blocker v1.1.0");
	},
    // This will only work with mozilla builds after 2004-07-27 10:15!
    shouldLoad: function (aContentType, aContentLocation, aRequestOrigin, aContext, aMimeTypeGuess)
    {
      if (aContentLocation.scheme == "chrome")
        return ACCEPT;

      if (aContentType == TYPE_STYLESHEET && aContentLocation.scheme == "http" && 
          aContext == "[object XMLStylesheetProcessingInstruction]") {
        return REJECT_TYPE;
      }
      if (!cbIsEnabled || !aContentLocation.scheme.match(/^[http|news]/i) || (aContentType < 2 || aContentType > 8))
        return ACCEPT;

      var contentId = permissionTypes[aContentType]; // Match string value for current permission type

      if (contentId != "unsupported") {
        var behavior = parseInt(behaviorPrefValues[aContentType]);

        if (behavior > 0) {
          if (aContentType == TYPE_IMAGE && aContext instanceof Components.interfaces.nsIDOMXULElement) {
            /***
              * cbCheckFavicons is controlled by "multizilla.content-blocker.check-favicons" 
              * and enables/disable favicons to be displayed in your bookmark menus. 
              */
            if (!cbCheckFavicons && aContentLocation.spec.indexOf("favicon.ico") != -1)
              return ACCEPT; // allow favicon
            // Don't block images for XUL generated requests from our Add Feed Window!
            if ('id' in aContext && aContext.id.match(/^FeedImage/))
              return ACCEPT;
          }
          // No need to check for Java Applets when Java is disabled!
          if (aContentType == TYPE_OBJECT && cbJavaEnabled == false) {
            // Check for 'application/x-java-vm' or 'application/x-java-applet'
            if (aMimeTypeGuess.indexOf("application/x-java-") != -1)
              return false;
          }
          var win = this.getWindowFromContext(aContext);
          var URI = gIOService.newURI(aContentLocation.spec, null, null);
          URI.spec = URI.spec.replace(/\?(.*)/, ''); // remove crufty parameters!
          var policy = new Object();
          policy.retValue = true;

          /* if (behavior == BEHAVIOR_ASK_ME) {
            dump("\nBEHAVIOR_ASK_ME");
            var browser = this.getBrowserForDocument(win.document);
            // Is this domain blocked before?
            if (browser) {
              dump("\ncall isContentBlocked()");
              aBrowser.isContentBlocked(eventTypes[aContentType], URI, policy);
              dump("\npolicy.retValue: " + policy.retValue);
              dump("\npolicy.filter: " + policy.filter);
              if (policy.retValue == false) {
                // dump("\npolicy.retValue: " +  policy.retValue + "\npolicy.filter: " + policy.filter);
                dump("\nAUTOMATIC_SERVER_REJECT");
                policy.filter = "AUTOMATIC_SERVER_BLOCK";
              }
            }
            else {
              dump("\nERROR: No browser found for: " + URI.spec);
            }
          } */
          if (policy.retValue == true) {
            var policy = this.checkPolicyFor(behavior, URI, aRequestOrigin, win, aContentType, contentId, aMimeTypeGuess);
          }
          if (policy.retValue == false) // we can't use else here!
          {
            var aNode = this.getElementNode(aContentType, aRequestOrigin, aContext);

            if (aNode) // For testing purpose only, needs a lot more love!
            {
              // dump("\nnodeName: " + aNode.nodeName);
              if (aNode.nodeName == 'IMG' || aNode.nodeName == 'AREA')
              {
                // dump("\nHiding: " + aNode.nodeName);
                aNode.hidden = true;
                // aNode.style.visibility = 'hidden';
                // aNode.style.display = 'none';
              }
              else if (aNode.nodeName == 'IFRAME') {
                // dump("\nHiding: " + aNode.nodeName);
                aNode.style.display = "none";
              }
            }
            try {
              var _document = Components.lookupMethod(win, "document").call(win);
              var browserElement = this.getBrowserForDocument(_document);
              // dump("\nbrowserElement: " + browserElement + "\nwin : " + win);
              var chromeWindow = browserElement ? browserElement.ownerDocument.defaultView : win;
              var _event = chromeWindow.document.createEvent('Events');
              // dump("\nchromeWindow: " + chromeWindow);
              _event.initEvent(eventTypes[aContentType], false, false);
              _event.filter = policy.filter;
              _event.blockedURI = URI;
              _event.browserElement = browserElement ? browserElement : aContext;
              _event.isXULElement = (aContext instanceof Components.interfaces.nsIDOMXULElement);
              // dump("\nFire Event on: " + chromeWindow + "\nfor aContext : " + aContext);
              chromeWindow.document.dispatchEvent(_event);
              return REJECT_SERVER;
            }
            catch(ex) {
              dump("\nCB-ERROR: " + ex);
              return REJECT_SERVER;
            }
          }
        }
      }
      return ACCEPT;
    },

    checkPolicyFor: function(behavior, aURI, context, aWindow, aContentType, aContentId, aMimeTypeGuess)
    {
      var policy = this.shouldAllowContent(behavior, aURI, aContentType, aContentId, aWindow, aMimeTypeGuess); 
      var capability = policy.retValue;

      if (capability == UNLISTED_PERMISSION_ENTRY) {
        if (behavior & BEHAVIOR_ASK_ME) {
          // dump("\nshouldAllowContent() = " + policy.retValue);
          policy.retValue = this.getUsersPermission(aURI, context, aWindow, aContentType, aContentId, aMimeTypeGuess);
          // dump("\ngetUsersPermission() = " + policy.retValue);
          if (policy.retValue & AMD_BLOCK_BUTTON_SELECTED) { // 0 = Block, add preferred filter
            if (policy.retValue & AMD_CHECKBOX_CHECKED)
              policy.filter = this.addPreferredFilter(aURI, aContentType, aContentId);
            /* else 
              policy.filter = "MANUAL_SERVER_BLOCK"; */
            policy.retValue = false;
          }
          else if (policy.retValue & AMD_ALLOW_BUTTON_SELECTED) { // 1 = Allow, add permission
            if (policy.retValue & AMD_CHECKBOX_CHECKED)
              this.updatePermissions(aURI, permissionmanager.ALLOW_ACTION, aContentId);
            policy.retValue = true;
            policy.filter = "";
          }
          else if (policy.retValue & AMD_MANUAL_BUTTON_SELECTED) { // 2 = Open Content Blocking Window
            this.openContentBlockingWindow(aURI, aWindow, aContentType, policy);
          }
          // Do we still need this?
          var _document = Components.lookupMethod(aWindow, "document").call(aWindow);
          var _event = Components.lookupMethod(_document, "createEvent").call(_document, 'Events');
          _event.initEvent("DOMModalDialogClosed", false, false);
          aWindow.dispatchEvent(_event);
        }
        else {
          policy.retValue = (behavior & BEHAVIOR_ALLOW || (behavior & BEHAVIOR_ALLOW_SAME_DOMAIN)) ? true : false;
        }
      }
      // dump("\n" + contentId + " policy.retValue: " + policy.retValue);
      return policy;
    },

    shouldAllowContent: function(behavior, aURI, aContentType, contentId, aWindow)
    {
       var policy = new Object();
       policy.permission = this.readPermission(behavior, aURI, contentId, aWindow); // ALLOW_ACTION = 1 / DENY_ACTION = 2
       policy.retValue = policy.permission;

       if (policy.permission == UNLISTED_PERMISSION_ENTRY && patterns[aContentType].length > 0)
       {
         for (i in patterns[aContentType])
         {
           if (patterns[aContentType][i] != "") {
             var regexp = new RegExp();
             var isFiltered = aURI.spec.match(regexp.compile(patterns[aContentType][i]), 'i');

             if (isFiltered) {
               // dump("\nisFiltered: " + isFiltered);
               policy.filter = patterns[aContentType][i];
               // dump("\nPattern match for: " + permissionTypes[aContentType] + " : " + policy.filter + "\nLoad denied for URL: " + aURI.spec);

               if ((behavior & BEHAVIOR_ASK_ME) || (behavior & BEHAVIOR_ALLOW) || (behavior & BEHAVIOR_ALLOW_SAME_DOMAIN)) {
                 // dump("\nbehavior & BEHAVIOR_ASK_ME || BEHAVIOR_ALLOW || BEHAVIOR_ALLOW_SAME_DOMAIN");
                 policy.retValue = false;
               }
               else if (behavior & BEHAVIOR_BLOCK) {
                 // dump("\nBEHAVIOR_BLOCK");
                 policy.retValue = true;
               }
             }
           }
         }
       }
       // dump("\npolicy.retValue: " + policy.retValue);
       return policy;
    },

    readPermission: function(behavior, aURI, contentId, aWindow)
    {
      var shouldLoad = true;

      if (aURI.scheme == "file") {
        aURI = gIOService.newURI("file:", null, null);
      }
      var permission = permissionmanager.testPermission(aURI, contentId);
      // dump("\npermission1: " + permission);
      if (behavior & BEHAVIOR_ALLOW_SAME_DOMAIN) {
        shouldLoad = (permission == permissionmanager.DENY_ACTION) ? false : true;
        var browser = this.getBrowserForDocument(aWindow.document);
        // dump("\naURI.hostPort: " + aURI.hostPort);
		// dump("\nbrowser: " + browser.contentDocument.domain);
        if (aURI.hostPort != browser.contentDocument.domain) {
          // dump("\nContent blocked for other domain " + aURI.hostPort);
          return false;
        }
      }
      if (permission == permissionmanager.UNKNOWN_ACTION)
        return (UNLISTED_PERMISSION_ENTRY);
      else if (behavior & BEHAVIOR_ALLOW)
        shouldLoad = (permission == permissionmanager.DENY_ACTION) ? false : true;
      else // if (behavior & BEHAVIOR_BLOCK) and BEHAVIOR_ASK_ME
        shouldLoad = (permission == permissionmanager.ALLOW_ACTION);
      /* if (!shouldLoad)
        dump("\nshouldAllowContent: " + contentId + " rejected!");
        dump("\nshouldLoad: " + shouldLoad + "\npermission: " + permission); */
      return shouldLoad; // true or false
    },

    addPreferredFilter: function(aURI, contentType, aContentId)
    {
      var URL = aURI.spec.replace(/([-\w]*:\/+)?/, ""); // trim scheme
      var isFileScheme = (aURI.scheme == "file");
      var hostPort = isFileScheme ? "file:" : aURI.hostPort.replace(/^www/i, ''); // trim third level domain;
      var preferredFilter = isFileScheme ? hostPort : this.trimTLDFromHostPort(hostPort);

      if (preferredFilter.charAt(0) != '.')
        preferredFilter = "/" + preferredFilter;

      preferredFilter = preferredFilter.replace(/\./g, '\.+')
      // dump("\npreferredFilter: " + preferredFilter);
      patterns[contentType].push(preferredFilter);
      patterns[contentType] = patterns[contentType].sort();
      this.writePatternPrefFor(contentType, aContentId);

      return preferredFilter;
    },

    trimTLDFromHostPort: function(aURL)
    {
      /***
        * 'sponsored' (sTLDs) and 'unsponsored' (uTLDs) TLDs, and one special TLD, .arpa
        * that we're interested in. Note that we don't strip two letter country code (ccTLDs)
        */
      var TLDs = ['com', 'org', 'net', 'mil', 'gov', 'edu', 'int', 'biz', 'info', 'name', 'pro', 'aero', 'coop', '.museum', 'arpa'];

      for (i in TLDs) {
        var matchedTLD = aURL.substring(aURL.lastIndexOf('.'), aURL.length);

         if (matchedTLD)
           return aURL.replace(matchedTLD, '.');
      }
      return aURL;
    },

    writePatternPrefFor: function(aContentType, aContentId)
    {
      try {
        var pattern = "";

        for (var p = 0; p < patterns[aContentType].length; p++) {
          pattern += patterns[aContentType][p] + ", "; // The extra space is to make wrap work!
        }
        pattern = pattern.replace(/, $/, ''); // Remove trailing comma and space
        var str = Components.classes["@mozilla.org/supports-string;1"].createInstance(Components.interfaces.nsISupportsString);
        // ".+macromedia.+" converted into ".macromedia." and "mo.*.+org" converted into "mo*.org"
        str.data = pattern.replace(/\.\+/g, '.').replace(/\.\*/g, "*");
        var prefIdentifier = cbPrefBranch + ".pattern.for." + aContentId;
        prefs.setComplexValue(prefIdentifier, Components.interfaces.nsISupportsString, str);
        pattern = null;
      } catch(ex) {
        dump("\nERROR in writePatternPrefFor(): " + ex);
      }
    },
	  
    getWindowFromContext: function(aContext)
    {
      var win = aContext;

      if (aContext.nodeType == 9) {
        if (aContext.defaultView == undefined)
          win = Components.lookupMethod(aContext, "defaultView").call(aContext);
        else 
          win = aContext.defaultView;
      }
      else {
        var _contentDocument, _ownerDocument;

        try {
          if (aContext.nodeType == undefined) {
            win = aContext;
          }
          if ('defaultView' in aContext) {
            win = Components.lookupMethod(aContext, "defaultView").call(aContext);
          }
          if ('ownerDocument' in aContext) {
            _ownerDocument = Components.lookupMethod(aContext, "ownerDocument").call(aContext);
            win = Components.lookupMethod(_ownerDocument, "defaultView").call(aContext);
          }
          if ('contentDocument' in aContext) {
            _contentDocument = Components.lookupMethod(aContext, "contentDocument").call(aContext);
            win = Components.lookupMethod(_contentDocument, "defaultView").call(aContext);            
          }
        } 
        catch(ex) {
          // dump("\nWARNING: " + ex);
          if (aContext.contentDocument == null) {
            _contentDocument = Components.lookupMethod(aContext, "contentDocument").call(aContext);
            win = Components.lookupMethod(_contentDocument, "defaultView").call(aContext);
          }
          if (aContext.ownerDocument == null) {
            _ownerDocument = Components.lookupMethod(aContext, "ownerDocument").call(aContext);
            win = Components.lookupMethod(_ownerDocument, "defaultView").call(aContext);            
          }
        }
      }
      return win;
    },

    getUsersPermission: function (aURI, aContext, aWindow, aContentType, aContentId, aMimeTypeGuess)
    {
      if (!aContext) // Sanity check
        aContext = aWindow.document;
      // Do we still need this?
      var _document = Components.lookupMethod(aWindow, "document").call(aWindow);
      var _event = Components.lookupMethod(_document, "createEvent").call(_document, 'Events');
      _event.initEvent("DOMWillOpenModalDialog", false, false);
      aWindow.dispatchEvent(_event);

      var mimeType = aMimeTypeGuess;

      if (!mimeType) {
        mimeType = this.getMimeTypeFor(aURI, aContext, aWindow, aContentId);
        mimeType = mimeType || "Undetermined mimeType";
      }
      var domain = ('domain' in aWindow.document) ? aWindow.top.document.domain : "Undetermined";
      var host = (aURI.scheme == "file") ? "file:" : host = aURI.hostPort;
      var confirmationMsg = "The site [" + domain + "] requests for [" + mimeType + "]\nfrom [" + host + "]\n\nDo you want to Allow or do you want to Block this request?";
      var title = "Confirmation Required for " + aContentId + "...";
      var checkbox = {value:false};
      var checkboxText = "Always Allow/Block " + aContentId + "'s from [" + host + "]";			  
      var usersResponse = promptService.confirmEx(aWindow, title, confirmationMsg,
                                                 (promptService.BUTTON_POS_0 * promptService.BUTTON_TITLE_IS_STRING) + 
                                                 (promptService.BUTTON_POS_1 * promptService.BUTTON_TITLE_IS_STRING),
                                                  "Allow", "Block", null, checkboxText, checkbox);

      usersResponse = (usersResponse == 1) ? 1 : 2; // 0 : 1; // 0 = Block / 1 = Allow

      if (checkbox.value)
        usersResponse = (usersResponse += 8);

      return usersResponse;
    },

    openContentBlockingWindow: function (aURI, aWindow, aContentType, policy)
    {
      aWindow.openDialog("chrome://multiviews/content/permission-manager/permissionManagerBlockingWindow.xul", 
                         "blockingWindow", "chrome, dependent, modal, dialog, centerscreen", aContentType, 2, aURI.spec, policy.filter, policy.permission);
    },

    getMimeTypeFor: function (aURI, aContext, aWindow, aContentId)
    {
      // dump("\nIN getMimeTypeFor();");

      if (!aContext) // Sanity check
        return null;

      var mimeType = ('type' in aContext && aContext.type) || 
                     ('codeType' in aContext && aContext.codeType) || 
                     ('contentType' in aContext && aContext.contentType);

      if (!mimeType) {
        try {
          mimeType = gMIMEService.getTypeFromURI(aURI);
        } 
        catch(ex) { // Invalid or unsupported file extension used!
          // This is used as fallback only! 
          if ('src' in aContext) {
            if (aContentId == 'image') { // Fallback for HTMLImageElement's
              mimeType = "image/unknown";
            }
            else if (aContentId == 'script') { // Fallback for HTMLScriptElement's
              var language = aContext.getAttribute('language'); // Check language attribute like "javascript(X.X)"

              if (language && language.match(/javascript/i)) {
                mimeType = "text/javascript";
              }
              else { // Check extension for <script src=/.../somefile.js"></script>
                var uri = aURI.QueryInterface(Components.interfaces.nsIURL);
                mimeType = (uri.fileExtension == 'js') ? "unspecified/javascript" : "unspecified/script";
              }
            }
            if (!mimeType) { // Did we find a mimetype?
              mimeType = this.getContentTypeFromHeaders(aContext.src); // No, check cache entries
            }
          }
          else if ('location' in aContext) { // documents
            mimeType = this.getContentTypeFromHeaders(aContext.location);
          }
        }
      }
      mimeType = (mimeType || aContentId + "/undetermined");
      // dump("\nmimeType: " + mimeType);
      return mimeType;
    },

    getContentTypeFromHeaders: function (aURL)
    {
      var match = null;

      try {
        var httpCacheSession = cacheService.createSession("HTTP", 0, true);
        var cacheEntryDescriptor = httpCacheSession.openCacheEntry(aURL, Components.interfaces.nsICache.ACCESS_READ, false); // open for READ, in non-blocking mode
        var headers = cacheEntryDescriptor.getMetaDataElement("response-head");
        match = /^Content-Type:\s*(.*?)\s*(?:\;|$)/mi.exec(headers);
        // dump("\nmatch[1]: "+match[1]);
      }
      catch (ex) {
       // dump("\nRequested Entry Not Cached: " + aURL);
      }
      return (match ? match[1] : match);
    },

    getWindowsByType: function(aType, aWindowList)
    {
      // var index = 0;
      var windowManager = Components.classes['@mozilla.org/appshell/window-mediator;1'].getService();
      var windowManagerInterface = windowManager.QueryInterface(Components.interfaces.nsIWindowMediator);
      var enumerator = windowManagerInterface.getEnumerator(aType);

      while (enumerator.hasMoreElements()) {
        var aChromeWindow = enumerator.getNext();
        aChromeWindow = aChromeWindow.QueryInterface(Components.interfaces.nsIDOMWindowInternal);
        aWindowList.push(aChromeWindow);
        // dump("\nwindow[" + index + "]: " + aChromeWindow.gBrowser.browsers);
        // index++;
      }
    },

    getBrowserForDocument: function(aDocument)
    {
      var retValue = null;
      var windowList = new Array();
      this.getWindowsByType("navigator:browser", windowList);

      for (window in windowList) {
        var _gBrowser = windowList[window].getBrowser();
        var browsers = _gBrowser.browsers;
        // dump("\nNumber of Browsers: " + browsers);
        for (var i = 0; i < browsers.length; i++) {
          var browser = _gBrowser.browsers[i];

          if (browser.contentDocument == aDocument) {
            retValue = _gBrowser.browsers[i];
            break;
          }
          if ("frames" in browser.contentWindow) {
            var frames = browser.contentWindow.frames.length;

            for (var frame = 0; frame < frames; frame++) {
              if (browser.contentWindow.frames[frame].length > 0) { // deep nested
                if (this.isFrameDocumentContainer(browser.contentWindow.frames[frame], aDocument)) {
                  return _gBrowser.browsers[i];
                }
              }
              if (browser.contentWindow.frames[frame].document == aDocument) {
                retValue = _gBrowser.browsers[i];
                break;
              }
            }
          }
        }
      }
      windowList = null; // Prevent leaks

      /* if (retValue == null)
        dump("\nBrowser Lookup Failed!"); */

      return retValue;
    },

    isFrameDocumentContainer: function(aFrame, aDocument)
    {
      if (!aFrame || !aDocument)
        return false;

      for (var i = 0; i < aFrame.length; i++) {
        if (aFrame.frames[i].length > 0) {
          if (this.isFrameDocumentContainer(aFrame[i], aDocument)) // recursive call
            return true;
        }
        if (aFrame.frames[i].document == aDocument)
          return true;
      }
      return false;
    },

    getElementNode: function (aContentType, aContext, aWindow)
    {
      try {
        if (aContentType == TYPE_SUBDOCUMENT)
          return aWindow.frameElement;
        else if (aContentType == TYPE_DOCUMENT)
          return aWindow.document.documentElement;
        else 
          return aContext.QueryInterface(Components.interfaces.nsIDOMElement);
      } catch (ex) {
        return aContext;
      }
    },

    updatePermissions: function (aURI, aShouldLoad, aContentId)
    {
      var permission = aShouldLoad ? permissionmanager.ALLOW_ACTION : permissionmanager.DENY_ACTION;
      // dump("\ncontentId: " + aContentId +"\npermissionmanager.add("+permission+")");

      if (aURI.scheme == "file")
        aURI = gIOService.newURI("file:", null, null);
      else 
        aURI.spec = aURI.spec.replace(/www\./i, '');
      permissionmanager.add(aURI, aContentId, permission);
    },

    updateMimeTypes: function (aMimeType)
    {
      if (!aMimeType) // Sanity check
        return;

      /* if (prefs.getBoolPref("multizilla.content-blocker.mimetypes.keep"))
        dump("\nStore MimeType"); */
    },

    setDebugStateTo: function (aState) 
    {
      cbDebug = aState;
	},

    shouldProcess: function (aContentType, aContentLocation, aRequestOrigin, aContext, aMimeType, aExtra)
    {
      // dump("\naContentType: " + aContentType + "\naContentLocation: " + aContentLocation.spec);
      return true;
    },

    checkChromeDisableState: function()
    {
      var rv;

      try { // Is MultiZIlla disabled in the Chrome Registry?
        var rdfService = Components.classes["@mozilla.org/rdf/rdf-service;1"].getService(Components.interfaces.nsIRDFService);
        var rdfDS = rdfService.GetDataSource("rdf:chrome");
        var resSelf = rdfService.GetResource("urn:mozilla:package:inspector");
        var resDisabled = rdfService.GetResource("http://www.mozilla.org/rdf/chrome#name");
        rv = rdfDS.GetTarget(resSelf, resDisabled, true);
          dump("\nrv: " + rv);
      } catch (ex) {
        dump("\nERROR: " + ex);
      }
    },

    QueryInterface: function (aIID)
    {
      // dump("\naIID: " + aIID);
      if (!aIID.equals(Components.interfaces.nsIContentPolicy) &&
          !aIID.equals(Components.interfaces.nsISupports) &&
          !aIID.equals(Components.interfaces.nsIFactory) &&
          !aIID.equals(Components.interfaces.nsIObserver)) {
          throw Components.results.NS_ERROR_NO_INTERFACE;
      }
      // dump("\nQueryInterface: " + counter++);
      return this;
    }
}
////////////////////////////////////////////////////////////////////////
//  mzContentBlocker Module (for XPCOM registration)
////////////////////////////////////////////////////////////////////////
var mzContentBlockerModule =
{
  firstTime:     true,
  mClassName:    "MultiZilla Content Blocker",
  mContractID:   "@multizilla.mozdev.org/contentBlocker;1",
  mClassID:      Components.ID("bc1f905a-07ac-434d-97cf-6b2a996bf00d"),
  // getClassObject: Return this component's factory object.
  getClassObject: function(aCompMgr, aCID, aIID)
  {
    /* dump("\ngetClassObject");
    var mzInstalled = null;

    try { // The Content Blocker needs MultiZilla, check if we're installed!
        var rdfService = Components.classes["@mozilla.org/rdf/rdf-service;1"].getService(Components.interfaces.nsIRDFService);
        var rdfDS = rdfService.GetDataSource("rdf:chrome");
        var resSelf = rdfService.GetResource("urn:mozilla:package:multiviews");
        var resDisabled = rdfService.GetResource("http://www.mozilla.org/rdf/chrome#name");
        mzInstalled = rdfDS.GetTarget(resSelf, resDisabled, true);
        dump("\nmzInstalled: " + mzInstalled);
    } catch (ex) {
      // dump("\nERROR: " + ex);
    }
    if (mzInstalled == null)
      throw Components.results.NS_ERROR_NOT_INITIALIZED; */
    if (!aCID.equals(this.mClassID))
      throw Components.results.NS_ERROR_NO_INTERFACE;
    if (!aIID.equals(Components.interfaces.nsIFactory))
      throw Components.results.NS_ERROR_NOT_IMPLEMENTED;
    return this.mFactory;
  },

  registerSelf: function(aCompMgr, aFileSpec, aLocation, aType)
  {
    // dump("\nregisterSelf");
    // dump("aFileSpec: " + aFileSpec + "\naLocation: " + aLocation + "\naType: " +aType);
    debug("Registering...");
    if (this.firstTime) {
      this.firstTime = false;
      throw Components.results.NS_ERROR_FACTORY_REGISTER_AGAIN;
    }
    aCompMgr = aCompMgr.QueryInterface(Components.interfaces.nsIComponentRegistrar);
    aCompMgr.registerFactoryLocation(this.mClassID, this.mClassName, this.mContractID, aFileSpec, aLocation, aType);
    this.getCategoryManager().addCategoryEntry("content-policy", this.mContractID, this.mContractID, true, true);
  },

  unregisterSelf: function(aCompMgr, aFileSpec, aLocation)
  {
    debug("UnRegistering...");
    aCompMgr = aCompMgr.QueryInterface(Components.interfaces.nsIComponentRegistrar);
    aCompMgr.unregisterFactoryLocation(this.mClassID, aFileSpec);
    this.getCategoryManager().deleteCategoryEntry("content-policy", this.mContractID, true);
  },

  canUnload: function(aCompMgr)
  {
    // dump("\ncanUnload");
    return true;
  },

  getCategoryManager: function()
  {
    // dump("\ngetCategoryManager");
    return Components.classes["@mozilla.org/categorymanager;1"].getService(Components.interfaces.nsICategoryManager);
  },
  //////////////////////////////////////////////////////////////////////
  //  mFactory : nsIFactory
  //////////////////////////////////////////////////////////////////////
  mFactory: { 
    createInstance: function (aOuter, aIID) {
      // dump("\ncreateInstance");
      if (aOuter != null) {
        throw Components.results.NS_ERROR_NO_AGGREGATION;
        // dump("\nthrow");
      }
      return (new mzContentBlocker()).QueryInterface(aIID);
    }
  },
};

function NSGetModule(aCompMgr, aFileSpec) 
{
  // dump("\nNSGetModule");
  return mzContentBlockerModule;
}

function addPrefListener(observer)
{
  try {
    var pbi = prefs.QueryInterface(Components.interfaces.nsIPrefBranchInternal);
        pbi.addObserver(cbPrefBranch, observer, false);
  } catch(ex) {
    debug("Pref Observer Failed!");
  }
}

function initBehaviorFor(aPrefName, aPrefSelector)
{
  // multizilla.content-blocker.behavior.type -> type
  var contentType = aPrefName.replace(/[-\w]*\./ig, '');
  var prefArray = (aPrefSelector == "behavior") ? behaviorPrefValues : mimePrefValues;

  for (i in permissionTypes) {
    if (permissionTypes[i] == contentType) {
      try { // Catch non-initialized prefs
        prefValue = prefs.getIntPref(aPrefName);
        dump("\nInit behavior pref for '" + contentType + "' (behavior = " + prefValue + ")");
      } catch (ex) {
        debug("ERROR Initializing behavior pref for: " + contentType + " Recovering...");
        prefValue = 0;
        prefs.setIntPref(aPrefName, prefValue);

        if (i > 0)
          i--;
      }
      prefArray[i] = prefValue;
    }
  }
}

function initPatternFor(aPrefName)
{
  // multizilla.content-blocker.pattern.for.type -> type
  var pattern, contentType = aPrefName.replace(/[-\w]*\./ig, '');

  for (i in permissionTypes) {
    pattern = new Array();

    if (permissionTypes[i] == contentType) {

      try {
        pattern = prefs.getComplexValue(aPrefName, Components.interfaces.nsISupportsString).data;
        pattern = pattern.replace(/\s/g, '').replace(/\./g, '\.+').replace(/\*/g, '.*');
        patterns[i] = pattern.split(',');
        dump("\nInit pattern pref for '" + contentType + "' (" + patterns[i].length + " filters activated)");
      }
      catch(ex) {
        debug("ERROR: Initializing patterns for: " + permissionTypes[i] + " Failed... \nRecovering...");
        var str = Components.classes["@mozilla.org/supports-string;1"].createInstance(Components.interfaces.nsISupportsString);
        str.data = "";
        prefs.setComplexValue(aPrefName, Components.interfaces.nsISupportsString, str);

        if (i > 0)
          i--;
      }
      break;
    }
  }
  pattern = null;
}	  

////////////////////////////////////////////////////////////////////////
//  Debug Helper function
////////////////////////////////////////////////////////////////////////
if (!cbDebug)
  debug = function(txt) {};
else
  debug = function(txt) {dump("\nCB: " + txt + "\n");};
