/*$Id: ActiveRuleSet.cpp,v 1.6 2007/02/13 00:11:14 jwrobel Exp $*/
/* ***** BEGIN LICENSE BLOCK *****
 *  This file is part of Firekeeper.
 *
 *  Copyright (C) 2006 Jan Wrobel <wrobel@blues.ath.cx>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * ***** END LICENSE BLOCK ***** */

#include "ActiveRuleSet.h"
#define OVECCOUNT 30

/*
  Create ActiveRuleSet using rules from Rules object
 */
ActiveRuleSet::ActiveRuleSet(Rules *rules): url(NULL), headers(NULL)
{
	TRACE("constructor %08x", this);
	/*At the beginning only rules without *_content options
	  are interesting, other rules will be added when multi-pattern search
	  engine matches their *_content patterns*/
	this->noContent = *rules->getNoContentRuleSet();
	this->rules = rules;
	rulesVersion = rules->getRulesVersion();
}

ActiveRuleSet::~ActiveRuleSet()
{
	TRACE("destructor %08x", this);
	if (url)
		free(url);
	if (headers)
		free(headers);
}

/*
  Match regular expression against given string.
 */
inline bool ActiveRuleSet::match(const RE &re_info, const char *s, int len)
{
	int rc;	
	int ovector[OVECCOUNT];	
	rc = pcre_exec(re_info.re_compiled, re_info.re_extra, s, len, 
		       0, 0, ovector, OVECCOUNT);	
        if (rc < 0)
		return false;
	return true;
}


/*
  Called by multi-pattern search engine when one of url_content options
  matched.
 */
int 
urlContentMatched(void *id, int index, void* ctx)
{
	TRACE("MATCHED");
	PatternMatchData *match = (PatternMatchData *) id;
	ActiveRuleSet *caller = (ActiveRuleSet *) ctx;
	Rule *rule = match->parent;
	map<Rule *, set<PatternMatchData *> >::iterator it;
	
	it = caller->urlMatchedParts.find(rule);
	
	if (it != caller->urlMatchedParts.end() &&
	    it->second.find(match) != it->second.end()){
		/*previously matched*/
		return 0;
	}
	caller->urlMatchedParts[rule].insert(match);

	
	if (caller->urlMatchedParts[rule].size() == rule->url_content.size()){
		caller->urlContentCorrect[rule->actionID].insert(rule);
	}
	return 0;
}

/*
  Called by multi-pattern search engine when one of headers_content options
  matched.
 */
int 
headersContentMatched(void *id, int index, void* ctx)
{
	TRACE("MATCHED");
	PatternMatchData *match = (PatternMatchData *) id;
	ActiveRuleSet *caller = (ActiveRuleSet *) ctx;
	Rule *rule = match->parent;
	map<Rule *, set<PatternMatchData *> >::iterator it;
	
	/*True if some url_content or url_re pattern specified by this rule
	  don't match this HTTP request*/
	if (caller->noMatch.find(rule) != caller->noMatch.end() ||
	    (rule->url_content.size() && 
	     caller->urlContentCorrect[rule->actionID].find(rule) ==
	     caller->urlContentCorrect[rule->actionID].end())){
		return 0;
	}


	it = caller->headersMatchedParts.find(rule);

	if (it != caller->headersMatchedParts.end() &&
	    it->second.find(match) != it->second.end()){
		/*previously matched*/
		return 0;
	}
	caller->headersMatchedParts[rule].insert(match);

	
	if (caller->headersMatchedParts[rule].size() == 
	    rule->headers_content.size()){
		caller->headersContentCorrect[rule->actionID].insert(rule);
	}
	return 0;
}

/*
  Called by multi-pattern search engine when one of body_content options
  matched.
 */
int 
bodyContentMatched(void *id, int index, void* ctx)
{
	TRACE("MATCHED");
	PatternMatchData *match = (PatternMatchData *) id;
	ActiveRuleSet *caller = (ActiveRuleSet *) ctx;
	Rule *rule = match->parent;
	map<Rule *, set<PatternMatchData *> >::iterator it;

	/*True if some url_content, headers_content, url_re or headers_re
	  pattern specified by this rule don't match this HTTP request*/
	if (caller->noMatch.find(rule) != caller->noMatch.end() ||
	    (rule->url_content.size() && 
	     caller->urlContentCorrect[rule->actionID].find(rule) ==
	     caller->urlContentCorrect[rule->actionID].end()) ||
	    (rule->headers_content.size() && 
	     caller->headersContentCorrect[rule->actionID].find(rule) ==
	     caller->headersContentCorrect[rule->actionID].end())
	    ){
		TRACE("wrong url or headers %d %d",
		      rule->url_content.size(), rule->headers_content.size());
		if (caller->noMatch.find(rule) != caller->noMatch.end())
			TRACE("in noMatch set");
		return 0;
	}
	
	
	it = caller->bodyMatchedParts.find(rule);
	
	if (it != caller->bodyMatchedParts.end() &&
	    it->second.find(match) != it->second.end()){
		/*previously matched*/
		TRACE("previously matched");
		return 0;
	}
	TRACE("inserting");
	caller->bodyMatchedParts[rule].insert(match);
	
	
	if (caller->bodyMatchedParts[rule].size() ==
	    rule->body_content.size()){
		TRACE("ok");
		caller->bodyContentCorrect[rule->actionID].insert(rule);
		list<const Rule *> result;
		
		/*check if url_re and headers_re specified by this rule match*/
		caller->checkUrlSet(caller->bodyContentCorrect[rule->actionID],
				    result);
		caller->checkHeadersSet(caller->
					bodyContentCorrect[rule->actionID], 
					result);
		FK_ASSERT(result.size() == 0);
	}
	return 0;
}

/*
  Remove all rules with url_re option that do not match URL of a current HTTP
  response. Find rules that have matching URL and do not have any headers
  or body related options to check.
  @rules: rules set to check and reduce
  @result: list to which rules with valid URL and without headers_re and 
  body_re options are added.
 */
void
ActiveRuleSet::checkUrlSet(set<Rule *> &rules, list<const Rule *> &result)
{
	set<Rule *>::iterator it;
	list<RE>::const_iterator it2;	
	list <set<Rule*>::iterator> to_erase;
	list <set<Rule*>::iterator>::iterator it3;
	
	TRACE("checking url");
	for(it = rules.begin(); it != rules.end(); ++it){
		const Rule *rule = *it;
		for(it2 = rule->urlREs.begin(); it2 != rule->urlREs.end(); 
		    ++it2){
			TRACE("Matching %s against %s", it2->pattern, url);
			if (!match(*it2, url, urllen)){
				TRACE("not matched");				
				to_erase.push_front(it);
				noMatch.insert(rule);
				break;
			}
			TRACE("matched");
		}
		if (it2 == rule->urlREs.end() && !rule->headersRule() && 
		    !rule->bodyRule()){
			result.push_back(rule);

			/*avoid alerting more then once*/
			to_erase.push_front(it);
		}
	}
	
	for(it3 = to_erase.begin(); it3 != to_erase.end(); ++it3){
		//erase rule from a rule set
		rules.erase(*it3);
	}

}

/*
  Remove all rules with headers_re option that do not match headers of a 
  current HTTP response. Find rules that have matching headers and do not 
  have any body related options to check.
  @rules: rules set to check and reduce
  @result: list to which rules with valid headers and without body_re options
  are added.
 */
void
ActiveRuleSet::checkHeadersSet(set<Rule *> &rules, list<const Rule *> &result)
{
	set<Rule *>::iterator it;
	list<RE>::const_iterator it2;	
	list <set<Rule*>::iterator> to_erase;
	list <set<Rule*>::iterator>::iterator it3;
	
	TRACE("checking headers");
	for(it = rules.begin(); it != rules.end(); ++it){
		const Rule *rule = *it;

		for(it2 = rule->headersREs.begin(); it2 != rule->headersREs.end(); ++it2){
			TRACE("Matching %s against %s", it2->pattern, headers);
			if (!match(*it2, headers, headerslen)){
				TRACE("not matched");				
				to_erase.push_front(it);
				noMatch.insert(rule);
				break;
			}
			TRACE("matched");
		}
		if (it2 == rule->headersREs.end() && !rule->bodyRule()){
			TRACE("inserting %d", rule->fid);
			result.push_back(rule);
			
			/*avoid alerting more then once*/
			to_erase.push_front(it);
		}
	}
	
	for(it3 = to_erase.begin(); it3 != to_erase.end(); ++it3){
		//erase rule from a rule set
		rules.erase(*it3);
	}
}

/*
  Find rules that have all body_re options matched against current HTTP 
  response body. (Do not remove rules that do not match from a set 
  because unlike URL and headers, respomse body can be splitted into 
  multiple parts  and body_re can match against any of these parts).
  @rules: rules set to check 
  @result: list to which rules with all body_re options matched are added.
 */
void
ActiveRuleSet::checkBodySet(set<Rule *> &rules, const char *body, int bodylen,
			     list<const Rule *> &result)
{
	set<Rule *>::iterator it;
	list<RE>::const_iterator it2;	
	list <set<Rule*>::iterator> to_erase;
	list <set<Rule*>::iterator>::iterator it3;
	
	TRACE("checking body");
	for(it = rules.begin(); it != rules.end(); ++it){
		const Rule *rule = *it;
		bool matched = true;
		for(it2 = rule->bodyREs.begin(); it2 != rule->bodyREs.end();
		    ++it2){
			if (bodyMatchedREs.find(&*it2) != bodyMatchedREs.end())
				continue;
			TRACE("Matching %s against body", it2->pattern, body);
			if (!match(*it2, body, bodylen)){
				TRACE("not matched");
				matched = false;
			}
			else{
				TRACE("matched");
				bodyMatchedREs.insert(&*it2);
			}
		}
		if (matched){
			TRACE("inserting %d", rule->fid);
			result.push_back(rule);	
			
			/*avoid alerting more then once*/
			to_erase.push_front(it);
		}
	}

	for(it3 = to_erase.begin(); it3 != to_erase.end(); ++it3){
		//erase rule from a rule set
		rules.erase(*it3);
	}
}

/*
  Check URL of a HTTP request/response.
  @url_arg: URL to check
  @urllen_arg: length of a URL
  @returns: list of Rules that match this URL and do not have 
  any options related to body and headers.
 */
list<const Rule*>
ActiveRuleSet::checkURL(const char *url_arg, int urllen_arg)
{
	list<const Rule *> result;
	this->url = strdup(url_arg);
	this->urllen = urllen_arg;
	
	if (rulesVersion < rules->getRulesVersion()){
		TRACE("not checking url - rules invalidated");
		return result;
	}

	/*check url_content patterns*/
	searchAPI->search_find(URL_CONTENT, (unsigned char *)url, urllen,
			       urlContentMatched, this);
	
	/*check url_re patterns (only for rules without *_content options 
	  or with matched url_content options)
	 */
	for(int i = 0; i < NACTIONS; i++){		
		checkUrlSet(noContent[i], result);
		checkUrlSet(urlContentCorrect[i], result);
		for(set<Rule *>::iterator it = urlContentCorrect[i].begin();
		    it != urlContentCorrect[i].end(); ++it)
			if ((*it)->headers_content.size() == 0 && 
			    (*it)->body_content.size() == 0)
				noContent[i].insert(*it);	
		
	}
	return result;	
}

/*
  Check headers of a HTTP response.
  @headers_arg: headers to check
  @headerslen_arg: length of a headers string
  @returns: list of rules with headers_* options that have matched headers 
  and URL and do not have any options related to body.
 */
list<const Rule*>
ActiveRuleSet::checkHeaders(const char *headers_arg, int headerslen_arg)
{
	list<const Rule *> result;
	this->headers = strdup(headers_arg);
	this->headerslen = headerslen_arg;
	
	if (rulesVersion < rules->getRulesVersion()){
		TRACE("not checking headers - rules invalidated");
		return result;
	}
	
	/*check headers_content patterns*/
	searchAPI->search_find(HEADERS_CONTENT, (unsigned char *)headers, 
			       headerslen,  headersContentMatched, this);

	/*check headers_re patterns (only for rules without *_content options 
	  or with matched url_content and headers_content options)
	 */
	for(int i = 0; i < NACTIONS; i++){
		checkUrlSet(headersContentCorrect[i], result);
		
		checkHeadersSet(noContent[i], result);
		checkHeadersSet(headersContentCorrect[i], result);

		for(set<Rule *>::iterator it = headersContentCorrect[i].
			    begin();it != headersContentCorrect[i].end(); ++it)
			if ((*it)->body_content.size() == 0)
				noContent[i].insert(*it);	
	}
	return result;

}

/*
  Check body of a HTTP response.
  @body: body to check
  @body_len: length of a body string
  @returns: list of rules with body_* options that have matched all URL, 
  headers and body options. 
 */
list<const Rule*>
ActiveRuleSet::checkBody(const char *body, int bodylen)
{
	list<const Rule *> result;
	TRACE("called");
	
	if (rulesVersion < rules->getRulesVersion()){
		TRACE("not checking body - rules invalidated");
		return result;
	}
	
	/*check body_content patterns*/
	TRACE("search API calling bodylen: %d", bodylen);
	searchAPI->search_find(BODY_CONTENT, (unsigned char *)body, bodylen, 
			       bodyContentMatched, this);
	
	/*check body_re patterns*/
	for(int i = 0; i < NACTIONS; i++){
		TRACE("checking action %d", i);
		checkBodySet(noContent[i], body, bodylen, result);
		TRACE("rules with matched body content: %d", bodyContentCorrect[i].size());
		checkBodySet(bodyContentCorrect[i], body, bodylen, result);
	}
	return result;

}

