/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is hultmann localization tools.
 *
 * The Initial Developer of the Original Code is
 * Jeferson Hultmann <hultmann@gmail.com>
 * Portions created by the Initial Developer are Copyright (C) 2005
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either of the GNU General Public License Version 2 or later (the "GPL"),
 * or the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Text;

namespace LocaleInpector
{

internal sealed class Escape
{

private List<string> m_log = new List<string>();

public Escape()
{
}

public string[] GetErrors()
{
    string[] arr = new string[m_log.Count];
    m_log.CopyTo(arr, 0);
    return arr;
}

private void Add(string id, string txt, string msg)
{
    if (id != null) {
        m_log.Add(id + " [" + msg + "] " + txt);
    }
}

// http://www.mozilla.org/projects/l10n/mlp_chrome.html#text
// http://java.sun.com/docs/books/jls/second_edition/html/lexical.doc.html#100850
// http://java.sun.com/docs/books/jls/second_edition/html/lexical.doc.html#101089
// http://java.sun.com/j2se/1.4.2/docs/api/java/util/Properties.html#load(java.io.InputStream)
// the raw input "\\u2297=\u2297" results in the eleven characters "\\u2297=ø" (\u2297 is the Unicode encoding of the character "").
// the raw input \u005cu005a results in the six characters \u005a, because 005c is the Unicode value for \.
public string UnEscape(string id, string txt)
{
    if (txt == null) {
        return null;
    }

    bool unicode = Config2.Current.GetPref("escapes.unicode") == "true";

    for (int idx = 0; idx < txt.Length; idx++) {
        if (unicode && txt[idx] > 127) {
            this.Add(id, txt, @"char 0x" + ((int) txt[idx]).ToString("x4") + " not escaped (" + txt[idx] + ")");
            break;
        } else if (txt[idx] < 32) {
            this.Add(id, txt, @"char 0x" + ((int) txt[idx]).ToString("x2") + " not escaped");
            break;
        }
    }

    // undecode \uabcd
    txt = this.UnEscapeUnicode(id, txt);

    System.Text.StringBuilder buf = new System.Text.StringBuilder(txt.Length);
    StringEnumerator read = new StringEnumerator(txt);
    while (read.MoveNext()) {

        if (read.Current != '\\') {
            buf.Append(read.Current);
            continue;
        }

        if (read.MoveNext() == false) {
            // end
            buf.Append('\\');
            this.Add(id,  txt, @"\ at end of string");
            continue;
        }

        switch (read.Current) {
            case 't': // tab
                buf.Append('\r');
                break;
            case 'n': // new line
                buf.Append('\n');
                break;
            case 'f': // form feed
                buf.Append('\f');
                break;
            case 'r': // cr
                buf.Append('\r');
                break;

            case '"':
            case '\'':
            case '\\':
            case ' ':
                buf.Append(read.Current);
                break;

            // MT
            case ':':
            case '<':
            case '#':
            case '=':
            case '!':
                if (Config2.Current.GetPref("escapes.unicode") == "true") {
                    this.Add(id, txt, @"Unnecessary escaped character \" + read.Current + ". Can be fixed by locale-fix");
                }
                buf.Append(read.Current);
                break;

            //
            default:
                buf.Append(read.Current);
                this.Add(id, txt, @"Unnecessary escaped character \" + read.Current);
                break;
        }
    }
    return buf.ToString();
}

/*
http://java.sun.com/docs/books/jls/second_edition/html/lexical.doc.html#100850

In addition to the processing implied by the grammar, for each raw input character
that is a backslash \, input processing must consider how many other \ characters
contiguously precede it, separating it from a non-\ character or the start of the
input stream. If this number is even, then the \ is eligible to begin a Unicode
escape; if the number is odd, then the \ is not eligible to begin a Unicode escape.
For example, the raw input "\\u2297=\u2297" results in the eleven characters
" \ \ u 2 2 9 7 =  " (\u2297 is the Unicode encoding of the character "").
*/
public string UnEscapeUnicode(string id, string txt)
{
    System.Text.StringBuilder buf = new System.Text.StringBuilder(txt.Length);
    StringEnumerator read = new StringEnumerator(txt);
    int backslashs = 0;

    while (read.MoveNext()) {
        switch (read.Current) {
            case 'u':
                break;
            case '\\':
                backslashs++;
                buf.Append('\\');
                continue;
            default:
                backslashs = 0;
                buf.Append(read.Current);
                continue;
        }

        // read.Current == 'u'
        // .u      NO .u
        // .\u     OK .*
        // .\\u    NO .\\u1234
        // .\\\u   OK .\\*
        // .\\\\u  NO .\\\\u1234
        // .\\\\\u OK .\\\\*
        if ((backslashs & 1) == 0) {
            backslashs = 0;
            buf.Append('u');
            continue;
        }

        // check escape code
        backslashs = 0;

        string escapeString = Next4Chars(read);
        if (IsValidUnicode(escapeString) == false) {
            buf.Append('u');
            buf.Append(escapeString);
            this.Add(id, txt, @"invalid unicode escape (\u" + escapeString + ")");
            continue;
        }

        // parse escape

        int code = int.Parse(escapeString, System.Globalization.NumberStyles.HexNumber);
        char myChar = (char) code;

        // warning
        switch (code) {
            case '"':
            case '\'':
            case '\\':
                break;
            default:
                if ((code < 128) && (code > 32)) {
                    this.Add(id, txt, @"unnecessary unicode escaped character: \u" + escapeString + " (" + myChar.ToString() + ")");
                }
                break;
        }

        // replace last char ==> \
        buf[buf.Length - 1] = myChar;
    }

    return buf.ToString();
}

private static string Next4Chars(StringEnumerator read)
{
    if (read.MoveNext() == false) {
        return string.Empty;
    }
    char[] gg = new char[4];
    gg[0] = read.Current;
    if (read.MoveNext() == false) {
        return new string(gg, 0, 1);
    }
    gg[1] = read.Current;
    if (read.MoveNext() == false) {
        return new string(gg, 0, 2);
    }
    gg[2] = read.Current;
    if (read.MoveNext() == false) {
        return new string(gg, 0, 3);
    }
    gg[3] = read.Current;
    return new string(gg, 0, 4);
}

private static bool IsValidUnicode(string gg)
{
    if (gg.Length != 4) {
        return false;
    }
    if (IsHex(gg[0]) == false) {
        return false;
    }
    if (IsHex(gg[1]) == false) {
        return false;
    }
    if (IsHex(gg[2]) == false) {
        return false;
    }
    if (IsHex(gg[3]) == false) {
        return false;
    }
    return true;
}

private static bool IsHex(char ch)
{
    return "0123456789abcdefABCDEF".IndexOf(ch) > -1;
}

}//class
}//ns