/*
 * Electric(tm) VLSI Design System
 *
 * File: dberror.c
 * Database error handler
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

#include "global.h"
#include "database.h"
#include "usr.h"

INTBIG db_lasterror;			/* last error message */
INTSML db_printerrors;			/* flag for printing internal errors */

static char *db_errortype[] =
{
	N_("no error"),								/* DBNOERROR */
	N_("no memory"),							/* DBNOMEM */
	N_("bad transposition"),					/* DBBADTRANS */
	N_("bad rotation"),							/* DBBADROT */
	N_("bad prototype"),						/* DBBADPROTO */
	N_("bad parent"),							/* DBBADPARENT */
	N_("invalid instance"),						/* DBBADINST */
	N_("invalid name"),							/* DBBADNAME */
	N_("bad width"),							/* DBBADWIDTH */
	N_("bad end A node/port"),					/* DBBADENDAN */
	N_("bad end B node/port"),					/* DBBADENDBN */
	N_("bad end A connection"),					/* DBBADENDAC */
	N_("bad end B connection"),					/* DBBADENDBC */
	N_("bad end A position"),					/* DBBADENDAP */
	N_("bad end B position"),					/* DBBADENDBP */
	N_("bad new width"),						/* DBBADNEWWID */
	N_("bad facet"),							/* DBBADFACET */
	N_("bad library"),							/* DBBADLIB */
	N_("bad size"),								/* DBBADSIZE */
	N_("bad object type"),						/* DBBADOBJECT */
	N_("bad sub port"),							/* DBBADSUBPORT */
	N_("still has arcs"),						/* DBHASARCS */
	N_("still has ports"),						/* DBHASPORTS */
	N_("facet has instances"),					/* DBHASINSTANCES */
	N_("recursive call"),						/* DBRECURSIVE */
	N_("arc not in port"),						/* DBNOTINPORT */
	N_("conflicts with primitive"),				/* DBCONFLICT */
	N_("port mismatch"),						/* DBPORTMM */
	N_("duplicate name"),						/* DBDUPLICATE */
	N_("primitive prototype"),					/* DBPRIMITIVE */
	N_("bad transformation matrix"),			/* DBBADTMAT */
	N_("variable does not exist"),				/* DBNOVAR */
	N_("variable cannot be changed"),			/* DBVARFIXED */
	N_("variable cannot be displayable array"),	/* DBVARARRDIS */
	N_("this is the last technology"),			/* DBLASTECH */
	N_("technology is still in use"),			/* DBTECINUSE */
	N_("sliding not allowed")					/* DBNOSLIDING */
};

/*
 * database error routine codes
 */
static char *db_routines[] =
{
	"addstringtoinfstr",	/*  1 */
	"addtechnology",		/*  2 */
	"addtoinfstr",			/*  3 */
	"allocarcinst",			/*  4 */
	"alloccell",			/*  5 */
	"allocgeom",			/*  6 */
	"alloclibrary",			/*  7 */
	"allocnodeinst",		/*  8 */
	"allocnodeproto",		/*  9 */
	"allocpolygon",			/* 10 */
	"allocportarcinst",		/* 11 */
	"allocportexpinst",		/* 12 */
	"allocportproto",		/* 13 */
	"allocstring",			/* 14 */
	"alloctechnology",		/* 15 */
	"allocview",			/* 16 */
	"copynodeproto",		/* 17 */
	"delind",				/* 18 */
	"delindkey",			/* 19 */
	"delvalkey",			/* 20 */
	"describevariable",		/* 21 */
	"extendpolygon",		/* 22 */
	"initinfstr",			/* 23 */
	"initobjlist",			/* 24 */
	"insind",				/* 25 */
	"insindkey",			/* 26 */
	"killarcinst",			/* 27 */
	"killlibrary",			/* 28 */
	"killnodeinst",			/* 29 */
	"killnodeproto",		/* 30 */
	"killportproto",		/* 31 */
	"killtechnology",		/* 32 */
	"makekey",				/* 33 */
	"modifyarcinst",		/* 34 */
	"moveportproto",		/* 35 */
	"newarcinst",			/* 36 */
	"newlibrary",			/* 37 */
	"newnodeinst",			/* 38 */
	"newnodeproto",			/* 39 */
	"newportproto",			/* 40 */
	"newview",				/* 41 */
	"replacearcinst",		/* 42 */
	"replacenodeinst",		/* 43 */
	"returninfstr",			/* 44 */
	"setind",				/* 45 */
	"setindkey",			/* 46 */
	"setval",				/* 47 */
	"setvalkey",			/* 48 */
	"transmult",			/* 49 */
	"xform",				/* 50 */
};

/*
 * routine to report a database error
 */
INTBIG db_error(INTBIG code)
{
	db_lasterror = code;
	if (db_printerrors != 0) telldatabaseerror();
	return(-1);
}

/*
 * routine to print an error message
 */
void telldatabaseerror(void)
{
	REGISTER INTSML routine, error;

	routine = (db_lasterror >> 16) - 1;
	error = db_lasterror & 0xFFFF;
	if (error == DBNOERROR) ttyputmsg(_("No error")); else
		ttyputmsg("%s: %s", db_routines[routine], _(db_errortype[error]));
}

/******************** ERROR REPORTING ********************/

/*
 * These are the routines to log errors:
 *   initerrorlogging(char *s, NODEPROTO *f1, NODEPROTO *f2) initialize for subsystem "s"
 *   e = logerror(char *msg, NODEPROTO *f, k)    log message "msg" in facet "f", sort key "k"
 *   addgeomtoerror(void *e, GEOM *g, INTSML s, INTSML l, NODEINST **p)
 *                                               add geom "g" to error "e" (show if "s" nonzero)
 *   addexporttoerror(void *e, PORTPROTO *p, INTSML s)  add export "pp" to error "e"
 *   addlinetoerror(void *e, x1, y1, x2, y2)     add line to error "e"
 *   addpolytoerror(void *e, POLYGON *p)         add polygon to error "e"
 *   addpointtoerror(void *e, x, y)              add point to error "e"
 * To report errors, call:
 *   termerrorlogging()                          complete error accumulation
 *   sorterrors()                                sort errors by key
 *   n = numerrors()                             returns number of errors
 *   s = reportnexterror(show, &g1, &g2)         report next error
 *   s = reportpreverror()                       report previous error
 *   s = reportcurrenterror(show, &g1, &g2)      report current error
 */

typedef enum {ERRORTYPEGEOM, ERRORTYPEEXPORT, ERRORTYPELINE, ERRORTYPEPOINT} ERRORHIGHLIGHTTYPE;

#define NOERRORHIGHLIGHT ((ERRORHIGHLIGHT *)-1)

typedef struct
{
	ERRORHIGHLIGHTTYPE type;
	GEOM              *geom;
	PORTPROTO         *pp;
	INTBIG             showgeom;
	INTBIG             x1, y1;
	INTBIG             x2, y2;
	INTSML             pathlen;
	NODEINST         **path;
} ERRORHIGHLIGHT;


#define NOERRORLIST ((ERRORLIST *)-1)

typedef struct iErrorList
{
	char              *message;
	NODEPROTO         *facet;
	INTBIG             sortkey;
	INTBIG             numhighlights;
	ERRORHIGHLIGHT   **highlights;
	struct iErrorList *preverrorlist;
	struct iErrorList *nexterrorlist;
} ERRORLIST;

static ERRORLIST *db_errorlistfree = NOERRORLIST;
static ERRORLIST *db_firsterrorlist = NOERRORLIST;
static ERRORLIST *db_curerrorlist = NOERRORLIST;
static ERRORLIST *db_nexterrorlist = NOERRORLIST;
static ERRORLIST *db_preverrorlist = NOERRORLIST;
static char       db_errorsystem[100] = "";
static NODEPROTO *db_errorfacet1 = NONODEPROTO;
static NODEPROTO *db_errorfacet2 = NONODEPROTO;


static INTSML db_addtoerrorlist(ERRORLIST *el, ERRORHIGHLIGHT *eh);
static int    db_errorlistascending(const void *e1, const void *e2);

/*
 * Routine to free all previously stored errors and initialize the system.
 * The errors are described by "system" and up to two facets "facet1" and
 * "facet2" (may be NONODEPROTO).
 */
void initerrorlogging(char *system, NODEPROTO *facet1, NODEPROTO *facet2)
{
	db_freeerrormemory();
	db_curerrorlist = NOERRORLIST;
	db_nexterrorlist = NOERRORLIST;
	db_preverrorlist = NOERRORLIST;
	strncpy(db_errorsystem, system, 100);
	db_errorfacet1 = facet1;
	db_errorfacet2 = facet2;
}

void db_freeerrormemory(void)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;
	REGISTER INTBIG i;

	while (db_firsterrorlist != NOERRORLIST)
	{
		el = db_firsterrorlist;
		db_firsterrorlist = el->nexterrorlist;
		efree(el->message);
		for(i=0; i<el->numhighlights; i++)
		{
			eh = el->highlights[i];
			if (eh->pathlen > 0) efree((char *)eh->path);
			efree((char *)eh);
		}
		if (el->numhighlights > 0) efree((char *)el->highlights);
		efree((char *)el);
	}
}

/*
 * Routine to create an error message with the text "message" applying to facet "facet".
 * Returns a pointer to the message (0 on error) which can be used to add highlights.
 */
void *logerror(char *message, NODEPROTO *facet, INTBIG sortkey)
{
	REGISTER ERRORLIST *el;

	if (db_errorlistfree != NOERRORLIST)
	{
		el = db_errorlistfree;
		db_errorlistfree = el->nexterrorlist;
	} else
	{
		el = (ERRORLIST *)emalloc(sizeof (ERRORLIST), db_cluster);
		if (el == 0) return(0);
	}
	if (allocstring(&el->message, message, db_cluster) != 0) return(0);
	el->facet = facet;
	el->sortkey = sortkey;
	el->numhighlights = 0;
	el->preverrorlist = NOERRORLIST;
	el->nexterrorlist = db_firsterrorlist;
	if (db_firsterrorlist != NOERRORLIST) db_firsterrorlist->preverrorlist = el;
	db_firsterrorlist = el;
	return((void *)el);
}

/*
 * Routine to add "geom" to the error in "errorlist".  Also adds a
 * hierarchical traversal path "path" (which is "pathlen" long).
 * Returns nonzero on error.
 */
INTSML addgeomtoerror(void *errorlist, GEOM *geom, INTSML showit, INTSML pathlen, NODEINST **path)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;
	REGISTER INTSML i;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPEGEOM;
	eh->geom = geom;
	eh->showgeom = showit;
	eh->pathlen = pathlen;
	if (pathlen > 0)
	{
		eh->path = (NODEINST **)emalloc(pathlen * (sizeof (NODEINST *)), db_cluster);
		if (eh->path == 0) return(1);
		for(i=0; i<pathlen; i++)
			eh->path[i] = path[i];
	}
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

/*
 * Routine to add "pp" to the error in "errorlist".  Returns nonzero on error.
 */
INTSML addexporttoerror(void *errorlist, PORTPROTO *pp, INTSML showit)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPEEXPORT;
	eh->pp = pp;
	eh->showgeom = showit;
	eh->pathlen = 0;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

/*
 * Routine to add line (x1,y1)=>(x2,y2) to the error in "errorlist".
 * Returns nonzero on error.
 */
INTSML addlinetoerror(void *errorlist, INTBIG x1, INTBIG y1, INTBIG x2, INTBIG y2)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPELINE;
	eh->x1 = x1;
	eh->y1 = y1;
	eh->x2 = x2;
	eh->y2 = y2;
	eh->pathlen = 0;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

/*
 * Routine to add polygon "poly" to the error in "errorlist".
 * Returns nonzero on error.
 */
INTSML addpolytoerror(void *errorlist, POLYGON *poly)
{
	REGISTER INTBIG i, prev;
	INTBIG lx, hx, ly, hy;

	if (isbox(poly, &lx, &hx, &ly, &hy) != 0)
	{
		if (addlinetoerror(errorlist, lx, ly, lx, hy) != 0) return(1);
		if (addlinetoerror(errorlist, lx, hy, hx, hy) != 0) return(1);
		if (addlinetoerror(errorlist, hx, hy, hx, ly) != 0) return(1);
		if (addlinetoerror(errorlist, hx, ly, lx, ly) != 0) return(1);
	} else
	{
		for(i=0; i<poly->count; i++)
		{
			if (i == 0) prev = poly->count-1; else prev = i-1;
			if (addlinetoerror(errorlist, poly->xv[prev], poly->yv[prev],
				poly->xv[i], poly->yv[i]) != 0) return(1);
		}
	}
	return(0);	
}

/*
 * Routine to add point (x,y) to the error in "errorlist".
 * Returns nonzero on error.
 */
INTSML addpointtoerror(void *errorlist, INTBIG x, INTBIG y)
{
	REGISTER ERRORLIST *el;
	REGISTER ERRORHIGHLIGHT *eh;

	el = (ERRORLIST *)errorlist;
	eh = (ERRORHIGHLIGHT *)emalloc(sizeof (ERRORHIGHLIGHT), db_cluster);
	if (eh == 0) return(1);
	eh->type = ERRORTYPEPOINT;
	eh->x1 = x;
	eh->y1 = y;
	eh->pathlen = 0;
	if (db_addtoerrorlist(el, eh) != 0) return(1);
	return(0);	
}

INTSML db_addtoerrorlist(ERRORLIST *el, ERRORHIGHLIGHT *eh)
{
	REGISTER ERRORHIGHLIGHT **neweh;
	REGISTER INTBIG i;

	neweh = (ERRORHIGHLIGHT **)emalloc((sizeof (ERRORHIGHLIGHT *)) * (el->numhighlights+1),
		db_cluster);
	if (neweh == 0) return(1);
	for(i=0; i<el->numhighlights; i++)
		neweh[i] = el->highlights[i];
	neweh[el->numhighlights] = eh;
	if (el->numhighlights > 0) efree((char *)el->highlights);
	el->highlights = neweh;
	el->numhighlights++;
	return(0);
}

/*
 * Routine called when all errors are logged.  Initializes pointers for replay of errors.
 */
void termerrorlogging(void)
{
	REGISTER INTBIG errs, olderrs;
	REGISTER VARIABLE *var;

	db_curerrorlist = NOERRORLIST;
	db_nexterrorlist = db_firsterrorlist;
	db_preverrorlist = NOERRORLIST;

	/* save the number of errors in "%V" */
	errs = numerrors();
	var = getval((INTBIG)us_tool, VTOOL, VINTEGER, "USER_local_capv");
	if (var == NOVARIABLE) olderrs = -1; else
		olderrs = var->addr;
	if (errs != olderrs)
	{
		setval((INTBIG)us_tool, VTOOL, "USER_local_capv", errs, VINTEGER|VDONTSAVE);
	}
}

void sorterrors(void)
{
	REGISTER ERRORLIST *ep, **sortedlist;
	REGISTER INTBIG total, i;

	total = 0;
	for(ep = db_firsterrorlist; ep != NOERRORLIST; ep = ep->nexterrorlist) total++;
	if (total == 0) return;
	sortedlist = (ERRORLIST **)emalloc(total * (sizeof (ERRORLIST *)), db_cluster);
	if (sortedlist == 0) return;
	i = 0;
	for(ep = db_firsterrorlist; ep != NOERRORLIST; ep = ep->nexterrorlist)
		sortedlist[i++] = ep;
	esort(sortedlist, total, sizeof (ERRORLIST *), db_errorlistascending);
	for(i=0; i<total; i++)
	{
		ep = sortedlist[i];
		if (i == 0) ep->preverrorlist = NOERRORLIST; else
			ep->preverrorlist = sortedlist[i-1];
		if (i < total-1) ep->nexterrorlist = sortedlist[i+1]; else
			ep->nexterrorlist = NOERRORLIST;
	}
	db_firsterrorlist = sortedlist[0];
	efree((char *)sortedlist);
}

/*
 * Helper routine for "sorterrors()" to sort errors by their key
 */
int db_errorlistascending(const void *e1, const void *e2)
{
	REGISTER ERRORLIST *er1, *er2;

	er1 = *((ERRORLIST **)e1);
	er2 = *((ERRORLIST **)e2);
	return(er1->sortkey - er2->sortkey);
}

/*
 * Routine to return the number of logged errors.
 */
INTBIG numerrors(void)
{
	REGISTER INTBIG errors;
	REGISTER ERRORLIST *el;

	errors = 0;
	for(el = db_firsterrorlist; el != NOERRORLIST; el = el->nexterrorlist)
		errors++;
	return(errors);
}

/*
 * Routine to advance to the next error and report it.
 */
char *reportnexterror(INTBIG showhigh, GEOM **g1, GEOM **g2)
{
	if (db_nexterrorlist == NOERRORLIST)
	{
		(void)initinfstr();
		(void)formatinfstr(_("No more %s errors"), db_errorsystem);
		return(returninfstr());
	}
	db_curerrorlist = db_nexterrorlist;
	db_nexterrorlist = db_curerrorlist->nexterrorlist;
	db_preverrorlist = db_curerrorlist->preverrorlist;
	return(reportcurrenterror(showhigh, g1, g2));
}

/*
 * Routine to back up to the previous error and report it.
 */
char *reportpreverror(void)
{
	if (db_preverrorlist == NOERRORLIST)
	{
		(void)initinfstr();
		(void)formatinfstr(_("No more %s errors"), db_errorsystem);
		return(returninfstr());
	}
	db_curerrorlist = db_preverrorlist;
	db_nexterrorlist = db_curerrorlist->nexterrorlist;
	db_preverrorlist = db_curerrorlist->preverrorlist;
	return(reportcurrenterror(1, 0, 0));
}

/*
 * Routine to return the error message associated with the current error.
 * Highlights associated graphics if "showhigh" is nonzero.  Fills "g1" and "g2"
 * with associated geometry modules (if nonzero).
 */
#define MAXFACETS 20

char *reportcurrenterror(INTBIG showhigh, GEOM **g1, GEOM **g2)
{
	REGISTER ERRORLIST *el;
	REGISTER NODEPROTO *facet, *np;
	NODEPROTO *facetlist[MAXFACETS];
	REGISTER INTBIG i, j, consize, numfacets, havegeoms, newwindows, count;
	REGISTER INTSML hierpathcount;
	REGISTER NODEINST **hierpath;
	INTBIG lx, hx, ly, hy;
	REGISTER ERRORHIGHLIGHT *eh;
	REGISTER GEOM *geom1, *geom2;
	REGISTER WINDOWPART *w;
	WINDOWPART *neww[4];

	el = db_curerrorlist;
	if (el == NOERRORLIST)
	{
		(void)initinfstr();
		(void)formatinfstr(_("No %s errors"), db_errorsystem);
		return(returninfstr());
	}

	/* turn off highlighting */
	if (showhigh != 0)
	{
		(void)asktool(us_tool, "clear");

		/* first figure out which facets need to be displayed */
		numfacets = 0;
		for(i=0; i<el->numhighlights; i++)
		{
			eh = el->highlights[i];
			facet = el->facet;
			hierpathcount = 0;
			switch (eh->type)
			{
				case ERRORTYPEGEOM:
					if (eh->showgeom == 0) facet = NONODEPROTO; else
					{
						facet = geomparent(eh->geom);
						if (eh->pathlen > 0)
						{
							hierpathcount = eh->pathlen;
							hierpath = eh->path;
						}
					}
					break;
				case ERRORTYPEEXPORT:
					if (eh->showgeom == 0) facet = NONODEPROTO; else
						facet = eh->pp->parent;
					break;
			}
			if (facet == NONODEPROTO) continue;
			for(j=0; j<numfacets; j++)
				if (facetlist[j] == facet) break;
			if (j < numfacets) continue;
			if (numfacets >= MAXFACETS) break;
			facetlist[numfacets] = facet;

			/* add this hierarchical path */
			np = facet;
			for(j=hierpathcount-1; j>=0; j--)
			{
				(void)setvalkey((INTBIG)np, VNODEPROTO, us_facet_descent_path, (INTBIG)hierpath[j],
					VNODEINST|VDONTSAVE);
				np = hierpath[j]->parent;
			}
			numfacets++;
		}

		/* be sure that all requested facets are shown */
		newwindows = 0;
		for(i=0; i<numfacets; i++)
		{
			/* see if the facet is already being displayed */
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
				if (w->curnodeproto == facetlist[i]) break;
			if (w != NOWINDOWPART)
			{
				/* already displayed: mark this facet done */
				facetlist[i] = NONODEPROTO;
				continue;
			}

			/* not displayed: find a window to use */
			for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
			{
				if (w->curnodeproto == NONODEPROTO) break;
				if (w->curnodeproto == db_errorfacet1 ||
					w->curnodeproto == db_errorfacet2) continue;
				for(j=0; j<numfacets; j++)
					if (w->curnodeproto == facetlist[j]) break;
				if (j >= numfacets) break;
			}
			if (w != NOWINDOWPART)
			{
				/* found a window: show the facet there */
				el_curwindowpart = w;
				us_fullview(facetlist[i], &lx, &hx, &ly, &hy);
				us_switchtofacet(facetlist[i], lx, hx, ly, hy, NONODEINST, NOPORTPROTO, 0, 0);
				facetlist[i] = NONODEPROTO;
				continue;
			}

			/* keep a count of the number of new windows needed */
			newwindows++;
		}
		while (newwindows > 0)
		{
			neww[0] = us_wantnewwindow(0);
			newwindows--;
			if (newwindows > 0)
			{
				el_curwindowpart = neww[0];
				neww[1] = us_splitcurrentwindow(2, 0, &neww[0]);
				newwindows--;
			}
			if (newwindows > 0)
			{
				el_curwindowpart = neww[0];
				neww[2] = us_splitcurrentwindow(1, 0, &neww[0]);
				newwindows--;
			}
			if (newwindows > 0)
			{
				el_curwindowpart = neww[1];
				neww[3] = us_splitcurrentwindow(1, 0, &neww[1]);
				newwindows--;
			}
			count = 0;
			for(i=0; i<numfacets; i++)
			{
				if (facetlist[i] == NONODEPROTO) continue;
				el_curwindowpart = neww[count++];
				us_fullview(facetlist[i], &lx, &hx, &ly, &hy);
				us_switchtofacet(facetlist[i], lx, hx, ly, hy, NONODEINST, NOPORTPROTO, 0, 0);
				facetlist[i] = NONODEPROTO;
				if (count >= 4) break;
			}
		}
	}

	/* first show the geometry associated with this error */
	geom1 = geom2 = NOGEOM;
	havegeoms = 0;
	for(i=0; i<el->numhighlights; i++)
	{
		eh = el->highlights[i];
		switch (eh->type)
		{
			case ERRORTYPEGEOM:
				if (geom1 == NOGEOM) geom1 = eh->geom; else
					if (geom2 == NOGEOM) geom2 = eh->geom;
				if (showhigh == 0 || eh->showgeom == 0) continue;

				/* include this geometry module in list to show */
				if (havegeoms == 0) (void)initinfstr(); else
					(void)addtoinfstr('\n');
				havegeoms++;
				(void)formatinfstr("FACET=%s FROM=0%lo;-1;0",
					describenodeproto(geomparent(eh->geom)), (INTBIG)eh->geom);
				break;
			case ERRORTYPEEXPORT:
				if (havegeoms == 0) (void)initinfstr(); else
					(void)addtoinfstr('\n');
				havegeoms++;
				(void)formatinfstr("FACET=%s TEXT=0%lo;0%lo;-",
					describenodeproto(eh->pp->parent), (INTBIG)eh->pp->subnodeinst->geom,
						(INTBIG)eh->pp);
				break;
		}
	}

	if (havegeoms != 0)
		(void)asktool(us_tool, "show-multiple", (INTBIG)returninfstr());

	/* now show the lines and points associated with this error */
	for(i=0; i<el->numhighlights; i++)
	{
		eh = el->highlights[i];
		switch (eh->type)
		{
			case ERRORTYPELINE:
				if (showhigh != 0)
					(void)asktool(us_tool, "show-line", eh->x1, eh->y1, eh->x2, eh->y2,
						el->facet);
				break;
			case ERRORTYPEPOINT:
				if (showhigh != 0)
				{
					consize = lambdaoffacet(el->facet) * 5;
					(void)asktool(us_tool, "show-line", eh->x1-consize, eh->y1-consize,
						eh->x1+consize, eh->y1+consize, el->facet);
					(void)asktool(us_tool, "show-line", eh->x1-consize, eh->y1+consize,
						eh->x1+consize, eh->y1-consize, el->facet);
				}
				break;
		}
	}

	/* return geometry if requested */
	if (g1 != 0) *g1 = geom1;
	if (g2 != 0) *g2 = geom2;

	/* return the error message */
	(void)initinfstr();
	(void)addstringtoinfstr(db_errorsystem);
	(void)addstringtoinfstr(": ");
	(void)addstringtoinfstr(el->message);
	return(returninfstr());
}
