/* dap3.c -- plotting routines */

/*  Copyright (C) 2001  Susan Bassein
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "externs.h"
#include "ps.h"
#include "dap_make.h"

extern dataobs dap_obs[];
extern FILE *dap_lst;
extern FILE *dap_err;
extern char *dap_title;
extern char *dap_dapname;

static void plot1(double x[], double y[], int nobs,
	char xvar[], char yvar[], int markv[], int nmark,
	pict p[], pict a[], int pn, char style[],
	double (*xfunct)(), double (*yfunct)(), int doaxes)
{
static char *title0 = NULL;
char *title1;
int overlay;
int n;
int v;
int s;
char axspec[3];
int t;
double pictr;

dap_swap();
if (!title0)
	title0 = dap_malloc(dap_linelen + 1, "");
title1 = dap_malloc(dap_linelen + 1, "");
for (s = 0; style[s] == ' '; s++)
	;
overlay = 0;
if (style[s] == 'o')
	{
	for (s++; style[s] == ' '; s++)
		;
	for ( ; '0' <= style[s] && style[s] <= '9'; s++)
		overlay = 10 * overlay + style[s] - '0';
	if (!overlay)
		overlay = -1;
	}
while (style[s] == ' ')
	s++;
strncpy(axspec, style + s, 2);
axspec[2] = '\0';
if (overlay == -1)
	{
	if (pn)
		pict_initpict(p + pn - 1, p + pn);
	else
		pict_initpict(NULL, p + pn);
	pict_initpict(p + pn, a);
	}
else if (overlay)
	{
	if (pn % overlay)
		pict_initpict(p + pn - 1, p + pn);
	else
		pict_initpict(NULL, p + pn);
	pict_initpict(p + pn, a + (pn / overlay));
	}
else
	{
	pict_initpict(NULL, p + pn);
	pict_initpict(p + pn, a + pn);
	}
for (n = 0; n < nobs; n++)
	pict_point(p + pn, x[n], y[n]);
strcpy(p[pn].pict_type, "CIRC");
title1[0] = '\0';
if (dap_title)
	strcpy(title1, dap_title);
if (overlay >= 0 && nmark)
	{
	for (v = 0; v < nmark; v++)
		{
		strcat(title1, " ");
		if (dap_obs[0].do_len[markv[v]] == DBL)
			sprintf(title1 + strlen(title1), "%g",
						dap_obs[0].do_dbl[markv[v]]);
		else if (dap_obs[0].do_len[markv[v]] == INT)
			sprintf(title1 + strlen(title1), "%d",
						dap_obs[0].do_int[markv[v]]);
		else
			strcat(title1, dap_obs[0].do_str[markv[v]]);
		}
	if (overlay > 0)
		{
		if (!(pn % overlay))
			strcpy(title0, title1);
		else
			{
			for (t = 0; title0[t] && title0[t] == title1[t]; t++)
				;
			title1[t] = '\0';
			}
		}
	}
if (doaxes)
	{
	if (overlay == -1)
		{
		pictr =
		0.05 * pict_autoaxes(p, xvar, yvar, axspec, xfunct, yfunct, title1, PORTRAIT);
		while (pn >= 0)
			p[pn--].pict_r = pictr;
		}
	else if (overlay)
		{
		pictr = 0.05 * pict_autoaxes(p + (pn / overlay) * overlay,
				xvar, yvar, axspec, xfunct, yfunct, title1, PORTRAIT);
		while (pn >= (pn / overlay) * overlay)
			p[pn--].pict_r = pictr;
		}
	else
		p[pn].pict_r = 0.05 * pict_autoaxes(p + pn, xvar, yvar,
					axspec, xfunct, yfunct, title1, PORTRAIT);
	}
dap_swap();
dap_free(title1);
}

static void plotparse(char *xyvar, char *xyname, char *xname, char *yname)
{
int n, xyn, xn, yn;
int ystart;

for (n = 0; xyvar[n] == ' '; n++)
	;
for (xyn = 0; xyvar[n] && xyvar[n] != ' ' && xyvar[n] != '`'; n++)
	{
	if (xyn < dap_namelen)
		xyname[xyn++] = xyvar[n];
	else
		{
		fprintf(dap_err, "(plotparse) X-variable name too long: %s\n", xyvar);
		exit(1);
		}
	}
strncpy(xname, xyvar, n);
xname[n] = '\0';
xyname[xyn++] = ' ';
while (xyvar[n] == ' ')
	n++;
if (xyvar[n] == '`')
	{
	for (n++; xyvar[n] == ' '; n++)
		;
	for (xn = 0; xyvar[n] && xyvar[n] != '`'; n++)
		{
		if (xn < dap_linelen)
			xname[xn++] = xyvar[n];
		else
			{
			fprintf(dap_err, "(plotparse) X-axis label too long: %s\n", xyvar);
			exit(1);
			}
		}
	while (xyvar[n] == ' ')
		n++;
	if (xyvar[n] == '`')
		{
		for (n++; xyvar[n] == ' '; n++)
			;
		}
	else
		{
		fprintf(dap_err, "(plotparse) Expected ` after x-axis label: %s\n", xyvar);
		exit(1);
		}
	xname[xn] = '\0';
	}
ystart = n;
for ( ; xyvar[n] && xyvar[n] != ' ' && xyvar[n] != '`'; n++)
	{
	if (xyn < 2 * (dap_namelen + 1))
		xyname[xyn++] = xyvar[n];
	else
		{
		fprintf(dap_err, "(plotparse) Y-variable name too long: %s\n", xyvar);
		exit(1);
		}
	}
xyname[xyn] = '\0';
if (yname)
	{
	strncpy(yname, xyvar + ystart, n);
	yname[n - ystart] = '\0';
	xyname[xyn++] = ' ';
	while (xyvar[n] == ' ')
		n++;
	if (xyvar[n] == '`')
		{
		for (n++; xyvar[n] == ' '; n++)
			;
		for (yn = 0; xyvar[n] && xyvar[n] != '`'; n++)
			{
			if (yn < dap_linelen)
				yname[yn++] = xyvar[n];
			else
				{
				fprintf(dap_err, "(plotparse) Y-axis label too long: %s\n", xyvar);
				exit(1);
				}
			}
		yname[yn] = '\0';
		}
	}
xyname[xyn] = '\0';
}

pict *plot(char *fname, char *xyvar, char *marks,
		char *style, double (*xfunct)(), double (*yfunct)(), int nplots)
{
pict *p;
pict *a;
int *markv;
int nmark;
int nobs;
static double *x, *y;
int pn;
int xyv[2];
char *xyname;
char *xname;
char *yname;
int s;
int overlay;
int more;

markv = (int *) dap_malloc(sizeof(int) * dap_maxvar, "");
p = (pict *) dap_malloc(2 * nplots * sizeof(pict), "");
a = p + nplots;
x = (double *) dap_malloc(dap_maxval * sizeof(double), "");
y = (double *) dap_malloc(dap_maxval * sizeof(double), "");
if (!fname)
	{
	fputs("(plot) No dataset name given.\n", dap_err);
	exit(1);
	}
inset(fname);
if (!xyvar)
	{
	fputs("(plot) No x and y variable list given.\n", dap_err);
	exit(1);
	}
xyname = dap_malloc(2 * (dap_namelen + 1), "");
xname = dap_malloc(dap_linelen + 1, "");
yname = dap_malloc(dap_linelen + 1, "");
plotparse(xyvar, xyname, xname, yname);
nmark = dap_list(marks, markv, dap_maxvar);
if (dap_list(xyname, xyv, dap_maxvar) != 2)
	{
	fprintf(dap_err, "(plot) Invalid x and y variable list: %s\n", xyvar);
	exit(1);
	}
if (dap_obs[0].do_len[xyv[0]] != DBL)
	{
	fprintf(dap_err, "(plot) x-variable is not double variable: %s\n",
			dap_obs[0].do_nam[xyv[0]]);
	exit(1);
	}
if (dap_obs[0].do_len[xyv[1]] != DBL)
	{
	fprintf(dap_err, "(plot) y-variable is not double variable: %s\n",
			dap_obs[0].do_nam[xyv[1]]);
	exit(1);
	}
for (s = 0; style[s] == ' '; s++)
	;
overlay = 0;
if (style && style[s] == 'o')
	{
	for (s++; style[s] == ' '; s++)
		;
	for ( ; '0' <= style[s] && style[s] <= '9'; s++)
		overlay = 10 * overlay + style[s] - '0';
	if (!overlay)
		overlay = -1;
	}
for (nobs = 0, pn = 0, more = 1; more; nobs++)
        {
	more = step();
        if (dap_newpart(markv, nmark))
		{
		if (pn < nplots)
			{
			plot1(x, y, nobs, xname, yname,
				markv, nmark, p, a, pn, style, xfunct, yfunct,
				(!more || !overlay || (overlay > 0 && !((pn + 1) % overlay))));
			pn++;
			}
		else
			{
			fprintf(dap_err,
				"(plot) More plots than specified by nplots (%d)\n",
						nplots);
			exit(1);
			}
		nobs = 0;
		}
	if (nobs < dap_maxval)
		{
		x[nobs] = dap_obs[0].do_dbl[xyv[0]];
		y[nobs] = dap_obs[0].do_dbl[xyv[1]];
		}
	else
		{
		fprintf(dap_err, "(plot) Too many points\n");
		exit(1);
		}
	}
dap_free(x);
dap_free(y);
dap_free(markv);
dap_free(xyname);
dap_free(xname);
dap_free(yname);
return p;
}

static int dblcmp(double *x, double *y)
{
if (*x < *y)
        return -1;
if (*x > *y)
        return 1;
return 0;
}

#define SQRTHALF 0.707106781186547524401
#define INVSQ2PI 0.398942280401432677940
#define INVSQRTPI 0.56418958354775628
#define TWOOSQRTPI 1.12837916709551257
#define SQRTPI 1.77245385090551602729

static double comb(int n, int k)
{
double dn, dk;
double c;

for (c = 1.0, dn = (double) n, dk = (double) k; dk >= 1.0;
		dn -= 1.0, dk -= 1.0)
	c *= dn / dk;
return c;
}

static double dnmk;
static double dkm1;

static double orderf(double t)
{
double x;
double tmp;

if (t == -1.0 || t == 1.0)
	return 0.0;
tmp = 1.0 - t * t;
x = t / sqrt(tmp);
return exp(dkm1 * log(probz(x)) + dnmk * log(probz(-x)) - 0.5 * x * x) *
		t / (tmp * tmp);
}

#define NSTEPS 128

static void geta(double a[], int n)
{
int k;
double dn;
double c;

for (k = 0; k < n / 2; k++)
	{
	dnmk = (double) (n - k - 1);
	dkm1 = (double) k;
	a[k] = ((double) k + 1) * comb(n, k + 1) * INVSQ2PI * dap_simp(&orderf, -1.0, 1.0, NSTEPS);
	}
dn = (double) n;
if (n <= 20)
	{
	for (a[0] = SQRTHALF; dn > 2.0; dn -= 2.0)
		a[0] *= (dn - 2.0) / (dn - 1.0);
	if (dn == 2.0)
		a[0] *= TWOOSQRTPI;
	else
		a[0] *= SQRTPI;
	}
else
	{
	for (a[0] = SQRTHALF; dn > 1.0; dn -= 2.0)
		a[0] *= (dn - 1.0) / dn;
	if (dn == 1.0)
		a[0] *= TWOOSQRTPI;
	else
		a[0] *= SQRTPI;
	}
for (c = 0.0, k = 1; k < n / 2; k++)
	c += a[k] * a[k];
c = sqrt((1.0 - 2.0 * a[0]) / (2.0 * c));
a[0] = -sqrt(a[0]);
for (k = 1; k < n / 2; k++)
	a[k] *= c;
}

#define NSIMUL 10000

typedef struct swvalstr
{
double swd;
struct swvalstr *swleft;
struct swvalstr *swright;
struct swvalstr *swup;
} swval;


static double probw(double swa[], int n, double w0)
{
int s;
int small;
swval *y;
double yval;
swval *py;
swval *ppy;
int ny;
double dny;
double sum, ss;
double vtmp, tmp;
double w;
int (*cmp)();

cmp = &dblcmp;
y = (swval *) dap_malloc(n * sizeof(swval), "");
for (s = 0, small = 0; s < NSIMUL; s++)
	{
	ny = 0;
	y[ny].swd = varnorm();
	y[ny].swleft = NULL;
	y[ny].swright = NULL;
	y[ny].swup = NULL;
	for (ny++; ny < n; ny++)
		{
		yval = varnorm();
		y[ny].swd = yval;
		y[ny].swleft = NULL;
		y[ny].swright = NULL;
		for (py = &y[0]; ; py = ppy)
			{
			if (yval >= py->swd)
				{
				ppy = py->swright;
				if (!ppy)
					{
					py->swright = &y[ny];
					y[ny].swup = py;
					break;
					}
				}
			else
				{
				ppy = py->swleft;
				if (!ppy)
					{
					py->swleft = &y[ny];
					y[ny].swup = py;
					break;
					}
				}
			}
		}
	for (py = &y[0]; (ppy = py->swleft); py = ppy)
		;
	for (ny = 0, sum = 0.0, ss = 0.0, w = 0.0; py; )
		{
		dny = (double) ny;
		vtmp = py->swd;
		if (ny < n / 2)
			w += swa[ny] * vtmp;
		else if (ny > n / 2)
			w -= swa[n - 1 - ny] * vtmp;
		py->swd = 0.0 / 0.0;
		if (ny)
			{
			tmp = sum - dny * vtmp;
			ss += tmp * tmp / (dny * (dny + 1.0));
			}
		sum += vtmp;
		ny++;
		if ((ppy = py->swright))
			{
			for (py = ppy; (ppy = py->swleft); py = ppy)
				;
			}
		else if ((ppy = py->swup))
			{
			for (py = ppy; py && !finite(py->swd); py = py->swup)
				;
			}
		else
			break;
		}
	if (ny < n)
		{
		fprintf(dap_err,
			"(probw) Only collected %d values from tree of %d values.\n",
					ny, n);
		exit(1);
		}
/*
	for (w = 0.0, ny = 0; ny < n / 2; ny++)
		w += swa[ny] * (y[ny] - y[n - 1 - ny]);
*/
	w *= w / ss;
	if (w < w0)
		small++;
	}
dap_free(y);
return ((double) small) / ((double) NSIMUL);
}

static void normal1(double x[], double y[], int nobs,
        char varname[], char varlabel[], int markv[], int nmark,
	pict *p, pict *l, pict *a, int pn)
{
int r;
double dr;
double dnobsp25;
double sum;
double ss;
double vtmp;
double tmp;
double sd;
double minx, maxx;
double *swa;
int k;
double w;
double prob;
static char caption[46];
int (*cmp)();

cmp = &dblcmp;
qsort((void *) y, (size_t) nobs, (size_t) sizeof(double), cmp);
dnobsp25 = ((double) nobs) + 0.25;
for (r = 0, sum = 0.0, ss = 0.0, minx = 0.0, maxx = 0.0; r < nobs; r++)
	{
	dr = (double) r;
	x[r] = -zpoint((dr + 0.625) / dnobsp25);
	if (x[r] < minx)
		minx = x[r];
	if (x[r] > maxx)
		maxx = x[r];
	vtmp = y[r];
	if (r)
		{
		tmp = sum - dr * vtmp;
		ss += tmp * tmp / (dr * (dr + 1.0));
		}
	sum += vtmp;
	}
sd = sqrt(ss / ((double) (nobs - 1)));
if (sd == 0.0)
	{
	fprintf(dap_err, "(normal1) Zero standard deviation for %s\n", varname);
	exit(1);
	}
if (6 < nobs && nobs <= 500)
	{
	swa = (double *) dap_malloc(nobs / 2 * sizeof(double), "");
	geta(swa, nobs);
	for (w = 0.0, k = 0; k < nobs / 2; k++)
		w += swa[k] * (y[k] - y[nobs - 1 - k]);
	w *= w / ss;
	if ((prob = probw(swa, nobs, w)) < 0.001)
		prob = 0.001;
	dap_free(swa);
	sprintf(caption, "q-q plot: W|0| = %.4f, P[W < W|0|] = %.3f", w, prob);
	}
else
	strcpy(caption, "q-q plot");
pict_initpict(NULL, p + pn);
pict_initpict(p + pn, l + pn);
pict_initpict(l + pn, a + pn);
for (r = 0; r < nobs; r++)
	pict_point(p + pn, x[r], y[r]);
strcpy(p[pn].pict_type, "CIRC");
sum /= (double) nobs;
strcpy(l[pn].pict_type, "LINE");
pict_line(l + pn, minx, sd * minx + sum, maxx, sd * maxx + sum);
p[pn].pict_r = 0.05 * pict_autoaxes(p + pn, "z", varlabel, "00", NULL, NULL, caption, PORTRAIT);
}

pict *normal(char *fname, char *variable, char *marks, int nplots)
{
char *varname;
char *varlabel;
int s, t;
pict *p;
pict *l;
pict *a;
int *markv;
int nmark;
int nobs;
double *x, *y;
int pn;
int vy;
int more;

varname = dap_malloc(dap_namelen + 1, "");
varlabel = dap_malloc(dap_linelen + 1, "");
markv = (int *) dap_malloc(sizeof(int) * dap_maxvar, "");
p = (pict *) dap_malloc(3 * nplots * sizeof(pict), "");
l = p + nplots;
a = p + 2 * nplots;
x = (double *) dap_malloc(dap_maxval * sizeof(double), "");
y = (double *) dap_malloc(dap_maxval * sizeof(double), "");
if (!variable)
	{
	fputs("(normal) No variable specified.\n", dap_err);
	exit(1);
	}
for (t = 0; variable[t] == ' '; t++)
	;
for (s = 0; variable[t] && variable[t] != ' ' && variable[t] != '`'; t++)
	{
	if (s < dap_namelen)
		varname[s++] = variable[t];
	else
		{
		fprintf(dap_err, "(normal) Variable name too long: %s\n", variable);
		exit(1);
		}
	}
varname[s] = '\0';
while (variable[t] == ' ')
	t++;
s = 0;
if (variable[t] == '`')
	{
	for (t++ ; variable[t] && variable[t] != ' ' && variable[t] != '`'; t++)
		{
		if (s < dap_linelen)
			varlabel[s++] = variable[t];
		else
			{
			fprintf(dap_err, "(normal) Variable label too long: %s\n", variable);
			exit(1);
			}
		}
	}
varlabel[s] = '\0';
if (!fname)
	{
	fputs("(normal) No dataset name given.\n", dap_err);
	exit(1);
	}
inset(fname);
nmark = dap_list(marks, markv, dap_maxvar);
if ((vy = dap_varnum(varname)) < 0)
	{
	fprintf(dap_err, "(normal) Variable unknown: %s\n", varname);
	exit(1);
	}
if (dap_obs[0].do_len[vy] != DBL)
	{
	fprintf(dap_err, "(normal) Variable is not double variable: %s\n", varname);
	exit(1);
	}
for (nobs = 0, pn = 0, more = 1; more; nobs++)
        {
	more = step();
        if (dap_newpart(markv, nmark))
		{
		if (pn < nplots)
			normal1(x, y, nobs, varname, varlabel, markv, nmark,
				p, l, a, pn++);
		else
			{
			fprintf(dap_err,
				"(normal) More plots than specified by nplots (%d)\n",
						nplots);
			exit(1);
			}
		nobs = 0;
		}
	if (nobs < dap_maxval)
		y[nobs] = dap_obs[0].do_dbl[vy];
	else
		{
		fprintf(dap_err, "(normal) Too many points\n");
		exit(1);
		}
	}
dap_free(x);
dap_free(y);
dap_free(varname);
dap_free(varlabel);
dap_free(markv);
return p;
}

static double arint(double x)
{
double i;
        
if (fabs(i = rint(x)) == 0.0)
        return 0.0; 
else    
        return i;
}

#define COUNT 0
#define PERCENT 1
#define FRACTION 2
#define UNSPEC 3

static void histo1(double x[], double xw[][2], int nobs, int varv[], int nbars,
	char xname[], char *style, double (*xfunct)(),
	int markv[], int nmark, pict *p, pict *a, int pn)
{
char *caption;
int s;
int w;
char *word;
char axspec[3];
double *h;
double *part;
int equal;
int height;
int whole;
int zero;
double xlen, xspace;
static char htitle[19];
double width;
double dnobs;
double dnbars;
int b;
int xn;
int xnm1;
int v;
int (*cmp)();

cmp = &dblcmp;
caption = dap_malloc(dap_linelen + 1, "");
word = dap_malloc(dap_namelen + 1, "");
h = (double *) dap_malloc(sizeof(double) * dap_maxbars, "");
part = (double *) dap_malloc(sizeof(double) * (dap_maxbars + 1), "");
if (!nbars)
	{
	fputs("(histo1) Number of bars is zero.\n", dap_err);
	exit(1);
	}
equal = 1;
height = UNSPEC;
whole = 0;
zero = 0;
htitle[0] = '\0';
axspec[0] = '\0';
if (style)
	{
	for (s = 0; style[s] == ' '; s++)
		;
	while (style[s])
		{
		for (w = 0; style[s] && style[s] != ' '; )
			{
			if (w < dap_namelen)
				word[w++] = style[s++];
			else
				{
				word[w] = '\0';
				fprintf(dap_err, "(histo1) Style word too long: %s\n", word);
				exit(1);
				}
			}
		word[w] = '\0';
		if (!strcmp(word, "EQUAL"))
			equal = 1;
		else if (!strcmp(word, "VARIABLE"))
			equal = 0;
		else if (!strcmp(word, "COUNT"))
			height = COUNT;
		else if (!strcmp(word, "PERCENT"))
			height = PERCENT;
		else if (!strcmp(word, "FRACTION"))
			height = FRACTION;
		else if (!strcmp(word, "ROUND"))
			whole = 1;
		else if (!strcmp(word, "ZERO"))
			zero = 1;
		else if (index("-+0nb", word[0]))
			{
			strncpy(axspec, word, 2);
			axspec[2] = '\0';
			}
		else
			{
			fprintf(dap_err, "(histo1) Unknown style parameter: %s\n", word);
			exit(1);
			}
		while (style[s] == ' ')
			s++;
		}
	}
if (equal)
	{
	switch (height)
		{
	case UNSPEC:
	case COUNT:
		strcpy(htitle, "Count");
		break;
	case PERCENT:
		strcpy(htitle, "Percent");
		break;
	case FRACTION:
		strcpy(htitle, "Fraction");
		break;
		}
	}
else
	{
	switch (height)
		{
	case COUNT:
		fputs("(histo1) Can't use count with variable width bars.\n",
					dap_err);
		exit(1);
	case PERCENT:
		strcpy(htitle, "Density (Percent)");
		break;
	case UNSPEC:
	case FRACTION:
		strcpy(htitle, "Density (Fraction)");
		break;
		}
	}
if (x)
	{
	qsort((void *) x, (size_t) nobs, (size_t) sizeof(double), cmp);
	part[0] = x[0];
	part[nbars] = x[nobs - 1];
	}
else
	{
	qsort((void *) xw, (size_t) nobs, (size_t) (2 * sizeof(double)), cmp);
	part[0] = xw[0][0];
	part[nbars] = xw[nobs - 1][0];
	}
dnobs = (double) nobs;
dnbars = (double) nbars;
if (zero)
	part[0] = 0.0;
if (whole)
	{
	xlen = 1e5 / (part[nbars] - part[0]);
	xlen = (arint(xlen * part[nbars]) - arint(xlen * part[0])) / xlen;
	if (xlen >= 1.0)
		{
		for (xspace = 1.0; dnbars * xspace < xlen; xspace *= dnbars)
			;
		xspace *= ceil(xlen / xspace) / dnbars;
		}
	else
		{
		for (xspace = 0.1; xspace / nbars > xlen; xspace /= nbars)
			;
		xspace *= ceil(xlen / xspace) / dnbars;
		}
	part[0] = floor(part[0] / xspace) * xspace;
	part[nbars] = ceil(part[nbars] / xspace) * xspace;
	}
if (equal)
	{
	width = (part[nbars] - part[0]) / dnbars;
	for (b = 1; b < nbars; b++)
		part[b] = part[0] + width * ((double) b);
	for (b = 0; b < nbars; b++)
		h[b] = 0.0;
	for (xn = 0, b = 0; xn < nobs; xn++)
		{
		if (x)
			{
			while (x[xn] > part[b + 1])
				b++;
			h[b] += 1.0;
			}
		else
			{
			while (xw[xn][0] > part[b + 1])
				b++;
			h[b] += xw[xn][1];
			}
		}
	for (b = 0; b < nbars; b++)
		{
		switch (height)
			{
		case PERCENT:
			h[b] *= 100.0;
		case FRACTION:
			h[b] /= dnobs;
			break;
		default:
			break;
			}
		}
	}
else
	{
	for (b = 1, xnm1 = 0; b < nbars; b++)
		{
		xn = (int) rint(dnobs * ((double) b) / dnbars);
		if (x)
			part[b] = x[xn];
		else
			part[b] = xw[xn][0];
		if (part[b] > part[b - 1])
			h[b] = ((double) (xn - xnm1)) / ((part[b] - part[b - 1]) * dnobs);
		else
			h[b] = 0.0;
		if (height == PERCENT)
			h[b] *= 100.0;
		xnm1 = xn;
		}
	}
pict_initpict(NULL, p + pn);
pict_initpict(p + pn, a + pn);
for (b = 0; b < nbars; b++)
	pict_rectangle(p + pn, part[b], 0.0, part[b + 1] - part[b], h[b]);
caption[0] = '\0';
if (dap_title)
	strcpy(caption, dap_title);
if (dap_title)
	{
	for (v = 0; v < nmark; v++)
		{
		if (caption[0] && !v)
			strcat(caption, ": ");
		else if (v)
			strcat(caption, " ");
		if (dap_obs[0].do_len[markv[v]] == DBL)
			sprintf(caption + strlen(caption), "%g",
						dap_obs[0].do_dbl[markv[v]]);
		else if (dap_obs[0].do_len[markv[v]] == INT)
			sprintf(caption + strlen(caption), "%d",
						dap_obs[0].do_int[markv[v]]);
		else
			strcat(caption, dap_obs[0].do_str[markv[v]]);
		}
	}
pict_autoaxes(p + pn, xname, htitle, axspec, xfunct, NULL, caption, PORTRAIT);
dap_free(caption);
dap_free(word);
dap_free(h);
dap_free(part);
}

pict *histogram(char *fname, char *vars, char *marks, int nbars,
				char *style, double (*xfunct)(), int nplots)
{
pict *p;
pict *a;
int *markv;
int nmark;
int varv[2];
int nvar;
int nobs;
double *x;
double (*xw)[2];
char *xwname;
char *xname;
int pn;
int v;
int mv;
int nnan;
int more;

markv = (int *) dap_malloc(sizeof(int) * dap_maxvar, "");
xwname = dap_malloc(2 * (dap_namelen + 1), "");
xname = dap_malloc(dap_linelen + 1, "");
p = (pict *) dap_malloc(2 * nplots * sizeof(pict), "");
a = p + nplots;
if (!fname)
	{
	fputs("(histogram) No dataset name given.\n", dap_err);
	exit(1);
	}
inset(fname);
nmark = dap_list(marks, markv, dap_maxvar);
if (!vars)
	{
	fputs("(histogram) No variable given.\n", dap_err);
	exit(1);
	}
plotparse(vars, xwname, xname, NULL);
nvar = dap_list(xwname, varv, dap_maxvar);
for (v = 0; v < nvar; v++)
	{
	if (dap_obs[0].do_len[varv[v]] != DBL)
		{
		fprintf(dap_err, "(histogram) Variable is not double variable: %s\n",
					dap_obs[0].do_nam[varv[v]]);
		exit(1);
		}
	}
if (nvar == 1)
	{
	x = (double *) dap_malloc(dap_maxval * sizeof(double), "");
	xw = (double (*)[2]) NULL;
	}
else if (nvar == 2)
	{
	xw = (double (*)[2]) dap_malloc(dap_maxval * 2 * sizeof(double), "");
	x = (double *) NULL;
	}
else
	{
	fprintf(dap_err,
	"(histogram) Variable list contains more than two variables: %s\n", vars);
	exit(1);
	}
for (nobs = 0, nnan = 0, pn = 0, more = 1; more; )
        {
	more = step();
        if (dap_newpart(markv, nmark))
                {
                dap_swap();
		if (nnan)
			{
			fprintf(dap_err, "(histogram) %d missing values for:", nnan);
			for (mv = 0; mv < nmark; mv++)
				{
				putc(' ', dap_err);
				if (dap_obs[0].do_len[markv[mv]] == DBL)
					fprintf(dap_err, "%g",
						dap_obs[0].do_dbl[markv[mv]]);
				else if (dap_obs[0].do_len[markv[mv]] == INT)
					fprintf(dap_err, "%d",
						dap_obs[0].do_int[markv[mv]]);
				else
					fputs(dap_obs[0].do_str[markv[mv]], dap_err);
				}
			putc('\n', dap_err);
			}
		if (nobs)
			{
			if (pn < nplots)
				histo1(x, xw, nobs, varv, nbars, xname,
					style, xfunct, markv, nmark, p, a, pn++);
			else
				{
				fprintf(dap_err,
				"(histogram) More plots than specified by nplots (%d)\n",
							nplots);
				exit(1);
				}
			}
                dap_swap();
                nobs = 0;
		nnan = 0;
                }
        if (nobs < dap_maxval)
		{
		if (nvar == 1)
			{
			x[nobs] = dap_obs[0].do_dbl[varv[0]];
			if (finite(x[nobs]))
				nobs++;
			else
				nnan++;
			}
		else
			{
			xw[nobs][0] = dap_obs[0].do_dbl[varv[0]];
			xw[nobs][1] = dap_obs[0].do_dbl[varv[1]];
			if (finite(xw[nobs][0]) && finite(xw[nobs][1]))
				nobs++;
			else
				nnan++;
			}
		}
        else
                {
                fprintf(dap_err, "(histogram) Too many points\n");
                exit(1);
                }
        }
if (nvar == 1)
	dap_free(x);
else
	dap_free(xw);
dap_free(markv);
dap_free(xwname);
dap_free(xname);
return p;
}

pict *plotlinreg(char *fname, char *ylist, char *x1list,
			char *marks, int nmarks, double level)
{
int varv[1];
char *mnsname;
char *regname;
char *srtarg;
char *srtname;
char *plotvars;
char *plotmarks;
pict *p;
int pn;

inset(fname);
dap_list(ylist, varv, 1);	/* check that it's only 1 variable */
dap_list(x1list, varv, 1);	/* check that it's only 1 variable */
mnsname = dap_malloc(strlen(fname) + 5, "");
strcpy(mnsname, fname);
strcat(mnsname, ".mns");
regname = dap_malloc(strlen(fname) + 5, "");
strcpy(regname, fname);
strcat(regname, ".reg");
srtname = dap_malloc(strlen(regname) + 5, "");
strcpy(srtname, regname);
strcat(srtname, ".srt");
means(fname, x1list, "STEP100", marks);
linreg(fname, ylist, "", x1list, marks, mnsname, level);
dataset(fname, regname, "APPEND");
srtarg = dap_malloc(strlen(marks) + strlen(x1list) + 9, "");
strcpy(srtarg, marks);
strcat(srtarg, " _type_ ");
strcat(srtarg, x1list);
sort(regname, srtarg, "");
plotvars = dap_malloc(strlen(x1list) + strlen(ylist) + 2, "");
strcpy(plotvars, x1list);
strcat(plotvars, " ");
strcat(plotvars, ylist);
plotmarks = dap_malloc(strlen(marks) + strlen("_type_") + 2, "");
strcpy(plotmarks, marks);
strcat(plotmarks, " _type_");
p = plot(srtname, plotvars, plotmarks, "o4bb", NULL, NULL, 4 * nmarks);
for (pn = 0; pn < nmarks; pn++)
	{
	strcpy(p[4 * pn + 0].pict_type, "LINE");
	strcpy(p[4 * pn + 2].pict_type, "LINE");
	strcpy(p[4 * pn + 3].pict_type, "LINE");
	p[4 * pn + 0].pict_dash = 4.0;
	p[4 * pn + 3].pict_dash = 4.0;
	}
dap_free(mnsname);
dap_free(regname);
dap_free(srtarg);
dap_free(srtname);
dap_free(plotvars);
dap_free(plotmarks);
return p;
}

pict *plotlogreg(char *fname, char *yspec, char *x1list, int ngroups,
					char *marks, int nmarks,  double level)
{
int varv[3];
char *trlname;
int trialsn;
char varspec[12];
char *grpname;
char *grparg;
char *grpvar;
char *mnsarg;
char *mnsname;
char *lgrname;
char *srtarg;
char *srtname;
char *plotvars;
char *plotmarks;
char *casevar;
int c;
int cs;
pict *p;
int pn;

trlname = dap_malloc(strlen(fname) + 5, "");
strcpy(trlname, fname);
strcat(trlname, ".trl");
grpname = dap_malloc(strlen(trlname) + 5, "");
strcpy(grpname, trlname);
strcat(grpname, ".grp");
srtname = dap_malloc(strlen(grpname) + 5, "");
strcpy(srtname, grpname);
strcat(srtname, ".srt");
mnsname = dap_malloc(strlen(srtname) + 5, "");
strcpy(mnsname, srtname);
strcat(mnsname, ".mns");
lgrname = dap_malloc(strlen(fname) + 5, "");
strcpy(lgrname, fname);
strcat(lgrname, ".lgr");
grparg = dap_malloc(strlen(x1list) + 14, "");
grpvar = dap_malloc(strlen(marks) + strlen(x1list) + 3, "");
casevar = dap_malloc(strlen(yspec) + 1, "");
mnsarg = dap_malloc(strlen(yspec) + 12 + strlen(x1list), "");
srtarg = dap_malloc(strlen(marks) + strlen(x1list) + 9, "");
plotvars = dap_malloc(strlen(x1list) + strlen(yspec) + 2, "");
plotmarks = dap_malloc(strlen(marks) + strlen("_type_") + 2, "");
inset(fname);
dap_list(x1list, varv, 1);	/* check that it's only 1 variable */
strcpy(grpvar, marks);
strcat(grpvar, " _");
strcat(grpvar, x1list);
strcpy(casevar, yspec);
for (cs = 0; casevar[cs] == ' '; cs++)
	;
for (c = 0; casevar[cs] && casevar[cs] != ' ' && casevar[cs] != '/'; )
	casevar[c++] = casevar[cs++];
casevar[c] = '\0';

inset(fname);
sprintf(varspec, "_ntrials %d", DBL);
trialsn = dap_vd(varspec, 0);
outset(trlname, "");
dap_parsey(yspec, varv);
while (step())
	{
	if (varv[1] >= 0)	/* number of trials variable */
		dap_obs[0].do_dbl[trialsn] = dap_obs[0].do_dbl[varv[1]];
	else
		dap_obs[0].do_dbl[trialsn] = -varv[1];
	output();
	}

strcpy(grparg, x1list);
sprintf(grparg + strlen(grparg), " %d COUNT", ngroups);

group(trlname, grparg, marks);

sort(grpname, grpvar, "");

strcpy(mnsarg, casevar);
strcat(mnsarg, " _ntrials ");
strcat(mnsarg, x1list);
means(srtname, mnsarg, "MEAN", grpvar);

inset(mnsname);
outset(grpname, "");
trialsn = dap_varnum("_ntrials");
dap_list(casevar, varv, 1);
while (step())
	{
	dap_obs[0].do_dbl[varv[0]] /= dap_obs[0].do_dbl[trialsn];
	output();
	}

means(fname, x1list, "STEP100", marks);

strcpy(mnsname, fname);
strcat(mnsname, ".mns");

logreg(fname, yspec, "", x1list, marks, mnsname, level);

dataset(grpname, lgrname, "APPEND");

strcpy(srtarg, marks);
strcat(srtarg, " _type_ ");
strcat(srtarg, x1list);

sort(lgrname, srtarg, "");

strcpy(srtname, lgrname);
strcat(srtname, ".srt");

strcpy(plotvars, x1list);
strcat(plotvars, " ");
strcat(plotvars, casevar);

strcpy(plotmarks, marks);
strcat(plotmarks, " _type_");

p = plot(srtname, plotvars, plotmarks, "o4bb", NULL, NULL, 4 * nmarks);
for (pn = 0; pn < nmarks; pn ++)
	{
	strcpy(p[4 * pn + 0].pict_type, "LINE");
	strcpy(p[4 * pn + 2].pict_type, "LINE");
	strcpy(p[4 * pn + 3].pict_type, "LINE");
	p[4 * pn + 0].pict_dash = 4.0;
	p[4 * pn + 3].pict_dash = 4.0;
	}
dap_free(trlname);
dap_free(grpname);
dap_free(grparg);
dap_free(grpvar);
dap_free(mnsarg);
dap_free(mnsname);
dap_free(lgrname);
dap_free(srtarg);
dap_free(srtname);
dap_free(plotvars);
dap_free(plotmarks);
dap_free(casevar);
return p;
}

pict *plotmeans(char *dataset, char *meanvar, char *varlist, char *errbar,
				char *partvars, int noverlay)
{
int meanv[1];
int *partv;
int npartv;
char *mnslist;
char *mnsname;
char *errname;
char *srtname;
char *ebar;
char *overstr;
int e;			/* index to ebar */
double scale;
int typen;
double mean;
double err;
int n;
char *srtarg;
char *plotvars;
char *plotparts;
pict *p;
int more;
int nparts;
int pn;

partv = NULL;
mean = 0.0;
err = 0.0;
if (partvars && partvars[0])
	{
	partv = (int *) dap_malloc(sizeof(int) * dap_maxvar, "");
	mnslist = dap_malloc(strlen(varlist) + strlen(partvars) + 2, "");
	strcpy(mnslist, partvars);
	strcat(mnslist, " ");
	strcat(mnslist, varlist);
	plotparts = dap_malloc(strlen(partvars) + 8, "");
	strcpy(plotparts, partvars);
	strcat(plotparts, " _type_");
	}
else
	{
	mnslist = varlist;
	plotparts = "_type_";
	}
mnsname = dap_malloc(strlen(dataset) + 5, "");
strcpy(mnsname, dataset);
strcat(mnsname, ".mns");
errname = dap_malloc(strlen(dataset) + 5, "");
strcpy(errname, dataset);
strcat(errname, ".err");
srtname = dap_malloc(strlen(errname) + 5, "");
strcpy(srtname, errname);
strcat(srtname, ".srt");
ebar = dap_malloc(strlen(errbar) + 6, "");
strcpy(ebar, "MEAN ");
strcat(ebar, errbar);
overstr = dap_malloc(10, "");
if (noverlay < 1)
	noverlay = 1;
sprintf(overstr, "o%d bb", 2 * noverlay);
srtarg = dap_malloc(strlen(mnslist) + 8, "");
if (partvars)
	{
	strcpy(srtarg, partvars);
	strcat(srtarg, " ");
	}
else
	srtarg[0] = '\0';
strcat(srtarg, "_type_ ");
strcat(srtarg, varlist);
plotvars = dap_malloc(strlen(meanvar) + strlen(varlist) + 2, "");
strcpy(plotvars, varlist);
strcat(plotvars, " ");
strcat(plotvars, meanvar);
for (e = 0; errbar[e] == ' '; e++)
	;
while (errbar[e] && errbar[e] != ' ')
	e++;
ebar[e + 5] = '\0';
while (errbar[e] == ' ')
	e++;
if (errbar[e])
	{
	if (sscanf(errbar + e, "%lf", &scale) != 1)
		{
		fprintf(stderr, "%s: bad scale in call to plotmeans: %s\n",
				dap_dapname, errbar + e);
		exit(1);
		}
	}
else
	scale = 1.0;
means(dataset, meanvar, ebar, mnslist);
inset(mnsname);
outset(errname, "");
dap_list(varlist, meanv, 1);	/* check that there's only one */
dap_list(meanvar, meanv, 1);
if (partvars)
	npartv = dap_list(partvars, partv, dap_maxvar);
else
	npartv = 0;
if ((typen = dap_varnum("_type_")) < 0)
	{
	fprintf(stderr, "%s: missing _type_ variable\n", dap_dapname);
	exit(1);
	}
for (n = 0, nparts = 0, more = 1; more; )
	{
	more = step();
	if (more)
		{
		if (!strcmp(dap_obs[0].do_str[typen], "MEAN"))
			mean = dap_obs[0].do_dbl[meanv[0]];
		else
			err = dap_obs[0].do_dbl[meanv[0]];
		if (++n == 2)
			{
			strcpy(dap_obs[0].do_str[typen], "MEAN");
			dap_obs[0].do_dbl[meanv[0]] = mean;
			output();
			strcpy(dap_obs[0].do_str[typen], "BAR");
			dap_obs[0].do_dbl[meanv[0]] = mean - err * scale;
			output();
			dap_obs[0].do_dbl[meanv[0]] = mean + err * scale;
			output();
			n = 0;
			}
		}
	if (dap_newpart(partv, npartv))
		nparts++;
	}
sort(errname, srtarg, "");
p = plot(srtname, plotvars, plotparts, overstr, NULL, NULL, 2 * nparts);
for (pn = 0; pn < nparts; pn++)
	strcpy(p[2 * pn].pict_type, "IBEA");
if (partvars && partvars[0])
	{
	dap_free(partv);
	dap_free(mnslist);
	dap_free(plotparts);
	}
dap_free(mnsname);
dap_free(errname);
dap_free(srtname);
dap_free(ebar);
dap_free(overstr);
dap_free(srtarg);
dap_free(plotvars);
return p;
}
