
package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link EndpointSpec}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableEndpointSpec.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableEndpointSpec implements EndpointSpec {
  private final @Nullable EndpointSpec.Mode mode;
  private final List<PortConfig> ports;
  private transient final EndpointSpec.Builder toBuilder;

  private ImmutableEndpointSpec(
      @Nullable EndpointSpec.Mode mode,
      List<PortConfig> ports) {
    this.mode = mode;
    this.ports = ports;
    this.toBuilder = Objects.requireNonNull(EndpointSpec.super.toBuilder(), "toBuilder");
  }

  /**
   * @return The value of the {@code mode} attribute
   */
  @JsonProperty("Mode")
  @Override
  public @Nullable EndpointSpec.Mode mode() {
    return mode;
  }

  /**
   * @return The value of the {@code ports} attribute
   */
  @JsonProperty("Ports")
  @Override
  public List<PortConfig> ports() {
    return ports;
  }

  /**
   * @return The computed-at-construction value of the {@code toBuilder} attribute
   */
  @JsonProperty(value = "toBuilder", required = true)
  @JsonIgnore
  @Override
  public EndpointSpec.Builder toBuilder() {
    return toBuilder;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link EndpointSpec#mode() mode} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for mode (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableEndpointSpec withMode(@Nullable EndpointSpec.Mode value) {
    if (this.mode == value) return this;
    return new ImmutableEndpointSpec(value, this.ports);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link EndpointSpec#ports() ports}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableEndpointSpec withPorts(PortConfig... elements) {
    List<PortConfig> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableEndpointSpec(this.mode, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link EndpointSpec#ports() ports}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of ports elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableEndpointSpec withPorts(Iterable<? extends PortConfig> elements) {
    if (this.ports == elements) return this;
    List<PortConfig> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableEndpointSpec(this.mode, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableEndpointSpec} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableEndpointSpec
        && equalsByValue((ImmutableEndpointSpec) another);
  }

  private boolean equalsByValue(ImmutableEndpointSpec another) {
    return Objects.equals(mode, another.mode)
        && ports.equals(another.ports);
  }

  /**
   * Computes a hash code from attributes: {@code mode}, {@code ports}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(mode);
    h += (h << 5) + ports.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code EndpointSpec} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "EndpointSpec{"
        + "mode=" + mode
        + ", ports=" + ports
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link EndpointSpec} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable EndpointSpec instance
   */
  public static ImmutableEndpointSpec copyOf(EndpointSpec instance) {
    if (instance instanceof ImmutableEndpointSpec) {
      return (ImmutableEndpointSpec) instance;
    }
    return ImmutableEndpointSpec.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableEndpointSpec ImmutableEndpointSpec}.
   * <pre>
   * ImmutableEndpointSpec.builder()
   *    .mode(org.mandas.docker.client.messages.swarm.EndpointSpec.Mode | null) // nullable {@link EndpointSpec#mode() mode}
   *    .port|addAllPorts(org.mandas.docker.client.messages.swarm.PortConfig) // {@link EndpointSpec#ports() ports} elements
   *    .build();
   * </pre>
   * @return A new ImmutableEndpointSpec builder
   */
  public static ImmutableEndpointSpec.Builder builder() {
    return new ImmutableEndpointSpec.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableEndpointSpec ImmutableEndpointSpec}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements EndpointSpec.Builder {
    private EndpointSpec.Mode mode;
    private List<PortConfig> ports = new ArrayList<PortConfig>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code EndpointSpec} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(EndpointSpec instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable EndpointSpec.Mode modeValue = instance.mode();
      if (modeValue != null) {
        mode(modeValue);
      }
      addAllPorts(instance.ports());
      return this;
    }

    /**
     * Initializes the value for the {@link EndpointSpec#mode() mode} attribute.
     * @param mode The value for mode (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Mode")
    public final Builder mode(@Nullable EndpointSpec.Mode mode) {
      this.mode = mode;
      return this;
    }

    /**
     * Adds one element to {@link EndpointSpec#ports() ports} list.
     * @param element A ports element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder port(PortConfig element) {
      this.ports.add(Objects.requireNonNull(element, "ports element"));
      return this;
    }

    /**
     * Adds elements to {@link EndpointSpec#ports() ports} list.
     * @param elements An array of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder ports(PortConfig... elements) {
      for (PortConfig element : elements) {
        this.ports.add(Objects.requireNonNull(element, "ports element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link EndpointSpec#ports() ports} list.
     * @param elements An iterable of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Ports")
    public final Builder ports(Iterable<? extends PortConfig> elements) {
      this.ports.clear();
      return addAllPorts(elements);
    }

    /**
     * Adds elements to {@link EndpointSpec#ports() ports} list.
     * @param elements An iterable of ports elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllPorts(Iterable<? extends PortConfig> elements) {
      for (PortConfig element : elements) {
        this.ports.add(Objects.requireNonNull(element, "ports element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableEndpointSpec ImmutableEndpointSpec}.
     * @return An immutable instance of EndpointSpec
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableEndpointSpec build() {
      return new ImmutableEndpointSpec(mode, createUnmodifiableList(true, ports));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
