/**
 * Copyright (c) 2010-2015, Denes Harmath, Istvan Rath and Daniel Varro
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Denes Harmath - initial API and implementation
 */
package org.eclipse.incquery.patternlanguage.validation.whitelist;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import java.util.HashSet;
import org.eclipse.xtend.lib.annotations.Data;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmOperation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * A whitelist that contains pure elements.
 */
@SuppressWarnings("all")
public class PureWhitelist {
  /**
   * Declares that a method is pure or all methods in a class or package are pure.
   */
  @Data
  public static class PureElement {
    public enum Type {
      METHOD,
      
      CLASS,
      
      PACKAGE;
    }
    
    private final String fullyQualifiedName;
    
    private final PureWhitelist.PureElement.Type type;
    
    public boolean covers(final JvmOperation jvmOperation) {
      boolean _xifexpression = false;
      boolean _eIsProxy = jvmOperation.eIsProxy();
      if (_eIsProxy) {
        _xifexpression = false;
      } else {
        boolean _xblockexpression = false;
        {
          String _switchResult = null;
          final PureWhitelist.PureElement.Type type = this.type;
          if (type != null) {
            switch (type) {
              case METHOD:
                _switchResult = jvmOperation.getIdentifier();
                break;
              case CLASS:
                JvmDeclaredType _declaringType = jvmOperation.getDeclaringType();
                String _identifier = null;
                if (_declaringType!=null) {
                  _identifier=_declaringType.getIdentifier();
                }
                _switchResult = _identifier;
                break;
              case PACKAGE:
                JvmDeclaredType _declaringType_1 = jvmOperation.getDeclaringType();
                String _packageName = null;
                if (_declaringType_1!=null) {
                  _packageName=_declaringType_1.getPackageName();
                }
                _switchResult = _packageName;
                break;
              default:
                break;
            }
          }
          final String qualifiedNameToCheck = _switchResult;
          _xblockexpression = Objects.equal(qualifiedNameToCheck, this.fullyQualifiedName);
        }
        _xifexpression = _xblockexpression;
      }
      return _xifexpression;
    }
    
    public PureElement(final String fullyQualifiedName, final PureWhitelist.PureElement.Type type) {
      super();
      this.fullyQualifiedName = fullyQualifiedName;
      this.type = type;
    }
    
    @Override
    @Pure
    public int hashCode() {
      final int prime = 31;
      int result = 1;
      result = prime * result + ((this.fullyQualifiedName== null) ? 0 : this.fullyQualifiedName.hashCode());
      result = prime * result + ((this.type== null) ? 0 : this.type.hashCode());
      return result;
    }
    
    @Override
    @Pure
    public boolean equals(final Object obj) {
      if (this == obj)
        return true;
      if (obj == null)
        return false;
      if (getClass() != obj.getClass())
        return false;
      PureWhitelist.PureElement other = (PureWhitelist.PureElement) obj;
      if (this.fullyQualifiedName == null) {
        if (other.fullyQualifiedName != null)
          return false;
      } else if (!this.fullyQualifiedName.equals(other.fullyQualifiedName))
        return false;
      if (this.type == null) {
        if (other.type != null)
          return false;
      } else if (!this.type.equals(other.type))
        return false;
      return true;
    }
    
    @Override
    @Pure
    public String toString() {
      ToStringBuilder b = new ToStringBuilder(this);
      b.add("fullyQualifiedName", this.fullyQualifiedName);
      b.add("type", this.type);
      return b.toString();
    }
    
    @Pure
    public String getFullyQualifiedName() {
      return this.fullyQualifiedName;
    }
    
    @Pure
    public PureWhitelist.PureElement.Type getType() {
      return this.type;
    }
  }
  
  public final static PureWhitelist INSTANCE = new PureWhitelist();
  
  private PureWhitelist() {
  }
  
  private final HashSet<PureWhitelist.PureElement> pureElements = CollectionLiterals.<PureWhitelist.PureElement>newHashSet();
  
  public boolean contains(final JvmOperation jvmOperation) {
    final Function1<PureWhitelist.PureElement, Boolean> _function = new Function1<PureWhitelist.PureElement, Boolean>() {
      @Override
      public Boolean apply(final PureWhitelist.PureElement it) {
        return Boolean.valueOf(it.covers(jvmOperation));
      }
    };
    return IterableExtensions.<PureWhitelist.PureElement>exists(this.pureElements, _function);
  }
  
  public void add(final PureWhitelist.PureElement pureElement) {
    this.pureElements.add(pureElement);
  }
  
  @VisibleForTesting
  public void clear() {
    this.pureElements.clear();
  }
}
