# Licensed to the Apache Software Foundation (ASF) under one
# or more contributor license agreements.  See the NOTICE file
# distributed with this work for additional information
# regarding copyright ownership.  The ASF licenses this file
# to you under the Apache License, Version 2.0 (the
# "License"); you may not use this file except in compliance
# with the License.  You may obtain a copy of the License at
#
#   http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing,
# software distributed under the License is distributed on an
# "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
# KIND, either express or implied.  See the License for the
# specific language governing permissions and limitations
# under the License.

# This is a list of all the functions that are not auto-generated.
# It contains all the meta data that describes the function.
templated_type_symbol_map = {
  'bool'      : 'b',
  'int8_t'    : 'a',
  'int16_t'   : 's',
  'int32_t'   : 'i',
  'int64_t'   : 'l',
  'float'     : 'f',
  'double'    : 'd',
  'string'    : 'NS_11StringValueE',
  'timestamp' : 'NS_14TimestampValueE'
}

# Generates the BE symbol for the Compute Function class_name::fn_name<templated_type>.
# Does not handle varargs.
# TODO: this is a stopgap. ComputeFunctions are being removed and we can use the
# symbol lookup code in the BE.
def symbol(class_name, fn_name, templated_type = None):
  sym = '_ZN6impala'
  sym += str(len(class_name)) + class_name
  sym += str(len(fn_name)) + fn_name
  if templated_type == None:
    sym += 'EPNS_4ExprEPNS_8TupleRowE'
  else:
    sym += 'I'
    sym += templated_type_symbol_map[templated_type]
    sym += 'EEPvPNS_4ExprEPNS_8TupleRowE'
  return sym

# The format is:
#   [sql aliases], <return_type>, [<args>], <backend symbol>,
# With an optional
#   <prepare symbol>, <close symbol>
#
# 'sql aliases' are the function names that can be used from sql. There must be at least
# one per function.
#
# The symbol can be empty for functions that are not yet implemented or are special-cased
# in Expr::CreateExpr() (i.e., functions that are implemented via a custom Expr class
# rather than a single function).
visible_functions = [
  [['udf_pi'], 'DOUBLE', [], 'impala::UdfBuiltins::Pi'],
  [['udf_abs'], 'DOUBLE', ['DOUBLE'], 'impala::UdfBuiltins::Abs'],
  [['udf_lower'], 'STRING', ['STRING'], 'impala::UdfBuiltins::Lower'],
  [['max_int'], 'INT', [],
   '_ZN6impala11UdfBuiltins6MaxIntEPN10impala_udf15FunctionContextE'],
  [['max_tinyint'], 'TINYINT', [],
   '_ZN6impala11UdfBuiltins10MaxTinyIntEPN10impala_udf15FunctionContextE'],
  [['max_smallint'], 'SMALLINT', [],
   '_ZN6impala11UdfBuiltins11MaxSmallIntEPN10impala_udf15FunctionContextE'],
  [['max_bigint'], 'BIGINT', [],
   '_ZN6impala11UdfBuiltins9MaxBigIntEPN10impala_udf15FunctionContextE'],
  [['min_int'], 'INT', [],
   '_ZN6impala11UdfBuiltins6MinIntEPN10impala_udf15FunctionContextE'],
  [['min_tinyint'], 'TINYINT', [],
   '_ZN6impala11UdfBuiltins10MinTinyIntEPN10impala_udf15FunctionContextE'],
  [['min_smallint'], 'SMALLINT', [],
   '_ZN6impala11UdfBuiltins11MinSmallIntEPN10impala_udf15FunctionContextE'],
  [['min_bigint'], 'BIGINT', [],
   '_ZN6impala11UdfBuiltins9MinBigIntEPN10impala_udf15FunctionContextE'],
  [['is_nan'], 'BOOLEAN', ['DOUBLE'],
   '_ZN6impala11UdfBuiltins5IsNanEPN10impala_udf15FunctionContextERKNS1_9DoubleValE'],
  [['is_inf'], 'BOOLEAN', ['DOUBLE'],
   '_ZN6impala11UdfBuiltins5IsInfEPN10impala_udf15FunctionContextERKNS1_9DoubleValE'],
  [['trunc'], 'TIMESTAMP', ['TIMESTAMP', 'STRING'],
   '_ZN6impala11UdfBuiltins17TruncForTimestampEPN10impala_udf15FunctionContextERKNS1_12TimestampValERKNS1_9StringValE',
   '_ZN6impala11UdfBuiltins24TruncForTimestampPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins22TruncForTimestampCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['trunc'], 'DATE', ['DATE', 'STRING'],
   '_ZN6impala11UdfBuiltins12TruncForDateEPN10impala_udf15FunctionContextERKNS1_7DateValERKNS1_9StringValE',
   '_ZN6impala11UdfBuiltins19TruncForDatePrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins17TruncForDateCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  # Don't add an entry for EXTRACT(STRING, TIMESTAMP). STRINGs may be used to represent
  # TIMESTAMPs meaning EXTRACT(STRING, STRING) is valid. If EXTRACT(STRING, TIMESTAMP)
  # is added, it takes precedence over the existing EXTRACT(TIMESTAMP, STRING)
  # which could break users.
  [['extract'], 'BIGINT', ['TIMESTAMP', 'STRING'],
   '_ZN6impala11UdfBuiltins19ExtractForTimestampEPN10impala_udf15FunctionContextERKNS1_12TimestampValERKNS1_9StringValE',
   '_ZN6impala11UdfBuiltins26ExtractForTimestampPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins24ExtractForTimestampCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['date_part'], 'BIGINT', ['STRING', 'TIMESTAMP'],
   '_ZN6impala11UdfBuiltins20DatePartForTimestampEPN10impala_udf15FunctionContextERKNS1_9StringValERKNS1_12TimestampValE',
   '_ZN6impala11UdfBuiltins27DatePartForTimestampPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins25DatePartForTimestampCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['extract'], 'BIGINT', ['DATE', 'STRING'],
   '_ZN6impala11UdfBuiltins14ExtractForDateEPN10impala_udf15FunctionContextERKNS1_7DateValERKNS1_9StringValE',
   '_ZN6impala11UdfBuiltins21ExtractForDatePrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins19ExtractForDateCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['date_part'], 'BIGINT', ['STRING', 'DATE'],
   '_ZN6impala11UdfBuiltins15DatePartForDateEPN10impala_udf15FunctionContextERKNS1_9StringValERKNS1_7DateValE',
   '_ZN6impala11UdfBuiltins22DatePartForDatePrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins20DatePartForDateCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],

  [['madlib_encode_vector'], 'STRING', ['STRING'],
    '_ZN6impala11UdfBuiltins12EncodeVectorEPN10impala_udf15FunctionContextERKNS1_9StringValE'],
  [['madlib_decode_vector'], 'STRING', ['STRING'],
    '_ZN6impala11UdfBuiltins12DecodeVectorEPN10impala_udf15FunctionContextERKNS1_9StringValE'],
  [['madlib_print_vector'], 'STRING', ['STRING'],
    '_ZN6impala11UdfBuiltins11PrintVectorEPN10impala_udf15FunctionContextERKNS1_9StringValE'],
  [['madlib_vector'], 'STRING', ['DOUBLE', '...'],
    '_ZN6impala11UdfBuiltins8ToVectorEPN10impala_udf15FunctionContextEiPKNS1_9DoubleValE'],
  [['madlib_vector_get'], 'DOUBLE', ['BIGINT', 'STRING'],
    '_ZN6impala11UdfBuiltins9VectorGetEPN10impala_udf15FunctionContextERKNS1_9BigIntValERKNS1_9StringValE'],

  # Timestamp functions
  [['monthname'], 'STRING', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions13LongMonthNameEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['next_day'], 'TIMESTAMP', ['TIMESTAMP', 'STRING'], '_ZN6impala18TimestampFunctions7NextDayEPN10impala_udf15FunctionContextERKNS1_12TimestampValERKNS1_9StringValE'],
  [['last_day'], 'TIMESTAMP', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions7LastDayEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['unix_timestamp'], 'BIGINT', ['STRING'], '_ZN6impala18TimestampFunctions14UnixFromStringEPN10impala_udf15FunctionContextERKNS1_9StringValE'],
  [['year'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions4YearEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['quarter'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions7QuarterEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['month'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions5MonthEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['dayofweek'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions9DayOfWeekEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['day', 'dayofmonth'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions10DayOfMonthEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['dayofyear'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions9DayOfYearEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['week', 'weekofyear'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions10WeekOfYearEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['hour'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions4HourEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['minute'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions6MinuteEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['second'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions6SecondEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['millisecond'], 'INT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions11MillisecondEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['to_date'], 'STRING', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions6ToDateEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['dayname'], 'STRING', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions11LongDayNameEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['date_trunc'], 'TIMESTAMP', ['STRING', 'TIMESTAMP'],
   '_ZN6impala11UdfBuiltins21DateTruncForTimestampEPN10impala_udf15FunctionContextERKNS1_9StringValERKNS1_12TimestampValE',
   '_ZN6impala11UdfBuiltins28DateTruncForTimestampPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins26DateTruncForTimestampCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['years_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9date_time14years_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['years_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9date_time14years_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['years_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9date_time14years_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['years_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9date_time14years_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_add', 'add_months'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb1EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_add', 'add_months'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb1EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb1EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb1EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['weeks_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9gregorian14weeks_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['weeks_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9gregorian14weeks_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['weeks_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9gregorian14weeks_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['weeks_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9gregorian14weeks_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['days_add', 'date_add', 'adddate'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9gregorian13date_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['days_add', 'date_add', 'adddate'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9gregorian13date_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['days_sub', 'date_sub', 'subdate'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9gregorian13date_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['days_sub', 'date_sub', 'subdate'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9gregorian13date_durationELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['hours_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost10posix_time5hoursELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['hours_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost10posix_time5hoursELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['hours_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost10posix_time5hoursELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['hours_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost10posix_time5hoursELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['minutes_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost10posix_time7minutesELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['minutes_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost10posix_time7minutesELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['minutes_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost10posix_time7minutesELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['minutes_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost10posix_time7minutesELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['seconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost10posix_time7secondsELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['seconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost10posix_time7secondsELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['seconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost10posix_time7secondsELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['seconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost10posix_time7secondsELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKS7_RKT0_'],
  [['milliseconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['milliseconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['milliseconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['milliseconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['microseconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['microseconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['microseconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['microseconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['nanoseconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['nanoseconds_add'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['nanoseconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['nanoseconds_sub'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9date_time18subsecond_durationINS4_10posix_time13time_durationELl1000000000EEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['datediff'], 'INT', ['TIMESTAMP', 'TIMESTAMP'], '_ZN6impala18TimestampFunctions8DateDiffEPN10impala_udf15FunctionContextERKNS1_12TimestampValES6_'],
  [['unix_timestamp'], 'BIGINT', [], '_ZN6impala18TimestampFunctions4UnixEPN10impala_udf15FunctionContextE'],
  [['unix_timestamp'], 'BIGINT', ['TIMESTAMP'], '_ZN6impala18TimestampFunctions4UnixEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['unix_timestamp'], 'BIGINT', ['STRING', 'STRING'], '_ZN6impala18TimestampFunctions4UnixEPN10impala_udf15FunctionContextERKNS1_9StringValES6_',
          '_ZN6impala18TimestampFunctions22UnixAndFromUnixPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
          '_ZN6impala18TimestampFunctions20UnixAndFromUnixCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['unix_micros_to_utc_timestamp'], 'TIMESTAMP', ['BIGINT'],
      '_ZN6impala18TimestampFunctions24UnixMicrosToUtcTimestampEPN10impala_udf15FunctionContextERKNS1_9BigIntValE'],
  [['utc_to_unix_micros'], 'BIGINT', ['TIMESTAMP'],
    '_ZN6impala18TimestampFunctions15UtcToUnixMicrosEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['from_unixtime'], 'STRING', ['INT'],
      '_ZN6impala18TimestampFunctions8FromUnixIN10impala_udf6IntValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['from_unixtime'], 'STRING', ['INT', 'STRING'],
      '_ZN6impala18TimestampFunctions8FromUnixIN10impala_udf6IntValEEENS2_9StringValEPNS2_15FunctionContextERKT_RKS4_',
      '_ZN6impala18TimestampFunctions22UnixAndFromUnixPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
     '_ZN6impala18TimestampFunctions20UnixAndFromUnixCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['from_unixtime'], 'STRING', ['BIGINT'],
      '_ZN6impala18TimestampFunctions8FromUnixIN10impala_udf9BigIntValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['from_unixtime'], 'STRING', ['BIGINT', 'STRING'],
      '_ZN6impala18TimestampFunctions8FromUnixIN10impala_udf9BigIntValEEENS2_9StringValEPNS2_15FunctionContextERKT_RKS4_',
      '_ZN6impala18TimestampFunctions22UnixAndFromUnixPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
      '_ZN6impala18TimestampFunctions20UnixAndFromUnixCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['now', 'current_timestamp'], 'TIMESTAMP', [], '_ZN6impala18TimestampFunctions3NowEPN10impala_udf15FunctionContextE'],
  [['utc_timestamp'], 'TIMESTAMP', [], '_ZN6impala18TimestampFunctions12UtcTimestampEPN10impala_udf15FunctionContextE'],
  [['from_utc_timestamp'], 'TIMESTAMP', ['TIMESTAMP', 'STRING'],
   "impala::TimestampFunctions::FromUtc"],
  [['to_utc_timestamp'], 'TIMESTAMP', ['TIMESTAMP', 'STRING'],
   "impala::TimestampFunctions::ToUtc"],
  [['timeofday'], 'STRING', [],"impala::TimestampFunctions::TimeOfDay"],
  [['timestamp_cmp'], 'INT', ['TIMESTAMP', 'TIMESTAMP'],
   "impala::TimestampFunctions::TimestampCmp"],
  [['int_months_between'], 'INT', ['TIMESTAMP', 'TIMESTAMP'],
   "impala::TimestampFunctions::IntMonthsBetween"],
  [['months_between'], 'DOUBLE', ['TIMESTAMP', 'TIMESTAMP'],
   "impala::TimestampFunctions::MonthsBetween"],
  [['to_timestamp'], 'TIMESTAMP', ['BIGINT'],
   '_ZN6impala18TimestampFunctions11ToTimestampEPN10impala_udf15FunctionContextERKNS1_9BigIntValE'],
  [['to_timestamp'], 'TIMESTAMP', ['STRING', 'STRING'],
   '_ZN6impala18TimestampFunctions11ToTimestampEPN10impala_udf15FunctionContextERKNS1_9StringValES6_',
   '_ZN6impala18TimestampFunctions22UnixAndFromUnixPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala18TimestampFunctions20UnixAndFromUnixCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['from_timestamp'], 'STRING', ['TIMESTAMP', 'STRING'],
   'impala::TimestampFunctions::FromTimestamp',
   '_ZN6impala18TimestampFunctions22UnixAndFromUnixPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala18TimestampFunctions20UnixAndFromUnixCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],

  # Date functions
  [['monthname'], 'STRING', ['DATE'], '_ZN6impala13DateFunctions13LongMonthNameEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['next_day'], 'DATE', ['DATE', 'STRING'], '_ZN6impala13DateFunctions7NextDayEPN10impala_udf15FunctionContextERKNS1_7DateValERKNS1_9StringValE'],
  [['last_day'], 'DATE', ['DATE'], '_ZN6impala13DateFunctions7LastDayEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['year'], 'INT', ['DATE'], '_ZN6impala13DateFunctions4YearEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['quarter'], 'INT', ['DATE'], '_ZN6impala13DateFunctions7QuarterEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['month'], 'INT', ['DATE'], '_ZN6impala13DateFunctions5MonthEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['dayofweek'], 'INT', ['DATE'], '_ZN6impala13DateFunctions9DayOfWeekEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['day', 'dayofmonth'], 'INT', ['DATE'], '_ZN6impala13DateFunctions10DayOfMonthEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['dayofyear'], 'INT', ['DATE'], '_ZN6impala13DateFunctions9DayOfYearEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['week', 'weekofyear'], 'INT', ['DATE'], '_ZN6impala13DateFunctions10WeekOfYearEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['dayname'], 'STRING', ['DATE'], '_ZN6impala13DateFunctions11LongDayNameEPN10impala_udf15FunctionContextERKNS1_7DateValE'],
  [['date_trunc'], 'DATE', ['STRING', 'DATE'],
   '_ZN6impala11UdfBuiltins16DateTruncForDateEPN10impala_udf15FunctionContextERKNS1_9StringValERKNS1_7DateValE',
   '_ZN6impala11UdfBuiltins23DateTruncForDatePrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala11UdfBuiltins21DateTruncForDateCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['months_add', 'add_months'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb1EN10impala_udf6IntValELb1EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['months_add', 'add_months'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb1EN10impala_udf9BigIntValELb1EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['months_sub'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb0EN10impala_udf6IntValELb1EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['months_sub'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb0EN10impala_udf9BigIntValELb1EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['years_add'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions11AddSubYearsILb1EN10impala_udf6IntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['years_add'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions11AddSubYearsILb1EN10impala_udf9BigIntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['years_sub'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions11AddSubYearsILb0EN10impala_udf6IntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['years_sub'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions11AddSubYearsILb0EN10impala_udf9BigIntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['weeks_add'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions11AddSubWeeksILb1EN10impala_udf6IntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['weeks_add'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions11AddSubWeeksILb1EN10impala_udf9BigIntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['weeks_sub'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions11AddSubWeeksILb0EN10impala_udf6IntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['weeks_sub'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions11AddSubWeeksILb0EN10impala_udf9BigIntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['days_add', 'date_add', 'adddate'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions10AddSubDaysILb1EN10impala_udf6IntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['days_add', 'date_add', 'adddate'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions10AddSubDaysILb1EN10impala_udf9BigIntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['days_sub', 'date_sub', 'subdate'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions10AddSubDaysILb0EN10impala_udf6IntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['days_sub', 'date_sub', 'subdate'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions10AddSubDaysILb0EN10impala_udf9BigIntValEEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['datediff'], 'INT', ['DATE', 'DATE'], '_ZN6impala13DateFunctions8DateDiffEPN10impala_udf15FunctionContextERKNS1_7DateValES6_'],
  [['current_date'], 'DATE', [], '_ZN6impala13DateFunctions11CurrentDateEPN10impala_udf15FunctionContextE'],
  [['date_cmp'], 'INT', ['DATE', 'DATE'], "_ZN6impala13DateFunctions7DateCmpEPN10impala_udf15FunctionContextERKNS1_7DateValES6_"],
  [['int_months_between'], 'INT', ['DATE', 'DATE'], "_ZN6impala13DateFunctions16IntMonthsBetweenEPN10impala_udf15FunctionContextERKNS1_7DateValES6_"],
  [['months_between'], 'DOUBLE', ['DATE', 'DATE'], "_ZN6impala13DateFunctions13MonthsBetweenEPN10impala_udf15FunctionContextERKNS1_7DateValES6_"],

  # Math builtin functions
  [['pi'], 'DOUBLE', [], 'impala::MathFunctions::Pi'],
  [['e'], 'DOUBLE', [], 'impala::MathFunctions::E'],
  [['abs'], 'BIGINT', ['BIGINT'], 'impala::MathFunctions::Abs'],
  [['abs'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Abs'],
  [['abs'], 'FLOAT', ['FLOAT'], 'impala::MathFunctions::Abs'],
  [['abs'], 'BIGINT', ['INT'], 'impala::MathFunctions::Abs'],
  [['abs'], 'INT', ['SMALLINT'], 'impala::MathFunctions::Abs'],
  [['abs'], 'SMALLINT', ['TINYINT'], 'impala::MathFunctions::Abs'],
  [['sign'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Sign'],
  [['sin'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Sin'],
  [['asin'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Asin'],
  [['cos'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Cos'],
  [['acos'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Acos'],
  [['tan'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Tan'],
  [['atan'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Atan'],
  [['atan2'], 'DOUBLE', ['DOUBLE','DOUBLE'], 'impala::MathFunctions::Atan2'],
  [['cosh'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Cosh'],
  [['tanh'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Tanh'],
  [['sinh'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Sinh'],
  [['cot'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Cot'],
  [['radians'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Radians'],
  [['degrees'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Degrees'],
  [['ceil', 'ceiling', 'dceil'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Ceil'],
  [['floor', 'dfloor'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Floor'],
  [['truncate','dtrunc','trunc'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Truncate'],
  [['round','dround'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Round'],
  [['round','dround'], 'DOUBLE', ['DOUBLE', 'BIGINT'], 'impala::MathFunctions::RoundUpTo'],
  [['exp', 'dexp'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Exp'],
  [['ln','dlog1'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Ln'],
  [['log10','dlog10'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Log10'],
  [['log2'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Log2'],
  [['log'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'impala::MathFunctions::Log'],
  [['pow', 'power','dpow','fpow'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'impala::MathFunctions::Pow'],
  [['sqrt','dsqrt'], 'DOUBLE', ['DOUBLE'], 'impala::MathFunctions::Sqrt'],
  [['rand','random'], 'DOUBLE', [], 'impala::MathFunctions::Rand',
   '_ZN6impala13MathFunctions11RandPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala13MathFunctions9RandCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['factorial'], 'BIGINT', ['TINYINT'], 'impala::Operators::Factorial_TinyIntVal'],
  [['factorial'], 'BIGINT', ['SMALLINT'], 'impala::Operators::Factorial_SmallIntVal'],
  [['factorial'], 'BIGINT', ['INT'], 'impala::Operators::Factorial_IntVal'],
  [['factorial'], 'BIGINT', ['BIGINT'], 'impala::Operators::Factorial_BigIntVal'],
  [['rand','random'], 'DOUBLE', ['BIGINT'], 'impala::MathFunctions::RandSeed',
   '_ZN6impala13MathFunctions11RandPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala13MathFunctions9RandCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['bin'], 'STRING', ['BIGINT'], 'impala::MathFunctions::Bin'],
  [['hex'], 'STRING', ['BIGINT'], 'impala::MathFunctions::HexInt'],
  [['hex'], 'STRING', ['STRING'], 'impala::MathFunctions::HexString'],
  [['unhex'], 'STRING', ['STRING'], 'impala::MathFunctions::Unhex'],
  [['conv'], 'STRING', ['BIGINT', 'TINYINT', 'TINYINT'],
   'impala::MathFunctions::ConvInt'],
  [['conv'], 'STRING', ['STRING', 'TINYINT', 'TINYINT'],
      'impala::MathFunctions::ConvString'],
  [['pmod'], 'BIGINT', ['BIGINT', 'BIGINT'], 'impala::MathFunctions::PmodBigInt'],
  [['pmod'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'impala::MathFunctions::PmodDouble'],
  [['fmod'], 'FLOAT', ['FLOAT', 'FLOAT'], 'impala::MathFunctions::FmodFloat'],
  [['fmod'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'impala::MathFunctions::FmodDouble'],
  [['mod'], 'TINYINT', ['TINYINT', 'TINYINT'], 'impala::Operators::Mod_TinyIntVal_TinyIntVal'],
  [['mod'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'impala::Operators::Mod_SmallIntVal_SmallIntVal'],
  [['mod'], 'INT', ['INT', 'INT'], 'impala::Operators::Mod_IntVal_IntVal'],
  [['mod'], 'BIGINT', ['BIGINT', 'BIGINT'], 'impala::Operators::Mod_BigIntVal_BigIntVal'],
  [['mod'], 'FLOAT', ['FLOAT', 'FLOAT'], 'impala::MathFunctions::FmodFloat'],
  [['mod'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], 'impala::MathFunctions::FmodDouble'],
  [['mod'], 'DECIMAL', ['DECIMAL', 'DECIMAL'], 'impala::DecimalOperators::Mod_DecimalVal_DecimalVal'],
  [['positive'], 'TINYINT', ['TINYINT'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf10TinyIntValEEET_PNS2_15FunctionContextERKS4_'],
  [['positive'], 'SMALLINT', ['SMALLINT'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf11SmallIntValEEET_PNS2_15FunctionContextERKS4_'],
  [['positive'], 'INT', ['INT'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf6IntValEEET_PNS2_15FunctionContextERKS4_'],
  [['positive'], 'BIGINT', ['BIGINT'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf9BigIntValEEET_PNS2_15FunctionContextERKS4_'],
  [['positive'], 'FLOAT', ['FLOAT'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf8FloatValEEET_PNS2_15FunctionContextERKS4_'],
  [['positive'], 'DOUBLE', ['DOUBLE'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf9DoubleValEEET_PNS2_15FunctionContextERKS4_'],
  [['positive'], 'DECIMAL', ['DECIMAL'],
   '_ZN6impala13MathFunctions8PositiveIN10impala_udf10DecimalValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'TINYINT', ['TINYINT'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf10TinyIntValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'SMALLINT', ['SMALLINT'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf11SmallIntValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'INT', ['INT'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf6IntValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'BIGINT', ['BIGINT'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf9BigIntValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'FLOAT', ['FLOAT'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf8FloatValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'DOUBLE', ['DOUBLE'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf9DoubleValEEET_PNS2_15FunctionContextERKS4_'],
  [['negative'], 'DECIMAL', ['DECIMAL'],
   '_ZN6impala13MathFunctions8NegativeIN10impala_udf10DecimalValEEET_PNS2_15FunctionContextERKS4_'],
  [['quotient'], 'BIGINT', ['BIGINT', 'BIGINT'],
      'impala::MathFunctions::QuotientBigInt'],
  [['quotient'], 'BIGINT', ['DOUBLE', 'DOUBLE'],
      'impala::MathFunctions::QuotientDouble'],
  [['least'], 'TINYINT', ['TINYINT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf10TinyIntValELb1EEET_PNS2_15FunctionContextEiPKS4_'],
  [['least'], 'SMALLINT', ['SMALLINT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf11SmallIntValELb1EEET_PNS2_15FunctionContextEiPKS4_'],
  [['least'], 'INT', ['INT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf6IntValELb1EEET_PNS2_15FunctionContextEiPKS4_'],
  [['least'], 'BIGINT', ['BIGINT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf9BigIntValELb1EEET_PNS2_15FunctionContextEiPKS4_'],
  [['least'], 'FLOAT', ['FLOAT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf8FloatValELb1EEET_PNS2_15FunctionContextEiPKS4_'],
  [['least'], 'DOUBLE', ['DOUBLE', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf9DoubleValELb1EEET_PNS2_15FunctionContextEiPKS4_'],
  [['least'], 'TIMESTAMP', ['TIMESTAMP', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb1EEEN10impala_udf12TimestampValEPNS2_15FunctionContextEiPKS3_'],
  [['least'], 'STRING', ['STRING', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb1EEEN10impala_udf9StringValEPNS2_15FunctionContextEiPKS3_'],
  [['least'], 'DECIMAL', ['DECIMAL', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb1EEEN10impala_udf10DecimalValEPNS2_15FunctionContextEiPKS3_'],
  [['least'], 'DATE', ['DATE', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb1EEEN10impala_udf7DateValEPNS2_15FunctionContextEiPKS3_'],
  [['greatest'], 'TINYINT', ['TINYINT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf10TinyIntValELb0EEET_PNS2_15FunctionContextEiPKS4_'],
  [['greatest'], 'SMALLINT', ['SMALLINT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf11SmallIntValELb0EEET_PNS2_15FunctionContextEiPKS4_'],
  [['greatest'], 'INT', ['INT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf6IntValELb0EEET_PNS2_15FunctionContextEiPKS4_'],
  [['greatest'], 'BIGINT', ['BIGINT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf9BigIntValELb0EEET_PNS2_15FunctionContextEiPKS4_'],
  [['greatest'], 'FLOAT', ['FLOAT', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf8FloatValELb0EEET_PNS2_15FunctionContextEiPKS4_'],
  [['greatest'], 'DOUBLE', ['DOUBLE', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestIN10impala_udf9DoubleValELb0EEET_PNS2_15FunctionContextEiPKS4_'],
  [['greatest'], 'TIMESTAMP', ['TIMESTAMP', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb0EEEN10impala_udf12TimestampValEPNS2_15FunctionContextEiPKS3_'],
  [['greatest'], 'STRING', ['STRING', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb0EEEN10impala_udf9StringValEPNS2_15FunctionContextEiPKS3_'],
  [['greatest'], 'DECIMAL', ['DECIMAL', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb0EEEN10impala_udf10DecimalValEPNS2_15FunctionContextEiPKS3_'],
  [['greatest'], 'DATE', ['DATE', '...'],
   '_ZN6impala13MathFunctions13LeastGreatestILb0EEEN10impala_udf7DateValEPNS2_15FunctionContextEiPKS3_'],
  [['width_bucket'], 'BIGINT', ['DECIMAL', 'DECIMAL', 'DECIMAL', 'INT'],
    '_ZN6impala13MathFunctions11WidthBucketEPN10impala_udf15FunctionContextERKNS1_10DecimalValES6_S6_RKNS1_6IntValE'],

  # Decimal Functions
  # TODO: oracle has decimal support for transcendental functions (e.g. sin()) to very
  # high precisions. Do we need them? It's unclear if other databases do the same.
  [['precision'], 'INT', ['DECIMAL'], 'impala::DecimalFunctions::Precision'],
  [['scale'], 'INT', ['DECIMAL'], 'impala::DecimalFunctions::Scale'],
  [['abs'], 'DECIMAL', ['DECIMAL'], 'impala::DecimalFunctions::Abs'],
  [['ceil', 'ceiling'], 'DECIMAL', ['DECIMAL'], 'impala::DecimalFunctions::Ceil'],
  [['floor','dfloor'], 'DECIMAL', ['DECIMAL'], 'impala::DecimalFunctions::Floor'],
  [['round','dround'], 'DECIMAL', ['DECIMAL'], 'impala::DecimalFunctions::Round'],
  # The reason why we declare the following function several times (instead of simply
  # setting the second argument to BIGINT and relying on implicit casting is because
  # we don't know if the first Decimal argument needs to be cast to double, or the second
  # argument needs to be cast to a BIGINT.
  [['round','dround'], 'DECIMAL', ['DECIMAL', 'TINYINT'], 'impala::DecimalFunctions::RoundTo'],
  [['round','dround'], 'DECIMAL', ['DECIMAL', 'SMALLINT'], 'impala::DecimalFunctions::RoundTo'],
  [['round','dround'], 'DECIMAL', ['DECIMAL', 'INT'], 'impala::DecimalFunctions::RoundTo'],
  [['round','dround'], 'DECIMAL', ['DECIMAL', 'BIGINT'], 'impala::DecimalFunctions::RoundTo'],
  [['truncate','dtrunc','trunc'], 'DECIMAL', ['DECIMAL'], 'impala::DecimalFunctions::Truncate'],
  [['truncate','dtrunc','trunc'], 'DECIMAL', ['DECIMAL', 'TINYINT'],
      'impala::DecimalFunctions::TruncateTo'],
  [['truncate','dtrunc','trunc'], 'DECIMAL', ['DECIMAL', 'SMALLINT'],
      'impala::DecimalFunctions::TruncateTo'],
  [['truncate','dtrunc','trunc'], 'DECIMAL', ['DECIMAL', 'INT'],
      'impala::DecimalFunctions::TruncateTo'],
  [['truncate','dtrunc','trunc'], 'DECIMAL', ['DECIMAL', 'BIGINT'],
      'impala::DecimalFunctions::TruncateTo'],

  # String builtin functions
  [['substr', 'substring'], 'STRING', ['STRING', 'BIGINT'],
   'impala::StringFunctions::Substring'],
  [['substr', 'substring'], 'STRING', ['STRING', 'BIGINT', 'BIGINT'],
   'impala::StringFunctions::Substring'],
  [['split_part'], 'STRING', ['STRING', 'STRING', 'BIGINT'],
   'impala::StringFunctions::SplitPart'],
  [['base64encode'], 'STRING', ['STRING'], 'impala::StringFunctions::Base64Encode'],
  [['base64decode'], 'STRING', ['STRING'], 'impala::StringFunctions::Base64Decode'],
  [['left', 'strleft'], 'STRING', ['STRING', 'BIGINT'], 'impala::StringFunctions::Left'],
  [['right', 'strright'], 'STRING', ['STRING', 'BIGINT'], 'impala::StringFunctions::Right'],
  [['space'], 'STRING', ['BIGINT'], 'impala::StringFunctions::Space'],
  [['repeat'], 'STRING', ['STRING', 'BIGINT'], 'impala::StringFunctions::Repeat'],
  [['lpad'], 'STRING', ['STRING', 'BIGINT', 'STRING'], 'impala::StringFunctions::Lpad'],
  [['rpad'], 'STRING', ['STRING', 'BIGINT', 'STRING'], 'impala::StringFunctions::Rpad'],
  [['length'], 'INT', ['STRING'], 'impala::StringFunctions::Length'],
  [['length'], 'INT', ['CHAR'], 'impala::StringFunctions::CharLength'],
  [['char_length'], 'INT', ['STRING'], 'impala::StringFunctions::Length'],
  [['character_length'], 'INT', ['STRING'], 'impala::StringFunctions::Length'],
  [['lower', 'lcase'], 'STRING', ['STRING'], 'impala::StringFunctions::Lower'],
  [['upper', 'ucase'], 'STRING', ['STRING'], 'impala::StringFunctions::Upper'],
  [['initcap'], 'STRING', ['STRING'], 'impala::StringFunctions::InitCap'],
  [['replace'], 'STRING', ['STRING', 'STRING', 'STRING'], 'impala::StringFunctions::Replace',
   '_ZN6impala15StringFunctions14ReplacePrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions12ReplaceCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['reverse'], 'STRING', ['STRING'], 'impala::StringFunctions::Reverse'],
  [['translate'], 'STRING', ['STRING', 'STRING', 'STRING'],
   'impala::StringFunctions::Translate'],
  [['trim'], 'STRING', ['STRING'], 'impala::StringFunctions::Trim',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['ltrim'], 'STRING', ['STRING'], 'impala::StringFunctions::Ltrim',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['rtrim'], 'STRING', ['STRING'], 'impala::StringFunctions::Rtrim',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['ltrim'], 'STRING', ['STRING', 'STRING'], 'impala::StringFunctions::LTrimString',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['rtrim'], 'STRING', ['STRING', 'STRING'], 'impala::StringFunctions::RTrimString',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['ascii'], 'INT', ['STRING'], 'impala::StringFunctions::Ascii'],
  [['instr'], 'INT', ['STRING', 'STRING'], 'impala::StringFunctions::Instr'],
  [['instr'], 'INT', ['STRING', 'STRING', 'BIGINT'], 'impala::StringFunctions::Instr'],
  [['instr'], 'INT', ['STRING', 'STRING', 'BIGINT', 'BIGINT'],
   'impala::StringFunctions::Instr'],
  [['locate'], 'INT', ['STRING', 'STRING'], 'impala::StringFunctions::Locate'],
  [['locate'], 'INT', ['STRING', 'STRING', 'BIGINT'],
   'impala::StringFunctions::LocatePos'],
  [['regexp_escape'], 'STRING', ['STRING'], 'impala::StringFunctions::RegexpEscape'],
  [['regexp_extract'], 'STRING', ['STRING', 'STRING', 'BIGINT'],
   'impala::StringFunctions::RegexpExtract',
   '_ZN6impala15StringFunctions13RegexpPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions11RegexpCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['regexp_replace'], 'STRING', ['STRING', 'STRING', 'STRING'],
   'impala::StringFunctions::RegexpReplace',
   '_ZN6impala15StringFunctions13RegexpPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions11RegexpCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['regexp_like'], 'BOOLEAN', ['STRING', 'STRING'],
   'impala::LikePredicate::Regex',
   '_ZN6impala13LikePredicate12RegexPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala13LikePredicate10RegexCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['regexp_like'], 'BOOLEAN', ['STRING', 'STRING', 'STRING'],
   'impala::LikePredicate::RegexpLike',
   '_ZN6impala13LikePredicate17RegexpLikePrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala13LikePredicate10RegexCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['regexp_match_count'], 'INT', ['STRING', 'STRING'],
   'impala::StringFunctions::RegexpMatchCount2Args',
   '_ZN6impala15StringFunctions23RegexpMatchCountPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions11RegexpCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['regexp_match_count'], 'INT', ['STRING', 'STRING', 'INT', 'STRING'],
   'impala::StringFunctions::RegexpMatchCount4Args',
   '_ZN6impala15StringFunctions23RegexpMatchCountPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions11RegexpCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['concat'], 'STRING', ['STRING', '...'], 'impala::StringFunctions::Concat'],
  [['concat_ws'], 'STRING', ['STRING', 'STRING', '...'],
   'impala::StringFunctions::ConcatWs'],
  [['find_in_set'], 'INT', ['STRING', 'STRING'], 'impala::StringFunctions::FindInSet'],
  [['parse_url'], 'STRING', ['STRING', 'STRING'], 'impala::StringFunctions::ParseUrl',
   '_ZN6impala15StringFunctions15ParseUrlPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions13ParseUrlCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['parse_url'], 'STRING', ['STRING', 'STRING', 'STRING'], 'impala::StringFunctions::ParseUrlKey',
   '_ZN6impala15StringFunctions15ParseUrlPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions13ParseUrlCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
# Netezza compatibility char functions
  [['chr'], 'STRING', ['INT'], 'impala::StringFunctions::Chr'],
  [['btrim'], 'STRING', ['STRING'], 'impala::StringFunctions::Trim',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['btrim'], 'STRING', ['STRING', 'STRING'], 'impala::StringFunctions::BTrimString',
   '_ZN6impala15StringFunctions11TrimPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala15StringFunctions9TrimCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['get_json_object'], 'STRING', ['STRING', 'STRING'],
   'impala::StringFunctions::GetJsonObject'],
  [['levenshtein', 'le_dst'], 'INT', ['STRING', 'STRING'],
   '_ZN6impala15StringFunctions11LevenshteinEPN10impala_udf15FunctionContextERKNS1_9StringValES6_'],
  [['damerau_levenshtein', 'dle_dst'], 'INT', ['STRING', 'STRING'],
   '_ZN6impala15StringFunctions18DamerauLevenshteinEPN10impala_udf15FunctionContextERKNS1_9StringValES6_'],
  [['jaro_distance', 'jaro_dst'], 'DOUBLE', ['STRING', 'STRING'],
   '_ZN6impala15StringFunctions12JaroDistanceEPN10impala_udf15FunctionContextERKNS1_9StringValES6_'],
  [['jaro_similarity', 'jaro_sim'], 'DOUBLE', ['STRING', 'STRING'],
   '_ZN6impala15StringFunctions14JaroSimilarityEPN10impala_udf15FunctionContextERKNS1_9StringValES6_'],
  [['jaro_winkler_distance', 'jw_dst'], 'DOUBLE', ['STRING', 'STRING'],
   '_ZN6impala15StringFunctions19JaroWinklerDistanceEPN10impala_udf15FunctionContextERKNS1_9StringValES6_'],
  [['jaro_winkler_distance', 'jw_dst'], 'DOUBLE', ['STRING', 'STRING', 'DOUBLE'],
   '_ZN6impala15StringFunctions19JaroWinklerDistanceEPN10impala_udf15FunctionContextERKNS1_9StringValES6_RKNS1_9DoubleValE'],
  [['jaro_winkler_distance', 'jw_dst'], 'DOUBLE', ['STRING', 'STRING', 'DOUBLE', 'DOUBLE'],
   '_ZN6impala15StringFunctions19JaroWinklerDistanceEPN10impala_udf15FunctionContextERKNS1_9StringValES6_RKNS1_9DoubleValES9_'],
  [['jaro_winkler_similarity', 'jw_sim'], 'DOUBLE', ['STRING', 'STRING'],
   '_ZN6impala15StringFunctions21JaroWinklerSimilarityEPN10impala_udf15FunctionContextERKNS1_9StringValES6_'],
  [['jaro_winkler_similarity', 'jw_sim'], 'DOUBLE', ['STRING', 'STRING', 'DOUBLE'],
   '_ZN6impala15StringFunctions21JaroWinklerSimilarityEPN10impala_udf15FunctionContextERKNS1_9StringValES6_RKNS1_9DoubleValE'],
  [['jaro_winkler_similarity', 'jw_sim'], 'DOUBLE', ['STRING', 'STRING', 'DOUBLE', 'DOUBLE'],
   '_ZN6impala15StringFunctions21JaroWinklerSimilarityEPN10impala_udf15FunctionContextERKNS1_9StringValES6_RKNS1_9DoubleValES9_'],

  # Conditional Functions
  # Some of these have empty symbols because the BE special-cases them based on the
  # function name
  [['if'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN', 'BOOLEAN'], ''],
  [['if'], 'TINYINT', ['BOOLEAN', 'TINYINT', 'TINYINT'], ''],
  [['if'], 'SMALLINT', ['BOOLEAN', 'SMALLINT', 'SMALLINT'], ''],
  [['if'], 'INT', ['BOOLEAN', 'INT', 'INT'], ''],
  [['if'], 'BIGINT', ['BOOLEAN', 'BIGINT', 'BIGINT'], ''],
  [['if'], 'FLOAT', ['BOOLEAN', 'FLOAT', 'FLOAT'], ''],
  [['if'], 'DOUBLE', ['BOOLEAN', 'DOUBLE', 'DOUBLE'], ''],
  [['if'], 'STRING', ['BOOLEAN', 'STRING', 'STRING'], ''],
  [['if'], 'TIMESTAMP', ['BOOLEAN', 'TIMESTAMP', 'TIMESTAMP'], ''],
  [['if'], 'DECIMAL', ['BOOLEAN', 'DECIMAL', 'DECIMAL'], ''],
  [['if'], 'DATE', ['BOOLEAN', 'DATE', 'DATE'], ''],

  [['zeroifnull'], 'TINYINT', ['TINYINT'], 'impala::ConditionalFunctions::ZeroIfNull'],
  [['zeroifnull'], 'SMALLINT', ['SMALLINT'], 'impala::ConditionalFunctions::ZeroIfNull'],
  [['zeroifnull'], 'INT', ['INT'], 'impala::ConditionalFunctions::ZeroIfNull'],
  [['zeroifnull'], 'BIGINT', ['BIGINT'], 'impala::ConditionalFunctions::ZeroIfNull'],
  [['zeroifnull'], 'FLOAT', ['FLOAT'], 'impala::ConditionalFunctions::ZeroIfNull'],
  [['zeroifnull'], 'DOUBLE', ['DOUBLE'], 'impala::ConditionalFunctions::ZeroIfNull'],
  [['zeroifnull'], 'DECIMAL', ['DECIMAL'], 'impala::ConditionalFunctions::ZeroIfNull'],

  [['nullifzero'], 'TINYINT', ['TINYINT'], 'impala::ConditionalFunctions::NullIfZero'],
  [['nullifzero'], 'SMALLINT', ['SMALLINT'], 'impala::ConditionalFunctions::NullIfZero'],
  [['nullifzero'], 'INT', ['INT'], 'impala::ConditionalFunctions::NullIfZero'],
  [['nullifzero'], 'BIGINT', ['BIGINT'], 'impala::ConditionalFunctions::NullIfZero'],
  [['nullifzero'], 'FLOAT', ['FLOAT'], 'impala::ConditionalFunctions::NullIfZero'],
  [['nullifzero'], 'DOUBLE', ['DOUBLE'], 'impala::ConditionalFunctions::NullIfZero'],
  [['nullifzero'], 'DECIMAL', ['DECIMAL'], 'impala::ConditionalFunctions::NullIfZero'],

  [['isnull', 'ifnull', 'nvl'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN'], ''],
  [['isnull', 'ifnull', 'nvl'], 'TINYINT', ['TINYINT', 'TINYINT'], ''],
  [['isnull', 'ifnull', 'nvl'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], ''],
  [['isnull', 'ifnull', 'nvl'], 'INT', ['INT', 'INT'], ''],
  [['isnull', 'ifnull', 'nvl'], 'BIGINT', ['BIGINT', 'BIGINT'], ''],
  [['isnull', 'ifnull', 'nvl'], 'FLOAT', ['FLOAT', 'FLOAT'], ''],
  [['isnull', 'ifnull', 'nvl'], 'DOUBLE', ['DOUBLE', 'DOUBLE'], ''],
  [['isnull', 'ifnull', 'nvl'], 'STRING', ['STRING', 'STRING'], ''],
  [['isnull', 'ifnull', 'nvl'], 'TIMESTAMP', ['TIMESTAMP', 'TIMESTAMP'], ''],
  [['isnull', 'ifnull', 'nvl'], 'DECIMAL', ['DECIMAL', 'DECIMAL'], ''],
  [['isnull', 'ifnull', 'nvl'], 'DATE', ['DATE', 'DATE'], ''],

  [['coalesce'], 'BOOLEAN', ['BOOLEAN', '...'], ''],
  [['coalesce'], 'TINYINT', ['TINYINT', '...'], ''],
  [['coalesce'], 'SMALLINT', ['SMALLINT', '...'], ''],
  [['coalesce'], 'INT', ['INT', '...'], ''],
  [['coalesce'], 'BIGINT', ['BIGINT', '...'], ''],
  [['coalesce'], 'FLOAT', ['FLOAT', '...'], ''],
  [['coalesce'], 'DOUBLE', ['DOUBLE', '...'], ''],
  [['coalesce'], 'STRING', ['STRING', '...'], ''],
  [['coalesce'], 'TIMESTAMP', ['TIMESTAMP', '...'], ''],
  [['coalesce'], 'DECIMAL', ['DECIMAL', '...'], ''],
  [['coalesce'], 'DATE', ['DATE', '...'], ''],

  [['istrue'], 'BOOLEAN', ['BOOLEAN'], 'impala::ConditionalFunctions::IsTrue'],
  [['isnottrue'], 'BOOLEAN', ['BOOLEAN'], 'impala::ConditionalFunctions::IsNotTrue'],
  [['isfalse'], 'BOOLEAN', ['BOOLEAN'], 'impala::ConditionalFunctions::IsFalse'],
  [['isnotfalse'], 'BOOLEAN', ['BOOLEAN'], 'impala::ConditionalFunctions::IsNotFalse'],

  # Utility functions
  [['uuid'], 'STRING', [],
   '_ZN6impala16UtilityFunctions4UuidEPN10impala_udf15FunctionContextE',
   '_ZN6impala16UtilityFunctions11UuidPrepareEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE',
   '_ZN6impala16UtilityFunctions9UuidCloseEPN10impala_udf15FunctionContextENS2_18FunctionStateScopeE'],
  [['current_database', 'current_catalog'], 'STRING', [], 'impala::UtilityFunctions::CurrentDatabase'],
  [['current_session', 'current_sid'], 'STRING', [], 'impala::UtilityFunctions::CurrentSession'],
  [['user', 'session_user'], 'STRING', [], 'impala::UtilityFunctions::User'],
  [['effective_user', 'current_user', 'logged_in_user'], 'STRING', [],
      'impala::UtilityFunctions::EffectiveUser'],
  [['sleep'], 'BOOLEAN', ['INT'], 'impala::UtilityFunctions::Sleep'],
  [['pid'], 'INT', [], 'impala::UtilityFunctions::Pid'],
  [['version'], 'STRING', [], 'impala::UtilityFunctions::Version'],
  [['coordinator'], 'STRING', [], 'impala::UtilityFunctions::Coordinator'],
  [['typeOf'], 'STRING', ['BOOLEAN'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf10BooleanValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['TINYINT'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf10TinyIntValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['SMALLINT'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf11SmallIntValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['INT'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf6IntValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['BIGINT'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf9BigIntValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['FLOAT'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf8FloatValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['DOUBLE'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf9DoubleValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['CHAR'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf9StringValEEES3_PNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['VARCHAR'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf9StringValEEES3_PNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['STRING'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf9StringValEEES3_PNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['TIMESTAMP'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf12TimestampValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['DECIMAL'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf10DecimalValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['typeOf'], 'STRING', ['DATE'], '_ZN6impala16UtilityFunctions6TypeOfIN10impala_udf7DateValEEENS2_9StringValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['TINYINT'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf10TinyIntValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['SMALLINT'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf11SmallIntValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['INT'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf6IntValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['BIGINT'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf9BigIntValEEES3_PNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['FLOAT'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf8FloatValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['DOUBLE'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf9DoubleValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['DATE'],
   '_ZN6impala16UtilityFunctions7FnvHashIN10impala_udf7DateValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['fnv_hash'], 'BIGINT', ['STRING'],
   '_ZN6impala16UtilityFunctions13FnvHashStringEPN10impala_udf15FunctionContextERKNS1_9StringValE'],
  [['fnv_hash'], 'BIGINT', ['TIMESTAMP'],
   '_ZN6impala16UtilityFunctions16FnvHashTimestampEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['fnv_hash'], 'BIGINT', ['DECIMAL'],
   '_ZN6impala16UtilityFunctions14FnvHashDecimalEPN10impala_udf15FunctionContextERKNS1_10DecimalValE'],
  [['murmur_hash'], 'BIGINT', ['TINYINT'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf10TinyIntValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['SMALLINT'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf11SmallIntValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['INT'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf6IntValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['BIGINT'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf9BigIntValEEES3_PNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['FLOAT'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf8FloatValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['DOUBLE'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf9DoubleValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['DATE'],
   '_ZN6impala16UtilityFunctions10MurmurHashIN10impala_udf7DateValEEENS2_9BigIntValEPNS2_15FunctionContextERKT_'],
  [['murmur_hash'], 'BIGINT', ['STRING'],
   '_ZN6impala16UtilityFunctions16MurmurHashStringEPN10impala_udf15FunctionContextERKNS1_9StringValE'],
  [['murmur_hash'], 'BIGINT', ['TIMESTAMP'],
   '_ZN6impala16UtilityFunctions19MurmurHashTimestampEPN10impala_udf15FunctionContextERKNS1_12TimestampValE'],
  [['murmur_hash'], 'BIGINT', ['DECIMAL'],
   '_ZN6impala16UtilityFunctions17MurmurHashDecimalEPN10impala_udf15FunctionContextERKNS1_10DecimalValE'],

  # (Non)NullValue functions
  [['nullvalue'], 'BOOLEAN', ['BOOLEAN'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf10BooleanValEEES3_PNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['TINYINT'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf10TinyIntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['SMALLINT'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf11SmallIntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['INT'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf6IntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['BIGINT'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf9BigIntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['FLOAT'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf8FloatValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['DOUBLE'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf9DoubleValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['STRING'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf9StringValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['TIMESTAMP'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf12TimestampValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nullvalue'], 'BOOLEAN', ['DECIMAL'], '_ZN6impala15IsNullPredicate6IsNullIN10impala_udf10DecimalValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],

  [['nonnullvalue'], 'BOOLEAN', ['BOOLEAN'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf10BooleanValEEES3_PNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['TINYINT'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf10TinyIntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['SMALLINT'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf11SmallIntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['INT'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf6IntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['BIGINT'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf9BigIntValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['FLOAT'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf8FloatValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['DOUBLE'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf9DoubleValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['STRING'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf9StringValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['TIMESTAMP'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf12TimestampValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],
  [['nonnullvalue'], 'BOOLEAN', ['DECIMAL'], '_ZN6impala15IsNullPredicate9IsNotNullIN10impala_udf10DecimalValEEENS2_10BooleanValEPNS2_15FunctionContextERKT_'],

  # Bit and Byte functions
  # For functions corresponding to builtin operators, we can reuse the implementations
  [['bitand'], 'TINYINT', ['TINYINT', 'TINYINT'], 'impala::Operators::Bitand_TinyIntVal_TinyIntVal'],
  [['bitand'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'impala::Operators::Bitand_SmallIntVal_SmallIntVal'],
  [['bitand'], 'INT', ['INT', 'INT'], 'impala::Operators::Bitand_IntVal_IntVal'],
  [['bitand'], 'BIGINT', ['BIGINT', 'BIGINT'], 'impala::Operators::Bitand_BigIntVal_BigIntVal'],
  [['bitor'], 'TINYINT', ['TINYINT', 'TINYINT'], 'impala::Operators::Bitor_TinyIntVal_TinyIntVal'],
  [['bitor'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'impala::Operators::Bitor_SmallIntVal_SmallIntVal'],
  [['bitor'], 'INT', ['INT', 'INT'], 'impala::Operators::Bitor_IntVal_IntVal'],
  [['bitor'], 'BIGINT', ['BIGINT', 'BIGINT'], 'impala::Operators::Bitor_BigIntVal_BigIntVal'],
  [['bitxor'], 'TINYINT', ['TINYINT', 'TINYINT'], 'impala::Operators::Bitxor_TinyIntVal_TinyIntVal'],
  [['bitxor'], 'SMALLINT', ['SMALLINT', 'SMALLINT'], 'impala::Operators::Bitxor_SmallIntVal_SmallIntVal'],
  [['bitxor'], 'INT', ['INT', 'INT'], 'impala::Operators::Bitxor_IntVal_IntVal'],
  [['bitxor'], 'BIGINT', ['BIGINT', 'BIGINT'], 'impala::Operators::Bitxor_BigIntVal_BigIntVal'],
  [['bitnot'], 'TINYINT', ['TINYINT'], 'impala::Operators::Bitnot_TinyIntVal'],
  [['bitnot'], 'SMALLINT', ['SMALLINT'], 'impala::Operators::Bitnot_SmallIntVal'],
  [['bitnot'], 'INT', ['INT'], 'impala::Operators::Bitnot_IntVal'],
  [['bitnot'], 'BIGINT', ['BIGINT'], 'impala::Operators::Bitnot_BigIntVal'],
  [['countset'], 'INT', ['TINYINT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf10TinyIntValEEENS2_6IntValEPNS2_15FunctionContextERKT_'],
  [['countset'], 'INT', ['SMALLINT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf11SmallIntValEEENS2_6IntValEPNS2_15FunctionContextERKT_'],
  [['countset'], 'INT', ['INT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf6IntValEEES3_PNS2_15FunctionContextERKT_'],
  [['countset'], 'INT', ['BIGINT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf9BigIntValEEENS2_6IntValEPNS2_15FunctionContextERKT_'],
  [['countset'], 'INT', ['TINYINT', 'INT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf10TinyIntValEEENS2_6IntValEPNS2_15FunctionContextERKT_RKS4_'],
  [['countset'], 'INT', ['SMALLINT', 'INT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf11SmallIntValEEENS2_6IntValEPNS2_15FunctionContextERKT_RKS4_'],
  [['countset'], 'INT', ['INT', 'INT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf6IntValEEES3_PNS2_15FunctionContextERKT_RKS3_'],
  [['countset'], 'INT', ['BIGINT', 'INT'], '_ZN6impala16BitByteFunctions8CountSetIN10impala_udf9BigIntValEEENS2_6IntValEPNS2_15FunctionContextERKT_RKS4_'],
  [['getbit'], 'TINYINT', ['TINYINT', 'INT'], '_ZN6impala16BitByteFunctions6GetBitIN10impala_udf10TinyIntValEEES3_PNS2_15FunctionContextERKT_RKNS2_6IntValE'],
  [['getbit'], 'TINYINT', ['SMALLINT', 'INT'], '_ZN6impala16BitByteFunctions6GetBitIN10impala_udf11SmallIntValEEENS2_10TinyIntValEPNS2_15FunctionContextERKT_RKNS2_6IntValE'],
  [['getbit'], 'TINYINT', ['INT', 'INT'], '_ZN6impala16BitByteFunctions6GetBitIN10impala_udf6IntValEEENS2_10TinyIntValEPNS2_15FunctionContextERKT_RKS3_'],
  [['getbit'], 'TINYINT', ['BIGINT', 'INT'], '_ZN6impala16BitByteFunctions6GetBitIN10impala_udf9BigIntValEEENS2_10TinyIntValEPNS2_15FunctionContextERKT_RKNS2_6IntValE'],
  [['rotateleft'], 'TINYINT', ['TINYINT', 'INT'], 'impala::BitByteFunctions::RotateLeft'],
  [['rotateleft'], 'SMALLINT', ['SMALLINT', 'INT'], 'impala::BitByteFunctions::RotateLeft'],
  [['rotateleft'], 'INT', ['INT', 'INT'], 'impala::BitByteFunctions::RotateLeft'],
  [['rotateleft'], 'BIGINT', ['BIGINT', 'INT'], 'impala::BitByteFunctions::RotateLeft'],
  [['rotateright'], 'TINYINT', ['TINYINT', 'INT'], 'impala::BitByteFunctions::RotateRight'],
  [['rotateright'], 'SMALLINT', ['SMALLINT', 'INT'], 'impala::BitByteFunctions::RotateRight'],
  [['rotateright'], 'INT', ['INT', 'INT'], 'impala::BitByteFunctions::RotateRight'],
  [['rotateright'], 'BIGINT', ['BIGINT', 'INT'], 'impala::BitByteFunctions::RotateRight'],
  [['setbit'], 'TINYINT', ['TINYINT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf10TinyIntValEEET_PNS2_15FunctionContextERKS4_RKNS2_6IntValE'],
  [['setbit'], 'SMALLINT', ['SMALLINT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf11SmallIntValEEET_PNS2_15FunctionContextERKS4_RKNS2_6IntValE'],
  [['setbit'], 'INT', ['INT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf6IntValEEET_PNS2_15FunctionContextERKS4_RKS3_'],
  [['setbit'], 'BIGINT', ['BIGINT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf9BigIntValEEET_PNS2_15FunctionContextERKS4_RKNS2_6IntValE'],
  [['setbit'], 'TINYINT', ['TINYINT', 'INT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf10TinyIntValEEET_PNS2_15FunctionContextERKS4_RKNS2_6IntValESB_'],
  [['setbit'], 'SMALLINT', ['SMALLINT', 'INT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf11SmallIntValEEET_PNS2_15FunctionContextERKS4_RKNS2_6IntValESB_'],
  [['setbit'], 'INT', ['INT', 'INT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf6IntValEEET_PNS2_15FunctionContextERKS4_RKS3_SA_'],
  [['setbit'], 'BIGINT', ['BIGINT', 'INT', 'INT'], '_ZN6impala16BitByteFunctions6SetBitIN10impala_udf9BigIntValEEET_PNS2_15FunctionContextERKS4_RKNS2_6IntValESB_'],
  [['shiftleft'], 'TINYINT', ['TINYINT', 'INT'], 'impala::BitByteFunctions::ShiftLeft'],
  [['shiftleft'], 'SMALLINT', ['SMALLINT', 'INT'], 'impala::BitByteFunctions::ShiftLeft'],
  [['shiftleft'], 'INT', ['INT', 'INT'], 'impala::BitByteFunctions::ShiftLeft'],
  [['shiftleft'], 'BIGINT', ['BIGINT', 'INT'], 'impala::BitByteFunctions::ShiftLeft'],
  [['shiftright'], 'TINYINT', ['TINYINT', 'INT'], 'impala::BitByteFunctions::ShiftRight'],
  [['shiftright'], 'SMALLINT', ['SMALLINT', 'INT'], 'impala::BitByteFunctions::ShiftRight'],
  [['shiftright'], 'INT', ['INT', 'INT'], 'impala::BitByteFunctions::ShiftRight'],
  [['shiftright'], 'BIGINT', ['BIGINT', 'INT'], 'impala::BitByteFunctions::ShiftRight'],

  # Mask functions
  [['mask_show_first_n'], 'STRING', ['STRING'], 'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'STRING', ['STRING', 'INT'], 'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'STRING', ['STRING', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'BIGINT', ['BIGINT'], 'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'BIGINT', ['BIGINT', 'INT'], 'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_first_n'], 'BIGINT', ['BIGINT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskShowFirstN'],
  [['mask_show_last_n'], 'STRING', ['STRING'], 'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'STRING', ['STRING', 'INT'], 'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'STRING', ['STRING', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'BIGINT', ['BIGINT'], 'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'BIGINT', ['BIGINT', 'INT'], 'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_show_last_n'], 'BIGINT', ['BIGINT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskShowLastN'],
  [['mask_first_n'], 'STRING', ['STRING'], 'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'STRING', ['STRING', 'INT'], 'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'STRING', ['STRING', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'BIGINT', ['BIGINT'], 'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'BIGINT', ['BIGINT', 'INT'], 'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_first_n'], 'BIGINT', ['BIGINT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskFirstN'],
  [['mask_last_n'], 'STRING', ['STRING'], 'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'STRING', ['STRING', 'INT'], 'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'STRING', ['STRING', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'STRING', ['STRING', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'BIGINT', ['BIGINT'], 'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'BIGINT', ['BIGINT', 'INT'], 'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'BIGINT', ['BIGINT', 'INT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask_last_n'], 'BIGINT', ['BIGINT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::MaskLastN'],
  [['mask'], 'STRING', ['STRING'], 'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'STRING'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'STRING', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'STRING', 'STRING', 'STRING', 'INT', 'STRING', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'STRING', ['STRING', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT'], 'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING'],
       'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT', 'INT', 'INT', 'INT', 'INT', 'INT'],
       'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT', 'STRING', 'STRING', 'STRING', 'STRING', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT', 'STRING', 'STRING', 'STRING', 'INT', 'STRING', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'BIGINT', ['BIGINT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'DATE', ['DATE'], 'impala::MaskFunctions::Mask'],
  [['mask'], 'DATE', ['DATE', 'STRING', 'STRING', 'STRING', 'STRING', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'DATE', ['DATE', 'STRING', 'STRING', 'STRING', 'STRING', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'DATE', ['DATE', 'STRING', 'STRING', 'STRING', 'INT', 'STRING', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'DATE', ['DATE', 'STRING', 'STRING', 'STRING', 'STRING', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask'], 'DATE', ['DATE', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT', 'INT'],
      'impala::MaskFunctions::Mask'],
  [['mask_hash'], 'STRING', ['STRING'], 'impala::MaskFunctions::MaskHash'],
  [['mask_hash'], 'BIGINT', ['BIGINT'], 'impala::MaskFunctions::MaskHash'],
  [['mask_hash'], 'DOUBLE', ['DOUBLE'], 'impala::MaskFunctions::MaskHash'],
  [['mask_hash'], 'BOOLEAN', ['BOOLEAN'], 'impala::MaskFunctions::MaskHash'],
  [['mask_hash'], 'TIMESTAMP', ['TIMESTAMP'], 'impala::MaskFunctions::MaskHash'],
  [['mask_hash'], 'DATE', ['DATE'], 'impala::MaskFunctions::MaskHash'],
]

invisible_functions = [
  [['months_add_interval'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf6IntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_add_interval'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb1EN10impala_udf9BigIntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_sub_interval'], 'TIMESTAMP', ['TIMESTAMP', 'INT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf6IntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],
  [['months_sub_interval'], 'TIMESTAMP', ['TIMESTAMP', 'BIGINT'],
      '_ZN6impala18TimestampFunctions6AddSubILb0EN10impala_udf9BigIntValEN5boost9date_time15months_durationINS4_9gregorian21greg_durations_configEEELb0EEENS2_12TimestampValEPNS2_15FunctionContextERKSA_RKT0_'],

  [['months_add_interval'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb1EN10impala_udf6IntValELb0EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['months_add_interval'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb1EN10impala_udf9BigIntValELb0EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['months_sub_interval'], 'DATE', ['DATE', 'INT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb0EN10impala_udf6IntValELb0EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],
  [['months_sub_interval'], 'DATE', ['DATE', 'BIGINT'],
      '_ZN6impala13DateFunctions12AddSubMonthsILb0EN10impala_udf9BigIntValELb0EEENS2_7DateValEPNS2_15FunctionContextERKS4_RKT0_'],

  [['distinctfrom'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN'], 'impala::Operators::DistinctFrom_BooleanVal_BooleanVal'],
  [['distinctfrom'], 'BOOLEAN', ['TINYINT', 'TINYINT'], 'impala::Operators::DistinctFrom_TinyIntVal_TinyIntVal'],
  [['distinctfrom'], 'BOOLEAN', ['SMALLINT', 'SMALLINT'], 'impala::Operators::DistinctFrom_SmallIntVal_SmallIntVal'],
  [['distinctfrom'], 'BOOLEAN', ['INT', 'INT'], 'impala::Operators::DistinctFrom_IntVal_IntVal'],
  [['distinctfrom'], 'BOOLEAN', ['BIGINT', 'BIGINT'], 'impala::Operators::DistinctFrom_BigIntVal_BigIntVal'],
  [['distinctfrom'], 'BOOLEAN', ['FLOAT', 'FLOAT'], 'impala::Operators::DistinctFrom_FloatVal_FloatVal'],
  [['distinctfrom'], 'BOOLEAN', ['DOUBLE', 'DOUBLE'], 'impala::Operators::DistinctFrom_DoubleVal_DoubleVal'],
  [['distinctfrom'], 'BOOLEAN', ['STRING', 'STRING'], 'impala::Operators::DistinctFrom_StringVal_StringVal'],
  [['distinctfrom'], 'BOOLEAN', ['TIMESTAMP', 'TIMESTAMP'], 'impala::Operators::DistinctFrom_TimestampVal_TimestampVal'],
  [['distinctfrom'], 'BOOLEAN', ['CHAR', 'CHAR'], 'impala::Operators::DistinctFrom_Char_Char'],
  [['distinctfrom'], 'BOOLEAN', ['DECIMAL', 'DECIMAL'], 'impala::DecimalOperators::DistinctFrom_DecimalVal_DecimalVal'],
  [['distinctfrom'], 'BOOLEAN', ['DATE', 'DATE'], 'impala::Operators::DistinctFrom_DateVal_DateVal'],

  [['notdistinct'], 'BOOLEAN', ['BOOLEAN', 'BOOLEAN'], 'impala::Operators::NotDistinct_BooleanVal_BooleanVal'],
  [['notdistinct'], 'BOOLEAN', ['TINYINT', 'TINYINT'], 'impala::Operators::NotDistinct_TinyIntVal_TinyIntVal'],
  [['notdistinct'], 'BOOLEAN', ['SMALLINT', 'SMALLINT'], 'impala::Operators::NotDistinct_SmallIntVal_SmallIntVal'],
  [['notdistinct'], 'BOOLEAN', ['INT', 'INT'], 'impala::Operators::NotDistinct_IntVal_IntVal'],
  [['notdistinct'], 'BOOLEAN', ['BIGINT', 'BIGINT'], 'impala::Operators::NotDistinct_BigIntVal_BigIntVal'],
  [['notdistinct'], 'BOOLEAN', ['FLOAT', 'FLOAT'], 'impala::Operators::NotDistinct_FloatVal_FloatVal'],
  [['notdistinct'], 'BOOLEAN', ['DOUBLE', 'DOUBLE'], 'impala::Operators::NotDistinct_DoubleVal_DoubleVal'],
  [['notdistinct'], 'BOOLEAN', ['STRING', 'STRING'], 'impala::Operators::NotDistinct_StringVal_StringVal'],
  [['notdistinct'], 'BOOLEAN', ['TIMESTAMP', 'TIMESTAMP'], 'impala::Operators::NotDistinct_TimestampVal_TimestampVal'],
  [['notdistinct'], 'BOOLEAN', ['CHAR', 'CHAR'], 'impala::Operators::NotDistinct_Char_Char'],
  [['notdistinct'], 'BOOLEAN', ['DECIMAL', 'DECIMAL'], 'impala::DecimalOperators::NotDistinct_DecimalVal_DecimalVal'],
  [['notdistinct'], 'BOOLEAN', ['DATE', 'DATE'], 'impala::Operators::NotDistinct_DateVal_DateVal'],
]
