/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.service.cli.operation;

import com.google.common.annotations.VisibleForTesting;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.InputStreamReader;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.common.ServerUtils;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.QueryInfo;
import org.apache.hadoop.hive.ql.QueryState;
import org.apache.hadoop.hive.ql.session.OperationLog;
import org.apache.hadoop.util.StringUtils;
import org.apache.hive.service.cli.OperationHandle;
import org.apache.hive.service.cli.operation.Operation;
import org.apache.hive.service.cli.operation.OperationManager;
import org.apache.hive.service.cli.operation.SQLOperation;
import org.apache.hive.service.cli.session.HiveSession;
import org.apache.hive.service.cli.session.SessionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OperationLogManager {
    private static final Logger LOG = LoggerFactory.getLogger(OperationLogManager.class);
    private static final String HISTORIC_DIR_SUFFIX = "_historic";
    private static String historicLogRootDir;
    private static long maxBytesToFetch;
    private final HiveConf hiveConf;
    private final SessionManager sessionManager;
    private final OperationManager operationManager;
    private OperationLogDirCleaner cleaner;
    private String historicParentLogDir;
    private String serverInstance;

    public OperationLogManager(SessionManager sessionManager, HiveConf hiveConf) {
        this.operationManager = sessionManager.getOperationManager();
        this.hiveConf = hiveConf;
        this.sessionManager = sessionManager;
        if (HiveConf.getBoolVar((Configuration)hiveConf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_ENABLED) && hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_LOGGING_OPERATION_ENABLED) && hiveConf.isWebUiQueryInfoCacheEnabled()) {
            this.initHistoricOperationLogRootDir();
            maxBytesToFetch = HiveConf.getSizeVar((Configuration)hiveConf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_FETCH_MAXBYTES);
            if (historicLogRootDir != null && !HiveConf.getBoolVar((Configuration)hiveConf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_IN_TEST)) {
                this.cleaner = new OperationLogDirCleaner();
                this.cleaner.start();
            }
        }
    }

    private String getServerInstance() {
        String hostname;
        try {
            hostname = ServerUtils.hostname();
        }
        catch (Exception e) {
            hostname = UUID.randomUUID().toString();
        }
        int serverPort = this.hiveConf.getIntVar(HiveConf.ConfVars.HIVE_SERVER2_THRIFT_PORT);
        return hostname + "_" + serverPort;
    }

    private void initHistoricOperationLogRootDir() {
        String origLogLoc = this.hiveConf.getVar(HiveConf.ConfVars.HIVE_SERVER2_LOGGING_OPERATION_LOG_LOCATION);
        File logLocation = new File(origLogLoc);
        this.historicParentLogDir = logLocation.getAbsolutePath() + HISTORIC_DIR_SUFFIX;
        this.serverInstance = this.getServerInstance();
        String logRootDir = this.historicParentLogDir + "/" + this.serverInstance + "_" + System.currentTimeMillis();
        File operationLogRootDir = new File(logRootDir);
        if (operationLogRootDir.exists() && !operationLogRootDir.isDirectory()) {
            LOG.warn("The historic operation log root directory exists, but it is not a directory: " + operationLogRootDir.getAbsolutePath());
            return;
        }
        if (!operationLogRootDir.exists() && !operationLogRootDir.mkdirs()) {
            LOG.warn("Unable to create historic operation log root directory: " + operationLogRootDir.getAbsolutePath());
            return;
        }
        historicLogRootDir = logRootDir;
    }

    public static OperationLog createOperationLog(Operation operation, QueryState queryState) {
        boolean isHistoricLogEnabled;
        HiveSession session = operation.getParentSession();
        File parentFile = session.getOperationLogSessionDir();
        boolean bl = isHistoricLogEnabled = historicLogRootDir != null;
        if (isHistoricLogEnabled && operation instanceof SQLOperation) {
            String sessionId = session.getSessionHandle().getHandleIdentifier().toString();
            parentFile = new File(historicLogRootDir + "/" + sessionId);
            if (!parentFile.exists()) {
                if (!parentFile.mkdirs()) {
                    LOG.warn("Unable to create the historic operation log session dir: " + parentFile + ", fall back to the original operation log session dir.");
                    parentFile = session.getOperationLogSessionDir();
                    isHistoricLogEnabled = false;
                }
            } else if (!parentFile.isDirectory()) {
                LOG.warn("The historic operation log session dir: " + parentFile + " is exist, but it's not a directory, fall back to the original operation log session dir.");
                parentFile = session.getOperationLogSessionDir();
                isHistoricLogEnabled = false;
            }
        }
        OperationHandle opHandle = operation.getHandle();
        File operationLogFile = new File(parentFile, queryState.getQueryId());
        HiveConf.setBoolVar((Configuration)queryState.getConf(), (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_ENABLED, (boolean)isHistoricLogEnabled);
        if (isHistoricLogEnabled) {
            HiveConf.setVar((Configuration)queryState.getConf(), (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_LOGGING_OPERATION_LOG_LOCATION, (String)historicLogRootDir);
            if (HiveConf.getBoolVar((Configuration)queryState.getConf(), (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_IN_TEST)) {
                HiveConf.setBoolVar((Configuration)queryState.getConf(), (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_TESTING_REMOVE_LOGS, (boolean)false);
            }
            LOG.info("The operation log location changes from {} to {}.", (Object)new File(session.getOperationLogSessionDir(), queryState.getQueryId()), (Object)operationLogFile);
        }
        OperationLog operationLog = new OperationLog(opHandle.toString(), operationLogFile, queryState.getConf());
        return operationLog;
    }

    private Set<String> getLiveSessions() {
        Collection<HiveSession> hiveSessions = this.sessionManager.getSessions();
        HashSet<String> liveSessions = new HashSet<String>();
        for (HiveSession session : hiveSessions) {
            liveSessions.add(session.getSessionHandle().getHandleIdentifier().toString());
        }
        return liveSessions;
    }

    private Set<String> getHistoricSessions() {
        File[] subFiles;
        assert (historicLogRootDir != null);
        File logDir = new File(historicLogRootDir);
        HashSet<String> results = new HashSet<String>();
        if (logDir.exists() && logDir.isDirectory() && (subFiles = logDir.listFiles()) != null) {
            for (File f : subFiles) {
                results.add(f.getName());
            }
        }
        return results;
    }

    @VisibleForTesting
    public List<File> getExpiredOperationLogFiles() {
        if (historicLogRootDir == null) {
            return Collections.emptyList();
        }
        ArrayList<File> results = new ArrayList<File>();
        Collection files = FileUtils.listFiles((File)new File(historicLogRootDir), null, (boolean)true);
        Set<String> queryIds = this.operationManager.getAllCachedQueryIds();
        for (File logFile : files) {
            if (queryIds.contains(logFile.getName())) continue;
            results.add(logFile);
        }
        return results;
    }

    @VisibleForTesting
    public List<File> getExpiredSessionLogDirs() {
        if (historicLogRootDir == null) {
            return Collections.emptyList();
        }
        ArrayList<File> results = new ArrayList<File>();
        Set<String> liveSessions = this.getLiveSessions();
        Set<String> historicSessions = this.getHistoricSessions();
        historicSessions.removeAll(liveSessions);
        Set<String> queryIds = this.operationManager.getAllCachedQueryIds();
        for (String sessionId : historicSessions) {
            File sessionLogDir = new File(historicLogRootDir, sessionId);
            if (!sessionLogDir.exists()) continue;
            File[] logFiles = sessionLogDir.listFiles();
            if (logFiles == null || logFiles.length == 0) {
                results.add(sessionLogDir);
                continue;
            }
            boolean found = false;
            for (File logFile : logFiles) {
                if (!queryIds.contains(logFile.getName())) continue;
                found = true;
                break;
            }
            if (found) continue;
            results.add(sessionLogDir);
        }
        return results;
    }

    private List<String> getFileNames(List<File> fileList) {
        ArrayList<String> results = new ArrayList<String>();
        for (File file : fileList) {
            results.add(file.getName());
        }
        return results;
    }

    @VisibleForTesting
    public void removeExpiredOperationLogAndDir() {
        if (historicLogRootDir == null) {
            return;
        }
        List<File> operationLogFiles = this.getExpiredOperationLogFiles();
        if (operationLogFiles.isEmpty()) {
            LOG.info("No expired operation logs found under the dir: {}", (Object)historicLogRootDir);
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Trying to delete the expired operation logs: {} ", this.getFileNames(operationLogFiles));
            }
            for (File logFile : operationLogFiles) {
                FileUtils.deleteQuietly((File)logFile);
            }
            LOG.info("Deleted {} expired operation logs", (Object)operationLogFiles.size());
        }
        List<File> sessionLogDirs = this.getExpiredSessionLogDirs();
        if (sessionLogDirs.isEmpty()) {
            LOG.info("No expired operation log session dir under the dir: {}", (Object)historicLogRootDir);
        } else {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Trying to delete the expired operation log session dirs: {} ", this.getFileNames(sessionLogDirs));
            }
            for (File logDir : sessionLogDirs) {
                FileUtils.deleteQuietly((File)logDir);
            }
            LOG.info("Deleted {} expired operation log session dirs", (Object)sessionLogDirs.size());
        }
    }

    private void deleteElderLogRootDirs() {
        File[] children = new File(this.historicParentLogDir).listFiles(new FileFilter(){

            @Override
            public boolean accept(File child) {
                return child.isDirectory() && child.getName().startsWith(OperationLogManager.this.serverInstance) && !child.getAbsolutePath().equals(historicLogRootDir);
            }
        });
        if (children == null || children.length == 0) {
            return;
        }
        for (File f : children) {
            FileUtils.deleteQuietly((File)f);
        }
    }

    public void stop() {
        if (this.cleaner != null) {
            this.cleaner.shutDown();
        }
    }

    private static boolean isHistoricOperationLogEnabled(String logLocation) {
        if (logLocation == null || historicLogRootDir == null) {
            return false;
        }
        return logLocation.startsWith(historicLogRootDir);
    }

    public static String getOperationLog(QueryInfo queryInfo) {
        String logLocation = queryInfo.getOperationLogLocation();
        StringBuilder builder = new StringBuilder();
        if (!OperationLogManager.isHistoricOperationLogEnabled(logLocation)) {
            if (logLocation == null) {
                return "Operation log is disabled, please set hive.server2.logging.operation.enabled = true to enable it";
            }
            builder.append("Operation Log - will be deleted after query completes, ").append("set hive.server2.historic.operation.log.enabled = true ").append("and hive.server2.webui.max.historic.queries > 0 to disable it").append(System.lineSeparator());
        }
        try (RandomAccessFile raf = new RandomAccessFile(logLocation, "r");){
            long fileLen = raf.length();
            long seekPos = 0L;
            if (fileLen > maxBytesToFetch) {
                seekPos = fileLen - maxBytesToFetch;
            }
            ByteBuffer buffer = ByteBuffer.allocate((int)maxBytesToFetch);
            int read = raf.getChannel().read(buffer, seekPos);
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(new ByteArrayInputStream(buffer.array(), 0, read)));){
                String line;
                while ((line = reader.readLine()) != null) {
                    builder.append(line).append(System.lineSeparator());
                }
            }
        }
        catch (Exception e) {
            builder.append(StringUtils.stringifyException((Throwable)e));
        }
        return builder.toString();
    }

    @VisibleForTesting
    public static String getHistoricLogDir() {
        return historicLogRootDir;
    }

    private class OperationLogDirCleaner
    extends Thread {
        private final long interval;
        private boolean shutdown = false;
        private final Object monitor = new Object();

        OperationLogDirCleaner() {
            long checkInterval = HiveConf.getTimeVar((Configuration)OperationLogManager.this.hiveConf, (HiveConf.ConfVars)HiveConf.ConfVars.HIVE_SERVER2_HISTORIC_OPERATION_LOG_CHECK_INTERVAL, (TimeUnit)TimeUnit.MILLISECONDS);
            this.interval = Math.max(checkInterval, 3000L);
            this.setName("Historic-OperationLogDir-Cleaner");
            this.setDaemon(true);
        }

        @Override
        public void run() {
            OperationLogManager.this.deleteElderLogRootDirs();
            this.sleepFor(this.interval);
            while (!this.shutdown) {
                try {
                    OperationLogManager.this.removeExpiredOperationLogAndDir();
                    this.sleepFor(this.interval);
                }
                catch (Exception e) {
                    LOG.warn("OperationLogDir cleaner caught exception: " + e.getMessage(), (Throwable)e);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void sleepFor(long interval) {
            Object object = this.monitor;
            synchronized (object) {
                if (this.shutdown) {
                    return;
                }
                try {
                    this.monitor.wait(interval);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void shutDown() {
            Object object = this.monitor;
            synchronized (object) {
                this.shutdown = true;
                this.monitor.notifyAll();
            }
        }
    }
}

