/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.hdfs.caching.internal;

import com.oracle.bmc.util.StreamUtils;
import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BufferedInputStreamMultiplexer {
    private static final Logger LOG = LoggerFactory.getLogger(BufferedInputStreamMultiplexer.class);
    private final InputStream source;
    private final Buffer buffer;

    public BufferedInputStreamMultiplexer(InputStream source, Buffer buffer) {
        this.source = new BufferedInputStream(source);
        this.buffer = buffer;
        buffer.setMultiplexer(this);
    }

    private synchronized int readFromSource() throws IOException {
        try {
            int read = this.source.read();
            if (read == -1) {
                this.buffer.closeSource();
                StreamUtils.closeQuietly((InputStream)this.source);
            } else {
                this.buffer.write(read);
            }
            return read;
        }
        catch (IOException ioe) {
            this.buffer.closeSource();
            StreamUtils.closeQuietly((InputStream)this.source);
            throw ioe;
        }
    }

    private synchronized int readFromSource(byte[] b, int off, int len) throws IOException {
        try {
            int read = this.source.read(b, off, len);
            if (read == -1) {
                this.buffer.closeSource();
            } else {
                this.buffer.write(b, off, read);
            }
            return read;
        }
        catch (IOException ioe) {
            this.buffer.closeSource();
            throw ioe;
        }
    }

    public MultiplexerInputStream getInputStream() throws FileNotFoundException {
        return this.buffer.getInputStream();
    }

    public static class FileBuffer
    extends AbstractBuffer {
        private final RandomAccessFile bufferFile;
        private final FileChannel channel;

        public FileBuffer(File bufferFile) throws FileNotFoundException {
            this.bufferFile = new RandomAccessFile(bufferFile, "rw");
            this.channel = this.bufferFile.getChannel();
        }

        @Override
        public synchronized void write(byte[] b, int off, int len) throws IOException {
            try {
                ByteBuffer bb = ByteBuffer.wrap(b, off, len);
                this.channel.write(bb);
                this.bytesWritten += (long)len;
                LOG.trace("Wrote {} more bytes, {} bytes written in total", (Object)len, (Object)this.bytesWritten);
            }
            catch (IOException ioe) {
                this.closeSource();
                throw ioe;
            }
        }

        @Override
        public MultiplexerInputStream getInputStream() {
            return new FileBufferInputStream();
        }

        public RandomAccessFile getBufferFile() {
            return this.bufferFile;
        }

        protected class FileBufferInputStream
        extends MultiplexerInputStream
        implements Closeable,
        AutoCloseable {
            private long bytesRead = 0L;

            protected FileBufferInputStream() {
            }

            @Override
            public int read(byte[] b) throws IOException {
                return this.read(b, 0, b.length);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                FileBuffer fileBuffer = FileBuffer.this;
                synchronized (fileBuffer) {
                    int bytesReadFromSource;
                    if (this.isStreamClosed) {
                        return -1;
                    }
                    int bytesReturned = 0;
                    int bytesAvailable = (int)Math.min(Integer.MAX_VALUE, Math.min((long)len, FileBuffer.this.bytesWritten - this.bytesRead));
                    if (bytesAvailable > 0) {
                        ByteBuffer bb = ByteBuffer.wrap(b, off, bytesAvailable);
                        int actuallyTransferred = FileBuffer.this.channel.read(bb, this.bytesRead);
                        assert (actuallyTransferred == bytesAvailable);
                        off += bytesAvailable;
                        len -= bytesAvailable;
                        this.bytesRead += (long)bytesAvailable;
                        bytesReturned += bytesAvailable;
                    }
                    while (len > 0 && !FileBuffer.this.isSourceClosed && (bytesReadFromSource = FileBuffer.this.multiplexer.readFromSource(b, off, len)) != -1) {
                        len -= bytesReadFromSource;
                        off += bytesReadFromSource;
                        this.bytesRead += (long)bytesReadFromSource;
                        bytesReturned += bytesReadFromSource;
                    }
                    if (bytesReturned == 0 && FileBuffer.this.isSourceClosed) {
                        return -1;
                    }
                    LOG.trace("FileBufferInputStream {} read {} bytes, {} bytes read in total", new Object[]{System.identityHashCode(this), bytesReturned, this.bytesRead});
                    return bytesReturned;
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int read() throws IOException {
                FileBuffer fileBuffer = FileBuffer.this;
                synchronized (fileBuffer) {
                    if (this.isStreamClosed) {
                        return -1;
                    }
                    if (this.bytesRead < FileBuffer.this.bytesWritten) {
                        ByteBuffer bb = ByteBuffer.allocate(1);
                        int read = FileBuffer.this.channel.read(bb, this.bytesRead);
                        if (read == -1) {
                            throw new IllegalStateException("There should have been more bytes available, but read() returned -1");
                        }
                        ++this.bytesRead;
                        bb.position(0);
                        LOG.trace("FileBufferInputStream {} read 1 byte, {} bytes read in total", (Object)System.identityHashCode(this), (Object)this.bytesRead);
                        return bb.get();
                    }
                    if (FileBuffer.this.isSourceClosed) {
                        return -1;
                    }
                    int read = FileBuffer.this.multiplexer.readFromSource();
                    if (read != -1) {
                        ++this.bytesRead;
                    }
                    LOG.trace("FileBufferInputStream {} read 1 byte, {} bytes read in total", (Object)System.identityHashCode(this), (Object)this.bytesRead);
                    return read;
                }
            }

            @Override
            public long getBytesRead() {
                return this.bytesRead;
            }
        }
    }

    public static class MemoryBuffer
    extends AbstractBuffer {
        private byte[] buf;

        public MemoryBuffer() {
            this(8192);
        }

        public MemoryBuffer(int initialSizeInBytes) {
            this.buf = new byte[initialSizeInBytes];
        }

        @Override
        public synchronized void write(byte[] b, int off, int len) {
            while (len > 0) {
                long remainingInBuffer = (long)this.buf.length - this.bytesWritten;
                if (remainingInBuffer == 0L) {
                    byte[] newBuffer = new byte[this.buf.length * 2];
                    if (newBuffer.length <= this.buf.length) {
                        throw new IllegalStateException("Buffer size overflow");
                    }
                    System.arraycopy(this.buf, 0, newBuffer, 0, this.buf.length);
                    this.buf = newBuffer;
                    remainingInBuffer = (long)this.buf.length - this.bytesWritten;
                }
                long bytesToCopy = Math.min((long)len, remainingInBuffer);
                System.arraycopy(b, off, this.buf, (int)this.bytesWritten, (int)bytesToCopy);
                this.bytesWritten += bytesToCopy;
                off = (int)((long)off + bytesToCopy);
                len = (int)((long)len - bytesToCopy);
            }
        }

        @Override
        public MultiplexerInputStream getInputStream() {
            return new MemoryBufferInputStream();
        }

        protected class MemoryBufferInputStream
        extends MultiplexerInputStream
        implements Closeable,
        AutoCloseable {
            private long bytesRead = 0L;

            protected MemoryBufferInputStream() {
            }

            @Override
            public int read(byte[] b) throws IOException {
                return this.read(b, 0, b.length);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                MemoryBuffer memoryBuffer = MemoryBuffer.this;
                synchronized (memoryBuffer) {
                    int nextByte;
                    if (this.isStreamClosed) {
                        return -1;
                    }
                    int bytesReturned = 0;
                    long bytesAvailable = Math.min((long)len, MemoryBuffer.this.bytesWritten - this.bytesRead);
                    if (bytesAvailable > 0L) {
                        System.arraycopy(MemoryBuffer.this.buf, (int)this.bytesRead, b, off, (int)bytesAvailable);
                        off = (int)((long)off + bytesAvailable);
                        len = (int)((long)len - bytesAvailable);
                        this.bytesRead += bytesAvailable;
                        bytesReturned = (int)((long)bytesReturned + bytesAvailable);
                    }
                    while (len > 0 && !MemoryBuffer.this.isSourceClosed && (nextByte = this.read()) != -1) {
                        b[off++] = (byte)nextByte;
                        --len;
                        ++bytesReturned;
                    }
                    if (bytesReturned == 0 && MemoryBuffer.this.isSourceClosed) {
                        return -1;
                    }
                    return bytesReturned;
                }
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public int read() throws IOException {
                MemoryBuffer memoryBuffer = MemoryBuffer.this;
                synchronized (memoryBuffer) {
                    if (this.isStreamClosed) {
                        return -1;
                    }
                    if (this.bytesRead < MemoryBuffer.this.bytesWritten) {
                        return MemoryBuffer.this.buf[(int)this.bytesRead++] & 0xFF;
                    }
                    if (MemoryBuffer.this.isSourceClosed) {
                        return -1;
                    }
                    int read = MemoryBuffer.this.multiplexer.readFromSource();
                    if (read == -1) {
                        return -1;
                    }
                    ++this.bytesRead;
                    return read & 0xFF;
                }
            }

            @Override
            public long getBytesRead() {
                return this.bytesRead;
            }
        }
    }

    protected static abstract class AbstractBuffer
    implements Buffer {
        protected volatile BufferedInputStreamMultiplexer multiplexer;
        protected volatile long bytesWritten = 0L;
        protected volatile boolean isSourceClosed = false;

        protected AbstractBuffer() {
        }

        @Override
        public void setMultiplexer(BufferedInputStreamMultiplexer multiplexer) {
            if (this.multiplexer != null) {
                throw new IllegalArgumentException("multiplexer can only be set once");
            }
            this.multiplexer = multiplexer;
        }

        @Override
        public void write(int b) throws IOException {
            this.write(new byte[]{(byte)b}, 0, 1);
        }

        @Override
        public synchronized void closeSource() {
            this.isSourceClosed = true;
        }

        @Override
        public long getBytesWritten() {
            return this.bytesWritten;
        }

        @Override
        public boolean isSourceClosed() {
            return this.isSourceClosed;
        }
    }

    public static interface Buffer {
        public void setMultiplexer(BufferedInputStreamMultiplexer var1);

        public void write(byte[] var1, int var2, int var3) throws IOException;

        public void write(int var1) throws IOException;

        public void closeSource();

        public MultiplexerInputStream getInputStream() throws FileNotFoundException;

        public boolean isSourceClosed();

        public long getBytesWritten();
    }

    public static abstract class MultiplexerInputStream
    extends InputStream {
        protected volatile boolean isStreamClosed = false;

        public abstract long getBytesRead();

        @Override
        public synchronized void close() {
            this.isStreamClosed = true;
        }
    }
}

