package org.apache.torque.templates.platform;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import org.apache.torque.templates.typemapping.SchemaType;
import org.apache.torque.templates.typemapping.SqlType;

/**
 * HSQLDB (formerly known as Hypersonic) Platform implementation.
 *
 * @author <a href="mailto:mpoeschl@marmot.at">Martin Poeschl</a>
 * @version $Id: PlatformHsqldbImpl.java 1405649 2012-11-04 22:07:22Z tfischer $
 */
public class PlatformHsqldbImpl extends PlatformDefaultImpl
{
    /** The date format for formatting database dates. */
    private static final String DATE_FORMAT = "''yyyy-MM-dd''";

    /** The date format for formatting database times. */
    private static final String TIME_FORMAT = "''HH:mm:ss''";

    /**
     * Default constructor.
     */
    public PlatformHsqldbImpl()
    {
        super();
        initialize();
    }

    /**
     * Initializes db specific domain mapping.
     */
    private void initialize()
    {
        setSchemaTypeToSqlTypeMapping(
                SchemaType.CHAR,
                new SqlType("VARCHAR", "1"));
        setSchemaTypeToSqlTypeMapping(
                SchemaType.BOOLEANCHAR,
                new SqlType("VARCHAR", "1"));
    }

    /**
     * @return The RDBMS-specific SQL fragment for autoincrement.
     * @see Platform#getAutoIncrement()
     */
    @Override
    public String getAutoIncrement()
    {
        return "GENERATED BY DEFAULT AS IDENTITY (START WITH 1)";
    }

    /**
     * Returns whether the "not null part" of the definition of a column
     * should be generated before the "autoincrement part" in a "create table"
     * statement.
     * @return false.
     * @see Platform#createNotNullBeforeAutoincrement()
     */
    @Override
    public boolean createNotNullBeforeAutoincrement()
    {
        return false;
    }

    @Override
    protected boolean escapeBackslashes()
    {
        return false;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getDateString(Date date)
    {
        SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        return dateFormat.format(date);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getTimeString(Date date)
    {
        SimpleDateFormat dateFormat = new SimpleDateFormat(TIME_FORMAT);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        return dateFormat.format(date);
    }

    /**
     * Returns whether the database has schema support where a schema
     * is not tied to a user (oracle) or database (mysql), but can be created
     * separately.
     *
     * @return this implementation returns true.
     */
    public boolean usesStandaloneSchema()
    {
        return true;
    }
}
