/*% c2latex -- converts C code with LaTeX comments into LaTeX input.
% @(#)c2latex.c	1.1 91/01/16 
% To create the documentation, compile this source file and feed this
% source file into the resulting program.  The output will be a file
% that can be processed using LaTeX.
\documentstyle{article}
\newif\ifshowprogram		% Set to true to include a listing of
\showprogramtrue		% the program.

\newcommand{\ctolatex}{{\tt c2latex}}
\newcommand{\MITREcopyright}{%
Copyright \copyright{} 1991 by John D. Ramsdell.

This program is free software; you can redistribute it and/or modify
it under the terms of the MITRE Corporation General Public License as 
published by the MITRE Corporation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but {\em without any warranty;} without even the implied warranty of
{\em merchantability} or {\em fitness for a particular purpose.}  See the
MITRE General Public License for more details.

A copy of the MITRE General Public License has been included in the
last comment of the C source file for \ctolatex{}.  You also can obtain
a copy by writing to the MITRE Corporation.}

\title{\ctolatex}
\author{John D. Ramsdell}
\date{Version 1.1 of 91/01/16%
}

\begin{document}
\maketitle

\ctolatex\footnote{\MITREcopyright} provides simple support for literate
programming in C\@.  Given a C source file in which the comments have
been written in \LaTeX{}, \ctolatex{} converts the C source file into a
\LaTeX{} source file.  It can be used to produce typeset listings of C
programs and/or documentation associated with the program.

The C source given to \ctolatex{} usually has the following form.  It
starts with a large comment containing \LaTeX{} commands that start a
document along with any initial text.  Then there is a sequence of
comment and code pairs, with the comment explaining the code to
follow.  The source file is ended by a comment containing \LaTeX{}
commands that finish the document.

\ctolatex{} produces \LaTeX{} source by implementing a small number of
rules.  A C comment that starts at the beginning of a line is copied
unmodified into the \LaTeX{} source file.  Otherwise, non-blank lines
are surrounded by a pair of formatting commands
(\verb-\begin{flushleft}- and \verb-\end{flushleft}-), and the lines
are separated by \verb-\\*-.  Each non-blank line is formatted using
\LaTeX's \verb-\verb- command, except comments within the line are
formatted in an \verb-\mbox-.

\ctolatex{} is invoked with the command:
\begin{center}\tt
\ctolatex{} [$\langle${\it C file name}$\rangle$ 
[$\langle${\it \LaTeX{} file name}$\rangle$]]
\end{center}
When $\langle${\it \LaTeX{} file name}$\rangle$ is omitted, the
\LaTeX{} goes to standard output.  When 
$\langle${\it C file name}$\rangle$ is omitted, C source is
read from standard input, and the
\LaTeX{} goes to standard output. 

This \LaTeX{} document was produced by using \ctolatex{} on its source file. 
*/
/*\ifshowprogram*/

/*
\newpage
\section{Program listing}
\ctolatex{} can be modified to handle \TeX{} macro packages other
than \LaTeX{} by modifying the following strings and
\verb-tex_putc-.*/ 
char *begin_comment = "%\\mbox{"; /* This pair is used */
char *end_comment = "}\\verb%";	/* to surround comments in code. */
char *begin_code = "\\begin{flushleft}\n"; /* This pair is used */
char *end_code = "\\end{flushleft}\n"; /* to surround code. */
char *code_line_separator = "\\\\* ";
char *begin_code_line = "\\verb%"; /* This pair is used */
char *end_code_line = "%";	/* to surround code lines. */
/* The comment markers for C. */
char *comment_start = "/*";     /* Comment characters for */
char *comment_end = "*/";	/* the C source language. */

#include <stdio.h>
typedef enum {FALSE, TRUE} bool;
void tex_putc();
void tex_puts();

/*\subsection{Filter}
The routine {\tt filter} implements the main loop.  By the time it
is called, input comes from {\tt stdin} and output goes to {\tt
stdout.}  The loop is traversed for each line of input.  The variable
{\tt sp} contains a count of the number of previously seen spaces.
This space count is required so that partial matches can be printed
before the spaces when a code line is to be printed.
*/
int filter()
{
  int c; char *match, *s;
  int sp = 0;			/* Buffered space count. */
  bool just_saw_code = FALSE;
  while (1) {
    c = match_string(comment_start, &match);
    if (*match == '\0') {	/* Found a comment. */
      if (just_saw_code) {
	fputs(end_code, stdout);
	just_saw_code = FALSE;
      } 
      c = put_comment(c);	/* Make sure nothing from */
      match = comment_start;	/* this match is printed. */
    }
    for (sp = 0; c == ' '; sp++) /* Count white space */
      c = getchar_xtab();	/* in case of code line. */
    if (match == comment_start && (c == '\n' || c == EOF)) { 
      if (just_saw_code) {	/* Found blank line, */
	fputs(end_code, stdout); /* or a comment which */
	just_saw_code = FALSE;	/*  terminates a line. */
      }
    }
    else {			/* Found a code line. */
      if (!just_saw_code) {
	fputs(begin_code, stdout);
	just_saw_code = TRUE;
      }
      else
	fputs(code_line_separator, stdout);
      fputs(begin_code_line, stdout);
      for (s = comment_start; s < match; s++) 
	tex_putc(*s, stdout);	/* Print partial match. */
      for (; sp > 0; sp--)
	putc(' ', stdout);	/* Print buffered spaces. */
      c = put_code_line(c);
      fputs(end_code_line, stdout);
    }
    if (c == EOF) break;
    putc(c, stdout);		/* print newline. */
  }
  if (just_saw_code) fputs(end_code, stdout);
  return 0;
}

/*\subsection{Match string}
\verb-match_string- matches input to a pattern.  When done, characters
in the pattern string with address less than \verb-*match- have been
matched with the input.  If \verb-*match- points to the end of string
character, a complete match was found. \verb-match_string- returns the
character after the last one that matched. */
int match_string (pattern, match)
     char *pattern;
     char **match;
{
  int c;
  for (;; pattern++) {
    c = getchar_xtab();
    if (*pattern != c || *pattern == '\0') {
      *match = pattern;
      return c;
    }
  }
}

/*\subsection{Put comment}*/
int put_comment(c)
     int c;
{
  char *match, *s;
  while (1) {
    if (c == *comment_end) {
      c = match_string(comment_end+1, &match);
      if (*match == '\0') return c; /* Comment end found. */
      for (s = comment_end; s < match; s++)
	putc(*s, stdout);	/* Print partial match. */
    }
    if (c == EOF) return fatal ("EOF within a comment.");
    putc(c, stdout);
    c = getchar_xtab();
  }
}

/*\subsection{Put quoted}*/
int put_quoted (q, m)
     int q;			/* Quote character. */
     char *m;			/* EOF message. */
{
  int c;
  tex_putc(q, stdout);
  while (1) {
    c = getchar_xtab();
    if (c == EOF) return fatal (m);
    else if (c == q) {
      tex_putc(c, stdout);
      return getchar_xtab();
    }
    else if (c == '\\') {	/* Backslash quotes within */
      tex_putc(c, stdout);	/* quoted text. */
      c = getchar_xtab();
      if (c == EOF) return fatal (m);
      tex_putc(c, stdout);
    }
    else
      tex_putc(c, stdout);
  }    
}

/*\subsection{Put code line}*/
int put_code_line (c)
     int c; 
{
  char *match, *s;
  while (1)
    switch (c) {
    case EOF: return c;
    case '\n': return c;
    case '"':
      c = put_quoted (c, "EOF within a string");
      break;
    case '\'':
      c = put_quoted (c, "EOF within a character");
      break;
    default:
      if (c == *comment_start) {
	c = match_string(comment_start+1, &match);
	if (*match == '\0') {	/* Found comment */
	  tex_puts(comment_start, stdout);
	  fputs(begin_comment, stdout);
	  c = put_comment(c);
	  fputs(end_comment, stdout);
	  tex_puts(comment_end, stdout);
	}
	else			/* Print partial match. */
	  for (s = comment_start; s < match; s++)
	    tex_putc(*s, stdout);
      }
      else {			/* Just print the character. */
	tex_putc(c, stdout);
	c = getchar_xtab();
      }
    }
}

/*\subsection{Fatal}*/
int lineno = 1;			/* Input source line number. */
int fatal(message)		/* Report fatal errors and exit. */
     char *message;
{
  void exit();
  fprintf(stderr, "Fatal error on line %d: %s\n",
	  lineno, message);
  exit (1); return 0;		/* Return keeps lint happy. */
}

/*\subsection{Getchar xtab}*/
/* All input is processed by \verb-getchar_xtab- so that TAB characters
can be expanded. \TeX{} treats TAB characters as a space---not what is
wanted. */
int getchar_xtab()
{
  int c;
  static int spaces = 0;	/* Spaces left to print a TAB. */
  static int column = 0;	/* Current input column. */
  if (spaces > 0) {
    spaces--;
    return ' ';
  }
  switch (c = getc(stdin)) {
  case '\t':
    spaces = 7 - (7&column);
    column += spaces + 1;
    return ' ';
  case '\n':
    lineno++;			/* for {\tt fatal} */
    column = 0;
    return c;
  default:
    column++;
    return c;
  }
}

/*\subsection{\TeX{} putc}
\verb-tex_putc- handles the case in which you want to print the
character that is used to bound the \verb-\verb- text. */
void tex_putc (c, f)
     int c;
     FILE *f;
{
  if (c == '%') fputs("%\\verb-%-\\verb%", f);
  else putc (c, f);
}

/*\subsection{\TeX{} puts}
\verb-tex_putc-'s the elements of a string.*/
void tex_puts(s, f)
     char *s;
     FILE *f;
{
  while (*s != '\0') tex_putc(*s++, f);
}

/*\subsection{Main}*/
int main (argc, argv)
     int argc;
     char *argv[];
{
  switch (argc) {
  case 3:
  case 2:
    if (NULL == freopen(argv[1], "r", stdin)) {
      fprintf(stderr, "Cannot open %s for reading.\n", argv[1]);
      break;
    }
    if (argc == 3 && NULL == freopen(argv[2], "w", stdout)) {
      fprintf(stderr, "Cannot open %s for writing.\n", argv[2]);
      break;
    }
  case 1:
    return filter();
  }
  fprintf(stderr, 
	  "Usage: %s [C file] [LaTeX file]\n",
	  argv[0]);
  return 1;
}
/*\fi*/
/*\end{document}*/
/*

	       MITRE Corporation General Public License
		       Version 1, October 1990

 Copyright (C) 1990 The MITRE Corporation
		    Burlington Road,
		    Bedford, MA. 01730
 
 Everyone is permitted to copy and distribute verbatim copies
 of this license document, but changing it is not allowed.

			    Preamble


  This General Public License is intended to guarantee  freedom to
share and change software--to make sure the software is free for
all its users.  The General Public License applies to the MITRE
Corporation's software.

  When we speak of free software, we are referring to freedom, not
price.  Specifically, the General Public License is designed to make
sure that you have the freedom to give away copies of free
software, that you receive source code or can get it if you want it,
that you can change the software or use pieces of it in new free
programs; and that you know you can do these things.

  To protect your rights, we need to make restrictions that forbid
anyone to deny you these rights or to ask you to surrender the rights.
These restrictions translate to certain responsibilities for you if you
distribute copies of the software, or if you modify it.

  For example, if you distribute copies of a such a program, whether
gratis or for a distribution fee, you must give the recipients all the rights
that you have.  You must make sure that they, too, receive or can get the
source code.  And you must tell them their rights.

  We protect your rights with two steps: (1) copyright the software, and
(2) offer you this license which gives you legal permission to copy,
distribute and/or modify the software.

  Also, for each author's protection and ours, we want to make certain
that everyone understands that there is no warranty for this free
software.  If the software is modified by someone else and passed on, we
want its recipients to know that what they have is not the original, so
that any problems introduced by others will not reflect on the original
authors' reputations.

  The precise terms and conditions for copying, distribution and
modification follow.

		    MITRE Corporation GENERAL PUBLIC LICENSE
   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION

  0. This License Agreement applies to any program or other work which
contains a notice placed by the copyright holder saying it may be
distributed under the terms of this General Public License.  The
"Program", below, refers to any such program or work, and a "work based
on the Program" means either the Program or any work containing the
Program or a portion of it, either verbatim or with modifications.  Each
licensee is addressed as "you".

  1. You may copy and distribute verbatim copies of the Program's source
code as you receive it, in any medium, provided that you conspicuously and
appropriately publish on each copy an appropriate copyright notice and
disclaimer of warranty; keep intact all the notices that refer to this
General Public License and to the absence of any warranty; and give any
other recipients of the Program a copy of this General Public License
along with the Program.  You may charge a fee for the physical act of
transferring a copy.

  2. You may modify your copy or copies of the Program or any portion of
it, and copy and distribute such modifications under the terms of Paragraph
1 above, provided that you also do the following:

    a) cause the modified files to carry prominent notices stating that
    you changed the files and the date of any change; and

    b) cause the whole of any work that you distribute or publish, that
    in whole or in part contains the Program or any part thereof, either
    with or without modifications, to be licensed at no charge to all
    third parties under the terms of this General Public License (except
    that you may choose to grant warranty protection to some or all
    third parties, at your option).

    c) If the modified program normally reads commands interactively when
    run, you must cause it, when started running for such interactive use
    in the simplest and most usual way, to print or display an
    announcement including an appropriate copyright notice and a notice
    that there is no warranty (or else, saying that you provide a
    warranty) and that users may redistribute the program under these
    conditions, and telling the user how to view a copy of this General
    Public License.

    d) You may charge a fee for the physical act of transferring a
    copy, and you may at your option offer warranty protection in
    exchange for a fee.

Mere aggregation of another independent work with the Program (or its
derivative) on a volume of a storage or distribution medium does not bring
the other work under the scope of these terms.

  3. You may copy and distribute the Program (or a portion or derivative of
it, under Paragraph 2) in object code or executable form under the terms of
Paragraphs 1 and 2 above provided that you also do one of the following:

    a) accompany it with the complete corresponding machine-readable
    source code, which must be distributed under the terms of
    Paragraphs 1 and 2 above; or,

    b) accompany it with a written offer, valid for at least three
    years, to give any third party free (except for a nominal charge
    for the cost of distribution) a complete machine-readable copy of the
    corresponding source code, to be distributed under the terms of
    Paragraphs 1 and 2 above; or,

    c) accompany it with the information you received as to where the
    corresponding source code may be obtained.  (This alternative is
    allowed only for noncommercial distribution and only if you
    received the program in object code or executable form alone.)

Source code for a work means the preferred form of the work for making
modifications to it.  For an executable file, complete source code means
all the source code for all modules it contains; but, as a special
exception, it need not include source code for modules which are standard
libraries that accompany the operating system on which the executable
file runs, or for standard header files or definitions files that
accompany that operating system.

  4. You may not copy, modify, sublicense, distribute or transfer the
Program except as expressly provided under this General Public License.
Any attempt otherwise to copy, modify, sublicense, distribute or transfer
the Program is void, and will automatically terminate your rights to use
the Program under this License.  However, parties who have received
copies, or rights to use copies, from you under this General Public
License will not have their licenses terminated so long as such parties
remain in full compliance.

  5. By copying, distributing or modifying the Program (or any work based
on the Program) you indicate your acceptance of this license to do so,
and all its terms and conditions.

  6. Each time you redistribute the Program (or any work based on the
Program), the recipient automatically receives a license from the original
licensor to copy, distribute or modify the Program subject to these
terms and conditions.  You may not impose any further restrictions on the
recipients' exercise of the rights granted herein.

  7. The MITRE Corporation may publish revised and/or new versions
of the General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

Each version is given a distinguishing version number.  If the Program
specifies a version number of the license which applies to it and "any
later version", you have the option of following the terms and
conditions either of that version or of any later version published by
the MITRE Corporation.  If the Program does not specify a version
number of the license, you may choose any version ever published by
the MITRE Corporation.

  8. If you wish to incorporate parts of the Program into other free
programs whose distribution conditions are different, write to the author
to ask for permission.  For software which is copyrighted by the MITRE
Corporation, write to the MITRE Corporation; we sometimes
make exceptions for this.  Our decision will be guided by the two goals
of preserving the free status of all derivatives of our  software.

			    NO WARRANTY

  9. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

  10. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGES.

		     END OF TERMS AND CONDITIONS

*/
