/*
 * Created on Mar 19, 2003
 *
 * @author henkel@cs.colorado.edu
 * 
 */
package bibtex.parser;

import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.LinkedList;

import bibtex.dom.BibtexAbstractValue;
import bibtex.dom.BibtexEntry;
import bibtex.dom.BibtexFile;

/**
 * @author henkel
 */
public final class BibtexParser {

	private PseudoLexer lexer;
	private BibtexFile bibtexFile;
	private LinkedList exceptions;

	/**
	 * returns the list of non-fatal exceptions that occured during parsing.
	 * Usually, these occur while parsing an entry. Usually, the remainder of the entry
	 * will be treated as part of a comment - thus the following entry will be parsed
	 * again.
	 *  
	 * @return List
	 */
	public ParseException[] getExceptions() {
		if (exceptions == null)
			return new ParseException[0];
		ParseException[] result = new ParseException[exceptions.size()];
		exceptions.toArray(result);
		return result;
	}

	/**
	 * parses the input into bibtexFile - don't forget to check getExceptions()
	 * afterwards (if you use !throwAllParseExceptions)...
	 * 
	 * @param bibtexFile
	 * @param input
	 * @param throwAllParseExceptions if this is false, non-critical exceptions will
	 *                      be accumulated and can be retrieved using getExceptions().
	 * @throws ParseException
	 * @throws IOException
	 */

	public void parse(
		BibtexFile bibtexFile,
		Reader input,
		boolean throwAllParseExceptions)
		throws ParseException, IOException {
		this.lexer = new PseudoLexer(input);
		this.bibtexFile = bibtexFile;
		this.exceptions = new LinkedList();
		while (true) {
			PseudoLexer.Token token = lexer.scanTopLevelCommentOrAtOrEOF();
			switch (token.choice) {
				case 0 : // top level comment
					bibtexFile.addEntry(
						BibtexFile.makeToplevelComment(token.content));
					break;
				case 1 : // @ sign
					if (throwAllParseExceptions)
						parseEntry();
					else {
						try {
							parseEntry();
						}
						catch (ParseException parseException) {
							exceptions.add(parseException);
						}
					}
					break;
				case 2 : // EOF
					return;
			}
		}
	}

	private final static char[] EXCEPTION_SET_NAMES =
		new char[] { '"', '#', '%', '\'', '(', ')', ',', '=', '{', '}' };
	//	private final static String[] ENTRY_TYPES =
	//		new String[] {
	//			"string",
	//			"preamble",
	//			"article",
	//			"book",
	//			"booklet",
	//			"conference",
	//			"inbook",
	//			"incollection",
	//			"inproceedings",
	//			"manual",
	//			"mastersthesis",
	//			"misc",
	//			"phdthesis",
	//			"proceedings",
	//			"techreport",
	//			"unpublished",
	//			"periodical" // not really standard but commonly used.
	//	};

	/**
	 * 
	 */
	private void parseEntry() throws ParseException, IOException {
		String entryType = lexer.scanEntryTypeName().toLowerCase();
		final int bracketChoice =
			lexer.scanAlternatives(new char[] { '{', '(' }, false).choice;

		if (entryType.equals("string")) {
			String stringName =
				lexer.scanLiteral(EXCEPTION_SET_NAMES, true, true).content;
			lexer.scan('=');
			BibtexAbstractValue value = parseValue();
			bibtexFile.addEntry(
				BibtexFile.makeStringDefinition(stringName, value));
		}
		else if (entryType.equals("preamble")) {
			BibtexAbstractValue value = parseValue();
			bibtexFile.addEntry(BibtexFile.makePreamble(value));
		}
		else {
			// All others
			String bibkey =
				lexer.scanLiteral(new char[] { ',' }, true, true).content;
			final BibtexEntry entry = BibtexFile.makeEntry(entryType, bibkey);
			bibtexFile.addEntry(entry);
			while (true) {
				lexer.enforceNoEof(
					"',' or corresponding closing bracket",
					true);
				//System.out.println("---------->'"+lexer.currentInputChar()+"'");
				if (lexer.currentInputChar() == ',') {
					lexer.scan(',');
					lexer.enforceNoEof("'}' or [FIELDNAME]", true);
					if (lexer.currentInputChar() == '}')
						break;
					String fieldName =
						lexer.scanLiteral(
							EXCEPTION_SET_NAMES,
							true,
							true).content;
					lexer.scan('=');
					BibtexAbstractValue value = parseValue();
					entry.addField(BibtexFile.makeField(fieldName, value));
				}
				else
					break;
			}
		}

		if (bracketChoice == 0)
			lexer.scan('}');
		else
			lexer.scan(')');
	}

	/**
	 * 
	 */
	private BibtexAbstractValue parseValue()
		throws ParseException, IOException {
		lexer.enforceNoEof("[STRING] or [STRINGREFERENCE] or [NUMBER]", true);
		char inputCharacter = lexer.currentInputChar();
		BibtexAbstractValue result;

		if (inputCharacter == '"') {
			result = parseQuotedString();
		}
		else if (inputCharacter == '{') {
			result = parseBracketedString();
		}
		else {
			String stringContent =
				lexer
					.scanLiteral(EXCEPTION_SET_NAMES, false, true)
					.content
					.trim();
			try {
				Integer.parseInt(stringContent);
				// we can parse it, so it's a number
				result = BibtexFile.makeString(stringContent);
			}
			catch (NumberFormatException nfe) {
				result = BibtexFile.makeReference(stringContent);
			}
		}

		lexer.enforceNoEof("'#' or something else", true);
		if (lexer.currentInputChar() == '#') {
			lexer.scan('#');
			return BibtexFile.makeConcatenatedValue(result, parseValue());
		}
		else {
			return result;
		}

	}

	/**
	 * @return BibtexAbstractValue
	 */
	private BibtexAbstractValue parseBracketedString()
		throws ParseException, IOException {
		final String string = lexer.scanBracketedString().content;
		return BibtexFile.makeString(
			string.substring(0, string.length() - 1));
	}

	/**
	 * @return BibtexAbstractValue
	 */
	private BibtexAbstractValue parseQuotedString()
		throws IOException, ParseException {
		return BibtexFile.makeString(lexer.scanQuotedString().content);
	}

	/**
	 * This is just for testing purposes. Look into bibtex.Main if you want to see an
	 * example that is kind of useful.
	 * 
	 * @param args
	 */

	public static void main(String[] args) {
		BibtexFile bibtexFile = new BibtexFile();
		BibtexParser parser = new BibtexParser();
		try {
			FileReader input = new FileReader(args[0]);

			parser.parse(bibtexFile, input, false);
		}
		catch (Exception e) {
			System.err.println("Fatal Exception: ");
			e.printStackTrace();
			System.out.println("Fatal completion.");
			return;
		}
		finally {
			ParseException[] exceptions = parser.getExceptions();
			if (exceptions.length > 0) {
				System.err.println("Non-fatal exceptions: ");
				for (int i = 0; i < exceptions.length; i++) {
					exceptions[i].printStackTrace();
					System.err.println("===================");
				}
				System.out.println("Almost normal completion.");
			}
		}
		//System.out.println("Dumping bibtexFile ... \n");
		//bibtexFile.printBibtex(new PrintWriter(System.out));
		System.out.println("Normal completion.");
	}
}
