package Geo::LookupPostcode::ES;

use 5.012;
use strict;
use warnings FATAL => 'all';
use utf8;

use Carp qw(croak);
require Exporter;

our @ISA = qw(Exporter);
our @EXPORT_OK = qw(lookup_es_postcode);

my %geo_provinces;

=encoding utf-8

=head1 NAME

Geo::LookupPostcode::ES - Get province and region codes for a Spanish postcode

=head1 SUBROUTINES/METHODS

=head2 lookup_es_postcode

Takes one character string argument: a postcode.

If successful, it returns a reference to a hash, with this structure:


    my $rh_province = lookup_es_postcode("03460");
    {
        province_name => $province_name,
    }
    
If it cannot find the province, it returns undef.

Note that the names may be anglicised.

=cut

sub lookup_es_postcode {
    croak "Expected one argument" if (@_ != 1);
    my ($postcode) = @_;

    # Do some cleanup of postcode:
    utf8::upgrade $postcode;
    $postcode =~ s/\s*//g;

    my $province_name = $geo_provinces{$postcode};
    if (defined($province_name)) {
        return { province_name => $province_name };
    }
    return;
}

# Data from https://en.wikipedia.org/wiki/List_of_postal_codes_in_Spain


BEGIN { %geo_provinces = (
    # $postcode => $name,
    '01' => 'Álava',
    '02' => 'Albacete',
    '03' => 'Alicante',
    '04' => 'Almería',
    '05' => 'Ávila',
    '06' => 'Badajoz',
    '07' => 'Balearic Islands',
    '08' => 'Barcelona',
    '09' => 'Burgos',
    '10' => 'Cáceres',
    '11' => 'Cádiz',
    '12' => 'Castellón',
    '13' => 'Ciudad Real',
    '14' => 'Córdoba',
    '15' => 'A Coruña',
    '16' => 'Cuenca',
    '17' => 'Girona',
    '18' => 'Granada',
    '19' => 'Guadalajara',
    '20' => 'Gipuzkoa',
    '21' => 'Huelva',
    '22' => 'Huesca',
    '23' => 'Jaén',
    '24' => 'León',
    '25' => 'Lleida',
    '26' => 'La Rioja',
    '27' => 'Lugo',
    '28' => 'Madrid',
    '29' => 'Málaga',
    '30' => 'Murcia',
    '31' => 'Navarre',
    '32' => 'Ourense',
    '33' => 'Asturias',
    '34' => 'Palencia',
    '35' => 'Las Palmas',
    '36' => 'Pontevedra',
    '37' => 'Salamanca',
    '38' => 'Santa Cruz de Tenerife',
    '39' => 'Cantabria',
    '40' => 'Segovia',
    '41' => 'Seville',
    '42' => 'Soria',
    '43' => 'Tarragona',
    '44' => 'Teruel',
    '45' => 'Toledo',
    '46' => 'Valencia',
    '47' => 'Valladolid',
    '48' => 'Biscay',
    '49' => 'Zamora',
    '50' => 'Zaragoza',
    '51' => 'Ceuta',
    '52' => 'Melilla',
    (map { sprintf("%05d", $_) => 'Alicante' } 3000..3099),
    '03158' => 'Catral',
    '03189' => 'Villamartin',
    (map { sprintf("%05d", $_) => 'Elche' } 3201..3299),
    '03400' => 'Villena',
    '03460' => 'Beneixama',
    '03500' => 'Benidorm',
    '03600' => 'Elda',
    '03610' => 'Petrer',
    '03690' => 'San Vicente del Raspeig',
    '03700' => 'Denia',
    '03710' => 'Calpe',
    '03801' => 'Alcoy',
    '03530' => 'La Nucia',
    '03759' => 'Benidoleig',
    '03730' => 'Javea',
    '08001' => 'Barcelona - el Raval',
    '08002' => 'Barcelona - Barri Gòtic',
    '08003' => 'Barcelona - Sant Pere, Santa Susana i la Ribera',
    '08004' => 'Barcelona - Barceloneta',
    '08005' => 'Barcelona - Poblenou',
    '08006' => 'Barcelona - Gràcia',
    '08110' => 'Montcada i Reixac',
    '08120' => 'La Llagosta',
    '08170' => 'Montornès del Vallès',
    '08301' => 'Mataró',
    '08302' => 'Mataró',
    '08303' => 'Mataró',
    '08304' => 'Mataró',
    '08320' => 'el Masnou',
    '08329' => 'Teià',
    '08370' => 'Calella',
    '08380' => 'Malgrat de Mar',
    '08397' => 'Pineda de Mar',
    '08398' => 'Santa Susanna, Catalonia',
    '08401' => 'Granollers',
    '08620' => 'Sant Vicenç dels Horts',
    '08630' => 'Abrera',
    '08640' => 'Olesa de Montserrat',
    '08690' => 'Santa Coloma de Cervelló',
    '08740' => 'Sant Andreu de la Barca',
    '08758' => 'Cervelló',
    '08800' => 'Vilanova i la Geltrú',
    '08810' => 'Sant Pere de Ribes',
    '08820' => 'el Prat de Llobregat',
    '08830' => 'Sant Boi de Llobregat',
    '08840' => 'Viladecans',
    '08850' => 'Gavà',
    '08859' => 'Begues',
    '08860' => 'Castelldefels',
    '08870' => 'Sitges',
    '08940' => 'Cornellà de Llobregat',
    '08950' => 'Esplugues de Llobregat',
    '08960' => 'Sant Just Desvern',
    '08970' => 'Sant Joan Despí',
    '08980' => 'Sant Feliu de Llobregat',
    '13300' => ' Valdepeñas',
    '13500' => ' Puertollano',
    '13600' => ' Alcázar de San Juan',
    '13200' => ' Manzanares, Ciudad Real',
    '17210' => 'Calella de Pallafrugell',
    '17211' => 'Llafranc',
    '17212' => 'Tamariu',
    '17220' => 'Sant Feliu de Guíxols',
    '17230' => 'Palamós',
    '17246' => 'Santa Cristina d\'Aro',
    '17248' => 'Platja d\'Aro',
    '17249' => 'Platja d\'Aro',
    '17250' => 'Castell-Platja d\'Aro',
    '17251' => 'Sant Antoni de Calonge',
    '17252' => 'Sant Antoni de Calonge',
    '17255' => 'Begur, Spain',
    '17256' => 'Pals',
    '17257' => 'Torroella de Montgrí',
    '17258' => 'L’Estartit',
    '17300' => 'Blanes',
    '17310' => 'Lloret de Mar',
    '17320' => 'Tossa de Mar',
    '18140' => 'La Zubia',
    '18230' => 'Atarfe',
    '18300' => 'Loja, Granada',
    '18314' => 'Fuente Camacho',
    '18412' => 'Bubión',
    '18697' => 'La Herradura',
    '18690' => 'Almuñécar',
    '18740' => 'Gualchos',
    '21400' => 'Ayamonte',
    '21700' => 'La Palma del Condado',
    '21710' => 'Bollullos Par del Condado',
    '25220' => 'Bell-Lloc d\'Urgell',
    '25250' => 'Bellpuig',
    '25280' => 'Solsona, Lleida',
    '25560' => 'Sort, Lleida',
    '25566' => 'Soriguera, Vilamur, Llagunes',
    '25567' => 'Llessui, Altron',
    '25568' => 'Enviny, Olp, Pujalt, Pallars',
    '25569' => 'Malmercat, Tornafort',
    '25570' => 'Ribera de Cardós',
    '25571' => 'Ainet de Cardós, Esterri de Cardós',
    '25572' => 'Estaon',
    '25573' => 'Ainet de Besan, Araós',
    '25574' => 'Alins, Tor, Pallars',
    '25575' => 'Àreu',
    '25576' => 'Lladorre',
    '25577' => 'Tavascan',
    '25580' => 'Esterri d\'Àneu',
    '25586' => 'Alòs d\'Isil, Isil, Lleida',
    '25587' => 'València d\'Àneu, Sorpe',
    '25588' => 'Escalarre, Unarre',
    '25589' => 'Son, Lleida',
    '25590' => 'Gerri de la Sal',
    '25591' => 'Peramea',
    '25592' => 'Baen, Lleida',
    '25593' => 'Baro, Lleida, Arcalís',
    '25594' => 'Rialp, Surp, Caregue, Roní, Montenartró',
    '25595' => 'Llavorsí, Farrera, Tírvia',
    '25596' => 'Escaló, Escart',
    '25597' => 'Espot, La Guingueta d\'Àneu, Jou, Lleida',
    '25620' => 'Tremp',
    '28100' => 'Alcobendas',
    '28109' => 'Alcobendas',
    # '28120' => 'San Sebastián de los Reyes', # collision with Colmenar Viejo
    '28700' => 'San Sebastián de los Reyes',
    '28701' => 'San Sebastián de los Reyes',
    # '28707' => 'San Sebastián de los Reyes', # collision with Colmenar Viejo
    '28709' => 'San Sebastián de los Reyes',
    '28120' => 'Colmenar Viejo',
    '28707' => 'Colmenar Viejo',
    '28770' => 'Colmenar Viejo',
    '28780' => 'Colmenar Viejo',
    '28220' => 'Majadahonda',
    '28223' => 'Pozuelo de Alarcón',
    '28224' => 'Pozuelo de Alarcón',
    '28230' => 'Las Rozas de Madrid',
    '28232' => 'Las Rozas de Madrid',
    '28290' => 'Las Rozas de Madrid',
    '28300' => 'Aranjuez',
    '28320' => 'Pinto, Madrid',
    '28340' => 'Valdemoro',
    '28400' => 'Collado Villalba',
    '28409' => 'Collado Villalba',
    '28410' => 'Manzanares el Real',
    '28500' => 'Arganda del Rey',
    '28529' => 'Arganda del Rey',
    '28520' => 'Rivas-Vaciamadrid',
    '28521' => 'Rivas-Vaciamadrid',
    '28522' => 'Rivas-Vaciamadrid',
    '28523' => 'Rivas-Vaciamadrid',
    # '28529' => 'Rivas-Vaciamadrid', # collision with Arganda del Rey
    '28600' => 'Navalcarnero',
    '28608' => 'Navalcarnero',
    '28660' => 'Boadilla del Monte',
    '28668' => 'Boadilla del Monte',
    '28669' => 'Boadilla del Monte',
    '28670' => 'Villaviciosa de Odón',
    '28679' => 'Villaviciosa de Odón',
    '28690' => 'Villanueva de la Cañada',
    '28691' => 'Villanueva de la Cañada',
    '28692' => 'Villanueva de la Cañada',
    '28760' => 'Tres Cantos',
    '28761' => 'Tres Cantos',
    '28790' => 'Tres Cantos',
    '28800' => 'Alcalá de Henares',
    '28801' => 'Alcalá de Henares',
    '28802' => 'Alcalá de Henares',
    '28803' => 'Alcalá de Henares',
    '28804' => 'Alcalá de Henares',
    '28805' => 'Alcalá de Henares',
    '28806' => 'Alcalá de Henares',
    '28807' => 'Alcalá de Henares',
    '28808' => 'Alcalá de Henares',
    '28809' => 'Alcalá de Henares',
    '28820' => 'Coslada',
    '28830' => 'San Fernando de Henares',
    '28831' => 'San Fernando de Henares',
    # '28850' => 'San Fernando de Henares',
    '28850' => 'Torrejón de Ardoz',
    '28860' => 'Paracuellos de Jarama',
    (map { $_ => 'Getafe' } 28900..28909),
    (map { $_ => 'Leganés' } 28910..28919),
    (map { $_ => 'Alcorcón' } 28920..28929),
    (map { $_ => 'Móstoles' } 28930..28939),
    (map { $_ => 'Fuenlabrada' } 28941..28949), # 28940 collides with Humanes de Madrid
    '28940' => 'Humanes de Madrid',
    '28970' => 'Humanes de Madrid',
    '28980' => 'Parla',
    '28981' => 'Parla',
    '29011' => 'Málaga',
    '29100' => 'Coín',
    '29120' => 'Alhaurín el Grande',
    '29130' => 'Alhaurín de la Torre',
    '29170' => 'Colmenar, Málaga',
    '29180' => 'Riogordo',
    '29194' => 'Alfarnate',
    # '29194' => 'Alfarnatejo',
    '29195' => 'Comares',
    '29197' => 'Totalán',
    '29400' => 'Ronda',
    '29480' => 'Gaucín',
    '29490' => 'Gaucín Estación',
    '29600' => 'Elviria',
    # '29600' => 'Marbella', # collides with Elviria
    '29610' => 'Ojén',
    '29620' => 'Torremolinos',
    '29630' => 'Benalmádena Costa',
    '29631' => 'Arroyo de la Miel',
    '29639' => 'Benalmádena Pueblo',
    '29641' => 'Fuengirola',
    '29640' => 'Las Lagunas de Mijas',
    '29649' => 'El Chaparral (Mijas Costa)',
    # '29649' => 'Mijas Costa',
    '29650' => 'Mijas',
    '29660' => 'Nueva Andalucia, Málaga',
    # '29660' => 'Puerto Banús', # collides with Nueva Andalucia, Málaga
    '29670' => 'San Pedro de Alcántara',
    '29678' => 'Cerro Artola',
    # '29678' => 'Guadalmina (San Pedro, Málaga)', # collides with Cerro Artola
    '29679' => 'Benahavís',
    '29680' => 'Estepona',
    '29688' => 'Atalaya Isdabe (Estepona)',
    # '29688' => 'Benamara, Urbanization (Estepona)',
    # '29688' => 'Benavista, El Pilar, El Paraíso, Málagaíso, Urbanization (Estepona)',
    # '29688' => 'Cancelada (Estepona)',
    '29689' => 'Cancelada Playa',
    '29690' => 'Casares, Málaga',
    '29691' => 'Manilva',
    '29692' => 'Castillo de Santa Catalina (Málaga) (San Luis de Sabinillas)',
    # '29692' => 'Puerto de La Duquesa',
    # '29692' => 'San Luis de Sabinillas',
    '29693' => 'Arroyo Vaquero, Urbanization (Estepona)',
    # '29693' => 'Bahia Dorada, Urbanization (Estepona)',
    # '29693' => 'Buenas Noches, Urbanization (Estepona)',
    '29700' => 'Vélez-Málaga',
    '29710' => 'Periana',
    '29711' => 'Alcaucín',
    '29712' => 'Viñuela',
    '29714' => 'Salares',
    '29715' => 'Sedella',
    '29716' => 'Canillas de Aceituno',
    '29717' => 'Arenas, Málaga',
    '29718' => 'Almáchar',
    # '29718' => 'Benemargosa',
    # '29718' => 'Cútar',
    # '29718' => 'El Borge',
    '29719' => 'Benamocarra',
    # '29719' => 'Iznate',
    '29730' => 'Rincón de la Victoria',
    '29740' => 'Torre del Mar',
    '29750' => 'Algarrobo, Spain',
    '29751' => 'Caleta de Velez',
    '29752' => 'Sayalonga',
    '29753' => 'Árchez',
    '29754' => 'Cómpeta',
    '29755' => 'Canillas de Albaida',
    '29770' => 'Torrox',
    '29780' => 'Nerja',
    '29787' => 'Cueva de Nerja',
    '29788' => 'Frigiliana',
    '29791' => 'Macharaviaya',
    (map { $_ => 'Murcia' } 30001..30012),
    '30100' => 'Espinardo',
    '30110' => 'Cabezo de Torres',
    '30120' => 'El Palmar, Murcia',
    '30130' => 'Beniel',
    '30139' => 'El Raal',
    '30140' => 'Santomera',
    '30148' => 'La Matanza, Murcia',
    '30149' => 'El Siscar',
    '30150' => 'La Alberca, Murcia',
    '30151' => 'Santo Ángel, MurciaÁngel',
    '30152' => 'Aljucer',
    '30153' => 'Corvera (Murcia)',
    '30154' => 'Valladolises',
    '30155' => 'Baños y Mendigo',
    '30156' => 'Los Martínez del Puerto',
    '30157' => 'Algezares',
    '30158' => 'Los Garres',
    '30160' => 'Monteagudo, Murcia',
    '30161' => 'Llano de Brujas',
    '30162' => 'Santa Cruz, Murcia',
    '30163' => 'Cobatillas, El Esparragal',
    '30164' => 'Cañada de San Pedro',
    '30165' => 'Rincón de Seca',
    '30166' => 'Nonduermas',
    '30167' => 'La Raya',
    '30168' => 'Era Alta',
    '30169' => 'San Ginés, Murciaés',
    '30170' => 'Mula, Spain',
    '30176' => 'Pliego',
    '30177' => 'Casas Nuevas',
    '30178' => 'Fuente Librilla',
    '30179' => 'Barqueros',
    '30180' => 'Bullas',
    '30189' => 'Chaparral, Murcia',
    '30190' => 'Albudeite',
    '30191' => 'Campos del Río',
    '30192' => 'Rodeo de Enmedio',
    '30193' => 'Yéchar, Puebla de Mula, Baños de Mula',
    '30194' => 'Niño de Mula',
    '30195' => 'Archivel, Noguericas',
    '30196' => 'Bajil',
    (map { $_ => 'Cartagena, Spain' } 30200..30205),
    '30400' => 'Caravaca de la Cruz',
    '30510' => 'Yecla',
    '30520' => 'Jumilla',
    '30550' => 'Abarán',
    '30559' => 'Abarán',
    '30800' => 'Lorca, Spain',
    '30813' => 'Lorca, Spain',
    '30815' => 'Lorca, Spain',
    '30840' => 'Alhama de Murcia',
    '30848' => 'Alhama de Murcia',
    '30849' => 'Alhama de Murcia',
    '35011' => 'Ciudad Alta, Las Palmas de Gran Canaria',
    '35017' => 'Las Palmas',
    '35018' => 'Las Palmas',
    '35100' => 'Playa del Inglés',
    '35220' => 'Jinamar - Telde',
    '35240' => 'Cerecinos del Carrizal',
    '35290' => 'Maspalomas',
    '35470' => 'La Aldea de San Nicolás',
    '35500' => 'Arrecife',
    '35508' => 'Costa Teguise',
    '35510' => 'Puerto del Carmen',
    '35520' => 'Haría (municipality)ía',
    # '35521..35529' => '''not used''',
    '35530' => 'Teguise (municipality)',
    # '35531..35549' => '''not used''',
    '35550' => 'San Bartolomé, Las Palmasé',
    # '35551&ndash;35559' => '''not used''',
    '35560' => 'Tinajo, Las Palmas',
    # '35561..35569' => '''not used''',
    '35570' => 'Yaiza, Las Palmas',
    # '35571' => '''not used''',
    '35572' => 'Tías, Las Palmasías',
    # '35573..35799' => '''not used''',
    '35600' => 'Puerto del Rosario',
    '35628' => 'Pájara and Tuineje, Las Palmas',
    # '35629' => '''not used''',
    '35630' => 'Antigua, Fuerteventura',
    # '35631..35636' => '''not used''',
    '35637' => 'Betancuria',
    # '35638 and 35639' => '''not used''',
    '35640' => 'La Oliva',
    # '35641..35699' => '''not used''',
    '37789' => 'Buenavista, Spain',
    '38000' => 'Santa Cruz de Tenerife',
    '41092' => 'Isla de la Cartuja',
    '41500' => 'Alcalá de Guadaíra',
    '41927' => 'Mairena del Aljarafe',
    (map { $_ => 'Valencia' } 46000..46025),
    '46100' => 'Burjassot',
    '46120' => 'Alboraya',
    '46134' => 'Foios',
    '46182' => 'Paterna(Municipality of Paterna)',
    '46530' => 'Puçol',
    '46200' => 'Paiporta',
    '46300' => 'Utiel and Aldea de Estenas',
    '46400' => 'Cullera and El Caño',
    '46410' => 'Sueca (city)',
    '46420' => 'El Perelló',
    '46500' => 'Sagunto',
    '46600' => 'Alzira, Valencia',
    '46700' => 'Gandia',
    '46710' => 'Daimuz',
    '46800' => 'Xàtiva',
    '46900' => 'Torrent, Valencia',
    '46901' => 'El Vedat de Torrent',
    '46920' => 'Mislata',
    '46940' => 'Manises',
    '46980' => 'Paterna',
    (map { $_ => 'Bilbao' } 48001..48015),
    # '48003' => 'Bilbao, Arrigorriaga',
    '48100' => 'Morga, Mungia',
    '48110' => 'Gatika',
    '48111' => 'Laukiz',
    '48112' => 'Maruri-Jatabe',
    '48113' => 'Gamiz-Fika',
    '48114' => 'Arrieta',
    '48115' => 'Morga',
    '48116' => 'Fruiz',
    '48120' => 'Meñaka',
    '48130' => 'Bakio',
    '48140' => 'Arantzazu, Igorre',
    '48141' => 'Dima, Spain',
    '48142' => 'Artea',
    '48143' => 'Areatza',
    '48144' => 'Zeanuri',
    '48145' => 'Ubide',
    '48150' => 'Sondika',
    '48160' => 'Loiu, Derio',
    '48170' => 'Zamudio',
    '48180' => 'Loiu',
    (map { $_ => 'Sopuerta' } 48190..48190), # 48191 collides with Galdames
    '48191' => 'Galdames',
    '48192' => 'Gordexola',
    '48195' => 'Larrabetzu',
    '48196' => 'Lezama',
    '48200' => 'Durango, Biscay, Garai, Biscay|Garay',
    '48210' => 'Otxandio',
    '48211' => 'Abadiño',
    '48212' => 'Mañaria',
    '48213' => 'Izurtza',
    '48215' => 'Iurreta',
    '48220' => 'Abadiño',
    '48230' => 'Elorrio',
    '48240' => 'Berriz',
    # '48240' => 'Abadiño',
    '48250' => 'Zaldibar',
    '48260' => 'Ermua',
    '48269' => 'Mallabia',
    '48270' => 'Markina-Xemein, Ziortza-Bolibar',
    '48277' => 'Etxebarria',
    '48280' => 'Mendexa, Lekeitio',
    '48287' => 'Ea, Biscay',
    '48288' => 'Ispaster',
    '48289' => 'Mendexa, Amoroto, Gizaburuaga',
    '48291' => 'Atxondo',
    '48300' => 'Gernika-Lumo, Ajangiz',
    '48309' => 'Errigoiti',
    '48310' => 'Elantxobe',
    '48311' => 'Ibarrangelu',
    '48312' => 'Nabarniz',
    '48313' => 'Ereño',
    '48314' => 'Gautegiz Arteaga',
    '48315' => 'Kortezubi',
    '48320' => 'Ajangiz',
    '48330' => 'Lemoa',
    '48340' => 'Amorebieta-Etxano',
    '48350' => 'Busturia',
    '48360' => 'Mundaka',
    '48370' => 'Bermeo',
    '48380' => 'Aulesti',
    '48381' => 'Munitibar-Arbatzegi Gerrikaitz,',
    '48382' => 'Mendata',
    '48383' => 'Arratzu',
    '48390' => 'Bedia',
    '48392' => 'Muxika',
    '48393' => 'Kortezubi, Forua',
    '48394' => 'Murueta',
    '48395' => 'Sukarrieta',
    '48410' => 'Orozko',
    '48450' => 'Etxebarri',
    '48460' => 'Orduña',
    '48480' => 'Zaratamo, Arrigorriaga',
    '48490' => 'Ugao-Miraballes, Zeberio',
    '48498' => 'Arrankudiaga, Arakaldo',
    '48499' => 'Zeberio',
    '48500' => 'Abanto y Ciérbana-Abanto Zierbena',
    '48540' => 'Abanto y Ciérbana-Abanto Zierbena',
    '48508' => 'Ciérbana-Zierbena',
    '48510' => 'Valle de Trápaga-Trapagaran',
    '48520' => 'Valle de Trápaga-Trapagaran',
    '48530' => 'Ortuella',
    '48550' => 'Muskiz',
    '48600' => 'Sopelana',
    '48610' => 'Urduliz',
    '48620' => 'Lemoiz, Plentzia',
    '48630' => 'Gorliz',
    '48640' => 'Berango',
    '48650' => 'Barrika',
    '48700' => 'Ondarroa',
    '48710' => 'Berriatua',
    '48800' => 'Balmaseda',
    '48810' => 'Alonsotegi',
    '48820' => 'Güeñes',
    # '48830' => 'Güeñes',
    '48840' => 'Güeñes',
    '48830' => 'Sodupe',
    '48850' => 'Zalla',
    '48860' => 'Zalla',
    '48879' => 'Trucios-Turtzioz, Artzentales, Sopuerta',
    '48880' => 'Trucios-Turtzioz',
    (map { $_ => 'Karrantza-Valle de Carranza' } 48890..48891),
    '48895' => 'Lanestosa',
    '48901' => 'Barakaldo',
    '48902' => 'Barakaldo',
    '48903' => 'Barakaldo',
    '48910' => 'Sestao',
    '48920' => 'Portugalete',
    '48930' => 'Getxo',
    '48940' => 'Leioa',
    '48950' => 'Erandio',
    '48960' => 'Galdakao',
    '48970' => 'Basauri',
    '48980' => 'Santurtzi',
    '48991' => 'Getxo',
    '48992' => 'Getxo',
    '48993' => 'Getxo',
    '51002' => 'Ceuta',
    '51003' => 'Ceuta',
    '51004' => 'Ceuta',
    '51005' => 'Ceuta',
    '51070' => 'Ceuta',
    '51071' => 'Ceuta',
    '51081' => 'Ceuta',
    '52000' => 'Melilla',
    '52001' => 'Melilla',
    '52002' => 'Melilla',
    '52003' => 'Melilla',
    '52004' => 'Melilla',
    '52005' => 'Melilla',
    '52006' => 'Melilla',
    '52070' => 'Melilla',
    '52071' => 'Melilla',
    '52081' => 'Melilla',
); }

1;
