# Top-level make includes for GARStow ports.
#
# Copyright (C) 2001 Nick Moffitt
# Copyright (C) 2004, 2005, 2006, 2007, 2008, 2009, 2010 Adam Sampson
# Copyright (C) 2010 Brian Gough
#
# Redistribution and/or use, with or without modification, is
# permitted.  This software is without warranty of any kind.  The
# author(s) shall not be liable in the event that use of the
# software causes damage.

ifneq "$(strip $(USE))" ""
# Rather than including this file, pull in a library file instead.

USE_NEXT := $(firstword $(USE))
USE := $(wordlist 2,$(words $(USE)),$(USE))
include ../../gar.lib/$(USE_NEXT)

else
# No library files to include.

# Comment this out to make much verbosity
#.SILENT:


GARDIR ?= ../..
GARDIR_ABSOLUTE := $(shell cd $(GARDIR) && pwd)
SCRIPTSDIR = $(GARDIR_ABSOLUTE)/gar.scripts
TEMPDIR = $(GARDIR_ABSOLUTE)/gar.tmp
FILEDIR ?= files
DOWNLOADDIR ?= download
COOKIEDIR ?= cookies
WORKDIR ?= work
LOGDIR ?= logs
WORKSRC ?= $(WORKDIR)/$(DISTNAME)
EXTRACTDIR ?= $(WORKDIR)
SCRATCHDIR ?= tmp
CHECKSUM_FILE ?= sha256sums
CHECKSUM_CMD = $(if $(findstring sha256,$(CHECKSUM_FILE)),sha256sum,md5sum)
MANIFEST_FILE ?= manifest
GPG_KEYRING ?= ./gpg-keyring
packagesdir ?= $(prefix)/packages

# When we use WORKOBJ, we need to know if it's been changed from the default,
# and know how to get back out of it to the port dir.
WORKOBJ ?= $(WORKSRC)
WORKOBJ_CHANGED_P = $(filter-out $(WORKSRC),$(WORKOBJ))
LEAVE_WORKOBJ = $(subst $(strip ) ,/,$(patsubst %,..,$(subst /, ,$(WORKOBJ))))

UPSTREAMNAME ?= $(GARNAME)
DISTNAME ?= $(UPSTREAMNAME)-$(GARVERSION)

ALLFILES ?= $(DISTFILES) $(PATCHFILES) $(SIGFILES)

# For rules that do nothing, display what dependencies they
# successfully completed
DONADA = @echo "	[$@] complete.  Finished rules: $+"

# TODO: write a stub rule to print out the name of a rule when it
# *does* do something, and handle indentation intelligently.

# Default sequence for "all" is:  fetch checksum extract patch configure build
all: build
	$(DONADA)


#################### DIRECTORY MAKERS ####################

# This is to make dirs as needed by the base rules
$(sort $(DOWNLOADDIR) $(COOKIEDIR) $(LOGDIR) $(WORKSRC) $(WORKDIR) $(EXTRACTDIR) $(FILEDIR) $(SCRATCHDIR)):
	@if test -d $@; then : ; else \
		mkdir -p $@; \
	fi

$(COOKIEDIR)/%:
	@if test -d $@; then : ; else \
		mkdir -p $@; \
	fi


# include the configuration file to override any of these variables
include $(GARDIR)/config.mk
include $(GARDIR)/gar.conf.mk
include $(GARDIR)/gar.env.mk
include $(GARDIR)/gar.master.mk
include $(GARDIR)/gar.lib.mk
-include $(GARDIR)/gar.site.mk

# These stubs are wildcarded, so that the port maintainer can
# define something like "pre-configure" and it won't conflict,
# while the configure target can call "pre-configure" safely even
# if the port maintainer hasn't defined it.
# 
# in addition to the pre-<target> rules, the maintainer may wish
# to set a "pre-everything" rule, which runs before the first
# actual target.
pre-%:
	@true

post-%:
	@true

custom-%:
	@true

# ========================= MAIN RULES ========================= 
# The main rules are the ones that the user can specify as a
# target on the "make" command-line.  Currently, they are:
#	fetch-list fetch checksum makesum extract checkpatch patch
#	build install reinstall uninstall package
# (some may not be complete yet).
#
# Each of these rules has dependencies that run in the following
# order:
# 	- run the previous main rule in the chain (e.g., install
# 	  depends on build)
#	- run the pre- rule for the target (e.g., configure would
#	  then run pre-configure)
#	- generate a set of files to depend on.  These are typically
#	  cookie files in $(COOKIEDIR), but in the case of fetch are
#	  actual downloaded files in $(DOWNLOADDIR)
# 	- run the post- rule for the target
# 
# The main rules also run the $(DONADA) code, which prints out
# what just happened when all the dependencies are finished.

# fetch-list	- Show list of files that would be retrieved by fetch.
# NOTE: DOES NOT RUN pre-everything!
fetch-list:
	@echo "Name: $(GARNAME)"
	@echo "Version: $(GARVERSION)"
	@echo "Location: $(firstword $(MASTER_SITES))$(MASTER_SUBDIR)"
	@echo "Distribution files: "
	@for i in $(DISTFILES); do echo "	$$i"; done
	@echo "Patch files: "
	@for i in $(PATCHFILES); do echo "	$$i"; done
	@echo "Signature files: "
	@for i in $(SIGFILES); do echo "	$$i"; done
	@echo "Dependencies: "
	@for i in $(LIBDEPS) $(BUILDDEPS) $(if $(USE_TESTS),$(TESTDEPS),) ; do echo "	$$i"; done

# fetch			- Retrieves $(DISTFILES) (and $(PATCHFILES) if defined)
#				  into $(DOWNLOADDIR) as necessary.
fetch: pre-everything custom-pre-everything $(COOKIEDIR) $(DOWNLOADDIR) $(addprefix dep-$(GARDIR)/,$(FETCHDEPS)) pre-fetch custom-pre-fetch $(addprefix $(DOWNLOADDIR)/,$(ALLFILES)) post-fetch custom-post-fetch
	$(DONADA)

# returns true if fetch has completed successfully, false
# otherwise
fetch-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/dep-$(GARDIR)/,$(FETCHDEPS)), test -e $(COOKIEFILE) ;)

# checksum		- Use $(CHECKSUMFILE) to ensure that your
# 				  distfiles are valid.
checksum: fetch pre-checksum custom-pre-checksum $(addprefix checksum-,$(filter-out $(NOCHECKSUM),$(ALLFILES))) post-checksum custom-post-checksum
	$(DONADA)

# returns true if checksum has completed successfully, false
# otherwise
checksum-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/checksum-,$(filter-out $(NOCHECKSUM),$(ALLFILES))), test -e $(COOKIEFILE) ;)

# makesum		- Generate distinfo (only do this for your own ports!).
makesum: fetch $(addprefix $(DOWNLOADDIR)/,$(filter-out $(NOCHECKSUM),$(ALLFILES))) checksig
	@if test "x$(addprefix $(DOWNLOADDIR)/,$(filter-out $(NOCHECKSUM),$(ALLFILES)))" != "x"; then \
		$(CHECKSUM_CMD) $(addprefix $(DOWNLOADDIR)/,$(filter-out $(NOCHECKSUM),$(ALLFILES))) > $(CHECKSUM_FILE); \
	fi

# I am always typing this by mistake
makesums: makesum

# checksig      - Use signatures in $(SIGFILES) to check downloaded files.
checksig: fetch $(COOKIEDIR) pre-checksig custom-pre-checksig $(addprefix checksig-,$(SIGFILES)) post-checksig custom-post-checksig
	$(DONADA)

checksig-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/checksig-,$(SIGFILES)), test -e $(COOKIEFILE) ;)

# garchive      - Copy downloaded files to $(GARCHIVEDIR)
garchive: checksum $(COOKIEDIR) pre-garchive custom-pre-garchive $(addprefix garchive-,$(ALLFILES))  post-garchive custom-post-garchive
	$(DONADA)

# extract		- Unpacks $(DISTFILES) into $(EXTRACTDIR) (patches are "zcatted" into the patch program)
extract: garchive $(EXTRACTDIR) $(COOKIEDIR) $(addprefix dep-$(GARDIR)/,$(EXTRACTDEPS)) pre-extract custom-pre-extract $(addprefix extract-,$(filter-out $(NOEXTRACT),$(DISTFILES))) post-extract custom-post-extract
	$(DONADA)

# returns true if extract has completed successfully, false
# otherwise
extract-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/extract-,$(filter-out $(NOEXTRACT),$(DISTFILES))), test -e $(COOKIEFILE) ;)

# patch			- Apply any provided patches to the source.
patch: extract $(WORKSRC) pre-patch custom-pre-patch $(addprefix patch-,$(PATCHFILES)) post-patch custom-post-patch
	$(DONADA)

# returns true if patch has completed successfully, false
# otherwise
patch-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/patch-,$(PATCHFILES)), test -e $(COOKIEFILE) ;)

# makepatch		- Grab the upstream source and diff against $(WORKSRC).  Since
# 				  diff returns 1 if there are differences, we remove the patch
# 				  file on "success".  Goofy diff.
makepatch: $(SCRATCHDIR) $(FILEDIR) $(FILEDIR)/gar-base.diff
	$(DONADA)

# this takes the changes you've made to a working directory,
# distills them to a patch, updates the checksum file, and tries
# out the build (assuming you've listed the gar-base.diff in your
# PATCHFILES).  This is way undocumented.  -NickM
beaujolais: makepatch makesum clean build
	$(DONADA)

# Provide PRE_ and POST_ variables for each of the scripts types.
# This is because it's useful to default the main _SCRIPTS variables to
# something (often something complicated) in a library .mk file, but some ports
# still want to have custom scripts.
ALL_CONFIGURE_SCRIPTS = $(PRE_CONFIGURE_SCRIPTS) $(CONFIGURE_SCRIPTS) $(POST_CONFIGURE_SCRIPTS)
ALL_BUILD_SCRIPTS = $(PRE_BUILD_SCRIPTS) $(BUILD_SCRIPTS) $(POST_BUILD_SCRIPTS)
ALL_TEST_SCRIPTS = $(PRE_TEST_SCRIPTS) $(TEST_SCRIPTS) $(POST_TEST_SCRIPTS)
ALL_INSTALL_SCRIPTS = $(PRE_INSTALL_SCRIPTS) $(INSTALL_SCRIPTS) $(POST_INSTALL_SCRIPTS)

# configure		- Runs either GNU configure, one or more local
# 				  configure scripts or nothing, depending on
# 				  what's available.
configure: patch $(addprefix dep-$(GARDIR)/,$(LIBDEPS) $(BUILDDEPS) $(if $(USE_TESTS),$(TESTDEPS),)) $(addprefix depcheck-,$(filter-out $(DEPCHECK_IGNORE),$(DEPCHECK_TARGETS))) pre-configure custom-pre-configure $(addprefix configure-,$(ALL_CONFIGURE_SCRIPTS)) post-configure custom-post-configure
	$(DONADA)

# returns true if configure has completed successfully, false
# otherwise
configure-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/configure-,$(ALL_CONFIGURE_SCRIPTS)), test -e $(COOKIEFILE) ;)

# build			- Actually compile the sources.
build: configure pre-build custom-pre-build $(addprefix build-,$(ALL_BUILD_SCRIPTS)) post-build custom-post-build
	$(DONADA)

# returns true if build has completed successfully, false
# otherwise
build-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/build-,$(ALL_BUILD_SCRIPTS)), test -e $(COOKIEFILE) ;)

# test          - Run a package's self-tests
test: build pre-test custom-pre-test $(addprefix test-,$(ALL_TEST_SCRIPTS)) post-test custom-post-test
	$(DONADA)

# I am always typing this by mistake - BJG
check: test

# returns true if test has completed successfully, false
# otherwise
test-p:
	@$(foreach COOKIEFILE,$(addprefix $(COOKIEDIR)/test-,$(ALL_TEST_SCRIPTS)), test -e $(COOKIEFILE) ;)

# install		- Install the results of a build.
install: $(if $(USE_TESTS),test,build) $(addprefix dep-$(GARDIR)/,$(INSTALLDEPS)) prepare-install pre-install custom-pre-install $(addprefix install-,$(ALL_INSTALL_SCRIPTS)) finish-install pre-stow custom-pre-stow $(addprefix sysinstall-,$(filter-out $(SYSINSTALL_IGNORE),$(SYSINSTALL_TARGETS))) post-install custom-post-install
	$(DONADA)

# returns true if install has completed successfully, false
# otherwise
install-p:
	@test -h $(dotgardir)/VERSION && test "`readlink $(dotgardir)/VERSION`" = $(PACKAGENAME) && echo $(PACKAGENAME) is installed

reinstall-p:
	@test -d $(packagesdir)/$(PACKAGENAME) && echo $(PACKAGENAME) is available to be reinstalled

reinstall: $(addprefix dep-$(GARDIR)/,$(LIBDEPS)) $(addprefix sysinstall-,$(filter-out $(SYSINSTALL_IGNORE),$(SYSINSTALL_TARGETS)))

# uninstall		- Remove the installation.
uninstall: sysinstall-uninstall
	$(DONADA)

uninstall-pkg: sysinstall-uninstall-pkg
	$(DONADA)

# Update a package: clean out the old stuff, download a new version,
# rebuild the checksums and install it.
update:
	$(MAKE) clean
	$(MAKE) makesums
	$(MAKE) install

# Install a package only if the source version is not the currently-installed
# version.
maybe-install:
	@$(MAKE) install-p >/dev/null 2>&1 || $(MAKE) install

# Check in a trivially-updated package.
ci:
	@$(SCRIPTSDIR)/gar-lint
	@$(WITH_LOCK)darcs darcs record -a -m '$(GARNAME) $(GARVERSION).' .

# Automatically find dependencies (after a configure).
find-deps:
	@$(PKG_CONFIG) --fpc-dump

check-deps:
	@$(PKG_CONFIG) --fpc-check $(GARNAME)

# Some extra dependency rules
DEPS=$(LIBDEPS) $(BUILDDEPS) $(if $(USE_TESTS),$(TESTDEPS),)

dep-list:
	@echo $(DEPS) 

checkdeps: $(addprefix checkdep-$(GARDIR)/,$(LIBDEPS) $(BUILDDEPS) $(if $(USE_TESTS),$(TESTDEPS),) $(INSTALLDEPS)) 
	$(DONADA)

checkdep-$(GARDIR)/%:
	$(MAKE) -C $(call pathsearch,$*,$(DEPPATH)) fetch-list

install-deps: $(addprefix dep-$(GARDIR)/,$(LIBDEPS) $(BUILDDEPS) $(if $(USE_TESTS),$(TESTDEPS),) $(INSTALLDEPS)) 
	$(DONADA)

# The clean rule.  It must be run if you want to re-download a
# file after a successful checksum (or just remove the checksum
# cookie, but that would be lame and unportable).
clean:
	@rm -rf $(DOWNLOADDIR) $(COOKIEDIR) $(LOGDIR) $(WORKSRC) $(WORKDIR) $(EXTRACTDIR) $(SCRATCHDIR) $(SCRATCHDIR)-$(COOKIEDIR) *~

# these targets do not have actual corresponding files
.PHONY: all fetch-list fetch checksum makesum garchive extract patch makepatch configure build test install clean beaujolais update

# A logging version of the rules, e.g. make build-log, respawns and sends output to $(LOGDIR)
%-log: $(LOGDIR)
	$(MAKE) $*-p || $(MAKE) $* >$(LOGDIR)/$*.log 2>&1

test-logs:
	@for f in `find . -name 'test*.log'` ; do \
		echo -e "===> test log start: $$f <===" ; \
		cat $$f ; \
		echo -e "===> test log end: $$f <===\n" ;  \
	done

.PHONY: $(addsuffix fetch checksum extract patch configure build test install,-log)

.NOTPARALLEL:

endif
