// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccscript.
// 
// The exception is that, if you link the ccscript library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccscript library code into it.
// 
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccscript.  If you copy code from other releases into a copy of
// ccscript, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccscript, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#include <cc++/config.h>
#include <cc++/misc.h>
#include <cc++/slog.h>
#include <cc++/url.h>
#include <cc++/export.h>
#include <cstdlib>
#include <cstdio>
#include "script.h"

#ifdef	USE_REGEX
#include <regex.h>
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using namespace std;
#endif

static long tens[] = {1, 10, 100, 1000, 10000, 100000, 1000000, 10000000};

Script::Attr *ScriptInterp::attr = NULL;
Script::Test *ScriptInterp::test = NULL;
Script::Fun *ScriptInterp::ifun = NULL;

static void logerror(const char *script, unsigned id, const char *msg)
{
	slog(Slog::levelError) << script;
	if(id)
		slog() << "(" << id << ")";
	slog() << ": " << msg << endl;
}

static void adjustValue(char *buffer, int diff)
{
	int value = atoi(buffer);

	sprintf(buffer, "%d", value + diff);
}

ScriptInterp::ScriptInterp(ScriptCommand *cmdset, size_t sym, size_t pg) :
ScriptSymbol(sym, pg)
{
	session = NULL;
	once = true;
	loop = true;
	signalmask = 0;
	stack = 0;
	cmd = cmdset;
	image = NULL;
	memset(temps, 0, sizeof(temps));

	for(tempidx = 0; tempidx < SCRIPT_TEMP_SPACE; ++tempidx)
		temps[tempidx] = new char[getSymbolSize() + 1];
	tempidx = 0;
	symsize = sym;
	pgsize = pg;
};

ScriptInterp::~ScriptInterp()
{
	for(tempidx = 0; tempidx < SCRIPT_TEMP_SPACE; ++tempidx)
		if(temps[tempidx])
			delete[] temps[tempidx];
}

char ScriptInterp::getPackToken(void)
{
	char *sym = getSymbol("script.token");
	if(!sym)
		sym = ",";

	if(!*sym)
		sym = ",";

	return *sym;
}

Script::Name *ScriptInterp::getScriptCopy(const char *name)
{
	char buffer[256];
	Name *scr;

	snprintf(buffer, 255, "%s::%p", name, this);
	scr = image->dupScript(name, buffer);
	if(scr && !strcmp(script[stack].script->name, name))
		script[stack].script = scr;
	return scr;
}

unsigned long ScriptInterp::getMask(void)
{
	return script[stack].line->mask & script[stack].script->mask;
}

Script::Name *ScriptInterp::getScriptImage(const char *name)
{
	char buffer[256];
	Name *scr = image->getScript(name);

	if(!scr)
		return NULL;

	if(scr->mode == Name::COPIED)
	{
		snprintf(buffer, 255, "%s::%p", name, this);
		scr = image->getScript(buffer);
	}
	return scr;
}

void ScriptInterp::getTrigger(bool jump)
{
	char buffer[256];
	Symbol *sym = NULL;
	Name *scr;
	const char *id;

	if(script[stack].local)
		sym = script[stack].local->getTrigger();

	if(sym)
		ScriptSymbol::getTrigger();
	else
		sym = ScriptSymbol::getTrigger();
	if(!jump || !sym)
		return;

	id = strchr(sym->id, '.');
	if(id)
		++id;
	else
		id = sym->id;

	setSymbol("script.trigger", sym->data);
	snprintf(buffer, 255, "%s::%s_%s", script[stack].script->name,
		id, sym->data);

	scr = getScriptImage(buffer);
	if(!scr)
	{
		snprintf(buffer, 256, "%s::%s",  script[stack].script->name, id);
		scr = getScriptImage(buffer);
	}
	if(!scr)
	{
		snprintf(buffer, 256, "%s::%s", id, sym->data);
		scr = getScriptImage(buffer);
	}
	if(scr)
	{
		once = true;
		script[stack].tranflag = script[stack].caseflag = false;
		script[stack].script = scr;
		script[stack].line = scr->first;
		script[stack].index = 0;
	}
}

bool ScriptInterp::scrEnable(void)
{
	char buffer[256];
	char *name = getKeyword("name");
	Name *scr;
	unsigned long mask, id;
	char *cp;

	if(!name)
		name = getValue(NULL);

	if(!name)
	{
		advance();
		return true;
	}

	if(!strncmp(name, "::", 2))
	{
		strcpy(buffer, script[stack].script->name);
		cp = strstr(buffer, "::");
		if(cp)
			*cp = 0;
		strcat(buffer, name);
		name = buffer;
	}
	scr = getScriptCopy(name);
	if(!scr)
	{
		error("script-not-found");
		return true;
	}

	while(NULL != (name = getValue(NULL)))
	{
		id = cmd->getTrapId(name);
		mask = cmd->getTrapMask(name);
		if(!mask)
		{
			error("handler-invalid");
			return true;
		}
		if(!scr->trap[id])
		{
			error("handler-not-found");
			return true;
		}
		scr->mask |= mask;
	}
	advance();
	return true;
}

bool ScriptInterp::scrDisable(void)
{
	char buffer[256];
	char *name = getKeyword("name");
	Name *scr;
	unsigned long mask, id;
	char *cp;

	if(!name)
		name = getValue(NULL);

	if(!name)
	{
		advance();
		return true;
	}

	if(!strncmp(name, "::", 2))
	{
		strcpy(buffer, script[stack].script->name);
		cp = strstr(buffer, "::");
		if(cp)
			*cp = 0;
		strcat(buffer, name);
		name = buffer;
	}
	scr = getScriptCopy(name);
	if(!scr)
	{
		error("script-not-found");
		return true;
	}

	while(NULL != (name = getValue(NULL)))
	{
		id = cmd->getTrapId(name);
		mask = cmd->getTrapMask(name);
		if(!mask)
		{
			error("handler-invalid");
			return true;
		}
		if(!scr->trap[id])
		{
			error("handler-not-found");
			return true;
		}
		scr->mask &= ~mask;
	}
	advance();
	return true;
}

void ScriptInterp::rewindTemp(void)
{
	tempidx = 0;
}

void ScriptInterp::setTemp(const char *value)
{
	char *rts = temps[tempidx];
	snprintf(temps[tempidx++], getSymbolSize() + 1, "%s", value);
	if(tempidx >= SCRIPT_TEMP_SPACE)
		tempidx = 0;
}

void ScriptInterp::advance(void)
{
	script[stack].line = script[stack].line->next;
	if(!script[stack].line && loop)
		script[stack].line = script[stack].script->first;
}

void ScriptInterp::error(const char *errmsg)
{
	char evtname[80];
	setSymbol("script.error", errmsg);
	snprintf(evtname, sizeof(evtname), "error:%s", errmsg);
	if(event(evtname))
		return;

	if(script[stack].script->mask & 0x02)
		trap(1);
	else
		advance();
}

void ScriptInterp::trap(const char *trapid)
{
	unsigned trap = cmd->getTrapId(trapid);
	if(!trap)
	{
		if(!image)
			return;

		if(!stricmp(trapid, "first") || !stricmp(trapid, "top"))
		{
			script[stack].tranflag = script[stack].caseflag = false;
			script[stack].line = script[stack].script->first;
			return;
		}
	}
	ScriptInterp::trap(trap);
}

void ScriptInterp::trap(unsigned id)
{
	Line *trap = NULL;

	if(!image)
		return;

	if(getGlobalTrap(id))
		return;

retry:
	trap = script[stack].script->trap[id];
	if(!trap && id)
	{
		advance();
		return;
	}
	if(!trap && !id && stack)
	{
		pull();
		goto retry;
	}

	script[stack].tranflag = script[stack].caseflag = false;
	script[stack].line = trap;
}

bool ScriptInterp::step(const char *trapname)
{
	unsigned long mask, cmask;
	bool rtn, sync, skip = false;
	Symbol *sym = NULL;

	if(!image)
		return true;

	script[stack].index = 0;

	if(session)
	{
		session->waitHandler();
		delete session;
		session = NULL;
	}

	if(trapname)
	{
		getTrigger(false);
		mask = cmd->getTrapMask(trapname);
	}
	else
	{
		mask = 0;
		getTrigger(true);
	}

	// auto return code

retry:
	while(!script[stack].line && stack)
	{
		if(script[stack - 1].local == script[stack].local)
			break;
		pull();
		if(script[stack].line)
		{
			advance();
			skip = true;
		}
	}

	if(!script[stack].line)
	{
		exit();
		return false;
	}
	

	cmask = script[stack].line->cmask;
	if(cmask)
	{
		if((cmask & script[stack].script->mask) != cmask)
		{
			skip = true;
			advance();
			goto retry;
		}
	}

	if(mask & script[stack].line->mask)
	{
		trap(trapname);
		return true;
	}

/*
	if(script[stack].line->prescan && skip)
		return true;
*/

	if(script[stack].line->error)
		sym = getEntry("script.error");
	if(sym)
		sym->flags.type = TRIGGER;
	sync = script[stack].line->sync;
	if(sync)
		setExclusive(true);

trans:
	rtn = execute(script[stack].line->method);
	if(script[stack].tranflag && rtn && !sync && script[stack].line)
	{
		script[stack].index = 0;
		goto trans;
	}

	if(sync)
		setExclusive(false);
	if(sym)
		sym->flags.type = NORMAL;

	return rtn;
}

bool ScriptInterp::event(const char *name)
{
	Name::Event *evt = script[stack].script->events;
	
	while(evt)
	{
		if(!stricmp(evt->name, name))
			break;
		evt = evt->next;
	}
	if(!evt)
	{
		name = strchr(name, ':');
		if(name)
			return event(++name);
		return false;
	}

	script[stack].tranflag = script[stack].caseflag = false;
	script[stack].line = evt->line;
	return true;
}

bool ScriptInterp::signal(const char *trapname)
{
	if(!image)
		return true;

	unsigned long mask = cmd->getTrapMask(trapname);
	mask &= script[stack].line->mask;
	mask &= script[stack].script->mask;
	if(!mask)
		return false;

	stop(mask);
	trap(trapname);
	return true;
}

bool ScriptInterp::signal(unsigned id)
{
	if(!image)
		return true;

	if(id >= TRAP_BITS)
		return false;

	unsigned long mask = cmd->getTrapMask(id);
	mask &= script[stack].line->mask;
	if(!mask)
	{
		signalmask |= id;
		return false;
	}

	stop(mask);
	trap(id);
	return true;
}

Script::Symbol *ScriptInterp::getLocal(const char *id, size_t size)
{
	Symbol *sym = NULL;

	if(script[stack].local)
		sym = script[stack].local->getEntry(id, size);

	if(!sym)
		sym = getEntry(id, size);

	return sym;
}

bool ScriptInterp::scrSkip(void)
{
	char *val;
	char *id = getValue(NULL);
	Line *line;
	int argc;
	bool cf = false; 

	if(!id)
	{
		advance();
		if(script[stack].line)
			if(script[stack].line->method == &ScriptInterp::scrCase)
				cf = true;

		advance();
		while(script[stack].line && cf)
		{
			if(script[stack].line->method != &ScriptInterp::scrCase)
				break;

			advance();
		}
		return true;
	}

	script[stack].line = script[stack].script->first;
	while(NULL != (line = script[stack].line))
	{
		advance();
		if(line->method != &ScriptInterp::scrLabel)
			continue;

		argc = 0;
		while(argc < line->argc)
		{
			val = getContent(line->args[argc++]);
			if(!stricmp(val, id))
				return true;
		}
	}
	return true;
}

bool ScriptInterp::scrData(void)
{
	while(script[stack].line->method == &ScriptInterp::scrData)
	{
		advance();
		if(!script[stack].line)
			return true;
	}
	return true;
}

bool ScriptInterp::scrSlog(void)
{
	unsigned id = getId();
	const char *member = getMember();
	char *val;

	if(!member)
		member = getKeyword("level");

	if(member)
	{
		if(!strnicmp(member, "err", 3))
			slog(Slog::levelError);
		else if(!strnicmp(member, "warn", 4))
			slog(Slog::levelWarning);
		else if(!stricmp(member, "debug"))
			slog(Slog::levelDebug);
		else if(!strnicmp(member, "crit", 4))
			slog(Slog::levelCritical);
		else
			slog(Slog::levelInfo);
	}
	else
		slog(Slog::levelInfo);

	slog() << script[stack].script->name;
	if(id)
		slog() << "(" << id << ")";
	slog() << ": ";
	while(NULL != (val = getValue(NULL)))
		slog() << val;
	slog() << endl;
	advance();
	return true;
}

bool ScriptInterp::scrUse(void)
{
#ifdef	HAVE_MODULES
	advance();
#else
	error("missing-dynloader-support");
#endif
	return true;
}

bool ScriptInterp::scrSelect(void)
{
	const char *value = getKeyword("id");
	const char *match = getMember();
	int len, l2;
	int offset = 0;

	enum
	{
		SEL_NORMAL,
		SEL_LENGTH,
		SEL_PREFIX,
		SEL_SUFFIX,
		SEL_DECIMAL,
		SEL_VALUE
	} sm = SEL_NORMAL;

	if(!match)
		match = getKeyword("match");

	if(!value)
		value = getValue(NULL);

	if(!value)
	{
		error("branch-no-select");
		return true;
	}

	if(match)
	{
		if(!strnicmp(match, "len", 3))
			sm = SEL_LENGTH;
		else if(!strnicmp(match, "pre", 3))
			sm = SEL_PREFIX;
		else if(!strnicmp(match, "suf", 3))
			sm = SEL_SUFFIX;
		else if(!strnicmp(match, "val", 3))
			sm = SEL_VALUE;
		else if(!strnicmp(match, "end", 3))
			sm = SEL_SUFFIX;
		else if(!strnicmp(match, "dec", 3))
			sm = SEL_DECIMAL;
		else
		{
			offset = atoi(match);
			sm = SEL_PREFIX;
		}
	}
	len = strlen(value);
	if(offset < 0)
		offset = len + offset;
	if(offset < 0)
		offset = 0;
	if(offset > len)
		offset = len;

	while(NULL != (match = getValue(NULL)))
	{
		l2 = strlen(match);
		switch(sm)
		{
		case SEL_SUFFIX:
			if(len < l2)
				break;
			if(!mapicmp(match, value + len - l2))
				return scrGoto();
			break;
		case SEL_PREFIX:
			if(!mapnicmp(match + offset, value, l2))
				return scrGoto();
			break;
		case SEL_NORMAL:
			if(!mapicmp(match, value))
				return scrGoto();
			break;
		case SEL_LENGTH:
			if(atoi(match) == len)
				return scrGoto();
			break;
		case SEL_DECIMAL:
			match = strchr(match, '.');
			value = strchr(value, '.');

			if(!match)
				match = "0";
			else
				++match;

			if(!value)
				value = "0";
			else
				++value;

		case SEL_VALUE:
			if(atoi(match) == atoi(value))
				return scrGoto();
			break;
		}
		getValue(NULL);
	}
	advance();
	return true;
}

bool ScriptInterp::scrOnce(void)
{
	if(getOnce())
		return scrGoto();
	advance();
	return true;
}

bool ScriptInterp::scrOn(void)
{
	char *sig = getValue(NULL);
	unsigned long mask;

	if(!sig)
	{
		error("on-no-signal");
		return true;
	}

	mask = cmd->getTrapMask(sig);
	if(!mask)
	{
		error("on-invalid-signal");
		return true;
	}
	if(signalmask & mask)
	{
		signalmask &= ~mask;
		return scrGoto();
	}
	advance();
	return true;
}

bool ScriptInterp::scrGoto(void)
{
	char namebuf[256];
	char *label = getOption(NULL);
	char *ext;
	int len;
	bool pvt = true;
	Name *scr;

	if(label && *label != '@')
		label = getContent(label);

	if(!label)
	{
		error("branch-failed");
		return true;
	}

	if(*label == '^')
	{
		initKeywords(0);
		trap(++label);
		return true;
	}

	if(*label == '@')
	{
		if(event(++label))
		{
			initKeywords(0);
			return true;
		}
		advance();
		return true;
	}

	len = strlen(label);
	if(!strncmp(label, "::", 2))
	{
		pvt = false;
		strcpy(namebuf, script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			strcpy(ext, label);
		else
			strcat(namebuf, label);
		label = namebuf;
	}
	else if(label[len - 1] == ':')
	{
		pvt = false;
		strcpy(namebuf, script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			strcpy(ext + 2, label);
		else
		{
			strcat(namebuf, "::");
			strcat(namebuf, label);
		}
		label = namebuf;
		len = strlen(label);
		label[len - 1] = 0;
	}

	scr = getScriptImage(label);
	if(!scr)
	{
		error("script-not-found");
		return true;
	}
	if(pvt && !scr->access)
	{
		error("script-private");
		return true;
	}
	if(scr->mode == Name::DATA)
	{
		error("script-data");
		return true;
	}
	initKeywords(0);
	once = true;

	script[stack].caseflag = script[stack].tranflag = false;
	script[stack].script = scr;
	script[stack].line = scr->first;
	script[stack].index = 0;
	return true;
}

bool ScriptInterp::scrTry(void)
{
	char namebuf[256];
	Name *scr;
	char *label, *ext;

	while(NULL != (label = getValue(NULL)))
	{
		if(!strncmp(label, "::", 2))
		{
			strcpy(namebuf, script[stack].script->name);
			ext = strstr(namebuf, "::");
			if(ext)
				strcpy(ext, label);
			else
				strcat(namebuf, label);
			label = namebuf;
		}
		scr = getScriptImage(label);
		if(!scr)
			continue;
		once = true;
		script[stack].caseflag = script[stack].tranflag = false;
		script[stack].script = scr;
		script[stack].line = scr->first;
		script[stack].index = 0;
		return true;
	}
	advance();
	return true;
}

bool ScriptInterp::scrArm(void)
{
	Symbol *opt;

	while(NULL != (opt = getVariable(0)))
	{
		if(opt->flags.type == NORMAL)
			opt->flags.type = TRIGGER;
	}
	advance();
	return true;
}

bool ScriptInterp::scrDisarm(void)
{
	Symbol *opt;

	while(NULL != (opt = getVariable(0)))
	{
		if(opt->flags.type == TRIGGER)
			opt->flags.type = NORMAL;
	}
	advance();
	return true;
}

bool ScriptInterp::scrSwap(void)
{
	Symbol *s1, *s2;
	s1 = getVariable(getSymbolSize());
	s2 = getVariable(getSymbolSize());
	bool b1 = false, b2 = false;

	if(!s1 || !s2)
	{
		error("symbol-not-found");
		return true;
	}
	if(script[stack].local)
	{
		if(strchr(s1->id, '.'))
			b1 = true;

		if(strchr(s2->id, '.'))
			b2 = true;

		if(b1 != b2)			
		{
			error("symbol-mixed-contexts");
			return true;
		}	
		if(!b1)
		{
			script[stack].local->swapSymbol(s1->id, s2->id);
			advance();
			return true;
		}
	}
	swapSymbol(s1->id, s2->id);
	advance();
	return true;
}

bool ScriptInterp::scrMap(void)
{
	Line *line;
	Name *scr;
	char *cp;
	const char *prefix = getMember();
	char *value = getKeyword("table");
	char namebuf[256];
	unsigned len;
	bool find = false;

	enum
	{
		MAP_PREFIX,
		MAP_SUFFIX,
		MAP_ABSOLUTE,
		MAP_VALUE
	}	mapmode = MAP_ABSOLUTE;

	if(!prefix)
		prefix = getKeyword("match");

	if(prefix)
        {
                if(!strnicmp(prefix, "pre", 3))
                        mapmode = MAP_PREFIX;
                else if(!strnicmp(prefix, "suf", 3))
                        mapmode = MAP_SUFFIX;
		else if(!strnicmp(prefix, "end", 3))
			mapmode = MAP_SUFFIX;
		else if(!strnicmp(prefix, "val", 3))
			mapmode = MAP_VALUE;
                else if(!strnicmp(prefix, "abs", 3))
                        mapmode = MAP_ABSOLUTE;
        }

	if(!value)
		value = script[stack].script->name;
	else if(!strnicmp(value, "::", 2))
        {
                strcpy(namebuf, script[stack].script->name);
                strcat(namebuf, value);
                value = namebuf;
        }

        scr = getScriptImage(value);
        if(!scr)
        {
                error("no-source-to-read");
                return true;
        }
	line = scr->first;

	value = getValue("*");
	len = strlen(value);
	
	while(line)
	{
		if(line->method != &ScriptInterp::scrData)
		{
			line = line->next;
			continue;
		}	
		cp = strchr(line->cmd, '.');
		if(!cp)
		{
			line = line->next;
			continue;
		}
		++cp;

		switch(mapmode)
		{
		case MAP_VALUE:
			if(atol(cp) == atol(value))
				find = true;
			break;
		case MAP_ABSOLUTE:
			if(!mapicmp(cp, value))
				find = true;
			break;
		case MAP_PREFIX:
			if(!mapnicmp(cp, value, len))
				find = true;
			break;
		case MAP_SUFFIX:
			if(strlen(cp) > len)
				break;

			cp = cp + strlen(cp) - len;
			if(!mapicmp(cp, value))
				find = true;
			break;
		}
						
		if(find)
			break;

		line = line->next;
	}

	if(line)
	{
		setLine(line);
		return scrGoto();		
	}

	error("no-map-data");
	return true;
}	

bool ScriptInterp::setData(const char *name)
{
	char namebuf[256];
	Name *scr;
	char *tok;

	if(!name)
		name = script[stack].script->name;
	
	if(!strnicmp(name, "::", 2))
	{
		strcpy(namebuf, script[stack].script->name);
		tok = strstr(namebuf, "::");
		if(!tok)
			tok = namebuf + strlen(namebuf);
		strcpy(tok, name);
		name = namebuf;
	}
	scr = getScriptImage(name);
	if(!scr)
		return false;

	script[stack].read = scr->first;
	return true;
}

bool ScriptInterp::scrRead(void)
{
	Symbol *sym = NULL;
	Line *rd;
	const char *mem = getMember();
	int argc = 0;
	char *value = NULL;
	unsigned offset = 0;
	bool packed = false;
	unsigned size = getSymbolSize();
	const char *sz = getKeyword("size");
	char packtoken = ',';
	unsigned row = 0;
	unsigned col = 0;
	unsigned count = 0;

	if(!mem)
		mem = "";

	if(sz)
		size = atoi(sz);

	if(!strnicmp(mem, "pack", 4))
	{
		packed = true;		
		packtoken = *getSymbol("script.token");
		value = getKeyword("token");
		if(value)
			packtoken = *value;
	}

	if(!stricmp(mem, "from"))
	{
		value = getKeyword("table");
		if(!value)
			value = getValue(script[stack].script->name);

		if(!setData(value))
		{
			error("no-source-to-read");
			return true;
		}
		advance();
		return true;
	}

	value = getKeyword("row");
	if(value)
		row = atoi(value);
	
	value = getKeyword("col");
	if(value)
		col = atoi(value);

	value = getKeyword("count");
	if(value)
		count = atoi(value);

	value = getKeyword("from");
	if(!value)
		value = getKeyword("table");

	if(value)
		if(!setData(value))
		{
			error("no-source-to-read");
			return true;
		}
	
rows:
	rd = script[stack].read;
	while(rd)
	{
		if(rd->method == &ScriptInterp::scrData)
			break;
		rd = rd->next;
	}
	if(!rd)
	{
		script[stack].read = NULL;
		error("end-of-data");
		return true;
	}
	else
		script[stack].read = rd->next;

	if(row--)
		goto rows;
	
	while(argc < rd->argc)
	{
		value = getContent(rd->args[argc++]);
		if(!value)
			break;
		
		if(col)
		{
			--col;
			continue;
		}

		if(!packed || !sym)
		{
			if(packed || sz)
				sym = getVariable(size);
			else
				sym = getVariable(strlen(value));
			if(!sym)
				break;
			if(sym->flags.readonly)
				continue;

			if(packed)
				sym->data[0] = 0;
		}

		if(packed)
		{
			if(offset)
				sym->data[offset++] = packtoken;
			snprintf(sym->data + offset, sym->flags.size - offset, "%s", value);
			offset = strlen(sym->data);
		}
		else
		{
			snprintf(sym->data, sym->flags.size + 1, "%s", value);
			sym->flags.initial = false;
			if(sym->flags.commit)
				commit(sym);
		}
		if(count)
			if(!--count)
				break;
	}
	if(sym && packed)
	{
		sym->flags.initial = false;
		if(sym->flags.commit)
			commit(sym);
	}

	advance();
	return true;
}

bool ScriptInterp::scrRepeat(void)
{
	unsigned short loop = script[stack].line->loop;
	Line *line;
	int index = script[stack].index;
	int count;

	script[stack].index = 0;
	count = atoi(getValue("0"));

	if(index >= count)
	{
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}
	script[stack].index = ++index;
	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrFordata(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	int index = script[stack].index;
	char *table = getKeyword("table");
	Line *rd, *line;
	int argc = 0;
	unsigned size = getSymbolSize();
	const char *sz = getKeyword("size");
	const char *value;

	if(sz)
		size = atoi(sz);

	script[stack].index = 0;
	if(!table)
		table = getValue(NULL);

	if(!index)
	{
		if(!setData(table))
			script[stack].read = NULL;
	}
	rd = script[stack].read;
	while(rd)
	{
		if(rd->method == &ScriptInterp::scrData)
			break;
		rd = rd->next;
	}		

	if(!rd)
	{
                line = script[stack].line->next;
                while(line)
                {
                        if(line->loop == loop)
                        {
                                script[stack].line = line;
                                advance();
                                return true;
                        }
                        line = line->next;
                }
                error("loop-overflow");
                return true;
	}

	script[stack].read = rd->next;
	while(argc < rd->argc)
	{
		value = getContent(rd->args[argc++]);
		if(!value)
			break;
		
		if(sz)
			sym = getVariable(size);
		else
			sym = getVariable(strlen(value));
		
		if(!sym)
			break;
		if(sym->flags.readonly)
			continue;

		snprintf(sym->data, sym->flags.size + 1, "%s", value);
		sym->flags.initial = false;
		if(sym->flags.commit)
			commit(sym); 
	}
	if(!push())
		return true;

	advance();
	return true;			
}	

bool ScriptInterp::scrForeach(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	Line *line;
	int index = script[stack].index;
	const char *value = getMember();
	char packtoken = getPackToken();
	unsigned len = 0;
	char *kw = getKeyword("token");

	if(kw && *kw)
		packtoken = *kw;

	script[stack].index = 0;

	sym = getVariable(getSymbolSize());

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	sym->data[sym->flags.size] = 0;

	if(value && !index)
		len = atoi(value);

	value = getValue();
	if(!value)
	{
		error("list-missing");
		return true;
	}

	while(value[index] && (len--) > 1)
	{
		while(value[index] && value[index] != packtoken)
			++index;
		if(value[index] == packtoken)
			++index;
	}

	if(!value[index])
	{
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}
	len = 0;
	while(value[index] && value[index] != packtoken && len < sym->flags.size)
		sym->data[len++] = value[index++];
	if(value[index] == packtoken)
		++index;
	sym->data[len] = 0;
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	script[stack].index = index;
	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrFor(void)
{
	Symbol *sym;
	unsigned short loop = script[stack].line->loop;
	Line *line;
	int index = script[stack].index;
	script[stack].index = 0;

	if(!index)
		++index;

	sym = getVariable(getSymbolSize());
	char *value;

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	sym->data[sym->flags.size] = 0;
	script[stack].index = index;
	value = getValue(NULL);
	if(!value)
	{
		line = script[stack].line->next;
		while(line)
		{
			if(line->loop == loop)
			{
				script[stack].line = line;
				advance();
				return true;
			}
			line = line->next;
		}
		error("loop-overflow");
		return true;
	}

	strncpy(sym->data, value, sym->flags.size);
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrDo(void)
{
	unsigned short loop = script[stack].line->loop;
	Line *line;

	script[stack].index = 0;	// always reset

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			line = script[stack].line->next;
			while(line)
			{
				if(line->loop == loop)
				{
					script[stack].line = line;
					advance();
					return true;
				}
				line = line->next;
			}
			error("loop-overflow");
			return true;
		}
	}

	if(!push())
		return true;

	advance();
	return true;
}

bool ScriptInterp::scrLoop(void)
{
	unsigned short loop;

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}

	loop = script[stack - 1].line->loop;
	if(!loop)
	{
		error("stack-not-loop");
		return true;
	}

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			script[stack - 1] = script[stack];
			--stack;
			advance();
			return true;
		}
	}

	--stack;
	return execute(script[stack].line->method);
}

bool ScriptInterp::scrContinue(void)
{
	Line *line;
	unsigned short loop;

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			advance();
			return true;
		}
	}

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}

	loop = script[stack - 1].line->loop;
	line = script[stack].line->next;

	if(!loop)
	{
		error("stack-not-loop");
		return true;
	}

	while(line)
	{
		if(line->loop == loop)
		{
			script[stack].line = line;
			return true;
		}
		line = line->next;
	}
	error("loop-overflow");
	return true;
}

bool ScriptInterp::scrBreak(void)
{
	Line *line;
	unsigned short loop;

	if(script[stack].line->argc)
	{
		if(!conditional())
		{
			advance();
			return true;
		}
	}

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}

	loop = script[stack - 1].line->loop;
	line = script[stack].line->next;

	if(!loop)
	{
		error("stack-not-loop");
		return true;
	}

	while(line)
	{
		if(line->loop == loop)
		{
			--stack;
			script[stack].line = line;
			advance();
			return true;
		}
		line = line->next;
	}
	error("loop-overflow");
	return true;
}

bool ScriptInterp::scrBegin(void)
{
	const char *mem = getMember();

	if(!mem)
		mem = "none";

	if(!stricmp(mem, "if"))
	{
		script[stack].tranflag = true;
		return scrIf();
	}

	if(script[stack].tranflag)
	{
		error("begin-already-in-transaction");
		return true;
	}
	
	script[stack].tranflag = true;
	advance();
	return true;
}

bool ScriptInterp::scrEnd(void)
{
	if(!script[stack].tranflag)
	{
		error("end-not-in-transaction");
		return true;
	}
	script[stack].tranflag = false;
	advance();
	return true;
}

bool ScriptInterp::scrEndcase(void)
{
	script[stack].caseflag = script[stack].tranflag = false;
	advance();
	return true;
}

bool ScriptInterp::scrEndif(void)
{
	script[stack].tranflag = false;
	advance();
	return true;
}

bool ScriptInterp::scrLabel(void)
{
	advance();
	return true;
}

bool ScriptInterp::scrElse(void)
{
	int level = 0;
	Line *line;

	advance();

	while(NULL != (line = script[stack].line))
	{
		advance();

		if(line->method == &ScriptInterp::scrThen)
			++level;
		else if(line->method == &ScriptInterp::scrEndif)
		{
			if(!level)
				return true;
		}
	}
	return true;
}

bool ScriptInterp::scrThen(void)
{
	int level = 0;
	Line *line;

	advance();

	while(NULL != (line = script[stack].line))
	{
		advance();

		if(line->method == &ScriptInterp::scrThen)
			++level;
		else if(line->method == &ScriptInterp::scrElse)
		{
			if(!level)
				return true;
		}
		else if(line->method == &ScriptInterp::scrEndif)
		{
			if(!level)
				return true;
			--level;
		}
	}
	return true;
}

bool ScriptInterp::scrIfThen(void)
{
	if(!conditional())
		advance();

	advance();
	return true;
}

bool ScriptInterp::scrCase(void)
{
	unsigned short loop = 0xffff;
	Line	*line;

	if(!script[stack].caseflag)
		if(conditional() || !script[stack].line->argc)
		{
			script[stack].caseflag = true;
			advance();
			while(script[stack].line)
			{
				if(script[stack].line->method == &ScriptInterp::scrCase)
					advance();
				else
					return true;
			}
			return true;
		}

	if(stack && script[stack].line->loop)
		loop = script[stack - 1].line->loop;

	advance();
	while(NULL != (line = script[stack].line))
	{
		if(line->loop == loop)
			return true;

		if(line->method == &ScriptInterp::scrCase && !script[stack].caseflag)
			return true;

		if(line->method == &ScriptInterp::scrEndcase)
			return true;

		advance();
	}
	return true;
}

bool ScriptInterp::ifGoto(void)
{
        if(script[stack].index < script[stack].line->argc)
                return scrGoto();
        advance();
        if(script[stack].line->method == &ScriptInterp::scrThen)
                advance();
        return true;
}

bool ScriptInterp::scrIf(void)
{
	if(conditional())
		return ifGoto();
	advance();
	return true;
}

bool ScriptInterp::scrLock(void)
{
	const char *id = getKeyword("id");
	const char *member = getMember();

	if(!id)
		id = getValue(NULL);

	if(!id)
	{
		error("no-lock-id");
		return true;
	}

	if(!member)
		member = "wait";

	if(!stricmp(member, "try"))
	{
		if(!locks.lock(this, id))
			error("lock-busy");
		else
			advance();
		return true;
	}

	if(!stricmp(member, "unlock") || !stricmp(member, "ulock") || !stricmp(member, "end"))
	{
		if(!locks.unlock(this, id))
			error("lock-invalid");
		else
			advance();
		if(!stricmp(member, "end"))
			script[stack].tranflag = false;
		return true;
	}

	if(!locks.lock(this, id))
		return true;

	if(!stricmp(member, "begin"))
		script[stack].tranflag = true;

	advance();
	return true;
}

bool ScriptInterp::scrCall(void)
{
	int index = script[stack].index;
	int id = 0;
	Symbol *sym;
	char symname[8];
	char *arg;
	const char *member = getMember();
	bool trans = false, local = false;
	ScriptSymbol *ref, *syms = script[stack].local;
	const char *name = getObject()->name;

	if(!strnicmp(script[stack].line->cmd, "call", 4))
		trans = true;

	if(!strnicmp(script[stack].line->cmd, "source", 6))
	{
		trans = true;
		if(!member)
			member = "local";
	}

	if(!member)
		member = "none";

	if(!push())
		return true;

	if(!stricmp(member, "global") || !stricmp(member, "public"))
		script[stack].local = NULL;
	else if(stricmp(member, "local") && stricmp(member, "protected"))
	{
		script[stack].local = new ScriptSymbol(symsize, pgsize);
		local = true;
	}
	else
	{
		script[stack].tranflag = trans;
		return scrGoto();
	}

	sprintf(symname, "%d", id++);
	if(local)
		script[stack].local->setConst(symname, name);
	else
	{ 
		sym = getLocal(symname, pgsize);
		if(sym)
		{
			sym->flags.initial = false;
			strcpy(sym->data, name);
		}
	}

	getValue(NULL);	// skip label

	while(NULL != (arg = getOption(NULL)))
	{
		sprintf(symname, "%d", id++);
		if(local && *arg != '&')
			script[stack].local->setConst(symname, arg);
		else if(*arg != '&')
		{
			sym = getLocal(symname, pgsize);
			if(sym)
			{
				sym->flags.initial = false;
				strcpy(sym->data, arg);
			}
		}
		sym = NULL;
		++arg;
		if(syms)
			sym = syms->getEntry(arg, 0);
		if(sym)
			ref = syms;
		else
			ref = this;
		sym = getLocal(symname, strlen(arg) + sizeof(ref));
		if(!sym->flags.initial)
			continue;
		enterMutex();
		*((ScriptSymbol **)(sym->data)) = ref;
		strcpy(sym->data + sizeof(ref), arg);
		sym->flags.initial = false;
		sym->flags.readonly = true;
		sym->flags.type = REF;
		leaveMutex();
	}

	script[stack].index = index;
	script[stack].tranflag = trans;
	return scrGoto();
}

bool ScriptInterp::scrPop(void)
{
	char *level = getValue(NULL);
	int max;

	if(level)
	{
		max = atoi(level);
		if(max < stack)
			stack = max;
		advance();
		return true;
	}

	if(stack < 1)
	{
		error("stack-underflow");
		return true;
	}
	if(script[stack].local != script[stack - 1].local)
	{
		if(script[stack].local)
			delete script[stack].local;
		script[stack].local = script[stack - 1].local;
	}
	script[stack - 1] = script[stack];
	--stack;
	advance();
	return true;
}

bool ScriptInterp::scrReturn(void)
{
	Line *line = getScript();
	char *label = getOption(NULL), *ext, *var;
	char namebuf[256];
	int len;
	int argc = 0;
	Name *scr;
	const char *member = getMember();
	bool exitflag = false;
	bool localflag = false;
	bool topflag = false;
	bool errflag = false;

	if(label && *label != '@')
		label = getContent(label);

	if(!member)
		member = "0";

	if(!stricmp(member, "exit"))
		exitflag = true;
	else if(!stricmp(member, "local"))
		localflag = true;
	else if(!stricmp(member, "top"))
	{
		localflag = true;
		topflag = true;
	}

	len = atoi(member);
	if(!len)
		len = getSymbolSize();

	tempidx = 0;
	while(argc < line->argc)
	{
		if(*line->args[argc++] != '=')
			continue;

		snprintf(temps[tempidx], getSymbolSize() + 1, 
			"%s", getContent(line->args[argc]));

//		line->args[argc] = temps[tempidx];
		if(tempidx++ >= SCRIPT_TEMP_SPACE)
			tempidx = 0;
		++argc;
	}

	do {
		if(!pull())
		{
			errflag = true;
			if(localflag)
				break;
			if(exitflag)
				scrExit();
			return true;
		}

	} while(script[stack].line->loop != 0 || topflag);

	argc = 0;
	tempidx = 0;
	while(argc < line->argc)
	{
		var = line->args[argc++];
		if(*var != '=')
			continue;

		++argc;
		if(*(++var) == '%')
			++var;
		ext = temps[tempidx++];
		if(tempidx >= SCRIPT_TEMP_SPACE)
			tempidx = 0;
		setSymbol(var, len);
		setSymbol(var, ext);
	}

retry:
	if(!label)
	{
		if(!errflag)
			advance();
		return true;
	}

	if(!*label)
	{
		if(!errflag)
			advance();
		return true;
	}
	if(*label == '@')
	{
		if(event(label + 1))
			return true;
	}
	if(*label == '^')
	{
		trap(++label);
		return true;
	}
        len = strlen(label);
        if(!strncmp(label, "::", 2))
        {
                strcpy(namebuf, script[stack].script->name);
                ext = strstr(namebuf, "::");
                if(ext)
                        strcpy(ext, label);
                else
                        strcat(namebuf, label);
                label = namebuf;
        }
        else if(label[len - 1] == ':')
        {
                strcpy(namebuf, script[stack].script->name);
                ext = strstr(namebuf, "::");
                if(ext)
                        strcpy(ext + 2, label);
                else
                {
                        strcat(namebuf, "::");
                        strcat(namebuf, label);
                }
                label = namebuf;
		len = strlen(label);
		label[len - 1] = 0;
        }

        scr = getScriptImage(label);
        if(!scr)
        {
		label = getValue(NULL);
		if(label)
			goto retry;
                error("script-not-found");
                return true;
        }
        once = true;
	script[stack].caseflag = script[stack].tranflag = false;
        script[stack].script = scr;
        script[stack].line = scr->first;
        script[stack].index = 0;
        return true;
}

bool ScriptInterp::scrExit(void)
{
	while(stack)
		pull();

	script[stack].line = NULL;
	return true;
}

bool ScriptInterp::scrRemove(void)
{
	Symbol *sym = getVariable(0);
	char *val;

	if(!sym)
	{
		error("symbol-missing");
		return true;
	}

	if(sym->flags.type != FIFO &&
	   sym->flags.type != STACK &&
	   sym->flags.type != SEQUENCE &&
	   sym->flags.type != CACHE)
	{
		error("symbol-invalid");
		return true;
	}

	while(NULL != (val = getValue()))
		removeSymbol(sym, val);

	advance();
	return true;
}

bool ScriptInterp::scrClear(void)
{
	Symbol *sym = getVariable(0);

	while(sym)
	{
		if(sym->flags.type == FIFO || sym->flags.type == SEQUENCE || sym->flags.type == STACK || sym->flags.type == CACHE)
		{
			sym->data[1] = sym->data[2] = 0;
			continue;
		}
		if(sym->flags.readonly && !sym->flags.commit)
		{
			sym = getVariable(0);
			continue;
		}
		sym->data[0] = 0;
		sym->flags.initial = true;
		if(sym->flags.commit)
			commit(sym);
		sym = getVariable(0);
	}
	advance();
	return true;
}

bool ScriptInterp::scrPost(void)
{
	Symbol *sym;
	char *opt = getOption(NULL);
	if(!opt)
	{
		error("symbol-missing");
		return true;
	}

	if(*opt != '%')
	{
		error("symbol-invalid");
		return true;
	}

	sym = getLocal(++opt, 0);
	if(!sym)
	{
		error("symbol-missing");
		return true;
	}

	if(sym->flags.type != FIFO && sym->flags.type != SEQUENCE && sym->flags.type != STACK && sym->flags.type != CACHE)
	{
		error("symbol-type-invalid");
		return true;
	}

	while(NULL != (opt = getValue(NULL)))
		postSymbol(sym, opt);

	advance();
	return true;
}

bool ScriptInterp::scrList(void)
{
	unsigned count = 0;
	Symbol *sym;
	const char *opt;
	const char *name = getOption(NULL);
	char def[96], buf[8];

	if(!name)
	{
		error("list-missing");
		return true;
	}

	if(*name != '%')
	{
		error("list-missing-array-name");
		return true;
	}
	++name;

	strcpy(def, name);
	strcat(def, ".#####");
	if(!setAlias(name, def))
	{
		error("array-unavailable");
		return true;
	}

	while(NULL != (opt = getOption(NULL)))
	{
		++count;
		snprintf(def, sizeof(def), "%s.%d", name, count);
		setConst(def, opt);
	}
	sprintf(buf, "%d", count);
	strcpy(def, name);
	strcat(def, ".count");
	setConst(def, buf);
	strcpy(def, name);
	strcat(def, ".limit");
	setConst(def, buf);
	strcpy(def, name);
	strcat(def, ".index");
	sym = getEntry(def, 5);
	if(!sym)
	{
		error("array-no-index");
		return true;
	}
	sym->flags.initial = false;
	sym->flags.commit = true;
	sym->flags.type = INDEX;
	strcpy(sym->data, "0");
	advance();
	return true;
}

bool ScriptInterp::scrArray(void)
{
	unsigned size = getSymbolSize();
	unsigned count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");
	char def[96], buf[8];
	unsigned index;
	Symbol *sym;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		size = atoi(mem);

	if(!count || !size)
	{
		error("array-no-size");
		return true;
	}

	while(NULL != (mem = getOption(NULL)))
	{
		strcpy(def, mem);
		strcat(def, ".#####");
		if(!setArray(mem, def))
		{
			error("array-unavailable");
			return true;
		}

		snprintf(def, sizeof(def), "%s.index", mem);
		sym = getEntry(def, 5);
		if(!sym)
		{
			error("array-no-index");
			return true;
		}
		sym->flags.initial = false;
		sym->flags.commit = true;
		sym->flags.type = INDEX;
		strcpy(sym->data, "0");

		sprintf(buf, "%d", count);
		snprintf(def, sizeof(def), "%s.count", mem);
		setConst(def, buf);
		snprintf(def, sizeof(def), "%s.limit", mem);
		setConst(def, buf);

		for(index = 1; index <= count; ++index)
		{
			snprintf(def, sizeof(def), "%s.%d", mem, index);
			setSymbol(def, size);
		}
	 }
	advance();
	return true;
}

bool ScriptInterp::scrFifo(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *kw = getKeyword("count");
	const char *mem = getMember();
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	while(NULL != (mem = getOption(NULL)))
	{
		if(strchr(mem, '.') || !script[stack].local)
			rtn = makeFifo(mem, count, rec);
		else
			rtn = script[stack].local->makeFifo(mem, count,  rec);
		if(!rtn)
		{
			error("fifo-make-failed");
			return true;
		}
	}
	advance();
	return true;
}

bool ScriptInterp::scrCounter(void)
{
	const char *id;
	int rtn;

	while(NULL != (id = getOption(NULL)))
	{
		if(strchr(id, '.') || !script[stack].local)
			rtn = makeCounter(id);
		else
			rtn = script[stack].local->makeCounter(id);
		if(!rtn)
		{
			error("counter-make-failed");
			return true;
		}
	}
	advance();
	return true;
}

bool ScriptInterp::scrStack(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	while(NULL != (mem = getOption(NULL)))
	{
		if(!makeStack(mem, count, rec))
		{
			error("stack-make-failed");
			return true;
		}
	}
	advance();
	return true;
}


bool ScriptInterp::scrSequence(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	mem = getOption(NULL);
	if(!mem)
	{
		error("symbol-missing");
		return true;
	}

	if(strchr(mem, '.') || !script[stack].local)
		rtn = makeSequence(mem, count, rec);
	else
		rtn = script[stack].local->makeSequence(mem, count, rec);

	if(!rtn)
	{
		error("sequence-make-failed");
		return true;
	}
	--script[stack].index;
	scrPost();
	return true;
}

bool ScriptInterp::scrCache(void)
{
	unsigned char rec = getSymbolSize() - 10;
	unsigned char count;
	const char *mem = getMember();
	const char *kw = getKeyword("count");
	int rtn;

	if(kw)
		count = atoi(kw);
	else
		count = atoi(getValue("0"));

	kw = getKeyword("size");
	if(kw)
		mem = kw;

	if(mem)
		rec = atoi(mem);
	else
		rec /= count;

	if(!count || !rec)
	{
		error("symbol-no-size");
		return true;
	}

	mem = getOption(NULL);
	if(!mem)
	{
		error("symbol-missing");
		return true;
	}

	if(strchr(mem, '.') || !script[stack].local)
		rtn = makeCache(mem, count, rec);
	else
		rtn = script[stack].local->makeCache(mem, count, rec);

	if(!rtn)
	{
		error("cache-make-failed");
		return true;
	}
	--script[stack].index;
	scrPost();
	return true;
}

bool ScriptInterp::scrVar(void)
{
	const char *opt;
	const char *sz = getMember();

	if(!sz)
		sz = getKeyword("size");

	if(!sz)
		sz = "65535";

	while(NULL != (opt = getOption(NULL)))
	{
		if(*opt == '%')
			++opt;
		setSymbol(opt, atoi(sz));
	}
	advance();
	return true;
}

bool ScriptInterp::scrSize(void)
{
	Symbol *sym;
	int size = atoi(getValue("0"));
	char *opt = getOption(NULL);

	if(!opt)
	{
		error("symbol-not-specified");
		return true;
	}

	if(!size)
	{
		error("symbol-no-size");
		return true;
	}

	while(opt)
	{
		if(*opt == '@')
		{
			opt = getSymbol(++opt);
			if(!opt)
			{
				opt = getOption(NULL);
				continue;
			}
		}
		else if(*opt != '%')
		{
			error("symbol-not-constant");
			return true;
		}

		if(*opt == '%')
			++opt;
		if(strchr(opt, '.') || !script[stack].local)
			sym = getEntry(opt, size);
		else
			sym = script[stack].local->getEntry(opt, size);
		opt = getOption(NULL);
	}

	advance();
	return true;
}

bool ScriptInterp::scrInc(void)
{
	Property *prop = NULL;
	Symbol *sym = getVariable(getSymbolSize());
	const char *prefix = getKeyword("prefix");
	int len;
	int value = 0;
	const char *opt;

	if(!prefix)
		prefix = "";

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	len = sym->flags.size;
	opt = getValue(NULL);
	if(opt)
	{
		while(opt)
		{
			value += atoi(opt);
			opt = getValue(NULL);
		}
	}
	else
	{
		opt = getKeyword("offset");
		if(opt)
			value = atoi(opt);
		else
			++value;
	}

	opt = getMember();
	if(strlen(sym->data) < strlen(prefix))
		strcpy(sym->data, prefix);
	else if(strlen(prefix))
		memcpy(sym->data, prefix, strlen(prefix));
	if(opt)
		prop = Property::find(opt);
	if(prop)
		prop->adjProperty(sym->data + strlen(prefix), sym->flags.size, value);
	else
		adjustValue(sym->data + strlen(prefix), value);
	if(sym->flags.commit)
		commit(sym);

	advance();
	return true;
}

bool ScriptInterp::scrDec(void)
{
	Property *prop = NULL;
	Symbol *sym = getVariable(getSymbolSize());
	const char *prefix = getKeyword("prefix");
	int len;
	int value = 0;
	const char *opt;

	if(!prefix)
		prefix = "";

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	len = sym->flags.size;
	opt = getValue(NULL);
	if(opt)
	{
		while(opt)
		{
			value -= atoi(opt);
			opt = getValue(NULL);
		}
	}
	else
	{
		opt = getKeyword("offset");
		if(opt)
			value -= atoi(opt);
		else
			--value;
	}

	opt = getMember();
	if(strlen(sym->data) < strlen(prefix))
		strcpy(sym->data, prefix);
	else if(strlen(prefix))
		memcpy(sym->data, prefix, strlen(prefix));
	if(opt)
		prop = Property::find(opt);
	if(prop)
		prop->adjProperty(sym->data + strlen(prefix), sym->flags.size, value);
	else
		adjustValue(sym->data + strlen(prefix), value);
	if(sym->flags.commit)
		commit(sym);

	advance();
	return true;
}

bool ScriptInterp::scrTryeach(void)
{
	char *gp = getValue(NULL);
	const char *mem = getMember();
	int offset = 0;
	char namebuf[256];
	char *np;
	Name *scr;
	char packtoken = getPackToken();

	if(mem)
		offset = atoi(mem);

	if(!gp)
	{
		error("list-not-found");
		return true;
	}

	while(*gp && (offset--) > 1)
	{
		gp = strchr(gp, packtoken);
		if(gp)
			++gp;
		else
			gp = "";
	}

	while(*gp)
	{
		np = namebuf;
		while(*gp && *gp != packtoken)
			*(np++) = *(gp++);

		*np = 0;
		if(*gp == packtoken)
			++gp;

		scr = getScriptImage(namebuf);
		if(!scr)
			continue;

		once = true;
		script[stack].caseflag = script[stack].tranflag = false;
		script[stack].script = scr;
		script[stack].line = scr->first;
		script[stack].index = 0;
		return true;
	}
	advance();
	return true;
}

bool ScriptInterp::scrDecimal(void)
{
	script[stack].decimal = atoi(getValue("0"));
	advance();
	return true;
}

bool ScriptInterp::scrError(void)
{
	error(getValue("no-arguments"));
	return true;
};

bool ScriptInterp::scrLoadable(void)
{
	Line *line = getScript();
	char *cmd = line->cmd;
	char keybuf[33];
	int len = 0;
	char *kw = keybuf;
	ScriptModule *mod = ScriptModule::first;

	while(len ++ < 32 && *cmd && *cmd != '.')
		*(kw++) = *(cmd++);
	*kw = 0;

	while(mod)
	{
		if(!stricmp(keybuf, mod->cmd))
			break;
		mod = mod->next;
	}

	if(!mod)
	{
		error("module-not-found");
		return true;
	}
	kw = mod->getSession(this, line, &session);
	if(line != getScript())
		return true;

	if(kw)
	{
		error(kw);
		return true;
	}
	if(!session)
	{
		advance();
		return true;
	}
	return false;
}

bool ScriptInterp::scrRef(void)
{
	Symbol *sym;
	char *ref;
	char refbuf[256];
	int len = 0;

	while(NULL != (ref = getValue(NULL)) && len < 250)
	{
		if(*ref == '%')
			++ref;

		if(script[stack].line->argc <= script[stack].index)
			break;

		if(len)
			snprintf(refbuf + len, sizeof(refbuf) - len,
				".%s", ref);
		else
			snprintf(refbuf + len, sizeof(refbuf) - len,
				"%s", ref);
		len = strlen(refbuf);
	}
	
	sym = getLocal(ref, len + sizeof(ScriptSymbol *));
	if(!sym->flags.initial)
	{
		error("alias-invalid-reference-object");
		return true;
	}
	advance();
	enterMutex();
	*((ScriptSymbol **)(sym->data)) = this;
	strcpy(sym->data + sizeof(ScriptSymbol *), refbuf);	
	sym->flags.initial = false;
	sym->flags.readonly = true;
	sym->flags.type = REF;
	leaveMutex();
	return true;
}	

bool ScriptInterp::scrAlias(void)
{
	char *sym;
	char *src;

	while(NULL != (sym = getValue(NULL)))
	{
		src = getValue(NULL);
		if(!src)
		{
			error("alias-no-source");
			return true;
		}

		if(!setAlias(sym, src))
		{
			error("alias-failure");
			return true;
		}
	}

	advance();
	return true;
}

bool ScriptInterp::scrConst(void)
{
	char *sym = getOption(NULL);
	char *val;
	int max = getSymbolSize();
	char buffer[max + 1];
	int count = 0;
	int idx = 0, len = 0;
	Line *line = getScript();

	buffer[0] = 0;
	if(!sym)
	{
		while(idx < line->argc)
		{
			val = line->args[idx++];
			if(*val != '=')
				continue;
			if(*(++val) == '%')
				++val;
			setConst(val, line->args[idx++]);
			++count;
		}

		if(count)
			advance();
		else
			error("const-not-specified");
		return true;
	}

	while(NULL != (val = getValue(NULL)) && len < max)
	{
		strncpy(buffer + len, val, max - len);
		buffer[max] = 0;
		len = strlen(buffer);
	}

	if(script[stack].local && !strchr(sym, '.'))
		script[stack].local->setConst(sym, buffer);
	else if(!setConst(sym, buffer))
	{
		error("const-not-set");
		return true;
	}
	advance();
	return true;
}

bool ScriptInterp::scrDup(void)
{
	const char *id;
	Symbol *src, *dup;
	while(NULL != (src = getVariable(0)))
	{
		dup = getVariable(src->flags.size);
		if(!dup)
		{
			error("no-target");
			return true;

		}
		enterMutex();
		if(!dup->flags.initial)
		{
			leaveMutex();
			error("target-exists");
			return true;
		}
		id = dup->id;
		memcpy(dup, src, sizeof(Symbol) + src->flags.size);
		dup->id = id;
		leaveMutex();
	}
	advance();
	return true;
}

bool ScriptInterp::scrNumber(void)
{
	unsigned prec = 0;
	const char *mem = getMember();
	Symbol *sym;
	const char *content;
	char *p;
	char fmt[13];
	unsigned max;
	long iNumber, hNumber, lNumber;
	char dec = *getSymbol("script.decimal");
	bool hex = false;

	if(mem)
	{
		prec = atoi(mem);
		if(!stricmp("hex", mem))
			hex = true;
	}
	else
		prec = script[stack].decimal;

	if(prec > 7)
		prec = 7;

	snprintf(fmt, sizeof(fmt), "%s%d%s", "%ld.%0", prec, "ld");

	while(NULL != (content = getOption(NULL)))
	{
		if(!stricmp(content, "-eq"))
			break;

		if(*content == '%')
			++content;

		sym = getLocal(content, 11);
		if(!sym)
			continue;

		if(!sym->flags.initial)
			continue;

		if(sym->flags.readonly)
			continue;

		if(hex)
			snprintf(sym->data, sym->flags.size + 1, "0x00000000");
		else if(prec)
			snprintf(sym->data, sym->flags.size + 1, fmt, 0, 0);
		else
			snprintf(sym->data, sym->flags.size + 1, "0");
		p = strchr(sym->data, '.');
		if(p)
			*p = dec;
		sym->flags.initial = false;
		if(sym->flags.commit)
			commit(sym);
	}

	if(!content)
	{
		advance();
		return true;
	}

	if(getExpression(&iNumber, 1, prec) != 1)
	{
		error("bad-expression");
		return true;
	}

	script[stack].index= 0;

	while(NULL != (content = getOption(NULL)))
	{
		if(!stricmp(content, "-eq"))
			break;
		
		if(*content == '%')
			++content;

		sym = getLocal(content, 0);

		if(!sym)
			continue;

		if(sym->flags.readonly)
			continue;


		hNumber = iNumber / tens[prec];
		lNumber = iNumber % tens[prec];
		if(lNumber < 0)
			lNumber = -lNumber;
		if(hex)
			snprintf(sym->data, sym->flags.size + 1, "0x%08lx", iNumber);
		else if(prec)
			snprintf(sym->data, sym->flags.size + 1, fmt, hNumber, lNumber);
		else
			snprintf(sym->data, sym->flags.size + 1, "%ld", iNumber);

		p = strchr(sym->data, '.');
		if(p)
			*p = dec;

		if(sym->flags.commit)
			commit(sym);
	}
	advance();
	return true;
}

bool ScriptInterp::scrSet(void)
{
	Property *prop = NULL;
	Symbol *sym;
	int size = 0;
	bool first = true, iflag = false;
	unsigned len, rlen = 0;
	const char *value;
	bool number = false;
	unsigned sz = 0, idx = 0, count = 0;
	Line *line = getScript();

	enum
	{
		VAL_FILL,
		VAL_RIGHT,
		VAL_NONE,
		VAL_MIN,
		VAL_MAX,
		VAL_SPLIT,
		VAL_CENTER
	}	minmax = VAL_NONE;

	if(!strnicmp(script[stack].line->cmd, "init", 4))
		iflag = true;

	value = getMember();
	if(value)
	{
		if(!stricmp(value, "size"))
			size = atoi(getValue("0"));
		else if(!stricmp(value, "min"))
			minmax = VAL_MIN;
		else if(!stricmp(value, "max"))
			minmax = VAL_MAX;
		else if(!stricmp(value, "right"))
			minmax = VAL_RIGHT;
		else if(!stricmp(value, "fill") || !stricmp(value, "left"))
			minmax = VAL_FILL;
		else if(!stricmp(value, "split"))
			minmax = VAL_SPLIT;
		else if(!stricmp(value, "center"))
			minmax = VAL_CENTER;
		else if(!strnicmp(value, "val", 3) || !strnicmp(value, "num", 3))
		{
			number = true;
			size = 11;
		}
		else if(NULL ==(prop = Property::find(value)))
			size = atoi(value);
	}
	else
	{
		value = getKeyword("justify");
		if(!value)
			value = "";
		if(!stricmp(value, "right"))
			minmax = VAL_RIGHT;
		else if(!stricmp(value, "center"))
			minmax = VAL_CENTER;
		else if(!stricmp(value, "left"))
			minmax = VAL_FILL;
	}
	value = getKeyword("size");
	if(value)
		sz = atoi(value);

	if(!size && prop)
		size = prop->getPropertySize();

	if(!size)
		size = getSymbolSize();

	if(!sz)
		sz = size;
	sym = getVariable(sz);
	if(!sym)
	{
		while(idx < line->argc)
		{
			value = line->args[idx++];
			if(*value != '=')
				continue;
			if(*(++value) == '%')
				++value;
			++count;
			sym = getEntry(value, size);
			value = line->args[idx++];
			if(!sym)
				continue;
			if(iflag && !sym->flags.initial)
				continue;
			setSymbol(sym->id, value);
		}
		if(count)
			advance();
		else
			error("symbol-not-found");
		return true;
	}

	switch(sym->flags.type)
	{
	case FIFO:
	case STACK:
	case CACHE:
	case SEQUENCE:
		script[stack].index = 0;
		return scrPost();
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	if(iflag && !sym->flags.initial)
	{
		advance();
		return true;
	}

	sym->data[sym->flags.size] = 0;
	len = 0;

	while((minmax == VAL_RIGHT || minmax == VAL_CENTER) && rlen < sym->flags.size)
		sym->data[rlen++] = ' ';

	while(NULL != (value = getValue(NULL)))
	{

		first = false;

		if(len >= sym->flags.size)
			break;

		if(minmax == VAL_MIN && atoi(value) >= atoi(sym->data) && len)
			continue;

		if(minmax == VAL_MAX && atoi(value) <= atoi(sym->data) && len)
			continue;
	
		if(minmax == VAL_RIGHT || minmax == VAL_CENTER)
		{
			rlen -= strlen(value);
			if(minmax == VAL_CENTER && rlen > 0)
				rlen /= 2;

			if(rlen < 0)
			{
				value -= rlen;
				rlen = 0;
			}
			strncpy(sym->data + rlen, value, strlen(value));
			if(minmax == VAL_CENTER)
				rlen = sym->flags.size;
		}
		else if(minmax == VAL_NONE || minmax == VAL_FILL || minmax == VAL_SPLIT)
			strncpy(sym->data + len, value, sym->flags.size - len);
		else
			strncpy(sym->data, value, sym->flags.size);
		sym->data[sym->flags.size] = 0;
		len = strlen(sym->data) - rlen;
		if(minmax == VAL_SPLIT)
		{
			rlen = len;
			while(rlen < sym->flags.size)
				sym->data[rlen++] = ' ';
			minmax = VAL_RIGHT;
			len = 0;
		}
	}

	while(len < sym->flags.size && minmax == VAL_FILL)
		sym->data[len++] = ' ';
	
	sym->data[len + rlen] = 0;	

	if(number)
		sprintf(sym->data, "%ld", atol(sym->data));
	else if(prop)
	{
		strcpy(temps[tempidx], sym->data);
		prop->setProperty(sym->data, temps[tempidx], sym->flags.size);
	}

	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	advance();
	return true;
}

bool ScriptInterp::scrDump(void)
{
	Symbol *index[64];
	unsigned pos = 0;
	unsigned count = gather(index, 63, getValue(NULL), NULL);	

	while(pos < count)
	{
		slog(Slog::levelDebug) << index[pos]->id << " = " << index[pos]->data << endl;
		++pos;
	}
	advance();
	return true;
}

bool ScriptInterp::scrGather(void)
{
	Symbol *sym = getVariable(getSymbolSize());
	char *suffix = getValue(NULL);
	Name *list[33];
	int count = 0, idx;

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	count = image->gather(suffix, list, 32);
	if(!count)
	{
		error("no-scripts-found");
		return true;
	}
	sym->data[0] = 0;
	for(idx = 0; idx < count; ++idx)
	{
		if(idx)
			strcat(sym->data, ",");
		strcat(sym->data, list[idx]->name);
	}
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);
	advance();
	return true;
}

bool ScriptInterp::scrFullpath(void)
{
	char *cp;
	char *pro;
	int len;
	Symbol *sym = getVariable(getSymbolSize());
	char *prefix = getValue(NULL);

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	if(!prefix)
	{
		error("no-prefix");
		return true;
	}

	cp = strchr(sym->data, '/');
	pro = strchr(sym->data, ':');
	if(cp == sym->data || pro < cp)
	{
		advance();
		return true;
	}
	len = strlen(prefix);
	if(sym->flags.size < len + strlen(sym->data) + 3)
	{
		error("no-space");
		return true;
	}
	cp = sym->data;
	sym->data[len++] = '/';
	while(*cp)
		sym->data[len++] = *(cp++);
	*cp = 0;
	memcpy(sym->data, prefix, strlen(prefix));
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);
	advance();
	return true;
}

bool ScriptInterp::scrDirname(void)
{
	char *cp;
	Symbol *sym = getVariable(1);

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	cp = strrchr(sym->data, '/');
	if(!cp)
		strcpy(sym->data, ".");
	else
		*cp = 0;

	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);

	advance();
	return true;
}

bool ScriptInterp::scrBasename(void)
{
	const char *ext;
	Symbol *sym = getVariable(0);
	char *cp;

	if(!sym)
	{
		error("symbol-not-found");
		return true;
	}

	if(sym->flags.readonly)
	{
		error("symbol-readonly");
		return true;
	}

	cp = strrchr(sym->data, '/');
	if(cp)
		strcpy(sym->data, cp + 1);

	while(NULL != (ext = getValue(NULL)))
	{
		cp = sym->data + strlen(sym->data) - strlen(ext) - 1;
		if(cp > sym->data)
			if(*cp == '.')
				if(!stricmp(cp + 1, ext))
					*cp = 0;
	}
	sym->flags.initial = false;
	if(sym->flags.commit)
		commit(sym);
	advance();
	return true;
}

long ScriptInterp::getRealValue(double d, unsigned prec)
{
	char buf[20];
	char *cp;
	char lval[9];
	int count;
	long rval;
	snprintf(buf, sizeof(buf), "%f", d);

	rval = atol(buf) * tens[prec];
	cp = strchr(buf, '.');
	if(!cp)
		return rval;
	count = strlen(++cp);
	if(count > prec)
		count = prec;
	strcpy(lval, "00000000");
	strncpy(lval, cp, count);
	lval[prec] = 0;
	if(rval < 0)
		return rval - atol(lval);
	return rval + atol(lval);
}

long ScriptInterp::getIntValue(const char *text, unsigned prec)
{
	Fun *fun = ifun;
	int count;
	char *arg;
	long *ival, rval;
	char lval[9];
	const char *orig = text;
	char dec = *getSymbol("script.decimal");

	if(!isalpha(*text))
	{
		if(!strnicmp("0x", text, 2))
		{
			rval = strtol(text, NULL, 16);
			return rval * tens[prec];
		}
		rval = atol(text);
		rval *= tens[prec];
		text = strchr(text, '.');
		if(!text)
			text = strrchr(orig, ',');
		if(!text)
			text = strrchr(orig, dec);
		if(!text)
			return rval;
		count = strlen(++text);
		if(count > prec)
			count = prec;
		strcpy(lval, "00000000");
		strncpy(lval, text, count);
		lval[prec] = 0;
		if(rval < 0)
			return rval - atol(lval);

		return rval + atol(lval);
	}	

	while(NULL != fun)
	{
		if(!stricmp(fun->id, text))
			break;
		fun = fun->next;
	}

	if(!fun)
		return 0;

	if(!fun->args)
		return fun->fn(NULL, prec);

	arg = getValue(NULL);
	if(!arg)
		return 0;

	if(stricmp(arg, "("))
		return 0;

	ival = new long[fun->args];

	count = getExpression(ival, fun->args, prec);
	if(count != fun->args)
		return 0;

	rval = fun->fn(ival, prec);	
	delete[] ival;
	return rval;
}	

long ScriptInterp::getInteger(long value, unsigned prec)
{
	return value / tens[prec];
}

long ScriptInterp::getTens(unsigned prec)
{
	return tens[prec];
}

double ScriptInterp::getDouble(long value, unsigned prec)
{
	return (double)value / (double)tens[prec];
}

int ScriptInterp::getExpression(long *vals, int max, unsigned prec)
{
	char estack[32];
	unsigned esp = 0;
	long vstack[32], val;
	const char *value;
	char **expr;
	int count = 0;

	static char *elist[] = {"*", "+", "-", "/", "%", NULL};

	vstack[0] = 0;

	while(NULL != (value = getValue(NULL)))
	{
		expr = elist;
		while(*expr)
		{
			if(!stricmp(*expr, value))
				break;
			++expr;
		}
		if(*expr)
			estack[esp] = *value;
		else
			estack[esp] = 0;

		if(!stricmp(value, "("))
		{
			if(esp < 31)
			{
				++esp;
				vstack[esp] = 0;
				continue;
			}
			return -1;
		}

		if(!stricmp(value, ","))
		{
			if(esp)
				return -1;
			if(count < max)
				*(vals++) = vstack[0];
			vstack[0] = 0;
			++count;
			continue;
		}
		if(!stricmp(value, ")"))
		{
			if(!esp)
			{
				if(count < max)
					*(vals++) = vstack[0];
				return ++count;
			}

			switch(estack[--esp])
			{
			case '+':
				vstack[esp] = vstack[esp] + vstack[esp + 1];
				break;
			case '*':
				vstack[esp] = vstack[esp] * vstack[esp + 1];
				break;
			case '/':
				if(vstack[esp + 1] == 0)
					return -1;
				vstack[esp] = vstack[esp] / vstack[esp + 1];
				break;
			case '-':
				vstack[esp] = vstack[esp] - vstack[esp + 1];
				break;
			case '%':
				vstack[esp] = vstack[esp] % vstack[esp + 1];
				break;
			default:
				vstack[esp] = vstack[esp + 1];
			}
			continue;
		}
	
		if(!*expr)
		{
			vstack[esp] = getIntValue(value, prec);
			continue;
		}

		value = getValue("0");
		if(!stricmp(value, "("))
		{
			if(esp > 31)
				return -1;
			vstack[++esp] = 0;
			continue;
		}

		val = getIntValue(value, prec);

		switch(estack[esp])
		{
		case '+':
			vstack[esp] = vstack[esp] + val;
			break;
		case '-':
			vstack[esp] = vstack[esp] - val;
			break;
		case '/':
			if(!val)
				return -1;
			vstack[esp] = vstack[esp] / val;
			break;
		case '*':
			vstack[esp] = vstack[esp] * val;
			break;
		case '%':
			vstack[esp] = vstack[esp] % atol(value);
			break;
		}
	}
	if(count < max)
		*(vals++) = vstack[esp];
	if(!esp)
		return ++count;
	else
		return -1;
}

bool ScriptInterp::conditional(void)
{
	Line *line = script[stack].line;
	char *joiner;
	bool rtn;
	bool andfalse = false;
	bool ortrue = false;

	for(;;)
	{
		rtn = expConditional();
		if(script[stack].index < line->argc)
			joiner = line->args[script[stack].index];
		else
			joiner = "";

		if(!stricmp(joiner, "and"))
		{
			if(!rtn)
				andfalse = true;
		}
		else if(!stricmp(joiner, "or"))
		{
			if(rtn)
				ortrue = true;
		}
		else
			break;

		++script[stack].index;
	}
	if(andfalse)
		return false;

	if(ortrue)
		return true;

	return rtn;
}
		

bool ScriptInterp::expConditional(void)
{
#ifdef	USE_REGEX
	regex_t *regex;
	bool rtn;
#endif

	Test *node = NULL;
	char *v1, *op = NULL, *v2;
	char n1[12], n2[12];
	int l1, l2;
	long ival;

	// no first parm, invalid

	v1 = getOption(NULL);
	if(!v1)
		return false;

	if(*v1 == '-' || *v1 == '!')
	{
		node = ScriptInterp::test;
		while(node)
		{
			if(!stricmp(node->id, v1 + 1))
				break;
			node = node->next;
		}
		if(node)
			op = (char *)node->id;
	}

	if(!strcmp(v1, "("))
	{
		getExpression(&ival, 1, script[stack].decimal);
		snprintf(n1, sizeof(n1), "%ld", ival);
		v1 = n1;
	}
	else if(!stricmp(v1, "-script") || !stricmp(v1, "!script"))
		op = v1;
	else if(!stricmp(v1, "-defined") || !stricmp(v1, "!undefined"))
		op = v1;
	else if(!stricmp(v1, "-empty") || !stricmp(v1, "!empty"))
		op = v1;
	else if(!stricmp(v1, "-module") || !stricmp(v1, "-installed") || !stricmp(v1, "-has"))
		op = "-module";
	else if(!stricmp(v1, "!module") || !stricmp(v1, "!installed"))
		op = "!module";
	else
		v1 = getContent(v1);

	// sym/label by itself, invalid

	if(!op)
		op = getValue(NULL);

	if(!op)
	{
		script[stack].index = 0;
		if(v1 && *v1)
			return true;
		return false;
	}

	// ifdef sym ... format assumed

	v2 = getOption(NULL);

	if(!v2)
	{
		script[stack].index = 1;
		if(v1 && *v1)
			return true;
		return false;
	}

	if(node)
	{
		if(*v1 == '!')
			return !node->handler(this, getContent(v2));
		else
			return node->handler(this, getContent(v2));
	}

	if(!stricmp(op, "-script"))
	{
		if(getScriptImage(getContent(v2)))
			return true;

		return false;
	}

	if(!stricmp(op, "!script"))
	{
		if(!getScriptImage(getContent(v2)))
			return true;
	
		return false;
	}

	if(!stricmp(op, "-module"))
	{
		if(cmd->getHandler(getContent(v2)))
			return true;
		return false;
	}

	if(!stricmp(op, "!module"))
	{
		if(cmd->getHandler(getContent(v2)))
			return false;
		return true;
	}

	if(!stricmp(op, "-defined"))
	{
		if(getSymbol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, "!defined"))
	{
		if(getSymbol(v2))
			return false;
		return true;
	}

	if(!stricmp(op, "-empty"))
	{
		v2 = getContent(v2);
		if(!v2)
			return true;

		if(!*v2)
			return true;

		return false;
	}

	if(!stricmp(op, "!empty"))
	{
		v2 = getContent(v2);
		if(!v2)
			return false;

		if(!*v2)
			return false;

		return true;
	}

	if(!strcmp(v2, "("))
	{
		getExpression(&ival, 1, script[stack].decimal);
		snprintf(n2, sizeof(n2), "%ld", ival);
		v2 = n2;
	}
	else
		v2 = getContent(v2);

	if(!v1)
		v1 = "";

	if(!v2)
		v2 = "";

	if(!stricmp(op, "=") || !stricmp(op, "-eq"))
	{
		if(atol(v1) == atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, "<>") || !stricmp(op, "-ne"))
	{
		if(atol(v1) != atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, "==") || !stricmp(op, ".eq."))
	{
		if(!stricmp(v1, v2))
			return true;

		return false;
	}

	if(!stricmp(op, "!=") || !stricmp(op, ".ne."))
	{
		if(stricmp(v1, v2))
			return true;

		return false;
	}


	if(!stricmp(op, "$") || !stricmp(op, ".in."))
	{
		if(strstr(v2, v1))
			return true;
		return false;
	}

	if(!stricmp(op, "!$"))
	{
		if(strstr(v2, v1))
			return false;

		return true;
	}

#ifdef	USE_REGEX
	if(!stricmp(op, "~") || !stricmp(op, "!~"))
	{
		script[stack].tranflag = false;
		rtn = false;
		regex = new regex_t;
		memset(regex, 0, sizeof(regex_t));

		if(regcomp(regex, v2, REG_ICASE|REG_NOSUB|REG_NEWLINE))
		{
			regfree(regex);
			delete regex;
			return false;
		}

		if(regexec(regex, v1, 0, NULL, 0))
			rtn = false;
		else
			rtn = true;

		regfree(regex);
		delete regex;
		if(*op == '!')
			return !rtn;
		return rtn;
	}
#endif

	if(!stricmp(op, "$<") || !stricmp(op, "$+") || !stricmp(op, ".prefix."))
	{
		if(!strnicmp(v1, v2, strlen(v1)))
			return true;
		return false;
	}

	if(!stricmp(op, "$>") || !stricmp(op, "$-") || !stricmp(op, ".suffix."))
	{
		l1 = strlen(v1);
		l2 = strlen(v2);
		if(l1 <= l2)
			if(!strnicmp(v1, v2 + l2 - l1, l1))
				return true;

		return false;
	}

	if(!stricmp(op, "<") || !stricmp(op, "-lt"))
	{
		if(atol(v1) < atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, ".le."))
	{
		if(stricmp(v1, v2) <= 0)
			return true;
		return false;
	}

	if(!stricmp(op, ".ge."))
	{
		if(stricmp(v1, v2) >= 0)
			return true;
		return false;
	}

	if(!stricmp(op, "<=") || !stricmp(op, "=<") || !stricmp(op, "-le"))
	{
		if(atol(v1) <= atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, ">") || !stricmp(op, "-gt"))
	{
		if(atol(v1) > atol(v2))
			return true;
		return false;
	}

	if(!stricmp(op, ">=") || !stricmp(op, "=>") || !stricmp(op, "-ge"))
	{
		if(atol(v1) >= atol(v2))
			return true;
		return false;
	}

	// if no op, assume ifdef format

	script[stack].index = 1;
	if(*v1)
		return true;

	return false;
}

bool ScriptInterp::push(void)
{
	if(stack >= (SCRIPT_STACK_SIZE - 1))
	{
		error("stack-overflow");
		return false;
	}

	script[stack + 1] = script[stack];
	script[stack + 1].caseflag = script[stack + 1].tranflag = false;
	++stack;
	return true;
}

bool ScriptInterp::pull(void)
{
	if(!stack)
	{
		error("stack-underflow");
		return false;
	}

	if(script[stack - 1].local != script[stack].local)
	{
		if(script[stack].local)
			delete script[stack].local;
	}

	--stack;
	return true;
}

bool ScriptInterp::getOnce(void)
{
	bool ret = once;
	once = false;
	return ret;
}

void ScriptInterp::setLine(Line *line)
{
	script[stack].line = line;
	script[stack].index = 0;
}

bool ScriptInterp::redirect(const char *scriptname)
{
	Name *scr;
	char namebuf[128];
	char *ext;

	if(!strncmp(scriptname, "::", 2))
	{
		strcpy(namebuf, script[stack].script->name);
		ext = strstr(namebuf, "::");
		if(ext)
			strcpy(ext, scriptname);
		else
			strcat(namebuf, scriptname);
	}
	else
		strcpy(namebuf, scriptname);

	scr = getScriptImage(namebuf);
	if(scr)
	{
		script[stack].script = scr;
		script[stack].line = scr->first;
		return true;
	}
	return false;
}

bool ScriptInterp::attach(const char *scriptname)
{
	ScriptImage::InitialList *ilist;
	ScriptModule *mod;
	session = NULL;
	stack = 0;
	lckcount = 0;
	cmd->enterMutex();
	image = cmd->active;
	if(!image)
	{
		cmd->leaveMutex();
		return false;
	}
	script[stack].local = NULL;
	script[stack].script = getScriptImage(scriptname);
	if(script[stack].script)
	{
		ilist = image->ilist;
		while(ilist)
		{
			setSymbol(ilist->name, ilist->size);
			setSymbol(ilist->name, ilist->value);
			ilist = ilist->next;
		}
		setSymbol("script.home", scriptname);
		mod = ScriptModule::first;
		while(mod)
		{
			mod->moduleAttach(this);
			mod = mod->next;
		}
		script[stack].line = script[stack].script->first;
		script[stack].index = 0;
		script[stack].read = NULL;
		script[stack].caseflag = script[stack].tranflag = false;
		script[stack].decimal = 0;
		++image->refcount;
		cmd->leaveMutex();
		return true;
	}
	cmd->leaveMutex();
	once = true;
	signalmask = 0;
	logerror(scriptname, getId(), "missing; attach failed");
	return false;
}

void ScriptInterp::detach(void)
{
	ScriptModule *mod;

	if(!image)
		return;

	if(session)
	{
		delete session;
		session = NULL;
	}

	cmd->enterMutex();
	--image->refcount;
	mod = ScriptModule::first;
	while(mod)
	{
		mod->moduleDetach(this);
		mod = mod->next;
	}
	if(image)
		if(!image->refcount && image != cmd->active)
			delete image;

	cmd->leaveMutex();
	image = NULL;

	while(stack)
		pull();
	locks.release(this);
}

int ScriptInterp::initKeywords(int size)
{
	int idx = 0;
	int count = 0;
	Line *line = script[stack].line;
	char *opt, *cp;
	Symbol *sym;

	if(!size)
		size = getSymbolSize();

	while(idx < line->argc)
	{
		opt = line->args[idx++];
		if(*opt != '=')
			continue;
		if(*(++opt) == '%')
			++opt;
		++count;
		cp = getContent(line->args[idx++]);
		if(!cp)
			continue;

		if(*cp == '&')
		{
			++cp;
			sym = getLocal(opt, strlen(cp) + sizeof(ScriptSymbol *));
			if(!sym->flags.initial)
				continue;

			enterMutex();
			*((ScriptSymbol **)(sym->data)) = this;
			strcpy(sym->data + sizeof(ScriptSymbol *), cp);
			sym->flags.initial = false;
			sym->flags.readonly = true;
			sym->flags.type = REF;
			leaveMutex();
			continue;
		}

		if(script[stack].local == NULL || strchr(opt, '.'))
		{
			setSymbol(opt, size);
			setSymbol(opt, cp);
			continue;
		}

		script[stack].local->setConst(opt, cp);
		script[stack].local->setSymbol(opt, cp);
	}
	return count;
}

char *ScriptInterp::getKeyword(const char *kw)
{
	int idx = 0;
	Line *line = script[stack].line;
	char *opt;
	while(idx < line->argc)
	{
		opt = line->args[idx++];
		if(*opt == '=')
		{
			if(!strnicmp(kw, opt + 1, strlen(kw)))
				return getContent(line->args[idx]);
			++idx;
		}
	}
	return NULL;
}

char *ScriptInterp::getOption(const char *def)
{
	for(;;)
	{
		if(script[stack].index >= script[stack].line->argc)
			return (char *)def;

		if(*script[stack].line->args[script[stack].index] != '=')
			break;

		script[stack].index += 2;
	}

	return script[stack].line->args[script[stack].index++];
}

Script::Symbol *ScriptInterp::getVariable(size_t size)
{
	char *opt;
	Symbol *sym;
	char alt[128];
	char *cp;
	bool local = true;

	for(;;)
	{
		if(script[stack].index >= script[stack].line->argc)
			return NULL;

		opt = script[stack].line->args[script[stack].index++];

		if(*opt != '=')
			break;

		++script[stack].index;
	}

	if(*opt != '%' && *opt != '@')
		return NULL;

	if(*opt == '@')
	{
		sym = getIndirect(++opt);
		if(sym)
			return sym;

		if(strchr(opt, '.'))
			sym = getEntry(opt);
		else
			sym = getLocal(opt);

		if(sym)
			opt = sym->data;
		else
			return NULL;
	}
	else
		++opt;

        if(strchr(opt + 2, '#'))
        {
                snprintf(alt, sizeof(alt), "%s",  opt);
                cp = strchr(alt + 2, '#');
                *(cp++) = 0;
                sym = getLocal(cp);
                if(!sym)
			return NULL;

                snprintf(alt + strlen(alt), sizeof(alt) - strlen(alt), ".%s", sym->data);
                opt = alt;
        }

	if(strchr(opt, '.'))
		local = false;

	if(!local)
		return getEntry(opt, size);
	else
		return getLocal(opt, size);
}

char *ScriptInterp::getContent(char *opt)
{
	Property *prop;
	Symbol *sym;
	char *ext = strrchr(opt, '.');
	char *buf, *dec;
	unsigned pos;
	char *cp;

	char alt[128];


	if(!opt)
		return NULL;

/*
	if(*opt == '!' && script[stack].line->prescan)
	{
		opt = temps[tempidx++];
		if(tempidx >= SCRIPT_TEMP_SPACE)
			tempidx = 0;
		return opt;
	}
*/

	if(*opt == '&' && strchr(opt + 2, '#'))
	{
		cp = temps[tempidx++];
		if(tempidx >= SCRIPT_TEMP_SPACE)
			tempidx = 0;

		snprintf(cp, getSymbolSize(), "%s", opt);
		opt = cp;
		cp = strchr(opt + 2, '#');
		*(cp++) = 0;
		sym = getLocal(cp);
		if(!sym)
			return NULL;
		snprintf(opt + strlen(opt), getSymbolSize() - strlen(opt), ".%s", sym->data);	
		return opt;
	}	

	if(opt[0] == '%' && !opt[1])
		return opt;

	if(opt[0] == '{')
		return ++opt;

	if(*opt != '%' && *opt != '@')
		return opt;

	if(*opt == '@')
	{
		++opt;
		sym = getIndirect(opt);
		if(sym)
			return readSymbol(sym);

		sym = getLocal(opt);
		if(!sym)
			return NULL;

		opt = sym->data;
	}
	else
		++opt;

	if(strchr(opt + 2, '#'))
	{
		snprintf(alt, sizeof(alt), "%s",  opt);
		cp = strchr(alt + 2, '#');
		*(cp++) = 0;
		sym = getLocal(cp);
		if(!sym)
			return NULL;

		snprintf(alt + strlen(alt), sizeof(alt) - 
				strlen(alt), ".%s", sym->data);
		opt = alt;
		ext = NULL;
	}

	sym = getLocal(opt);
	if(sym)
		return readSymbol(sym);

	if(!ext)
		return NULL;

	buf = new char[strlen(opt) + 1];
	strcpy(buf, opt);
	*strrchr(buf, '.') = 0;

	sym = getLocal(buf);
	delete[] buf;
	if(!sym)
		return NULL;

	if(++tempidx >= SCRIPT_TEMP_SPACE)
		tempidx = 0;
	buf = temps[tempidx];

	if((pos = atoi(ext + 1)) > 0)
	{
		char packtoken = getPackToken();

		dec = sym->data;
		while(--pos && dec)
		{
			dec = strchr(dec, packtoken);
			if(dec)
				++dec;
		}
		if(dec)
			strcpy(buf, dec);
		else
			strcpy(buf, "");
		dec = strchr(buf, packtoken);
		if(dec)
			*dec = 0;
	}
	else if(!strnicmp(ext, ".len", 4))
		sprintf(buf, "%d", strlen(sym->data));
	else if(!stricmp(ext, ".size"))
		sprintf(buf, "%d", sym->flags.size);
	else if(!strnicmp(ext, ".val", 4) || !strnicmp(ext, ".int", 4))
		sprintf(buf, "%d", atoi(sym->data));
	else if(!strnicmp(ext, ".dec", 4))
	{
		dec = strchr(sym->data, '.');
		if(dec)
			++dec;
		else
			dec = "0";
		sprintf(buf, "%d", atoi(dec));
	}
	else if(!stricmp(ext, ".bool"))
	{
		strcpy(buf, "false");
		switch(*sym->data)
		{
		case 'y':
		case 'Y':
		case 't':
		case 'T':
			strcpy(buf, "true");
			break;
		default:
			if(atoi(sym->data))
				strcpy(buf, "true");
		}
	}
	else if(!stricmp(ext, ".count"))
		switch(sym->flags.type)
		{
		case CACHE:
		case SEQUENCE:
		case FIFO:
		case STACK:
			sprintf(buf, "%d", sym->data[3]);
			break;
		default:
			return "1";
		}
	else if(NULL != (prop = Property::find(ext + 1)))
		prop->getProperty(sym->data, buf, getSymbolSize());
	else if(!stricmp(ext, ".max"))
		switch(sym->flags.type)
		{
		case CACHE:
		case SEQUENCE:
		case FIFO:
		case STACK:
			sprintf(buf, "%d", sym->data[4]);
			break;
		default:
			sprintf(buf, "%d", sym->flags.size);
		}
	else if(!stricmp(ext, ".type"))
		switch(sym->flags.type)
		{
		case CACHE:
			return "cache";
		case ALIAS:
			return "alias";
		case ARRAY:
			return "array";
		case INDEX:
			return "index";
		case SEQUENCE:
			return "sequence";
		case STACK:
			return "stack";
		case COUNTER:
			return "counter"; 
		case FIFO:
			return "fifo";
		default:
			return "string";
		}
	else
		buf = NULL;

	return buf;
}

char *ScriptInterp::getTempBuffer(void)
{
	char *tmp = temps[tempidx++];
	if(tempidx >= SCRIPT_TEMP_SPACE)
		tempidx = 0;

	tmp[0] = 0;
	return tmp;
}

char *ScriptInterp::getString(void)
{
	int index = script[stack].index;
	unsigned len = 0;
	unsigned max = getSymbolSize();
	char *tmp = getTempBuffer();
	char *opt;

	while(NULL != (opt = getOption(NULL)) && len < max)
	{
		script[stack].index = index;
		if(!stricmp(opt, ",") || !stricmp(opt, ")"))
			break;

		snprintf(tmp + len, max - len + 1, "%s", getValue(NULL));
		index = script[stack].index;
		len = strlen(tmp);
	}
	return tmp;
}

char *ScriptInterp::getValue(const char *def)
{
	char *opt = getOption(NULL);
	Attr *a;

	if(!opt)
		return (char *)def;

	if(*opt == '+')
		a = attr;
	else
		a = NULL;

	while(a)
	{
		if(!stricmp(a->id, opt + 1))
			break;
		a = a->next;
	}
	
	if(a)
		return a->meta(this, opt);

	opt = getContent(opt);
	if(!opt)
		return (char *)def;

	return opt;
}	

Script::Line *ScriptInterp::getPrescan(void)
{
	if(!script[stack].line)
		return NULL;

	if(script[stack].line->prescan)
		return script[stack].line;

	return NULL;
}

const char *ScriptInterp::getMember(void)
{
	char *cmd = script[stack].line->cmd;

	while(*cmd && *cmd != '.')
		++cmd;

	if(*cmd)
		return ++cmd;
	return NULL;
}

#ifdef	CCXX_NAMESPACES
};
#endif
