/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.mqtt.cs.session.infly;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.netty.handler.codec.mqtt.MqttFixedHeader;
import io.netty.handler.codec.mqtt.MqttMessage;
import io.netty.handler.codec.mqtt.MqttMessageIdVariableHeader;
import io.netty.handler.codec.mqtt.MqttMessageType;
import io.netty.handler.codec.mqtt.MqttQoS;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import org.apache.rocketmq.common.ThreadFactoryImpl;
import org.apache.rocketmq.mqtt.common.facade.LmqQueueStore;
import org.apache.rocketmq.mqtt.common.model.Message;
import org.apache.rocketmq.mqtt.common.model.Queue;
import org.apache.rocketmq.mqtt.common.model.Subscription;
import org.apache.rocketmq.mqtt.cs.channel.ChannelManager;
import org.apache.rocketmq.mqtt.cs.config.ConnectConf;
import org.apache.rocketmq.mqtt.cs.session.QueueFresh;
import org.apache.rocketmq.mqtt.cs.session.Session;
import org.apache.rocketmq.mqtt.cs.session.infly.InFlyCache;
import org.apache.rocketmq.mqtt.cs.session.infly.MqttMsgId;
import org.apache.rocketmq.mqtt.cs.session.infly.PushAction;
import org.apache.rocketmq.mqtt.cs.session.loop.SessionLoop;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

@Component
public class RetryDriver {
    private static Logger logger = LoggerFactory.getLogger(RetryDriver.class);
    @Resource
    private InFlyCache inFlyCache;
    @Resource
    private MqttMsgId mqttMsgId;
    @Resource
    private SessionLoop sessionLoop;
    @Resource
    private PushAction pushAction;
    @Resource
    private ChannelManager channelManager;
    @Resource
    private ConnectConf connectConf;
    @Resource
    private LmqQueueStore lmqQueueStore;
    @Resource
    private QueueFresh queueFresh;
    private Cache<String, RetryMessage> retryCache;
    private static final int MAX_CACHE = 50000;
    private int scheduleDelaySecs = 3;
    private long messageRetryInterval = 3000L;
    private Map<String, Map<Integer, RetryMessage>> sessionNoWaitRetryMsgMap = new ConcurrentHashMap<String, Map<Integer, RetryMessage>>(16);
    private ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(2, (ThreadFactory)new ThreadFactoryImpl("retry_msg_thread_"));

    @PostConstruct
    public void init() {
        this.retryCache = Caffeine.newBuilder().maximumSize(50000L).removalListener((key, value, cause) -> {
            if (value == null || key == null) {
                return;
            }
            if (cause.wasEvicted()) {
                this.saveRetryQueue((String)key, (RetryMessage)value);
            }
        }).build();
        this.scheduler.scheduleWithFixedDelay(() -> this.doRetryCache(), this.scheduleDelaySecs, this.connectConf.getRetryIntervalSeconds(), TimeUnit.SECONDS);
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            ConcurrentMap map = this.retryCache.asMap();
            if (map == null) {
                return;
            }
            for (Map.Entry entry : map.entrySet()) {
                this.saveRetryQueue((String)entry.getKey(), (RetryMessage)entry.getValue());
            }
        }));
    }

    public void unloadSession(Session session) {
        if (session == null) {
            return;
        }
        Map<Integer, RetryMessage> map = this.sessionNoWaitRetryMsgMap.remove(session.getChannelId());
        if (map == null || map.isEmpty()) {
            return;
        }
        for (Map.Entry<Integer, RetryMessage> entry : map.entrySet()) {
            String cacheKey = this.buildKey(entry.getKey(), session.getChannelId());
            this.retryCache.invalidate((Object)cacheKey);
            RetryMessage retryMessage = entry.getValue();
            this.saveRetryQueue(cacheKey, retryMessage);
        }
    }

    private void saveRetryQueue(String key, RetryMessage retryMessage) {
        Message message = retryMessage.message.copy();
        message.setFirstTopic(this.lmqQueueStore.getClientRetryTopic());
        Session session = retryMessage.session;
        int mqttMsgId = retryMessage.mqttMsgId;
        String clientId = session.getClientId();
        if (message.getRetry() >= this.connectConf.getMaxRetryTime()) {
            this.pushAction.rollNext(session, retryMessage.mqttMsgId);
            return;
        }
        String retryQueue = Subscription.newRetrySubscription((String)clientId).toQueueName();
        CompletableFuture result = this.lmqQueueStore.putMessage(new HashSet<String>(Arrays.asList(retryQueue)), message);
        result.whenComplete((storeResult, throwable) -> {
            if (throwable != null) {
                this.retryCache.put((Object)key, (Object)retryMessage);
                return;
            }
            long queueId = storeResult.getQueue().getQueueId();
            String brokerName = storeResult.getQueue().getBrokerName();
            this.pushAction.rollNext(session, mqttMsgId);
            this.scheduler.schedule(() -> {
                Subscription subscription = Subscription.newRetrySubscription((String)clientId);
                List<Session> sessionList = this.sessionLoop.getSessionList(clientId);
                if (sessionList != null) {
                    for (Session eachSession : sessionList) {
                        Set<Queue> set = this.queueFresh.freshQueue(eachSession, subscription);
                        if (set == null || set.isEmpty()) continue;
                        for (Queue queue : set) {
                            if (!Objects.equals(queue.getBrokerName(), brokerName)) continue;
                            this.sessionLoop.notifyPullMessage(eachSession, subscription, queue);
                        }
                    }
                }
            }, (long)this.scheduleDelaySecs, TimeUnit.SECONDS);
        });
    }

    private void doRetryCache() {
        try {
            for (Map.Entry entry : this.retryCache.asMap().entrySet()) {
                try {
                    RetryMessage retryMessage = (RetryMessage)entry.getValue();
                    Message message = retryMessage.message;
                    Session session = retryMessage.session;
                    int mqttMsgId = retryMessage.mqttMsgId;
                    if (System.currentTimeMillis() - retryMessage.timestamp < this.messageRetryInterval) continue;
                    if (MqttMessageType.PUBLISH.equals((Object)retryMessage.mqttMessageType)) {
                        if (session == null || session.isDestroyed()) {
                            this.cleanRetryMessage(mqttMsgId, session.getChannelId());
                            continue;
                        }
                        if (retryMessage.mountTimeout()) {
                            this.saveRetryQueue((String)entry.getKey(), retryMessage);
                            this.cleanRetryMessage(mqttMsgId, session.getChannelId());
                            continue;
                        }
                        this.pushAction.write(session, message, mqttMsgId, retryMessage.qos, retryMessage.subscription);
                        retryMessage.timestamp = System.currentTimeMillis();
                        retryMessage.localRetryTime++;
                        continue;
                    }
                    if (MqttMessageType.PUBREL.equals((Object)retryMessage.mqttMessageType)) {
                        if (session == null || session.isDestroyed() || retryMessage.mountRelTimeout()) {
                            this.retryCache.invalidate(entry.getKey());
                            logger.error("failed to retry pub rel more times,{},{}", (Object)session.getClientId(), (Object)mqttMsgId);
                            this.pushAction.rollNextByAck(session, mqttMsgId);
                            continue;
                        }
                        MqttFixedHeader pubRelMqttFixedHeader = new MqttFixedHeader(MqttMessageType.PUBREL, false, MqttQoS.valueOf((int)retryMessage.qos), false, 0);
                        MqttMessage pubRelMqttMessage = new MqttMessage(pubRelMqttFixedHeader, (Object)MqttMessageIdVariableHeader.from((int)mqttMsgId));
                        session.getChannel().writeAndFlush((Object)pubRelMqttMessage);
                        retryMessage.localRetryTime++;
                        retryMessage.timestamp = System.currentTimeMillis();
                        logger.warn("retryPubRel:{},{}", (Object)session.getClientId(), (Object)mqttMsgId);
                        continue;
                    }
                    logger.error("error retry message type:{}", (Object)retryMessage.mqttMessageType);
                }
                catch (Exception e) {
                    logger.error("", (Throwable)e);
                }
            }
        }
        catch (Exception e) {
            logger.error("", (Throwable)e);
        }
    }

    public void mountPublish(int mqttMsgId, Message message, int qos, String channelId, Subscription subscription) {
        Map<Integer, RetryMessage> old;
        Session session = this.sessionLoop.getSession(channelId);
        if (session == null) {
            return;
        }
        RetryMessage retryMessage = new RetryMessage(session, message, qos, mqttMsgId, MqttMessageType.PUBLISH, subscription);
        this.retryCache.put((Object)this.buildKey(mqttMsgId, channelId), (Object)retryMessage);
        Map<Integer, RetryMessage> noWaitRetryMsgMap = this.sessionNoWaitRetryMsgMap.get(channelId);
        if (noWaitRetryMsgMap == null && (old = this.sessionNoWaitRetryMsgMap.putIfAbsent(channelId, noWaitRetryMsgMap = new ConcurrentHashMap<Integer, RetryMessage>(2))) != null) {
            noWaitRetryMsgMap = old;
        }
        if (!subscription.isRetry() && noWaitRetryMsgMap.size() < this.connectConf.getSizeOfNotRollWhenAckSlow()) {
            noWaitRetryMsgMap.put(mqttMsgId, retryMessage);
            this.pushAction.rollNextNoWaitRetry(session, mqttMsgId);
        }
    }

    private RetryMessage cleanRetryMessage(int mqttMsgId, String channelId) {
        Map<Integer, RetryMessage> retryMsgMap = this.sessionNoWaitRetryMsgMap.get(channelId);
        if (retryMsgMap != null) {
            retryMsgMap.remove(mqttMsgId);
        }
        String key = this.buildKey(mqttMsgId, channelId);
        return this.unMount(key);
    }

    public void mountPubRel(int mqttMsgId, String channelId) {
        Session session = this.sessionLoop.getSession(channelId);
        if (session == null) {
            return;
        }
        RetryMessage retryMessage = new RetryMessage(session, null, MqttQoS.AT_LEAST_ONCE.value(), mqttMsgId, MqttMessageType.PUBREL, null);
        this.retryCache.put((Object)this.buildKey(mqttMsgId, channelId), (Object)retryMessage);
    }

    public RetryMessage unMountPublish(int mqttMsgId, String channelId) {
        RetryMessage retryMessage = this.cleanRetryMessage(mqttMsgId, channelId);
        return retryMessage;
    }

    public RetryMessage unMountPubRel(int mqttMsgId, String channelId) {
        String key = this.buildKey(mqttMsgId, channelId);
        return this.unMount(key);
    }

    private RetryMessage unMount(String key) {
        RetryMessage message = (RetryMessage)this.retryCache.getIfPresent((Object)key);
        if (message != null) {
            this.retryCache.invalidate((Object)key);
        }
        return message;
    }

    public boolean needRetryBefore(Subscription subscription, Queue queue, Session session) {
        Map<Integer, InFlyCache.PendingDown> pendingDowns = this.inFlyCache.getPendingDownCache().all(session.getChannelId());
        if (pendingDowns == null || pendingDowns.isEmpty()) {
            return false;
        }
        InFlyCache.PendingDown pendingDown = null;
        for (Map.Entry<Integer, InFlyCache.PendingDown> entry : pendingDowns.entrySet()) {
            InFlyCache.PendingDown each = entry.getValue();
            if (!each.getSubscription().equals((Object)subscription) || !each.getQueue().equals((Object)queue)) continue;
            pendingDown = each;
            break;
        }
        return pendingDown != null;
    }

    private String buildKey(int mqttMsgId, String channelId) {
        StringBuilder sb = new StringBuilder();
        sb.append(String.valueOf(mqttMsgId));
        sb.append("_");
        sb.append(channelId);
        return sb.toString();
    }

    public class RetryMessage {
        private Session session;
        private Message message;
        private Subscription subscription;
        private int qos;
        private int mqttMsgId;
        private MqttMessageType mqttMessageType;
        private int localRetryTime = 0;
        private static final int MAX_LOCAL_RETRY = 1;
        private long timestamp = System.currentTimeMillis();

        public RetryMessage(Session session, Message message, int qos, int mqttMsgId, MqttMessageType mqttMessageType, Subscription subscription) {
            this.session = session;
            this.message = message;
            this.qos = qos;
            this.mqttMsgId = mqttMsgId;
            this.mqttMessageType = mqttMessageType;
            this.subscription = subscription;
        }

        private boolean mountTimeout() {
            return this.localRetryTime >= 1;
        }

        private boolean mountRelTimeout() {
            return this.localRetryTime >= 3;
        }
    }
}

