/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Objects;
import org.apache.flink.annotation.PublicEvolving;
import org.apache.flink.table.api.ValidationException;

@PublicEvolving
public class IntervalFreshness {
    private static final String SECOND_CRON_EXPRESSION_TEMPLATE = "0/%s * * * * ? *";
    private static final String MINUTE_CRON_EXPRESSION_TEMPLATE = "0 0/%s * * * ? *";
    private static final String HOUR_CRON_EXPRESSION_TEMPLATE = "0 0 0/%s * * ? *";
    private static final String ONE_DAY_CRON_EXPRESSION_TEMPLATE = "0 0 0 * * ? *";
    private static final long SECOND_CRON_UPPER_BOUND = 60L;
    private static final long MINUTE_CRON_UPPER_BOUND = 60L;
    private static final long HOUR_CRON_UPPER_BOUND = 24L;
    private final int interval;
    private final TimeUnit timeUnit;

    private IntervalFreshness(int interval, TimeUnit timeUnit) {
        this.interval = interval;
        this.timeUnit = timeUnit;
    }

    public static IntervalFreshness of(String interval, TimeUnit timeUnit) {
        int validateIntervalInput = IntervalFreshness.validateIntervalInput(interval);
        return new IntervalFreshness(validateIntervalInput, timeUnit);
    }

    private static int validateIntervalInput(String interval) {
        int parsedInt;
        try {
            parsedInt = Integer.parseInt(interval);
        }
        catch (Exception e) {
            String errorMessage = String.format("The freshness interval currently only supports positive integer type values. But was: %s", interval);
            throw new ValidationException(errorMessage, e);
        }
        if (parsedInt <= 0) {
            String errorMessage = String.format("The freshness interval currently only supports positive integer type values. But was: %s", interval);
            throw new ValidationException(errorMessage);
        }
        return parsedInt;
    }

    public static IntervalFreshness ofSecond(String interval) {
        return IntervalFreshness.of(interval, TimeUnit.SECOND);
    }

    public static IntervalFreshness ofMinute(String interval) {
        return IntervalFreshness.of(interval, TimeUnit.MINUTE);
    }

    public static IntervalFreshness ofHour(String interval) {
        return IntervalFreshness.of(interval, TimeUnit.HOUR);
    }

    public static IntervalFreshness ofDay(String interval) {
        return IntervalFreshness.of(interval, TimeUnit.DAY);
    }

    public static void validateFreshnessForCron(IntervalFreshness intervalFreshness) {
        switch (intervalFreshness.getTimeUnit()) {
            case SECOND: {
                IntervalFreshness.validateCronConstraints(intervalFreshness, 60L);
                break;
            }
            case MINUTE: {
                IntervalFreshness.validateCronConstraints(intervalFreshness, 60L);
                break;
            }
            case HOUR: {
                IntervalFreshness.validateCronConstraints(intervalFreshness, 24L);
                break;
            }
            case DAY: {
                IntervalFreshness.validateDayConstraints(intervalFreshness);
                break;
            }
            default: {
                throw new ValidationException(String.format("Unknown freshness time unit: %s.", new Object[]{intervalFreshness.getTimeUnit()}));
            }
        }
    }

    public static String convertFreshnessToCron(IntervalFreshness intervalFreshness) {
        IntervalFreshness.validateFreshnessForCron(intervalFreshness);
        switch (intervalFreshness.getTimeUnit()) {
            case SECOND: {
                return String.format(SECOND_CRON_EXPRESSION_TEMPLATE, intervalFreshness.getIntervalInt());
            }
            case MINUTE: {
                return String.format(MINUTE_CRON_EXPRESSION_TEMPLATE, intervalFreshness.getIntervalInt());
            }
            case HOUR: {
                return String.format(HOUR_CRON_EXPRESSION_TEMPLATE, intervalFreshness.getIntervalInt());
            }
            case DAY: {
                return ONE_DAY_CRON_EXPRESSION_TEMPLATE;
            }
        }
        throw new ValidationException(String.format("Unknown freshness time unit: %s.", new Object[]{intervalFreshness.getTimeUnit()}));
    }

    private static void validateCronConstraints(IntervalFreshness intervalFreshness, long cronUpperBound) {
        int interval = intervalFreshness.getIntervalInt();
        TimeUnit timeUnit = intervalFreshness.getTimeUnit();
        if ((long)interval >= cronUpperBound) {
            throw new ValidationException(String.format("In full refresh mode, freshness must be less than %s when the time unit is %s.", new Object[]{cronUpperBound, timeUnit}));
        }
        if (cronUpperBound % (long)interval != 0L) {
            throw new ValidationException(String.format("In full refresh mode, only freshness that are factors of %s are currently supported when the time unit is %s.", new Object[]{cronUpperBound, timeUnit}));
        }
    }

    private static void validateDayConstraints(IntervalFreshness intervalFreshness) {
        int interval = intervalFreshness.getIntervalInt();
        if (interval > 1) {
            throw new ValidationException("In full refresh mode, freshness must be 1 when the time unit is DAY.");
        }
    }

    public static IntervalFreshness fromDuration(Duration duration) {
        if (duration.equals(duration.truncatedTo(ChronoUnit.DAYS))) {
            return new IntervalFreshness((int)duration.toDays(), TimeUnit.DAY);
        }
        if (duration.equals(duration.truncatedTo(ChronoUnit.HOURS))) {
            return new IntervalFreshness((int)duration.toHours(), TimeUnit.HOUR);
        }
        if (duration.equals(duration.truncatedTo(ChronoUnit.MINUTES))) {
            return new IntervalFreshness((int)duration.toMinutes(), TimeUnit.MINUTE);
        }
        return new IntervalFreshness((int)duration.getSeconds(), TimeUnit.SECOND);
    }

    @Deprecated
    public String getInterval() {
        return String.valueOf(this.interval);
    }

    public int getIntervalInt() {
        return this.interval;
    }

    public TimeUnit getTimeUnit() {
        return this.timeUnit;
    }

    public Duration toDuration() {
        switch (this.timeUnit) {
            case SECOND: {
                return Duration.ofSeconds(this.interval);
            }
            case MINUTE: {
                return Duration.ofMinutes(this.interval);
            }
            case HOUR: {
                return Duration.ofHours(this.interval);
            }
            case DAY: {
                return Duration.ofDays(this.interval);
            }
        }
        throw new IllegalStateException("Unexpected value: " + String.valueOf((Object)this.timeUnit));
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        IntervalFreshness that = (IntervalFreshness)o;
        return Objects.equals(this.interval, that.interval) && this.timeUnit == that.timeUnit;
    }

    public String toString() {
        return "INTERVAL '" + this.interval + "' " + String.valueOf((Object)this.timeUnit);
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.interval, this.timeUnit});
    }

    @PublicEvolving
    public static enum TimeUnit {
        SECOND,
        MINUTE,
        HOUR,
        DAY;

    }
}

