// BMP_QUAN.CPP - MS-Windows BMP Color Quantization Class

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "bmp_quan.h"

static char BQ_FormatErr[] = "file %s invalid BMP format";
static char BQ_IdentErr[] = "file names are identical";
static char BQ_OpenErr[] = "could not open file %s";
static char BQ_OutOfMem[] = "out of memory";
static char BQ_ReadErr[] = "could not read from file %s";
static char BQ_TypeErr[] = "file %s not 24-bit BMP file";
static char BQ_WriteErr[] = "could not write to file %s";

// Reduce number of colors
BOOL BMP_Quantize::Quantize( char *pin, char *pout )
{
  BOOL status = TRUE;   // Return status

  // Save file name pointers
  pin_name = pin;
  pout_name = pout;

  // Validate file names
  if (strcmp(pin, pout) == 0)
  {
    ReportError(BQ_IdentErr, NULL);
    status = FALSE;
  }
 
  // Open input file
  if (status == TRUE)
    if ((pin_file = fopen(pin_name, "rb")) == NULL)
    {
      ReportError(BQ_OpenErr, pin_name);
      status = FALSE;
    }

  // Open output file
  if (status == TRUE)
    if ((pout_file = fopen(pout_name, "wb")) == NULL)
    {
      ReportError(BQ_OpenErr, pout_name);
      status = FALSE;
    }

  // Read bitmap file header
  if (status == TRUE)
    if (fread(&f_header, sizeof(DIB_FileHeader), 1,
        pin_file) != 1)
    {
      ReportError(BQ_ReadErr, pin_name);
      status = FALSE;
    }

  // Check for 'BM' signature
  if (status == TRUE)
    if (f_header.bfType != (UINT) 0x4d42)
    {
      ReportError(BQ_FormatErr, pin_name);
      status = FALSE;
    }

  // Read bitmap info header
  if (status == TRUE)
    if (fread(&i_header, sizeof(DIB_InfoHeader), 1,
        pin_file) != 1)
    {
      ReportError(BQ_ReadErr, pin_name);
      status = FALSE;
    }

  // Check for 24-bit bitmap
  if (status == TRUE)
    if (i_header.biBitCount != 24)
    {
      ReportError(BQ_TypeErr, pin_name);
      status = FALSE;
    }

  // Set input file bitmap offset
  in_offset = f_header.bfOffBits;

  // Set bitmap dimensions
  width = (int) i_header.biWidth;
  height = (int) i_header.biHeight;

  // Allocate input scan line buffer
  if (status == TRUE)
  {
    in_len = GetScanLineWidth((size_t) width * 3);
    if ((pin_scan = new BYTE[in_len]) == NULL)
    {
      ReportError(BQ_OutOfMem, NULL);
      status = FALSE;
    }
  }

  // Allocate output scan line buffer
  if (status == TRUE)
  {
    out_len = GetScanLineWidth((size_t) width);
    if ((pout_scan = new BYTE[out_len]) == NULL)
    {
      ReportError(BQ_OutOfMem, NULL);
      status = FALSE;
    }
  }

  // Set file pointer to beginning of bitmap data
  if (status == TRUE)
    if (Reset() == FALSE)
    {
      ReportError(BQ_ReadErr, pin_name);
      status = FALSE;
    }

  // Build color quantization octree
  if (status == TRUE)
  {
    fputs("\nBuilding color quantization octree ...\n",
        stderr);
    if (BuildTree() == FALSE)
    {
      ReportError(BQ_OutOfMem, NULL);
      status = FALSE;
    }
  }

  // Initialize color palette
  if (status == TRUE)
  {
    fputs("Initializing color palette ...\n", stderr);
    InitPalette();
  }

  // Update bitmap file header
  f_header.bfOffBits = (DWORD) sizeof(DIB_FileHeader) +
      (DWORD) sizeof(DIB_InfoHeader) + (DWORD)
      (sizeof(DIB_Palette) * 256);
  f_header.bfSize = f_header.bfOffBits + (DWORD) width *
      (DWORD) height;

  // Write bitmap file header
  if (status == TRUE)
    if (fwrite(&f_header, sizeof(DIB_FileHeader), 1,
        pout_file) != 1)
    {
      ReportError(BQ_WriteErr, pout_name);
      status = FALSE;
    }

  // Update bitmap info header
  i_header.biBitCount = 8;
  i_header.biClrUsed = 256;
  i_header.biClrImportant = num_color;

  // Write bitmap info header
  if (status == TRUE)
    if (fwrite(&i_header, sizeof(DIB_InfoHeader), 1,
        pout_file) != 1)
    {
      ReportError(BQ_WriteErr, pout_name);
      status = FALSE;
    }

  // Write bitmap palette
  if (status == TRUE)
    if (fwrite(&dib_pal, sizeof(DIB_Palette), 256,
        pout_file) != 256)
    {
      ReportError(BQ_WriteErr, pout_name);
      status = FALSE;
    }

  // Map 24-bit RGB bitmap to 8-bit palette bitmap
  if (status == TRUE)
  {
    fputs("Mapping colors to 8-bit palette ...\n", stderr);
    status = MapColors();

    fputs("\nColor quantization complete\n", stderr);
  }

  if (pin_file != NULL)         // Close input file
    (void) fclose(pin_file);

  if (pout_file != NULL)        // Close output file
  {
    (void) fclose(pout_file);

    if (status == FALSE)
    {
      // Remove output file if error
      (void) remove(pout_name);
    }
  }

  if (pin_scan != NULL)     // Delete bitmap buffer
    delete [] pin_scan;

  if (pout_scan != NULL)    // Delete scan line buffer
    delete [] pout_scan;

  DeleteTree();     // Delete color quantization octree

  return status;
}

// Initialize color palette
void BMP_Quantize::InitPalette()
{
  int i;    // Loop index
  
  // Fill octree color palette entries
  FillPalette(proot, &num_color);

  // Copy octree palette entries to DIB palette
  for (i = 0; i < num_color; i++)
  {
    dib_pal[i].rgbBlue = Palette[i].GetBlue();
    dib_pal[i].rgbGreen = Palette[i].GetGreen();
    dib_pal[i].rgbRed = Palette[i].GetRed();
  }
}

// Map 24-bit RGB colors to 8-bit color palette entries
BOOL BMP_Quantize::MapColors()
{
  int row;              // Row counter
  int col;              // Column counter
  BOOL status = TRUE;   // Return status
  BYTE qcolor;          // Quantized pixel color
  OctColor color;       // 24-bit RGB color

  // Quantize the bitmap colors
  Reset();
  for (row = 0; row < height; row++)
  {
    for (col = 0; col < width; col++)
    {
      // Get 24-bit RGB pixel color
      if (GetPixel(col, row, color) == FALSE)
      {
        ReportError(BQ_ReadErr, pin_name);
        status = FALSE;
        break;
      }
      
      // Get color palette index
      qcolor = QuantizeColor(proot, color);

      // Set 8-bit palette pixel color
      if (SetPixel(col, row, qcolor) == FALSE)
      {
        ReportError(BQ_WriteErr, pout_name);
        status = FALSE;
        break;
      }
    }

    if (status == FALSE)
      break;
  }

  return status;
}

