/* ============ */
/* tstsort.c	*/
/* ============ */
#
#include "defcodes.h"
#
#include <stdio.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/timeb.h>

#include "etime.h"
#include "tsrtdefs.h"
# undef PRIVATE
# if defined(PROFILE)
#	define	PRIVATE
# else
#	define	PRIVATE static
# endif

# undef PUBLIC
#define PUBLIC	extern			/* for function decalrations */

/* Kill 'unreferenced local function removed' */
#pragma warning(disable:4505)
/* ------------------- */
/* FUNCTION PROTOTYPES */
/* ------------------- */
# undef F
# if defined(__STDC__) || defined(__PROTO__)
#	define	F( P )	P
# else
#	define	F( P )	()
# endif
/* INDENT OFF */

PRIVATE void	(*SortFun) F((void *, size_t, size_t,
		int (*)(const void *, const void *)));

PRIVATE void	AbortGracefully F((int));
PRIVATE int	ChkSrtdAry	F((ULONG, char *));
PRIVATE int	CmprFun 	F((const void *, const void *));
PUBLIC	double	GenGausRand	F((void));
PRIVATE int	InitSortRun	F((SORT_TEST_STRU  *));
PRIVATE void	PrintDirections F((char *, char *));
PRIVATE void	PrintSortTarget F((char *));
PUBLIC	double	round		F((double));
PRIVATE void	SkipThisTarget	F((int));
PRIVATE void	SortThisTarget	F((int));
PUBLIC	void	_Wait		F((char *));

# undef F

#define DEFAULT_START	00L
#define DEFAULT_STEP	100L
#define DEFAULT_STOP	2000L
#define MAX_SORT	65535

#define MODIFIED	1
#define NOT_MODIFIED	0

/* ====================================== */
/* Externally Referenced Global Variables */
/* ====================================== */

/* ------------------------------------ */
/* Counts Exchanges (done in each sort) */
/* ------------------------------------ */
ULONG	SwapCtr;

/* =================================== */
/* Locally Referenced Global Variables */
/* =================================== */

static	ULONG		CmprCtr;	/* Counts Comparisons */

static	int		SortList;	/* Bit Vector of Sort Codes */
static	size_t		SortSize = MAX_SORT;
static	long		SortTime = 0;
static	int		PlotTimer;	/* TRUE if Timing desired */
static	ETIME_STRU	TimerStru;

static	long		StartSize= 0,	/* Starting size of sort target */
			StepSize = 100, /* Increment for size of target */
			StopSize = 2000;/* Final Size of sort target	*/

static	double		TotCmprs;	/* Total comparisons during run	*/
static	double		TotSwaps;	/* Total exchanges during run	*/

/*  ------------------------------  */
/*  Sort Objects are designated as: */
/*  ------------------------------  */
/*  R   Random			    */
/*  S   Already Sorted		    */
/*  V   Reverse Sorted		    */
/*  H   Reverse Sorted by Halves    */
/*  F   First Item Out of Sort	    */
/*  T   Every Third Item Inverted   */
/*  Q   All Items Equal		    */
/*  E   Every Eighth Item Different */
/*  A   Alternating Pairs	    */
/*  B   Boresight Simulation	    */
/*  ------------------------------  */

/* ----------------------------------------------- */
/* Sort Codes - Letters that identify sort targets */
/*		when tstsort() sees one on the     */
/*		command line			   */
/* ----------------------------------------------- */
static	char	SortCodes[] = "RSVHFTQEAB";

/* ------------------------------------------------ */
/* Data Types - Labels for sort targets. They are   */
/*		printed on the activity-count recap */
/*		and in the plot-data file when user */
/*		enters P for PlotOpt on the command */
/*		line.				    */
/* ------------------------------------------------ */
static	char	*DataTypes[] =
{
    "Random-Uniform",
    "Already-Sorted",
    "Reverse-Sorted",
    "Reverse-Sorted-by-Halves",
    "First-Item-Out-of-Order",
    "Every-Third-Item-Inverted",
    "All-Items-Equal",
    "Every-Eighth-Item-Different",
    "Alternating-Pairs",
    "Bore-Sight-Simulation"
};

#define	WANT(OptionList, OptNum)	\
	(OptionList &			\
	(1 << ((sizeof(SortCodes) - 2) - OptNum)))

#define	RANDOM		0
#define	ALRDY_SRTD	1
#define	RVRS_SRTD       2
#define	RVRS_HLVS       3
#define	FRST_OUT	4
#define	THRD_INV	5
#define	ALL_SAME	6
#define	EVRY_8TH	7
#define	ALT_PAIRS	8
#define	BORST_SIM	9

typedef ULONG TST_TYP;
TST_TYP	SortTgt[MAX_SORT];

/* INDENT ON */
#include "gausrand.c"
#include "setslist.c"
#include "clongarg.c"
/* ==================================================================== */
/* AbortGracefully - does the right thing at <CTRL> C or <CTRL> break	*/
/* ==================================================================== */
PRIVATE void
AbortGracefully(int unused)
{
    if (unused || !unused)
	exit(1);
}
/* ==================================================================== */
/* ChkSrtdAry - returns TRUE if target array is sorted			*/
/* ==================================================================== */
PRIVATE int
ChkSrtdAry(ULONG Num, char *SortObject)
{
    ULONG   ArrayCtr;
    int     RetVal = TRUE;

    for (ArrayCtr = 1; ArrayCtr < Num; ++ArrayCtr)
    {
	if (CmprFun(SortTgt + (ArrayCtr - 1), SortTgt + ArrayCtr) > 0)
	{
	    E2("Sort Check Failed on %s at index %lu\n",
		SortObject, ArrayCtr);
	    D4("SortTgt[%lu] = %u, SortTgt[%lu] = %u\n",
		ArrayCtr - 1, SortTgt[ArrayCtr - 1],
		ArrayCtr, SortTgt[ArrayCtr]);

	    RetVal = FALSE;
	    break;
	}
    }

    return(RetVal);
}
/* ==================================================================== */
/* CmprFun - compares an integer at x with one at y			*/
/* ==================================================================== */

typedef TST_TYP *PTR_TYP;

PRIVATE int
CmprFun(const void *x, const void *y)
{
    ++CmprCtr;

    return((*(PTR_TYP) x > *(PTR_TYP) y) ? 1 :
	(*(PTR_TYP) x < *(PTR_TYP) y) ? -1 : 0);
}
/* ==================================================================== */
/* GenAltPairs - generates data of alternating pairs - 1, 0, 1, 0, ...	*/
/* ==================================================================== */
PRIVATE void
GenAltPairs(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;

    for (i = 0; i < SortSize; ++i)
    {
	SortTgt[i] = 1;
	if (++i < SortSize)
	{
	    SortTgt[i] = 0;
	}
    }
}
/* ==================================================================== */
/* GenBoreSgtTgt - generates data that simulates boresight data 	*/
/* ==================================================================== */
PRIVATE void
GenBoreSgtTgt(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;
    ULONG   BiasHi = 0, BiasLo = 0, BoreSgtSum = 0;

    double  BiasSum = 0;

#define BORESIGHT_BASE	180.0
#define ONE_SIGMA_ERR	0.5

    srand(2);

    for (i = 0; i < SortSize; ++i)
    {
	double	Bias = round(ONE_SIGMA_ERR * GenGausRand());

	if (Bias < 0)
	{
	    ++BiasLo;
	}
	else if (Bias > 0)
	{
	    ++BiasHi;
	}

	BiasSum += Bias;
	SortTgt[i] = (TST_TYP) (BORESIGHT_BASE + Bias);

	BoreSgtSum += SortTgt[i];
    }

    D3("Mean Bore Sight = %f, Lo Count = %ld, Hi Count %ld, ",
	(double) BoreSgtSum / (double) SortSize, BiasLo, BiasHi);
    D1("Mean Bias = %f\n", BiasSum / (double) SortSize);
}
/* ==================================================================== */
/* GenEqualTgt - generates sort target of equal (or nearly so) data	*/
/* ==================================================================== */
PRIVATE void
GenEqualTgt(TST_TYP *SortTgt, size_t SortSize, int ModifyOpt)
{
    /* ModifyOpt is 0 if no changes in the array are wanted */

    ULONG   i;

    for (i = 0; i < SortSize; ++i)
    {
	SortTgt[i] = SortSize;
    }

    if (ModifyOpt)
    {
	/* Change every eighth element */

	for (i = 0; i < SortSize; ++i)
	{
	    if ((i & 7) == 0)
	    {
		if (i < SortSize / 2)
		{
		    SortTgt[i] -= (TST_TYP) (i + 1);
		}
		else
		{
		    SortTgt[i] -= (TST_TYP) (i + 1) / 2;
		}
	    }
	}
    }
}
/* ==================================================================== */
/* GenEvryThrdInv - generates sort target w/ every third item inverted	*/
/* ==================================================================== */
PRIVATE void
GenEvryThrdInv(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;
    TST_TYP j = 3;

    for (i = 1; i <= SortSize; ++i)
    {
	if (i < j)
	{
	    SortTgt[i - 1] = (TST_TYP) i;
	}
	else
	{
	    j += 3;
	    SortTgt[i - 1] = (TST_TYP) (i - 2);
	}
    }
}
/* ==================================================================== */
/* GenFrstItmOut - generates sorted array except first item is largest		*/
/* ==================================================================== */
PRIVATE void
GenFrstItmOut(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;

    for (i = 1; i < SortSize; ++i)
    {
	SortTgt[i] = (TST_TYP) (i + 1);
    }

    SortTgt[0] = SortSize;
}
/* ==================================================================== */
/* GenRndmTgt - generates array of random numbers into SortTgt		*/
/* ==================================================================== */
PRIVATE void
GenRndmTgt(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;
# if FIXED
    srand(2);
# else
    srand((unsigned) time(NULL));
# endif

    for (i = 0; i < SortSize; ++i)
    {
	SortTgt[i] = (TST_TYP) rand() * (TST_TYP) rand();
    }
}
/* ==================================================================== */
/* GenRevSrtHlv - generates reverse-sorted-by-halves target		*/
/* ==================================================================== */
PRIVATE void
GenRevSrtHlv(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;
    TST_TYP j;

    j = SortSize;
    for (i = 0; i < (SortSize+1) / 2; ++i)
    {
	SortTgt[i] = j;
	j -= 2;
    }
    j = SortSize - 1;
    for (; i < SortSize; ++i)
    {
	SortTgt[i] = j;
	j -= 2;
    }
}
/* ==================================================================== */
/* GenRevSrtdTgt - generates reverse-sorted target array		*/
/* ==================================================================== */
PRIVATE void
GenRevSrtdTgt(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;

    for (i = 0; i < SortSize; ++i)
    {
	SortTgt[SortSize - (i + 1)] = (TST_TYP) (i + 1);
    }
}
/* ==================================================================== */
/* GenSortedTgt - generates sorted array (sequential list)		*/
/* ==================================================================== */
PRIVATE void
GenSortedTgt(TST_TYP *SortTgt, size_t SortSize)
{
    ULONG   i;

    for (i = 0; i < SortSize; ++i)
    {
	SortTgt[i] = (TST_TYP) (i + 1);
    }
}
/* ====================================================================	*/
/* GetElapsedTime - Calculates difference in StartTime and StopTime	*/
/* ====================================================================	*/
static	void
GetElapsedTime( struct _timeb *StartTime,
		struct _timeb *StopTime,
		ULONG  *ElapsedSecs,
		USHORT *ElapsedMillSec)
{
    double  MyElapsedTime, MyStartTime, MyStopTime;
    double  MyElapsedSecs, MyElapsedMillSec;

    MyStartTime = StartTime->time * 1000.0 +
		  StartTime->millitm;
    MyStopTime  = StopTime->time * 1000.0 +
		  StopTime->millitm;

    MyElapsedTime = fabs(MyStopTime - MyStartTime)/1000.0;

    MyElapsedMillSec = 1000 * modf(MyElapsedTime,
	&MyElapsedSecs);

    *ElapsedSecs    = (ULONG)  MyElapsedSecs;
    *ElapsedMillSec = (USHORT) MyElapsedMillSec;
}
/* ==================================================================== */
/* InitSortRun - Initialize variables - local & global - for next sort	*/
/* ==================================================================== */
PRIVATE int
InitSortRun(SORT_TEST_STRU *SortParms)
{
    int     SortOpt = FALSE;

    if (SortParms->argc <= 1)
    {
	PrintDirections(SortParms->ExecName,
		SortParms->ExecLabel);
	SortOpt = FALSE;
    }
    else
    {
	int	PlotArg = toupper(*(SortParms->argv[1]));

	SortFun = SortParms->SortFun;

	PlotTimer = (PlotArg == PRTIM);

	if (!PlotTimer && PlotArg != NOTIM)
	{
	    AX(("Unknown Timing Argument = %c\n",
		    *(SortParms->argv[1])));
	    exit(1);
	}

	/* --------------------------------- */
	/* Get Sort Test Range and Step Size */
	/* --------------------------------- */
	SortParms->argv += 2;
	if (CvtAscLong(*SortParms->argv, DEFAULT_STEP, &StepSize))
	{
	    if (StepSize <= 0)
	    {
		StepSize = DEFAULT_STEP;
	    }
	    if (CvtAscLong(*++(SortParms->argv), DEFAULT_START, &StartSize))
	    {
		if (StartSize < 0)
		{
		    StartSize = DEFAULT_START;
		}
		if (CvtAscLong(*++(SortParms->argv), DEFAULT_STOP, &StopSize))
		{
		    ++(SortParms->argv);

		    if (StopSize < 0)
		    {
			StopSize = DEFAULT_STOP;
		    }
		}
	    }
	}

	SetSortObjects(SortParms->argv, SortParms->OKCodes, &SortList);

	SortOpt = TRUE;
    }
    return(SortOpt);
}
/* ==================================================================== */
/* PrintDirections - provides directions on How to Use tstsort		*/
/* ==================================================================== */
PRIVATE void
PrintDirections(char *ExecName, char *ExecLabel)
{
    /* Clear Screen */
    AX(("[0m[1m[2J"));
    AX(("\r%s: Tests %s sorting method\n", ExecName, ExecLabel));
    AX(("Syntax:\n"));
    AX(("   %s [PlotOpt] [Step] [Start] [Stop] <+->[SortTgts]\n", ExecName));
    AX(("   PlotOpt: Plot Data Option\n"));
    AX(("      P  Print Plot Data  (millisecond timer activated)\n"));
    AX(("      N  Don't Bother (millisecond timer not activated)\n"));
    AX(("   Step:    Step Size, in Items, Between Sort Passes\n"));
    AX(("   Start:   Number of Items in First Sort Pass\n"));
    AX(("   Stop:    Number of Items in Final Sort Pass\n"));
    AX(("   SrtTgts: One or More Letters Designating Sort Targets\n"));
    AX(("            Not required if you want all the sort targets.\n\n"));
    AX(("            +SrtTgts: Exclusive Designation of Sort Target(s)\n"));
    AX(("            -SrtTgts: Omit Designated Sort Targets\n\n"));
    AX(("Sort Targets are designated as follows (lower case OK):\n"));
    AX(("R   Random                     T   Every 3rd Item Inverted\n"));
    AX(("S   Sorted                     Q   All Items Equal\n"));
    AX(("V   Reverse Sorted             E   Every 8th Item Different\n"));
    AX(("H   Reverse Sorted by Halves   A   Alternating Pairs\n"));
    AX(("F   First Item Out of Sort     B   Boresight Simulation\n\n"));
    AX(("Sort Activity Counts are printed on stderr.\n"));
    AX(("Plot data are printed on stdout.\n"));
    _Wait("Press <CR> to Continue");
    AX(("------------------------------------------------\n"));
    AX(("Examples:\n"));
    AX(("%s P 100 0 2000 +RSVHF > [Outfile]\n", ExecName));
    AX(("   Tests Sort Function for arrays ranging in size\n"));
    AX(("   from 0 to 2000 elements in steps of 100 elements.\n"));
    AX(("   Activates millisecond timer, writes plot data.\n"));
    AX(("   Sorts only the targets specified\n\n"));
    AX(("%s N 100 0 2000 -TQEAB > [Outfile]\n", ExecName));
    AX(("   Tests Sort Function without plot-timing output\n"));
    AX(("   for arrays ranging in size from 0 to 2000 elements\n"));
    AX(("   in steps of 100 elements.\n"));
    AX(("   Excludes specified sort targets TQEAB.\n\n"));
}
/* ==================================================================== */
/* PrintSortTarget - prints sort target with UserLabel			*/
/* ==================================================================== */
PRIVATE void
PrintSortTarget(char *UserLabel)
{
    ULONG   i;

    (E1("%s\n", UserLabel));

    for (i = 1; i <= SortSize; ++i)
    {
	(E2("%5lu  %5lu\n", i, SortTgt[i - 1]));
    }

    fflush(NULL);
}
/* ====================================================================	*/
/* SetStartTime - Rolls clock over to next second tick			*/
/* ====================================================================	*/
static	void
SetStartTime(struct _timeb *StartTime)
{
    time_t  Seconds = time((time_t *) NULL);

    while (Seconds == time((time_t *) NULL))
	    ;
    _ftime(StartTime);
}
/* ==================================================================== */
/* SkipThisTarget - Prints activity recap that indicates NO SORT	*/
/* ==================================================================== */
PRIVATE void
SkipThisTarget(int TgtIdx)
{
    E5("%*s %10s  (%c) %s\n", (TgtIdx == 0) ? 13 : 20,
	"NO-SORT", "NO-SORT",
	SortCodes[TgtIdx], DataTypes[TgtIdx]);
}
/* ==================================================================== */
/* SortThisTarget - Executes next sort sequence with timekeeping added	*/
/* ==================================================================== */
PRIVATE void
SortThisTarget(int TgtIdx)
{
    CmprCtr = SwapCtr = 0;

    PX(("Starting Sort on %s\n", DataTypes[TgtIdx]));

    P(PrintSortTarget("UNSORTED ARRAY"));

    if (PlotTimer)
    {
	ElapsedTimer(START_TIMER, &TimerStru);
	(*SortFun) (SortTgt, SortSize, sizeof(TST_TYP), CmprFun);
	SortTime = ElapsedTimer(READ_TIMER, &TimerStru);

	AX((" %5.0f", (((double) SortTime / 1024.) * 1000.0)));
    }
    else
    {
	(*SortFun) (SortTgt, SortSize, sizeof(TST_TYP), CmprFun);
    }

    TotCmprs += CmprCtr;
    TotSwaps += SwapCtr;

    E5("%*lu %10lu  (%c) %s\n", (TgtIdx == 0) ? 13 : 20,
	CmprCtr, SwapCtr, SortCodes[TgtIdx], DataTypes[TgtIdx]);

    PX(("\t%10f", (double)CmprCtr + (double)SwapCtr));

    fflush(NULL);
    if (ChkSrtdAry(SortSize, DataTypes[TgtIdx]) == TRUE)
    {
	P(PrintSortTarget("SORTED ARRAY"));
    }
    else
    {
	(PrintSortTarget("BADLY-SORTED ARRAY"));
	exit(1);
    }
}
/* ==================================================================== */
/* tstsort - tests sort functions					*/
/* ==================================================================== */
void
tstsort(SORT_TEST_STRU *SortParms)
{
    int     SortOpt;
    long    NextSize;
    USHORT  ElapsedMillSec;
    time_t  ElapsedSeconds, Hrs, Mins, Secs;
    struct  _timeb  StartTime, StopTime;

    /* -------------------- */
    /* Execute exit() on ^C */
    /* -------------------- */
    signal(SIGINT, AbortGracefully);

    SortOpt = InitSortRun(SortParms);

    if (SortOpt)
    {
# if defined(REDIRECT)
	/* ------------------------------------ */
	/* This section redirects stderr output */
	/* console to disk file sort_std.err.	*/
	/* ------------------------------------ */
	char    StdErrName[] = "sort_std.err";
	if (freopen(StdErrName, "a", stderr))
	{
	    char    *FullPathName;
	    FullPathName =
		_fullpath(NULL, StdErrName, 0);

	    AX(("File stderr redirected to %s\n",
		FullPathName));
	    free(FullPathName);
	}
	else
	{
	    ("tstsort(): Could not redirect stderr\n");
	    perror(StdErrName);
	    exit(1);
	}

# endif

	E3("Generating Targets in [%ld (%ld) %ld]\n",
	    StartSize, StepSize, StopSize);
	E1("Sort Function is %s\n", SortParms->ExecLabel);
	E1("PlotTimer Option is %s\n",
		(PlotTimer) ? "TRUE" : "FALSE");

	if (PlotTimer)
	{
	    int     i;

	    /* ------------------------------------ */
	    /* Put Individual Timing Data on stsout */
	    /* ------------------------------------ */
	    AX(("METHOD\t%s\n", SortParms->ExecLabel));
	    AX(("DATA TYPES\t%d\n", NUM_ITEMS(DataTypes)));

	    for (i = 1; i <= NUM_ITEMS(DataTypes); ++i)
	    {
		AX(("%2d:  %s\n", i, DataTypes[i - 1]));
	    }
	    AX(("*     N   "));
	    for (i = 1; i <= NUM_ITEMS(DataTypes); ++i)
	    {
		AX(("%3d   ", i));
	    }
	    AX(("\n"));
	}

    	SetStartTime(&StartTime);

	for (NextSize = StartSize;
	     NextSize <= StopSize;
	     NextSize += StepSize)
	{
	    struct  _timeb  SuiteTime;

	    if (NextSize > MAX_SORT)
	    {
		break;
	    }
	    E("\n  Items  Comparisons  Exchanges  Code/Target I.D.\n");
	    E1("%7ld", NextSize);

	    SortSize = (size_t) NextSize;
	    TotCmprs = TotSwaps =  0;
	    if (PlotTimer)
	    {
		AX(("%7u", SortSize));
	    }

	    SetStartTime(&SuiteTime);

	    if (WANT(SortList, 0))
	    {
		GenRndmTgt(SortTgt, SortSize);
		SortThisTarget(0);
	    }
	    else
	    {
		SkipThisTarget(0);
	    }

	    if (WANT(SortList, 1))
	    {
		GenSortedTgt(SortTgt, SortSize);
		SortThisTarget(1);
	    }
	    else
	    {
		SkipThisTarget(1);
	    }

	    if (WANT(SortList, 2))
	    {
		GenRevSrtdTgt(SortTgt, SortSize);
		SortThisTarget(2);
	    }
	    else
	    {
		SkipThisTarget(2);
	    }

	    if (WANT(SortList, 3))
	    {
		GenRevSrtHlv(SortTgt, SortSize);
		SortThisTarget(3);
	    }
	    else
	    {
		SkipThisTarget(3);
	    }

	    if (WANT(SortList, 4))
	    {
		GenFrstItmOut(SortTgt, SortSize);
		SortThisTarget(4);
	    }
	    else
	    {
		SkipThisTarget(4);
	    }

	    if (WANT(SortList, 5))
	    {
		GenEvryThrdInv(SortTgt, SortSize);
		SortThisTarget(5);
	    }
	    else
	    {
		SkipThisTarget(5);
	    }

	    if (WANT(SortList, 6))
	    {
		GenEqualTgt(SortTgt, SortSize, NOT_MODIFIED);
		SortThisTarget(6);
	    }
	    else
	    {
		SkipThisTarget(6);
	    }

	    if (WANT(SortList, 7))
	    {
		GenEqualTgt(SortTgt, SortSize, MODIFIED);
		SortThisTarget(7);
	    }
	    else
	    {
		SkipThisTarget(7);
	    }

	    if (WANT(SortList, 8))
	    {
		GenAltPairs(SortTgt, SortSize);
		SortThisTarget(8);
	    }
	    else
	    {
		SkipThisTarget(8);
	    }

	    if (WANT(SortList, 9))
	    {
		GenBoreSgtTgt(SortTgt, SortSize);
		SortThisTarget(9);
	    }
	    else
	    {
		SkipThisTarget(9);
	    }

	    if (PlotTimer)
	    {
		AX(("\n"));
	    }

	    _ftime(&StopTime);
	    GetElapsedTime(&SuiteTime, &StopTime,
		&ElapsedSeconds, &ElapsedMillSec);

	    E3("Suite Stats: %u Items/Target, %lu.%.2u Seconds\n",
		SortSize, ElapsedSeconds, ElapsedMillSec/10);
	    E2("\t     %.f Comparisons, %.f Exchanges\n",
		TotCmprs, TotSwaps);
	}

	E("\n");

	_ftime(&StopTime);

    	GetElapsedTime(&StartTime, &StopTime,
	    &ElapsedSeconds, &ElapsedMillSec);

    	Hrs  = ElapsedSeconds/3600;
    	Mins = ElapsedSeconds/60 - Hrs*60;
    	Secs = ElapsedSeconds - 60 * (Hrs * 60 + Mins);

    	E4("Total Sort Time: %lu Hrs, %lu Mins, %lu.%.2u Secs\n\n",
	    Hrs, Mins, Secs, ElapsedMillSec/10);
    }
}
