# ============= #
# prepgraf.awk	#
# =============================================================	#
# This program prepares plot-data files for program grafix	#
# using data created by a run of a program that calls function	#
# tstsort(). If the plot-option parameter on the command line	#
# is a P, a file of the appropriate format will be created on	#
# stdout.							#
#								#
# To run this program, say 					#
#								#
#   awk -f prepgraf.awk  file1	file2  ...			#
#	 							#
# where file1 ... are produced by runs on sort programs that	#
# use the same command line contents. You cannot specify more	#
# than five files because grafix cannot plot more than five	#
# curves at a time.						#
#								#
# I have prepared five files that can be used as a test,	#
#								#
#	bq10kplt.tim						#
#	fq10kplt.tim						#
#	h10kplt.tim						#
#	s10kplt.tim						#
#	sq10kplt.tim						#
#								#
# and prepgraf.bat to use them.					#
#								#
# One output file will be created for each data type.		#
#								#
# The names of the output files are of the form			#
#								#
#   GRAFnn_m.DAT						#
#								#
# where nn is 1 or 2 random digits and m is a digit or letter.	#
# The prefix 'grafnn_' is a constant for the run.		#
#                                                               #
# ============================================================= #
BEGIN\
{
    GetSortMethod();
    GetDataTypes();
    GetPlotData();
    WritePlotFiles();
}
# ======================================
# Error - Prints error message and exits
# ======================================
function Error(s)
{
    print "Error: " s;
    exit;
}
# ==============================================================
# GetDataTypes - Get types of data from second line of each file
# ==============================================================
function GetDataTypes(		i, j, RdStat)
{
    for (i = 1; i < ARGC; ++i)
    {
	RdStat = getline < ARGV[i];
	if (RdStat <= 0)
	{
	    RdErr(RdStat, ARGV[i]);
	}
	if (!($1 == "DATA" && $2 == "TYPES"))
	{
	    printf("File %s - " , ARGV[i]);
	    Error("Second line must begin with 'DATA TYPES'");
	}
	else
	{
	    if (1 == i)
	    {
		NumDataTypes = $3;
		for (j = 1; j <= NumDataTypes; ++j)
		{
		    RdStat = getline < ARGV[i];
		    if (RdStat <= 0)
		    {
			printf("Not Enough Labels for DATA TYPES\n");
			RdErr(RdStat, ARGV[i]);
		    }
		    DataTypes[j] = $2;
		}
		RdStat = getline < ARGV[i];
	    }
	    else
	    {
		if ($3 != NumDataTypes)
		{
		    printf("Count of DATA TYPES in file %s ", ARGV[i]);
		    printf("doesn't match master file %s\n",  ARGV[1]);
		    Error("Expected Count of " NumDataTypes ", saw " $3);
		}
		for (j = 1; j <= NumDataTypes; ++j)
		{
		    RdStat = getline < ARGV[i];
		    if (RdStat <= 0)
		    {
			printf("Not Enough Labels for DATA TYPES\n");
			RdErr(RdStat, ARGV[i]);
		    }
		    if ($2 == DataTypes[j])
		    {
			continue;
		    }

		    printf("DATA TYPES in file %s don't match master file %s\n",
			ARGV[i], ARGV[1]);
		    Error("Expected " DataTypes[j] ", saw " $j);
		}
		RdStat = getline < ARGV[i];
	    }
	}

	LineCt[i] += NumDataTypes + 2;
    }
}
# ==============================================================
# GetPlotData - Extracts data from plot files into memory arrays
# ==============================================================
function GetPlotData(		i, j, RdStat)
{
# Get independent variables from first file and synchronize with other files.
# Get dependent variables from all files. Must have NumDataTypes variables.
#
    for (i = 1; i < ARGC; ++i)
    {
	NumPts = 0;
	for (; ;)
	{
	    RdStat = getline < ARGV[i];
	    if (RdStat <= 0)
	    {
		if (RdStat == 0)	# Reached EOF
		{
		    break;
		}
		else
		{
		    RdErr(RdStat, ARGV[i]);
		}
	    }

	    ++LineCt[i];

	    if ((NF - 1) < NumDataTypes)
	    {
		printf("File %s - ", ARGV[i]);
		print $0
		Error("Too few items in data line # " LineCt[i]);
	    }

	    ++NumPts;

	    if (1 == i)
	    {
		X[NumPts] = $1;
	    }
	    else if (X[NumPts] != $1)
	    {
		printf("Independent Variable Mismatch, File %s, line # %d\n",
		    ARGV[i], LineCt[i]);
		Error("Expected = " X[NumPts] ", Actual = " $i);
	    }

	    for (j = 1; j <= NumDataTypes; ++j)
	    {
		Y[j, NumPts, i] = $(j+1);
	    }
	}
    }
}
# =====================================================================
# GetSortMethod - Gets name of sort method from first line of each file
# =====================================================================
function GetSortMethod(		i, j, RdStat)
{
    for (i = 1; i < ARGC; ++i)
    {
	RdStat = getline < ARGV[i];
	if (RdStat <= 0)
	{
	    RdErr(RdStat, ARGV[i]);
	}
	if ($1 == "METHOD")
	{
	    for (j = 2; j < NF; ++j)
	    {
		Methods[i] = Methods[i] $j " ";
	    }

	    Methods[i] = Methods[i] $j;
	}
	else
	{
	    printf("File %s - " , ARGV[i]);
	    Error("Keyword (first field) on first line must be METHOD");
	}

	MethodCt = ARGC - 1;
	++LineCt[i];
    }
}
# ====================================================================
# PrintDataList - Writes X & Y[i] data for each point to file PlotSpec
# ====================================================================
function PrintDataList(X, Y, PlotNum, NumPts, MethodCt, PlotSpec,   i, j)
{
    for (i = 1; i <= NumPts; ++i)
    {
	printf("%6s", X[i]) >> PlotSpec;
	for (j = 1; j <= MethodCt; ++j)
	{
	    printf("%10s", Y[PlotNum, i, j]) >> PlotSpec;
	}
	printf("\n") >> PlotSpec;
    }
}
# =========================================================
# PrintHeader - Writes header data for DataType to PlotSpec
# =========================================================
function PrintHeader(MethodCt, NumPts, DataType, Methods, PlotSpec,   i)
{
# "Format=XYY_ASCII Curves=%d Points=%d Title=%s Data;\
#  Xaxis=Number of Elements;\
#  Yaxis=Comparisons + Exchanges;\
#  Label=%s; %s; %s; %s; %s; %s;"\	<== Max of 5
#  "End of Labels"

    printf("\"Format=XYY_ASCII Curves=%d Points=%d Title=%s;\\\n",
      MethodCt, NumPts, DataType) >> PlotSpec;
    printf("Xaxis=Number of Elements;\\\n") >> PlotSpec;
    printf("Yaxis=Time in Milliseconds;\\\n") >> PlotSpec;
    printf("Label=") >> PlotSpec;
    for (i = 1; i < MethodCt; ++i)
    {
	printf("%s;\\\n", Methods[i]) >> PlotSpec;
    }
    printf("%s;\"\n\"End of Labels\"\n", Methods[MethodCt]) >> PlotSpec;
}
# =====================================================
# RdErr - Abandons job on read error with error message
# =====================================================
function RdErr(Status, FileName)
{
    if (0 == Status)
    {
	Error("Unexpected EOF on file: " FileName);
    }
    else if (-1 == Status)
    {
	Error("Read error on file: " FileName);
    }
    else
    {
	Error("Unknown error code in RdErr: " Status);
    }
}
# ========================================================
# WritePlotFiles - Writes plot files for each type of data
# ========================================================
function WritePlotFiles(	i, PlotSpec, Prefix)
{
    srand();
    Prefix = "graf" int(100 * rand()) "_";

    for (i = 1; i <= NumDataTypes; ++i)
    {
#	PlotSpec = Prefix i ".dat";
	PlotSpec = sprintf("%s%0X.dat", Prefix, i);
	printf("Plot file # %d = %s\n", i, PlotSpec);
	printf("") > PlotSpec;
	PrintHeader(MethodCt, NumPts, DataTypes[i], Methods, PlotSpec);
	PrintDataList(X, Y, i, NumPts, MethodCt, PlotSpec);
    }
}
