#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <ctype.h>

#include "as_util.h"

typedef int BOOL;

#ifndef FALSE
#define FALSE 0
#define TRUE  1
#endif

static int    giCurrarg;
static int    giArgc;
static char **gaArgv;
static int    giArgerror = ARG_NOINIT;
static int   *gaDone;

static int    TestUsage( char *parm );

/***************
* Function  : ArgCheck()
* Purpose   : check argument-vector for non-processed arguments; return each
*             no-processed arguments in subsequent calls
* Parameters: void
* Return    : pointer to first non-processed argument found, NULL if all
*             arguments are processed
****************/

char *ArgCheck( void )
{
    int i;

    for ( i = 1; i < giArgc; i++ ) {
        if ( gaDone[i] == 0 ) {
            gaDone[i] = 1;
            return gaArgv[i];
        }
    }

    return NULL;
}

/***************
* Function  : ArgErr()
* Purpose   : return character representation of last error occurred
* Parameters: none
* Return    : pointer to errorstring
****************/

char *ArgErr( void )
{

    switch ( giArgerror ) {
        case ARG_ERROR      : return gaArgv[ giCurrarg ];
        case ARG_NOINIT     : return "ARG_NOINIT";
        case ARG_NOTENOUGH  : return "ARG_NOTENOUGH";
        case ARG_NODELIM    : return gaArgv[ giCurrarg ];
        case ARG_INVTYPE    : return gaArgv[ giCurrarg ];
        case ARG_INVDATE    : return gaArgv[ giCurrarg ];
        case ARG_NOTFOUND   : return "ARG_NOTFOUND";
        case ARG_USAGE      : return "ARG_USAGE";
    }

    return NULL;
}

/***************
* Function  : ArgGet()
* Purpose   : Search an argument in gaArgv based on a keyword; type indicates
*             what to do with the value of the found argument;
*             if type = ARG_SWITCH no more parameters are necessary
*             if type = ARG_DATE a date-format and a pointer to a date_t are
*                expected
*             all other types: 3rd parameter is a pointer to an appropriate
*                buffer
* Parameters: key, string to be searched for in argument-vector
*             type, indicates type of parameter-value
* Return    : ARG_OK or value of giArgerror
****************/

int ArgGet( char *key, int type, ... )
{
    va_list buf;
    char    *p, *k;
    BOOL    part;
    int     i;
    date_t  date;

    va_start( buf, type );

    if ( giArgerror && giArgerror != ARG_NOTFOUND ) {
        return ARG_ERROR;
    }

    for ( i = 1; i < giArgc; i++ ) {
        giCurrarg = i;
        if ( gaDone[i] ) {
            continue;
        }

        if ( gaArgv[i][0] != '/' && gaArgv[i][0] != '-' ) {
            return ( giArgerror = ARG_NODELIM );
        }
        p = gaArgv[i] + 1;
        k = key;
        part = FALSE;                       /* no partial keyword            */
        while ( *k && *p && *p != '=' ) {
            if ( *k == '*' ) {
                k++;
                part = TRUE;                /* partial allowed               */
            }
            if ( *p != *k && isalpha(*p) && ( *p | 0x20 ) != ( *k | 0x20 ) ) {
                break;
            }
            p++;
            k++;
        }
        if ( type == ARG_SWITCH ) {
            if ( *p == '\0' && ( *k == '\0' || *k == '*' || part == TRUE ) )
            {
                gaDone[i] = 1;
                return ( giArgerror = ARG_OK );
            }
        }
        else if ( *p == '=' ) {             /* found end of parametername    */
            if ( *k == '\0' || *k == '*' || part == TRUE ) {
                switch ( type ) {
                case ARG_STRING :
                    strcpy( va_arg( buf, char *), p + 1 );
                    break;
                case ARG_CHAR   :
                    *( va_arg( buf, char *) ) = *( p + 1 );
                    break;
                case ARG_INT    :
                    *( va_arg( buf, int *) ) = atoi( p + 1 );
                    break;
                case ARG_LONG   :
                    *( va_arg( buf, long *) ) = atol( p + 1 );
                    break;
                case ARG_FLOAT  :
                    *( va_arg( buf, float *) ) = (float)atof( p + 1 );
                    break;
                case ARG_DOUBLE :
                    *( va_arg( buf, double *) ) = atof( p + 1 );
                    break;
                case ARG_DATE   :
                    date = str2base( va_arg( buf, char *), p + 1 );
                    *( va_arg( buf, date_t *) ) = date;
                    if ( date < 1.0 ) {
                        return ( giArgerror = ARG_INVDATE );
                    }
                    break;
                default         :
                    return ( giArgerror = ARG_INVTYPE );
                }
                gaDone[i] = 1;
                return ( giArgerror = ARG_OK );
            }
        }
    }

    return ( giArgerror = ARG_NOTFOUND );
}

/***************
* Function  : ArgInit()
* Purpose   : initialize global giArgc and gaArgv; take some preliminary tests
* Parameters: argc, number of arguments on commandline
*             argv, argument-vector
*             mandatory, number of arguments expexted (apart from argv[0])
* Return    : ARG_NOTENOUGH or ARG_USAGE or number of arguments found (after
*             decomposition )
****************/

int ArgInit( int argc, char *argv[], int mandatory )
{
    int     i, cnt;

    if ( gaArgv ) {                    /* was initialized, so clean up first */
        free( gaArgv );
        free( gaDone );
    }

    if ( argc == 1 ) {
        return ( giArgerror = ARG_NOARGS );
    }

    gaArgv = ( char **)calloc( giArgc = argc, sizeof( char *) );

    cnt = giArgc = argc;

    for ( i = 0; i < cnt; i++ ) {
        gaArgv[i] = argv[i];
    }

    if ( giArgc < mandatory + 1 ) {
        return ( giArgerror = ARG_NOTENOUGH );
    }

    gaDone = ( int * )calloc( giArgc, sizeof( int ) );

    if ( TestUsage( gaArgv[1] ) ) {
        return ( giArgerror = ARG_USAGE );
    }

    giArgerror = ARG_OK;

    return ( giArgc - 1 );                  /* all args except progname      */
}

/***************
* Function  : ArgInitPrgm()
* Purpose   : see ArgInit; additionally argv[0] is split with SplitFile()
* Parameters: argc, number of arguments on commandline
*             argv, argument-vector
*             mandatory, number of arguments expexted (apart from argv[0])
*             file, pointer to a file-structure (disk,dir,name,ext)
* Return    : returnvalue of ArgInit()
****************/

int   ArgInitPrgm( int argc, char *argv[], int mandatory, FILE_p file )
{

    SplitFile( argv[0], file );             /* split the programname         */

    return ArgInit( argc, argv, mandatory ); /* initialize the arguments     */
}

/***************
* Function  : SplitFile()
* Purpose   : split a filename into its building blocks
* Parameters: name, compound filename
*             file, pointer to a file-structure (disk,dir,name,ext)
* Return    :
****************/

void    SplitFile( char *name, FILE_p file )
{
    char *d, *dir, *fn, *ext, *p, *q;

    memset( file, 0, sizeof( FILE_t ) );    /* initialize the structure      */

 /* cut off drive-name or (conceiled) logical */
    if ( d = strchr( name, ':' ) ) {
        strncpy( file->disk, name, d - name + 1 );
        *( file->disk + ( d - name + 1 ) ) = '\0';
        dir = d + 1;
    }
    else {
        dir = name;
    }

    if ( *dir == '[' ) {                    /* VMS directory                 */
	p = strrchr( dir, ']' );
        strncpy( file->dir, dir, p - dir + 1 );
        *( file->dir + ( p - dir + 1 ) ) = '\0';
        fn = p + 1;
    }
    else if ( *dir == '/' ) {              /* UNIX directory                */
        p = strrchr( dir, '/' );
        strncpy( file->dir, dir, p - dir + 1 );
        *( file->dir + ( p - dir + 1 ) ) = '\0';
        fn = p + 1;
    }
    else if ( *dir == '\\' ) {              /* DOS directory                 */
        p = strrchr( dir, '\\' );
        strncpy( file->dir, dir, p - dir + 1 );
        *( file->dir + ( p - dir + 1 ) ) = '\0';
        fn = p + 1;
    }

    ext = strrchr( fn, '.' );               /* search the extension          */
    if ( ext == NULL ) {
        strcpy( file->name, fn );           /* no extension found            */
    }
    else {
        strncpy( file->name, fn, ext - fn );
        *( file->name + ( ext - fn ) ) = '\0';
        strcpy( file->ext, ext + 1 );
    }

    return;
}

/***************
* Function  : TestUsage()
* Purpose   : test if a parameter is a cry for help ( /? or /h*elp )
* Parameters: pointer to parameter-string
* Return    : TRUE (help is wanted) or FALSE (no help wanted)
****************/

static int TestUsage( char *parm )
{
    char *p, *q = "help";

DBUG( "AS_ARGS> TestUsage: " );

    if ( strcmp( parm, "/?" ) == 0 ) {
        return 1;                           /* that's obvious                */
    }

    p = parm + 1;

    if ( *p == '\0' ) return 0;             /* can never be /help            */

    while ( *p && *q && ( *p | 0x20 ) == *q ) {
        p++, q++;
    }

    return ( *p == '\0' );   /* if p is at end of parm, parm was like /h*elp */
}

/***************
* Function   : main()
* Purpose    : test the above functions
* Commandline: /r[eal]=999.99 /g[etal]=999 /t*ext=string /swi[tch]
* Return     :
****************/

#ifdef DRIVER

main( int argc, char *argv[] )
{
    int    getal;
    double real;
    char   buffer[80];
    char   *p;
    int    rc;
    date_t date;
    FILE_t file;

    rc = ArgInitPrgm( argc, argv, 4, &file );
    printf( "Drive: %s\n", file.disk );
    printf( "Dir  : %s\n", file.dir );
    printf( "Name : %s\n", file.name );
    printf( "Ext  : %s\n", file.ext  );

    if ( rc == ARG_NOTENOUGH ) {
        printf( "Not enough arguments: %d %d\n", argc, 4 );
        exit( ARG_NOTENOUGH );
    }
    else if ( rc == ARG_USAGE ) {
        printf( "\aUsage....\n" );
        exit( ARG_USAGE );
    }

    rc = ArgGet( "RE*AL", ARG_DOUBLE, &real );
    printf( "REAL: %d %f\n", rc, real );

    rc = ArgGet( "G*ETAL", ARG_INT, &getal );
    printf( "GETAL: %d %d\n", rc, getal );

    rc = ArgGet( "T*EXT", ARG_STRING, buffer );
    printf( "TEXT: %d %s\n", rc, buffer );

    rc = ArgGet( "SWI*TCH", ARG_SWITCH );
    printf( "SWITCH: %d\n", rc );

    rc = ArgGet( "SD", ARG_DATE, "%d1-%m5-%y1", &date );
    printf( "DATE: %d %s\n", rc, base2str( date, "%d1-%m5-%y1", buffer) );

    SplitFile( "PICA:[PICATTNS.C.NLS]IPTLP012.EXE", &file );
    printf( "Drive: %s\n", file.disk );
    printf( "Dir  : %s\n", file.dir );
    printf( "Name : %s\n", file.name );
    printf( "Ext  : %s\n", file.ext  );

    SplitFile( "/disk2/picaport/utl/dbcheck/dbrcheck.c", &file );
    printf( "Drive: %s\n", file.disk );
    printf( "Dir  : %s\n", file.dir );
    printf( "Name : %s\n", file.name );
    printf( "Ext  : %s\n", file.ext  );

    SplitFile( "D:\\disk2\\picaport\\utl\\dbcheck\\dbrcheck.c", &file );
    printf( "Drive: %s\n", file.disk );
    printf( "Dir  : %s\n", file.dir );
    printf( "Name : %s\n", file.name );
    printf( "Ext  : %s\n", file.ext  );

    while ( p = ArgCheck() ) {
        printf( "Unknown parameter: %s\n", p );
    }

    exit(0);
}
#endif

