// tdbuf.hpp: TableDataBuffer class (LISTING 4)
#ifndef TDBUF_HPP
#define TDBUF_HPP
#include "tbldata.hpp"

// Buffer class for data tables
template <class cellType> class TableDataBuffer:
        public TableData<cellType>
{
public:
    TableDataBuffer (TableData<cellType> *const prev,
            const int nRows, const int nCols):
            TableData<cellType> (prev),
            NumRows (nRows),
            NumCols (nCols)
    {
        // Allocate the buffer
        TableBuf = new cellType[nRows * nCols];
        assert (TableBuf != 0);
        // Allocate storage for row & col headings too.
        RowHeading = new char [nRows * MaxHeadingLen];
        assert (RowHeading != 0);
        ColHeading = new char [nCols * MaxHeadingLen];
        assert (ColHeading != 0);
    }

    virtual ~TableDataBuffer (void)
    {
        delete []TableBuf;
        delete []RowHeading;
        delete []ColHeading;
    }

    virtual const cellType GetCell (const int row,
            const int col)
    {
        ValidateCoord (row, col);
        return (RefCell (row, col));
    }

    virtual void PutCell (const int row, const int col,
            const cellType &value)
    {
        ValidateCoord (row, col);
        RefCell (row, col) = value;
    }

    virtual const int GetNumRows (void) const
    { return (NumRows); }

    virtual const int GetNumCols (void) const
    { return (NumCols); }

    virtual const char &GetRowHeading (const int row)
    {
        ValidateRow (row);
        return (RefRowHeading (row));
    }

    virtual const char &GetColHeading (const int col)
    {
        ValidateCol (col);
        return (RefColHeading (col));
    }

    virtual void ValidateCoord (const int row,
            const int col) const
    { ValidateRow (row); ValidateCol (col); }

    virtual void ValidateRow (const int row) const
    { assert ( (row >= 0 && row < GetNumRows()) ); }

    virtual void ValidateCol (const int col) const
    { assert ( (col >= 0 && col < GetNumCols()) ); }

protected:
    enum { MaxHeadingLen = 40 };

    cellType &RefCell (int row, int col)
    { return (TableBuf[row * GetNumCols() + col]); }

    char &RefRowHeading (int row)
    { return (RowHeading[row * MaxHeadingLen]); }

    char &RefColHeading (int col)
    { return (ColHeading[col * MaxHeadingLen]); }

private:
    int NumRows, NumCols;
    cellType *TableBuf; // buffer holds all cells
    char *RowHeading;   // array of row headings
    char *ColHeading;   // array of col headings
};
#endif
