/* Listing 2		Fixed Point Trig Functions */


#include <stdio.h>
#include <stdlib.h>



#define DEGREES_90            0x0400
#define DEGREES_180           0x0800



typedef short Trig;     /*  describes a 2.14 fixed point number */
typedef short Angle;    /*  describes a 12.4 fixed point number */



static Trig cosineTable[ DEGREES_90 + 1 ] =
/*
	This example uses a circle divided uniformly into 4096 "degrees".  The
	cosine values for the first quadrant are stored here.  Only the first
	quadrant is necessary because the lookup routine is able to take advantage
	of cosine's symmetry.
	
	This table was generated by another small program using floating point.
	All it needed to do was compute the cosine every (1/4096)(2*pi) radians
	and convert it into a 2.14 fixed point number.  The values stored here
	are truncated, but some applications may benefit from rounding, instead.
	
	Note that this table contains one more entry than there are divisions in
	a quadrant.  This is to ensure that all angles in the range [0,90] are
	accounted for.
*/
{
   0x4000, 0x3fff, 0x3fff, 0x3fff, 0x3fff, 0x3fff, 0x3fff, 0x3fff,
   0x3ffe, 0x3ffe, 0x3ffe, 0x3ffd, 0x3ffd, 0x3ffc, 0x3ffc, 0x3ffb,
   0x3ffb, 0x3ffa, 0x3ff9, 0x3ff9, 0x3ff8, 0x3ff7, 0x3ff6, 0x3ff5,
   0x3ff4, 0x3ff3, 0x3ff2, 0x3ff1, 0x3ff0, 0x3fef, 0x3fee, 0x3fed,
   0x3fec, 0x3feb, 0x3fe9, 0x3fe8, 0x3fe7, 0x3fe5, 0x3fe4, 0x3fe2,
   0x3fe1, 0x3fdf, 0x3fde, 0x3fdc, 0x3fda, 0x3fd8, 0x3fd7, 0x3fd5,
   0x3fd3, 0x3fd1, 0x3fcf, 0x3fcd, 0x3fcb, 0x3fc9, 0x3fc7, 0x3fc5,
   0x3fc3, 0x3fc1, 0x3fbf, 0x3fbc, 0x3fba, 0x3fb8, 0x3fb5, 0x3fb3,
   0x3fb1, 0x3fae, 0x3fac, 0x3fa9, 0x3fa6, 0x3fa4, 0x3fa1, 0x3f9e,
   0x3f9c, 0x3f99, 0x3f96, 0x3f93, 0x3f90, 0x3f8d, 0x3f8a, 0x3f87,
   0x3f84, 0x3f81, 0x3f7e, 0x3f7b, 0x3f78, 0x3f74, 0x3f71, 0x3f6e,
   0x3f6a, 0x3f67, 0x3f64, 0x3f60, 0x3f5d, 0x3f59, 0x3f55, 0x3f52,
   0x3f4e, 0x3f4a, 0x3f47, 0x3f43, 0x3f3f, 0x3f3b, 0x3f37, 0x3f33,
   0x3f2f, 0x3f2b, 0x3f27, 0x3f23, 0x3f1f, 0x3f1b, 0x3f17, 0x3f13,
   0x3f0e, 0x3f0a, 0x3f06, 0x3f01, 0x3efd, 0x3ef8, 0x3ef4, 0x3eef,
   0x3eeb, 0x3ee6, 0x3ee1, 0x3edd, 0x3ed8, 0x3ed3, 0x3ece, 0x3eca,
   0x3ec5, 0x3ec0, 0x3ebb, 0x3eb6, 0x3eb1, 0x3eac, 0x3ea7, 0x3ea1,
   0x3e9c, 0x3e97, 0x3e92, 0x3e8c, 0x3e87, 0x3e82, 0x3e7c, 0x3e77,
   0x3e71, 0x3e6c, 0x3e66, 0x3e61, 0x3e5b, 0x3e55, 0x3e50, 0x3e4a,
   0x3e44, 0x3e3e, 0x3e38, 0x3e33, 0x3e2d, 0x3e27, 0x3e21, 0x3e1b,
   0x3e14, 0x3e0e, 0x3e08, 0x3e02, 0x3dfc, 0x3df5, 0x3def, 0x3de9,
   0x3de2, 0x3ddc, 0x3dd6, 0x3dcf, 0x3dc9, 0x3dc2, 0x3dbb, 0x3db5,
   0x3dae, 0x3da7, 0x3da1, 0x3d9a, 0x3d93, 0x3d8c, 0x3d85, 0x3d7e,
   0x3d77, 0x3d70, 0x3d69, 0x3d62, 0x3d5b, 0x3d54, 0x3d4d, 0x3d45,
   0x3d3e, 0x3d37, 0x3d2f, 0x3d28, 0x3d21, 0x3d19, 0x3d12, 0x3d0a,
   0x3d02, 0x3cfb, 0x3cf3, 0x3cec, 0x3ce4, 0x3cdc, 0x3cd4, 0x3ccc,
   0x3cc5, 0x3cbd, 0x3cb5, 0x3cad, 0x3ca5, 0x3c9d, 0x3c95, 0x3c8c,
   0x3c84, 0x3c7c, 0x3c74, 0x3c6c, 0x3c63, 0x3c5b, 0x3c53, 0x3c4a,
   0x3c42, 0x3c39, 0x3c31, 0x3c28, 0x3c20, 0x3c17, 0x3c0e, 0x3c06,
   0x3bfd, 0x3bf4, 0x3beb, 0x3be2, 0x3bda, 0x3bd1, 0x3bc8, 0x3bbf,
   0x3bb6, 0x3bad, 0x3ba3, 0x3b9a, 0x3b91, 0x3b88, 0x3b7f, 0x3b75,
   0x3b6c, 0x3b63, 0x3b59, 0x3b50, 0x3b47, 0x3b3d, 0x3b34, 0x3b2a,
   0x3b20, 0x3b17, 0x3b0d, 0x3b03, 0x3afa, 0x3af0, 0x3ae6, 0x3adc,
   0x3ad2, 0x3ac8, 0x3abe, 0x3ab4, 0x3aaa, 0x3aa0, 0x3a96, 0x3a8c,
   0x3a82, 0x3a78, 0x3a6d, 0x3a63, 0x3a59, 0x3a4f, 0x3a44, 0x3a3a,
   0x3a2f, 0x3a25, 0x3a1a, 0x3a10, 0x3a05, 0x39fb, 0x39f0, 0x39e5,
   0x39da, 0x39d0, 0x39c5, 0x39ba, 0x39af, 0x39a4, 0x3999, 0x398e,
   0x3983, 0x3978, 0x396d, 0x3962, 0x3957, 0x394c, 0x3941, 0x3935,
   0x392a, 0x391f, 0x3913, 0x3908, 0x38fd, 0x38f1, 0x38e6, 0x38da,
   0x38cf, 0x38c3, 0x38b7, 0x38ac, 0x38a0, 0x3894, 0x3889, 0x387d,
   0x3871, 0x3865, 0x3859, 0x384d, 0x3841, 0x3835, 0x3829, 0x381d,
   0x3811, 0x3805, 0x37f9, 0x37ed, 0x37e0, 0x37d4, 0x37c8, 0x37bb,
   0x37af, 0x37a3, 0x3796, 0x378a, 0x377d, 0x3771, 0x3764, 0x3757,
   0x374b, 0x373e, 0x3731, 0x3725, 0x3718, 0x370b, 0x36fe, 0x36f1,
   0x36e5, 0x36d8, 0x36cb, 0x36be, 0x36b1, 0x36a4, 0x3696, 0x3689,
   0x367c, 0x366f, 0x3662, 0x3654, 0x3647, 0x363a, 0x362c, 0x361f,
   0x3612, 0x3604, 0x35f7, 0x35e9, 0x35dc, 0x35ce, 0x35c0, 0x35b3,
   0x35a5, 0x3597, 0x3589, 0x357c, 0x356e, 0x3560, 0x3552, 0x3544,
   0x3536, 0x3528, 0x351a, 0x350c, 0x34fe, 0x34f0, 0x34e2, 0x34d4,
   0x34c6, 0x34b7, 0x34a9, 0x349b, 0x348c, 0x347e, 0x3470, 0x3461,
   0x3453, 0x3444, 0x3436, 0x3427, 0x3419, 0x340a, 0x33fb, 0x33ed,
   0x33de, 0x33cf, 0x33c1, 0x33b2, 0x33a3, 0x3394, 0x3385, 0x3376,
   0x3367, 0x3358, 0x3349, 0x333a, 0x332b, 0x331c, 0x330d, 0x32fe,
   0x32ee, 0x32df, 0x32d0, 0x32c1, 0x32b1, 0x32a2, 0x3293, 0x3283,
   0x3274, 0x3264, 0x3255, 0x3245, 0x3236, 0x3226, 0x3216, 0x3207,
   0x31f7, 0x31e7, 0x31d8, 0x31c8, 0x31b8, 0x31a8, 0x3198, 0x3188,
   0x3179, 0x3169, 0x3159, 0x3149, 0x3138, 0x3128, 0x3118, 0x3108,
   0x30f8, 0x30e8, 0x30d8, 0x30c7, 0x30b7, 0x30a7, 0x3096, 0x3086,
   0x3076, 0x3065, 0x3055, 0x3044, 0x3034, 0x3023, 0x3013, 0x3002,
   0x2ff1, 0x2fe1, 0x2fd0, 0x2fbf, 0x2faf, 0x2f9e, 0x2f8d, 0x2f7c,
   0x2f6b, 0x2f5a, 0x2f49, 0x2f38, 0x2f28, 0x2f16, 0x2f05, 0x2ef4,
   0x2ee3, 0x2ed2, 0x2ec1, 0x2eb0, 0x2e9f, 0x2e8d, 0x2e7c, 0x2e6b,
   0x2e5a, 0x2e48, 0x2e37, 0x2e25, 0x2e14, 0x2e03, 0x2df1, 0x2de0,
   0x2dce, 0x2dbc, 0x2dab, 0x2d99, 0x2d88, 0x2d76, 0x2d64, 0x2d52,
   0x2d41, 0x2d2f, 0x2d1d, 0x2d0b, 0x2cf9, 0x2ce8, 0x2cd6, 0x2cc4,
   0x2cb2, 0x2ca0, 0x2c8e, 0x2c7c, 0x2c6a, 0x2c57, 0x2c45, 0x2c33,
   0x2c21, 0x2c0f, 0x2bfc, 0x2bea, 0x2bd8, 0x2bc6, 0x2bb3, 0x2ba1,
   0x2b8e, 0x2b7c, 0x2b6a, 0x2b57, 0x2b45, 0x2b32, 0x2b20, 0x2b0d,
   0x2afa, 0x2ae8, 0x2ad5, 0x2ac2, 0x2ab0, 0x2a9d, 0x2a8a, 0x2a77,
   0x2a65, 0x2a52, 0x2a3f, 0x2a2c, 0x2a19, 0x2a06, 0x29f3, 0x29e0,
   0x29cd, 0x29ba, 0x29a7, 0x2994, 0x2981, 0x296e, 0x295a, 0x2947,
   0x2934, 0x2921, 0x290e, 0x28fa, 0x28e7, 0x28d4, 0x28c0, 0x28ad,
   0x2899, 0x2886, 0x2872, 0x285f, 0x284b, 0x2838, 0x2824, 0x2811,
   0x27fd, 0x27ea, 0x27d6, 0x27c2, 0x27af, 0x279b, 0x2787, 0x2773,
   0x275f, 0x274c, 0x2738, 0x2724, 0x2710, 0x26fc, 0x26e8, 0x26d4,
   0x26c0, 0x26ac, 0x2698, 0x2684, 0x2670, 0x265c, 0x2648, 0x2634,
   0x261f, 0x260b, 0x25f7, 0x25e3, 0x25cf, 0x25ba, 0x25a6, 0x2592,
   0x257d, 0x2569, 0x2554, 0x2540, 0x252c, 0x2517, 0x2503, 0x24ee,
   0x24da, 0x24c5, 0x24b0, 0x249c, 0x2487, 0x2473, 0x245e, 0x2449,
   0x2434, 0x2420, 0x240b, 0x23f6, 0x23e1, 0x23cd, 0x23b8, 0x23a3,
   0x238e, 0x2379, 0x2364, 0x234f, 0x233a, 0x2325, 0x2310, 0x22fb,
   0x22e6, 0x22d1, 0x22bc, 0x22a7, 0x2292, 0x227d, 0x2267, 0x2252,
   0x223d, 0x2228, 0x2212, 0x21fd, 0x21e8, 0x21d2, 0x21bd, 0x21a8,
   0x2192, 0x217d, 0x2168, 0x2152, 0x213d, 0x2127, 0x2112, 0x20fc,
   0x20e7, 0x20d1, 0x20bb, 0x20a6, 0x2090, 0x207b, 0x2065, 0x204f,
   0x2039, 0x2024, 0x200e, 0x1ff8, 0x1fe2, 0x1fcd, 0x1fb7, 0x1fa1,
   0x1f8b, 0x1f75, 0x1f5f, 0x1f49, 0x1f34, 0x1f1e, 0x1f08, 0x1ef2,
   0x1edc, 0x1ec6, 0x1eb0, 0x1e99, 0x1e83, 0x1e6d, 0x1e57, 0x1e41,
   0x1e2b, 0x1e15, 0x1dfe, 0x1de8, 0x1dd2, 0x1dbc, 0x1da6, 0x1d8f,
   0x1d79, 0x1d63, 0x1d4c, 0x1d36, 0x1d20, 0x1d09, 0x1cf3, 0x1cdc,
   0x1cc6, 0x1caf, 0x1c99, 0x1c83, 0x1c6c, 0x1c55, 0x1c3f, 0x1c28,
   0x1c12, 0x1bfb, 0x1be5, 0x1bce, 0x1bb7, 0x1ba1, 0x1b8a, 0x1b73,
   0x1b5d, 0x1b46, 0x1b2f, 0x1b18, 0x1b02, 0x1aeb, 0x1ad4, 0x1abd,
   0x1aa6, 0x1a8f, 0x1a79, 0x1a62, 0x1a4b, 0x1a34, 0x1a1d, 0x1a06,
   0x19ef, 0x19d8, 0x19c1, 0x19aa, 0x1993, 0x197c, 0x1965, 0x194e,
   0x1937, 0x1920, 0x1908, 0x18f1, 0x18da, 0x18c3, 0x18ac, 0x1895,
   0x187d, 0x1866, 0x184f, 0x1838, 0x1820, 0x1809, 0x17f2, 0x17da,
   0x17c3, 0x17ac, 0x1794, 0x177d, 0x1766, 0x174e, 0x1737, 0x171f,
   0x1708, 0x16f1, 0x16d9, 0x16c2, 0x16aa, 0x1693, 0x167b, 0x1664,
   0x164c, 0x1634, 0x161d, 0x1605, 0x15ee, 0x15d6, 0x15be, 0x15a7,
   0x158f, 0x1577, 0x1560, 0x1548, 0x1530, 0x1519, 0x1501, 0x14e9,
   0x14d1, 0x14ba, 0x14a2, 0x148a, 0x1472, 0x145a, 0x1443, 0x142b,
   0x1413, 0x13fb, 0x13e3, 0x13cb, 0x13b3, 0x139b, 0x1383, 0x136c,
   0x1354, 0x133c, 0x1324, 0x130c, 0x12f4, 0x12dc, 0x12c4, 0x12ac,
   0x1294, 0x127b, 0x1263, 0x124b, 0x1233, 0x121b, 0x1203, 0x11eb,
   0x11d3, 0x11bb, 0x11a2, 0x118a, 0x1172, 0x115a, 0x1142, 0x112a,
   0x1111, 0x10f9, 0x10e1, 0x10c9, 0x10b0, 0x1098, 0x1080, 0x1068,
   0x104f, 0x1037, 0x101f, 0x1006, 0x0fee, 0x0fd6, 0x0fbd, 0x0fa5,
   0x0f8c, 0x0f74, 0x0f5c, 0x0f43, 0x0f2b, 0x0f12, 0x0efa, 0x0ee2,
   0x0ec9, 0x0eb1, 0x0e98, 0x0e80, 0x0e67, 0x0e4f, 0x0e36, 0x0e1e,
   0x0e05, 0x0ded, 0x0dd4, 0x0dbc, 0x0da3, 0x0d8b, 0x0d72, 0x0d59,
   0x0d41, 0x0d28, 0x0d10, 0x0cf7, 0x0cde, 0x0cc6, 0x0cad, 0x0c95,
   0x0c7c, 0x0c63, 0x0c4b, 0x0c32, 0x0c19, 0x0c01, 0x0be8, 0x0bcf,
   0x0bb6, 0x0b9e, 0x0b85, 0x0b6c, 0x0b54, 0x0b3b, 0x0b22, 0x0b09,
   0x0af1, 0x0ad8, 0x0abf, 0x0aa6, 0x0a8d, 0x0a75, 0x0a5c, 0x0a43,
   0x0a2a, 0x0a11, 0x09f9, 0x09e0, 0x09c7, 0x09ae, 0x0995, 0x097c,
   0x0964, 0x094b, 0x0932, 0x0919, 0x0900, 0x08e7, 0x08ce, 0x08b5,
   0x089c, 0x0884, 0x086b, 0x0852, 0x0839, 0x0820, 0x0807, 0x07ee,
   0x07d5, 0x07bc, 0x07a3, 0x078a, 0x0771, 0x0758, 0x073f, 0x0726,
   0x070d, 0x06f4, 0x06db, 0x06c2, 0x06a9, 0x0690, 0x0677, 0x065e,
   0x0645, 0x062c, 0x0613, 0x05fa, 0x05e1, 0x05c8, 0x05af, 0x0596,
   0x057d, 0x0564, 0x054b, 0x0532, 0x0519, 0x0500, 0x04e7, 0x04ce,
   0x04b5, 0x049c, 0x0483, 0x046a, 0x0451, 0x0437, 0x041e, 0x0405,
   0x03ec, 0x03d3, 0x03ba, 0x03a1, 0x0388, 0x036f, 0x0356, 0x033d,
   0x0323, 0x030a, 0x02f1, 0x02d8, 0x02bf, 0x02a6, 0x028d, 0x0274,
   0x025b, 0x0241, 0x0228, 0x020f, 0x01f6, 0x01dd, 0x01c4, 0x01ab,
   0x0192, 0x0178, 0x015f, 0x0146, 0x012d, 0x0114, 0x00fb, 0x00e2,
   0x00c9, 0x00af, 0x0096, 0x007d, 0x0064, 0x004b, 0x0032, 0x0019,
   0x0000
};



Trig cosine( Angle angle )
/*
	This routine returns the cosine of angle, returning a fixed point value in
	the range [-1, 1]. Since this is performed with a simple table lookup, the
	Trig type returned may have any format desired.  This example treats it as
	a 2.14 fixed point number.
	
	The Angle type describes a 12.4 fixed point number.  This divides the unit
	circle into 4096 integral "degrees".  Since every angle may be represented
	in 16 bits, overflow is impossible: the implicit modulus of limited-
	precision integer math guarantees this.
*/
{
   Trig value;

   /*  Convert angle from 12.4 fixed point to integer. */
   angle >>= 4;

   /*  Since cos(x) == cos(-x), combine both cases. */
   if( 0 > angle )
      angle = -angle;

   /*  
   	   Check which quadrant contains angle.  This isn't
   	   strictly necessary, but by doing so this routine
   	   is able to exploit the symmetry of the cosine
   	   function, reducing the size of the lookup table
   	   by three quarters.
   */
   if( DEGREES_90 < angle )
   {
      /*
	  	  If not in first quadrant, adjust lookup index
          by noting that cos(x) = -cos(180 - x).
	  */
      angle = DEGREES_180 - angle;
      value = -cosineTable[ angle ];
   }
   else
      value = cosineTable[ angle ];

   return( value );
}



Trig sine( Angle angle )
/*
	This routine returns the sine of angle.  Like the routine above, it
	returns a Trig value, which represents a 2.14 fixed point number.
*/
{
   /*
   	   Make use of the fact that sin(x) = cos(x - 90).
   	   Note that the constant must be converted to 12.4
       fixed point before it may be subtracted.
   */
   angle -= (DEGREES_90 << 4);

   return( cosine( angle ) );
}


