#' Flip a coin
#'
#' @param n How many coins to flip.
#'   Defaults to `NULL`, which interactively asks how many to flip.
#' @param animate If `TRUE`, play an animation before revealing the result.
#'   Defaults to `TRUE` if the session is [interactive] and `FALSE` otherwise.
#' @param header If `TRUE`, prints a header for the game.
#'   Defaults to `TRUE` if the session is [interactive] and `FALSE` otherwise.
#'
#' @return A character vector of `"heads"` or `"tails"` results.
#' @export
#'
#' @examples
#' play_coin_flip(1)
#'
#' if (rlang::is_interactive()) play_coin_flip()
play_coin_flip <- function(
    n = NULL,
    animate = rlang::is_interactive(),
    header = rlang::is_interactive()
) {
  if (isTRUE(header)) h1("\U0001fa99 Coin Flip \U0001fa99")

  selection <- NULL

  repeat {
    n <- suppressWarnings(as.numeric(n))

    if (rlang::is_integerish(n, finite = TRUE) && length(n) == 1) break

    if (!rlang::is_interactive()) {
      cli::cli_abort("{.arg n} must be a single integer.")
    } else if (!is.null(selection)) {
      cat_over("Please enter a single integer.")
    }

    selection <- input("How many coins do you want to flip? ")
    cat0("\b\b\r", strrep(" ", 36), strrep(" ", stringr::str_width(selection)))
    cat0(strrep("\b", 36), strrep("\b", stringr::str_width(selection)))
    n <- selection
  }

  result <- flip_coin(n = n, animate = animate)

  if (rlang::is_interactive()) {
    selection <- input("Press [ENTER] to flip again or [ESC] to quit. ")
    cat0("\b\b\r", strrep(" ", 46), strrep(" ", stringr::str_width(selection)))
    cat0(strrep("\b", 46), strrep("\b", stringr::str_width(selection)))

    play_coin_flip(n = n, animate = animate, header = FALSE)
  } else {
    invisible(result)
  }
}

flip_coin <- function(n = 1, animate = TRUE) {
  n <- suppressWarnings(as.numeric(n))

  if (!rlang::is_integerish(n) || length(n) != 1) {
    cli::cli_abort("{.arg n} must be a single integer.")
  }

  result <- sample(c("heads", "tails"), n, replace = TRUE)

  coins <- character(length(result))
  coins[result == "heads"] <- "\u24bd"
  coins[result == "tails"] <- "\u24c9"

  if (isTRUE(animate)) {
    coin_flip_animation(coins)
  } else {
    cat(coins, "\n")
  }

  if (length(result) > 1) {
    if (length(unique(result)) == 1) {
      cat(length(result), stringr::str_to_title(unique(result)))
    } else {
      heads <- sum(result == "heads")
      tails <- sum(result == "tails")
      cat(heads, "Heads,", tails, "Tails")
    }
  } else {
    cat0(stringr::str_to_title(result))
  }

  invisible(result)
}

coin_flip_animation <- function(coins) {
  n <- length(coins)

  cat_over(strrep("  ", n), strrep("  ", n), strrep("\u2b2d ", n), sep = "\n")
  utils::flush.console()
  Sys.sleep(0.1)

  cat(strrep("\b", n * 6 + 3))
  cat(strrep("  ", n), strrep("\u2500 ", n), strrep("  ", n), sep = "\n")
  utils::flush.console()
  Sys.sleep(0.1)

  cat(strrep("\b", n * 6 + 3))
  cat(strrep("\u2b2d ", n), strrep("  ", n), strrep("  ", n), sep = "\n")
  utils::flush.console()
  Sys.sleep(0.1)

  cat(strrep("\b", n * 6 + 3))
  cat(strrep("  ", n), strrep("\u2500 ", n), strrep("  ", n), sep = "\n")
  utils::flush.console()
  Sys.sleep(0.1)

  cat(strrep("\b", n * 6 + 3))
  cat(strrep("  ", n), strrep("  ", n), paste(coins, collapse = " "), sep = "\n")
  utils::flush.console()
}
