% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hrf-functions.R
\name{hrf_basis_lwu}
\alias{hrf_basis_lwu}
\title{LWU HRF Basis for Taylor Expansion}
\usage{
hrf_basis_lwu(theta0, t, normalize_primary = "none")
}
\arguments{
\item{theta0}{A numeric vector of length 3 specifying the expansion point
\code{c(tau0, sigma0, rho0)} for the LWU parameters.}

\item{t}{A numeric vector of time points (in seconds) at which to evaluate the basis.}

\item{normalize_primary}{Character string, one of \code{"none"} or \code{"height"}.
If \code{"height"}, the primary HRF column (\code{h0(t)}) is normalized to have a
peak absolute value of 1. For Taylor expansion fitting as described in Fit_LRU.md,
this should typically be \code{"none"} as the scaling is absorbed by the beta coefficient.
Default is \code{"none"}.}
}
\value{
A numeric matrix of dimension \code{length(t) x 4}. The columns represent:
  \itemize{
    \item \code{h0}: LWU HRF evaluated at theta0
    \item \code{d_tau}: Partial derivative with respect to tau at theta0
    \item \code{d_sigma}: Partial derivative with respect to sigma at theta0
    \item \code{d_rho}: Partial derivative with respect to rho at theta0
  }
}
\description{
Constructs the basis set for the Lag-Width-Undershoot (LWU) HRF model,
intended for Taylor expansion-based fitting. The basis consists of the
LWU HRF evaluated at a given expansion point \code{theta0}, and its
partial derivatives with respect to its parameters (tau, sigma, rho).
}
\examples{
t_points <- seq(0, 30, by = 0.5)
theta0_default <- c(tau = 6, sigma = 1, rho = 0.35)

# Generate the basis set
lwu_basis <- hrf_basis_lwu(theta0_default, t_points)
dim(lwu_basis) # Should be length(t_points) x 4
head(lwu_basis)

# Plot the basis functions
matplot(t_points, lwu_basis, type = "l", lty = 1,
        main = "LWU HRF Basis Functions", ylab = "Value", xlab = "Time (s)")
legend("topright", colnames(lwu_basis), col = 1:4, lty = 1, cex = 0.8)

# Example with primary HRF normalization (not typical for Taylor fitting step)
lwu_basis_norm_h0 <- hrf_basis_lwu(theta0_default, t_points, normalize_primary = "height")
plot(t_points, lwu_basis_norm_h0[,1], type="l", main="Normalized h0 in Basis")
max(abs(lwu_basis_norm_h0[,1])) # Should be 1
}
\seealso{
\code{\link{hrf_lwu}}, \code{\link[numDeriv]{grad}}

Other hrf_functions: 
\code{\link{hrf_bspline}()},
\code{\link{hrf_gamma}()},
\code{\link{hrf_gaussian}()},
\code{\link{hrf_inv_logit}()},
\code{\link{hrf_lwu}()},
\code{\link{hrf_mexhat}()},
\code{\link{hrf_sine}()},
\code{\link{hrf_spmg1}()},
\code{\link{hrf_time}()}
}
\concept{hrf_functions}
